/*!
 * This module contains logic to reconstruct the reference sequence of a given read without
 * a reference fasta file present. For this it parses the query sequence, the cigar elements
 * and the MD tag. 
 * 
 * It is adapted from the Pysam implementation:
 * https://github.com/pysam-developers/pysam/blob/3e3c8b0b5ac066d692e5c720a85d293efc825200/pysam/libcalignedsegment.pyx#L1971
 */

use helper::logger::get_log_vector_sample;
use noodles::sam::alignment::record::cigar::{op::Kind::{ 
    Deletion, HardClip, Insertion, Match, Pad, SequenceMatch, SequenceMismatch, Skip, SoftClip
}, Op};

use crate::{error::loader_errors::bam_errors::RefSeqReconstructError};

// Decimal ascii codes for relevant operations 
const A: u8 = 65;
const C: u8 = 67;
const G: u8 = 71;
const T: u8 = 84;
const N: u8 = 78;
const CIRCUMFLEX: u8 = 94;
const HYPHEN: u8 = 45;
const ZERO: u8 = 48;
const NINE: u8 = 57;

/// Reconstructs the reference sequence from a query sequence using CIGAR and MD tag information.
///
/// This function builds a reference sequence by first creating an intermediate sequence from 
/// the query and CIGAR data, then applying corrections based on the MD tag information.
/// The implementation follows the approach used in Pysam.
///
/// # Arguments
///
/// * `query_sequence` - The query sequence (read) as bytes
/// * `cigar` - Vector of CIGAR operations describing the alignment
/// * `md_tag` - MD tag as bytes, describing mismatches and deletions
///
/// # Returns
///
/// * `Ok(Vec<u8>)` - The reconstructed reference sequence
/// * `Err(RefSeqReconstructError)` - If the reconstruction fails due to invalid input
///
/// # Examples
///
/// ```ignore
/// let query = b"ACGTACGT";
/// let cigar = vec![Match(4), Op::new(Deletion, 1), Match(4)];
/// let md = b"4^T4";
/// let reference = build_reference_sequence(query, &cigar, md)?;
/// assert_eq!(reference, b"ACGTTACGT");
/// ```
pub fn build_reference_sequence(
    query_sequence: &[u8],
    cigar: &[Op],
    md_tag: &[u8]
) -> Result<Vec<u8>, RefSeqReconstructError> {
    log::debug!(
        "build_reference_sequence input: query_sequence = {}, cigar = {}, md_tag = {}",
        get_log_vector_sample(query_sequence, 10),
        get_log_vector_sample(cigar, 10),
        get_log_vector_sample(md_tag, 10)
    );

    if query_sequence.len() < 1 {
        return Err(RefSeqReconstructError::EmptyQuery);
    }
    
    let mut reference_idx = 0;
    let mut reference_sequence = Vec::new();
    
    // Reconstruct an intermediate reference sequence that contains the following elements:
    // - Matches, Equal (no need to change later on)
    // - Diff (mismatched base; correct base must be extracted from MD)
    // - Del ('-' as placeholder; deleted base from the reference must be extracted from MD)
    let mut query_idx = 0;
    for op in cigar {
        let len = op.len();
        match op.kind() {
            // Match / Equal / Diff => All present in reference sequence
            // (Mismatches will be corrected in the second iteration)
            Match | SequenceMatch | SequenceMismatch => {
                for _ in 0..len {
                    if query_idx >= query_sequence.len() {
                        return Err(RefSeqReconstructError::QueryOutOfBounds(
                            query_idx as usize, query_sequence.len()
                        ));
                    }

                    reference_sequence.push(query_sequence[query_idx]);
                    query_idx += 1;
                }
                reference_idx += len;
            }
            // Del (deleted from the reference) => Present in the reference sequence
            // ('-' placeholders will be filled in the second iteration)
            Deletion => {
                for _ in 0..len {
                    reference_sequence.push(HYPHEN);
                }
                reference_idx += len;
            }
            // Insertions must be regarded in the query sequence, but are not present in
            // the reference sequence => Must be ignored when reconstructing the reference
            // sequence
            Insertion | SoftClip | Pad => {
                for _ in 0..len {
                    if query_idx > query_sequence.len() {
                        return Err(RefSeqReconstructError::QueryOutOfBounds(
                            reference_idx as usize, query_sequence.len()
                        ));
                    }
                    query_idx += 1;
                }
            }
            // Clipping and reference skips do not affect either sequence, so they can
            // be ignored
            HardClip | Skip => {}
        }
    }

    // Fill the corresponding bases into the intermediate sequence 
    // from above where needed (Diff, Del)
    let mut reference_idx = 0;
    let mut md_idx = 0;
    let mut n_matches = 0;
    
    while md_idx < md_tag.len() {
        let c = md_tag[md_idx];

        check_char(&c)?;

        // Extract a number from the MD string. Digits that follow up on each
        // other are iteratively combined by multiplying them by 10 (shifting 
        // from 10^i to 10^(i+1)) and adding the latest digit to the 10^0 position.
        // This also ensures that isolated zeros are ignored, as 0 * 10 + 0 = 0
        if is_ascii_digit(&c) {
            n_matches = n_matches * 10 + decode_ascii_digit(&c)?;
            md_idx += 1;
        } 
        // If the current value doesn't correspond to a numeric character the previously 
        // collected n_matches value can be processed (if two non-match operations follow
        // up on each other n_matches is 0 and the for loop gets skipped) and the current
        // non-numerical character can be processed (deletion or mismatch)
        else {
            // Each match is a position in the intermediate sequence that can be ignored
            // The same goes for insertions (lowercase characters in the intermediate seq)
            // => Move along the intermediate sequence until a deletion or mismatch occurs
            process_matches(&n_matches, &mut reference_idx, &reference_sequence);

            // Reset the n_matches variable so a new number can be reconstructed from digits
            n_matches = 0;

            // Handle the deletion case ('^')
            if c == CIRCUMFLEX {
                md_idx += 1;
                while md_idx < md_tag.len() && is_base(&md_tag[md_idx]) {
                    if reference_idx >= reference_sequence.len() {
                        return Err(RefSeqReconstructError::ReferenceOutOfBounds(
                            reference_idx, reference_sequence.len()
                        ));
                    }
                    // replace the '-' placeholder by the deleted base
                    reference_sequence[reference_idx] = md_tag[md_idx];
                    reference_idx += 1;
                    md_idx += 1;
                }
            }
            // Handle the mismatch case ('A'/'C'/'G'/'T'/'N')
            else {
                if reference_idx >= reference_sequence.len() {
                    return Err(RefSeqReconstructError::ReferenceOutOfBounds(
                        reference_idx, reference_sequence.len()
                    ));
                }
                let mut mismatch_ref_base = c;
                if is_base(&mismatch_ref_base) {
                    // Transform the upper-case base from the MD string into lower-case
                    mismatch_ref_base += 32;
                } 
                // To make sure that no unexpected base sneaks in
                else {
                    return Err(RefSeqReconstructError::InvalidChar(mismatch_ref_base));
                }

                // Insert the reference base into the intermediate sequence
                reference_sequence[reference_idx] = mismatch_ref_base;
                reference_idx += 1;
                md_idx += 1;
            }
        }
    }
    // Process the remaining matches in case the MD string ends with a number
    // while again ignoring insertions
    process_matches(&n_matches, &mut reference_idx, &reference_sequence);

    log::debug!(
        "rescale output: reference_sequence = {}",
        get_log_vector_sample(&reference_sequence, 10),
    );

    Ok(reference_sequence)
}

/// Determine if a byte is a valid element of a MD String. 
/// Valid elements are: Digits (0 to 9), 'A', 'C', 'G', 'T', 'N', '^'
///  
/// # Arguments
/// 
/// * `c` - The byte to check
/// 
/// # Returns
///
/// * `Ok(())` - if the byte is valid
/// * `Err(RefSeqReconstructError)` - otherwise
fn check_char(c: &u8) -> Result<(), RefSeqReconstructError> {
    if is_ascii_digit(c) || is_base(c) || *c == CIRCUMFLEX {
        Ok(())
    } else {
        Err(RefSeqReconstructError::InvalidChar(*c))
    }
}

/// Determines if a byte represents an ASCII digit (0-9).
///
/// # Arguments
///
/// * `c` - The byte to check
///
/// # Returns
///
/// * `true` if the byte is a digit (0-9)
/// * `false` otherwise
fn is_ascii_digit(c: &u8) -> bool {
    if c >= &ZERO && c <= &NINE {
        true
    } else {
        false
    }
}

/// Converts an ASCII digit character to its numeric value.
///
/// # Arguments
///
/// * `d` - The byte representing an ASCII digit
///
/// # Returns
///
/// * `Ok(u32)` - The numeric value of the digit
/// * `Err(RefSeqReconstructError)` - If the byte is not a valid ASCII digit
fn decode_ascii_digit(d: &u8) -> Result<u32, RefSeqReconstructError> {
    if d < &ZERO || d > &NINE {
        Err(RefSeqReconstructError::InvalidChar(*d))
    } else {
        Ok((d - ZERO) as u32)
    }
}

/// Checks if a byte represents an uppercase nucleotide base (A, C, G, T, N).
///
/// # Arguments
///
/// * `c` - The byte to check
///
/// # Returns
///
/// * `true` if the byte is an uppercase base
/// * `false` otherwise
fn is_base(c: &u8) -> bool {
    if c == &A || c == &C || c == &G || c == &T || c == &N {
        true
    } else {
        false
    }
}

/// Advances the reference index by the number of matches from the MD tag.
///
/// This helper function advances the index in the reference sequence by skipping
/// over matched positions. It handles bounds checking to avoid out-of-bounds access.
///
/// # Arguments
///
/// * `n_matches` - Number of matches to process
/// * `reference_idx` - Current position in the reference sequence (will be updated)
/// * `reference_sequence` - The reference sequence being constructed
fn process_matches(n_matches: &u32, reference_idx: &mut usize, reference_sequence: &Vec<u8>) {
    for _ in 0..*n_matches {
        if *reference_idx < reference_sequence.len() {
            *reference_idx += 1;
        }
    }
}



#[cfg(test)]
mod test {
    use super::*;

    /// https://github.com/pysam-developers/pysam/blob/3e3c8b0b5ac066d692e5c720a85d293efc825200/tests/AlignedSegment_test.py#L1309
    #[test]
    fn test_match_only() {
        let cigar = vec![Op::new(Match, 21)];
        let query_sequence = vec![b'A'; 21];
        let md_tag = b"5C0T0G05C0G0T5";

        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = b"AAAAActgAAAAAcgtAAAAA".to_vec();
        assert_eq!(expected, reference);


        let cigar = vec![Op::new(Match, 21)];
        let query_sequence = vec![b'A'; 21];
        let md_tag = b"5CTG5CGT5";

        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = b"AAAAActgAAAAAcgtAAAAA".to_vec();
        assert_eq!(expected, reference);


        let cigar = vec![Op::new(Match, 11)];
        let query_sequence = vec![b'A'; 11];
        let md_tag = b"CTG5CGT";

        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = b"ctgAAAAAcgt".to_vec();
        assert_eq!(expected, reference);
    }

    /// https://github.com/pysam-developers/pysam/blob/3e3c8b0b5ac066d692e5c720a85d293efc825200/tests/AlignedSegment_test.py#L1328
    #[test]
    fn test_insertions() {
        // insertions are silent in the reference sequence
        let cigar = vec![Op::new(Match, 5), Op::new(Insertion, 1), Op::new(Match, 5)];
        let query_sequence = vec![b'A', b'A', b'A', b'A', b'A', b'C', b'A', b'A', b'A', b'A', b'A'];
        let md_tag = b"10";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = vec![b'A'; 10];
        assert_eq!(expected, reference);
        
        let cigar = vec![Op::new(Insertion, 1), Op::new(Match, 10)];
        let query_sequence = vec![b'C', b'A', b'A', b'A', b'A', b'A', b'A', b'A', b'A', b'A', b'A'];
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = vec![b'A'; 10];
        assert_eq!(expected, reference);
        
        let cigar = vec![Op::new(Match, 10), Op::new(Insertion, 1)];
        let query_sequence = vec![b'A', b'A', b'A', b'A', b'A', b'A', b'A', b'A', b'A', b'A', b'C'];
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = vec![b'A'; 10];
        assert_eq!(expected, reference);
    }

    /// https://github.com/pysam-developers/pysam/blob/3e3c8b0b5ac066d692e5c720a85d293efc825200/tests/AlignedSegment_test.py#L1358
    #[test]
    fn test_deletions() {
        let cigar = vec![Op::new(Match, 5), Op::new(Deletion, 1), Op::new(Match, 5)];
        let query_sequence = vec![b'A'; 10];
        let md_tag = b"5^C5";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let mut expected = vec![b'A'; 11];
        expected[5] = b'C';
        assert_eq!(expected, reference);
        
        let cigar = vec![Op::new(Match, 5), Op::new(Deletion, 3), Op::new(Match, 5)];
        let query_sequence = vec![b'A'; 10];
        let md_tag = b"5^CCC5";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let mut expected = vec![b'A'; 13];
        expected[5] = b'C';
        expected[6] = b'C';
        expected[7] = b'C';
        assert_eq!(expected, reference);
    }
    
    /// https://github.com/pysam-developers/pysam/blob/3e3c8b0b5ac066d692e5c720a85d293efc825200/tests/AlignedSegment_test.py#L1358
    #[test]
    fn test_ref_skipping() {
        let cigar = vec![Op::new(Match, 5), Op::new(Skip, 1), Op::new(Match, 5)];
        let query_sequence = vec![b'A'; 10];
        let md_tag = b"10";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = vec![b'A'; 10];
        assert_eq!(expected, reference);
        
        let cigar = vec![Op::new(Match, 5), Op::new(Skip, 3), Op::new(Match, 5)];
        let query_sequence = vec![b'A'; 10];
        let md_tag = b"10";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = vec![b'A'; 10];
        assert_eq!(expected, reference);
    }
    
    /// https://github.com/pysam-developers/pysam/blob/3e3c8b0b5ac066d692e5c720a85d293efc825200/tests/AlignedSegment_test.py#L1371
    #[test]
    fn test_soft_clipping() {
        // softclipping
        let cigar = vec![Op::new(SoftClip, 5), Op::new(Match, 5), Op::new(Deletion, 1), Op::new(Match, 5), Op::new(SoftClip, 5)];
        let query_sequence = vec![b'G', b'G', b'G', b'G', b'G', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'G', b'G', b'G', b'G', b'G'];
        let md_tag = b"5^C5";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let mut expected = vec![b'A'; 11];
        expected[5] = b'C';
        assert_eq!(expected, reference);
        
        // all together
        let cigar = vec![Op::new(SoftClip, 5), Op::new(Match, 5), Op::new(Deletion, 1), Op::new(Match, 5), Op::new(Insertion, 1), Op::new(Match, 5), Op::new(SoftClip, 5)];
        let query_sequence = vec![b'G', b'G', b'G', b'G', b'G', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'A', b'G', b'G', b'G', b'G', 
                                           b'G'];
        let md_tag = b"2C2^T10";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = b"AAcAATAAAAAAAAAA".to_vec();
        assert_eq!(expected, reference);
    }

    /// https://github.com/pysam-developers/pysam/blob/3e3c8b0b5ac066d692e5c720a85d293efc825200/tests/AlignedSegment_test.py#L1386
    #[test]
    fn test_complex() {
        let cigar = vec![Op::new(SoftClip, 5), Op::new(Match, 5), Op::new(Insertion, 1), Op::new(Deletion, 2), Op::new(Match, 5), Op::new(SoftClip, 5)];
        let query_sequence = vec![b'G', b'G', b'G', b'G', b'G', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'A', b'G', b'G', b'G', b'G', 
                                           b'G'];
        let md_tag = b"2C2^TC5";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = b"AAcAATCAAAAA".to_vec();
        assert_eq!(expected, reference);
        
        let cigar = vec![Op::new(SoftClip, 5), Op::new(Match, 5), Op::new(Deletion, 2), Op::new(Insertion, 1), Op::new(Match, 5), Op::new(SoftClip, 5)];
        let query_sequence = vec![b'G', b'G', b'G', b'G', b'G', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'A', b'A', b'A', b'A', b'A', 
                                           b'A', b'G', b'G', b'G', b'G', 
                                           b'G'];
        let md_tag = b"2C2^TC5";
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = b"AAcAATCAAAAA".to_vec();
        assert_eq!(expected, reference);
        
        // insertion in reference overlapping deletion in reference
        // read: AACCCCA---AAA
        // ref:  AA----AGGGAAA
        let cigar = vec![Op::new(Match, 2), Op::new(Insertion, 4), Op::new(Match, 1), Op::new(Deletion, 3), Op::new(Match, 3)];
        let md_tag = b"3^GGG3";
        let query_sequence = b"AACCCCAAAA".to_vec();
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = b"AAAGGGAAA".to_vec();
        assert_eq!(expected, reference);
        
        let cigar = vec![Op::new(Match, 5), Op::new(Deletion, 2), Op::new(Insertion, 2), Op::new(Match, 2)];
        let md_tag = b"4C^TT2";
        let query_sequence = vec![b'A'; 9];
        let reference = build_reference_sequence(&query_sequence, &cigar, md_tag).unwrap();
        let expected = b"AAAAcTTAA".to_vec();
        assert_eq!(expected, reference);
    }

    /// Transform a vector of cigar tuples into a vector of Cigar element
    /// 
    /// Helper function to easily copy pysam output into the tests
    fn tuplevec_to_cigarvec(vec: Vec<(u32, usize)>) -> Vec<Op> {
        let mut cigar_vec = Vec::with_capacity(vec.len());
        for (cig, len) in vec {
            let el = match cig {
                0 => Op::new(Match, len),
                1 => Op::new(Insertion, len),
                2 => Op::new(Deletion, len),
                3 => Op::new(Skip, len),
                4 => Op::new(SoftClip, len),
                5 => Op::new(HardClip, len),
                6 => Op::new(Pad, len),
                7 => Op::new(SequenceMatch, len),
                8 => Op::new(SequenceMismatch, len),
                _ => panic!("Invalid cigar index")
            };
            cigar_vec.push(el);
        }
        cigar_vec
    }

    /// Test (some of) the reads from the example data
    #[test]
    fn test_example_data() {
        let cigar_raw = vec![(4, 36), (0, 655), (2, 1), (0, 23), (2, 1), (0, 664), (2, 1), (0, 517), (1, 1), (0, 3), (1, 1), (0, 166), (2, 4), (0, 107), (2, 1), (0, 169), (1, 6), (0, 206), (2, 1), (0, 39), (2, 2), (0, 4), (1, 1), (0, 241), (2, 1), (0, 4), (2, 2), (0, 502), (1, 2), (0, 122), (1, 1), (0, 2), (2, 1), (0, 59), (2, 1), (0, 60), (2, 1), (0, 13), (2, 1), (0, 19), (1, 1), (0, 168), (2, 2), (0, 33), (1, 2), (0, 486), (2, 1), (0, 496), (1, 3), (0, 462), (1, 1), (0, 31), (2, 1), (0, 14), (1, 2), (0, 518), (1, 1), (0, 2), (1, 1), (0, 7), (1, 1), (0, 910), (2, 1), (0, 54), (4, 86)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"ATGCTGATATTGCGGCGTCTGCTTGGGTGTTTAACCTGGCTTTAACCCACCCATGTGGGCTTGGTATACTGAAGGTAAAGCTCCAGTGCTGGAGACGCACAGTGGCTACTGGCCCCCAGAGTAGGGCACACTCAAGAGATGACTCTGGTCTCAAGATGGTGCTGTGTTGCTGCAACTTAGGTCTCTGGGGGCGGGAAGCCTTCACCTTGTACTCCTAGTCCAGGACATGCAGCTGTGTAAATTCCTGGCAGTACTCCATACTGGCCTCAGGGCTTGCAAATACTGTGAGATTTTCCTTAAGACTCTGTGTTTGTGATGGCAGTGGGGCTGGTGGGGATCTTCTGCTTACCTTCACTCTGCAATGAGAAGTCCTTCCTGTCTCTGGGCCAGTTCAACCCAGGTGGGGGAGAAGGGACTGCAGAGGCTGAGTGCTTCCATGCTGTGGGGCCTTCCAGTCACCACAGGTGTGTCTCCTCCACTCCTTTGCTGCGCTCCAGGACTATCCCTTTGATACTCTAATCAGACCTTAGATGTGTATTTGTTCCTGGGTTTTCTTGCAGGGGAGATGAGCACAAGGCATCTCTAGTCACCCATCTTGCTTGATTTAGCATAACTCGTAAGGGCCCTAGGATTTTCAGAACAGTCAATGATCATTGGCCTCAACTTAAAGTCACCAGTTTCATTCGCCCCTAAACAGAATTTGCCTGTGCCTTGAGCTTTGAAACCATGCATTGACTTCTCTCCAAATATAAAAGTCCTGGATGGCATCTTCTTCCAATAAAAAAGGCTATCTTGTCTACAGTGAAAATCTGTTGTTTCGTGTAGCCACCTTCATCAATTGTCTCAGCTACATCTTCTGGATAACTTGCTGCTGCTGGATAACAGCACTTGCGTTTCACCTTGCACTTTTATGTTATGGAGATGGCTTCTTTCCTTAAACTTCATGAACCAACCTGTTAGCTTCAAGCTTTTCTTCTGCAACTTACTTACCTCTCTCAGCCATCACTGAATTGCAGAGAGAGGGGGCCTTGTCCTGGATTAGGCTTTGGCTTAAGGAAAGGCTATGGCTGCTTTGATCTTCTATCCAGACCGCTAAAACCTTCTCCATATTAGCAATCAGCCTGTTTTGCTTCCTCATCATTTGTATGTTCACTGGAGTAGCAGTCTTAATTTTCTTTGAGAATATATACACGTATTTTGCATTCGCAACTTGTCTGACTGGTGCAAGAAGTCTAGCCTTCCTCACAATCATTTTTAGCTTTTAGTTTAAAGTGAGAGGCTTCTGACTCTTCCTTTCACTAGGACACTTAGAAACCCTCATAGACTTATTAAATGGCCTACTTTCAATATTGTTGTGTCTTAGGAAGTAGGGAGGCCCAGAAGAGGCAGAGAGATGGGGAATGGCTACTCAGTGAAGCAGTCAAAAAACACATGGCATTGATTAAGGTAGTTGTCCTATAGGACATGGTTCATGTTCATGTTGCCCCCAAACAATTACAATAGTAACGTCAACGATCACTGATCACAGATCATAAGAGATAGGATAACAACAAAAAAGTTTAAAATAGGCTGGGCGTGGTGGCTCACGCCTGTAATCCCACCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCATTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGTACAGTAAAGCGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAAAAGACTACTGATGTTATTAATTGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAAGGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTCTGGGTCGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTCTCTCCCCAGATCTTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAGTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGCTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGTACTTTATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCATCTCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTATGACCTGTTCCTTTTTATCGAACTAGATCCAAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATGTTTACATCTAGCTTTTAATTTATTTCTACACCTTACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAGAGCTTACAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATGAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCTCCAAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGCTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACACACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAAAAATACAGTCAGACAGGTGACATAATTGAAAATTACTTTTCTGGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACACTCTGACCCCTTCCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGTGATGCATCCAGATTAAGTAACTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATCGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAAGAGAGAAATTTTACAGCTGGGCCTCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAAACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGAGGGGGGTATGAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAGGTTAAACACCCAAGCAGACGCCGAAGATAGAGCGACAGGCAAGTAACAAAGACCAACCGAACCAAGTAACAAGCCACATAAACAC";
        let md_tag = b"155T499^A3G5C0G12^A664^A8G27G483C4T160^TATG107^G375^A3A35^TT7T237^T4^TT8T2A174A141T175C0C120^G59^A60^T13^T8C178^AT35G4C478^A5A0G1G0G0T378T107G197G192T73G26^C4T525T6T548C278G85^G54";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"TGGCTTTAACCCACCCATGTGGGCTTGGTATACTGAAGGTAAAGCTCCAGTGCTGGAGACGCACAGTGGCTACTGGCCCCCAGAGTAGGGCACACTCAAGAGATGACTCTGGTCTCAAGATGGTGCTGTGTTGCTGCAACTTAGGTCTCTGGGGGtGGGAAGCCTTCACCTTGTACTCCTAGTCCAGGACATGCAGCTGTGTAAATTCCTGGCAGTACTCCATACTGGCCTCAGGGCTTGCAAATACTGTGAGATTTTCCTTAAGACTCTGTGTTTGTGATGGCAGTGGGGCTGGTGGGGATCTTCTGCTTACCTTCACTCTGCAATGAGAAGTCCTTCCTGTCTCTGGGCCAGTTCAACCCAGGTGGGGGAGAAGGGACTGCAGAGGCTGAGTGCTTCCATGCTGTGGGGCCTTCCAGTCACCACAGGTGTGTCTCCTCCACTCCTTTGCTGCGCTCCAGGACTATCCCTTTGATACTCTAATCAGACCTTAGATGTGTATTTGTTCCTGGGTTTTCTTGCAGGGGAGATGAGCACAAGGCATCTCTAGTCACCCATCTTGCTTGATTTAGCATAACTCGTAAGGGCCCTAGGATTTTCAGAACAGTCAATGATCATTGGCCTCAACTTAAAGTCACCAGTTTCATTCGCCCCTAAAAgAGAATcgGCCTGTGCCTTGAAGCTTTGAAACCATGCATTGACTTCTCTCCAAATATAAAAGTCCTGGATGGCATCTTCTTCCAATAAAAAAGGCTATCTTGTCTACAGTGAAAATCTGTTGTTTCGTGTAGCCACCTTCATCAATTGTCTCAGCTACATCTTCTGGATAACTTGCTGCTGCTGGATAACAGCACTTGCGTTTCACCTTGCACTTTTATGTTATGGAGATGGCTTCTTTCCTTAAACTTCATGAACCAACCTGTTAGCTTCAAGCTTTTCTTCTGCAACTTACTTACCTCTCTCAGCCATCACTGAATTGCAGAGAGAGGGGGCCTTGTCCTGGATTAGGCTTTGGCTTAAGGAAAGGCTATGGCTGCTTTGATCTTCTATCCAGACCGCTAAAACCTTCTCCATATTAGCAATCAGCCTGTTTTGCTTCCTCATCATTTGTATGTTCACTGGAGTAGCAGTCTTAATTTTCTTTGAGAATATATACACGTATTTTGCATTCGCAACTTGTCTGACTGGTGCAAGAAGTCTAGCCTTCCTCACAATCATTTTTAGCTTTTAGTTTAAAGTGAGAGGCTTCTGACTCTTCCTTTCACTAGGACACTTAGAAACCCTCATAGACTTATTAAATGGCCTACTTTCAATATTGTTGTGTCTTAGGAAGTAGGGAGGCCCAAGAAGAGGgAGAGAGATGGGGAATGGCTACTCAGTGgAGCAGTCAAAAAACACATGGCATTGATTAAGGTAGTTGTCCTATAGGACATGGTTCATGTTCATGTTGCCCCCAAACAATTACAATAGTAACGTCAACGATCACTGATCACAGATCATAAGAGATAGGATAACAACAAAAAAGTTTAAAATAGGCTGGGCGTGGTGGCTCACGCCTGTAATCCCACCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCATTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACAcAAAGtGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATGTATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAAAaGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTtGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCAtATaTTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAaTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGtTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTccATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCCTTTTTTATCcAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAgACCTcACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAagTggtAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATtAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCgAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGgTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACtCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAgAATACAGTCAGACAGGTGACATAATTCGAAAtTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACtCCTACCtCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGcGATGCATCCAGATTAAGTAACTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATCGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAAGAGAGAAATTTTACAGCTGGGgCTCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAAACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGTATGAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCA";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 73), (0, 1100), (2, 3), (0, 684), (2, 4), (0, 277), (1, 2), (0, 605), (1, 1), (0, 4), (1, 2), (0, 462), (2, 1), (0, 5), (2, 2), (0, 180), (2, 1), (0, 734), (1, 3), (0, 374), (2, 1), (0, 755), (1, 3), (0, 114), (1, 1), (0, 250), (4, 57)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"TTTTTCTTCGTTCAGTTACGTATTGCTACTTGCCTGTCGCTCTATCTTCGGCGTCTGCTTGGGTGTTTAACCTTACTGTGAGATTTTCCTTAAGACTCTGTGTTTGTGATGGCAGTGGGGCTGGTGGGGATCTTCTGCTTACCTTCACTCTGCAATGAGAAGTCCTTCCTGTCTCTGGGCCAGTTCAACCCAGGTGGGGGAGAAGGGACTGCAGAGGCTGAGTGCTTCCATGCTGTGGGGCCTTCCAGTCACCACAGGTGTGTCTCCTCCACTCCTTTGCTGCGCTCCAGGACTATCCCTTTGATACTCTAATCAGACCTTAGATGTGTATTTGTTCCTGGGTTTTCTTGCAGGGGAGATGAGCACAAGGCATCTCTAGTCACCCATCTTGCTTGATTTAGCATAACTCGTAAGGGCCCTAGGATTTTCAGAACAGTCAATGATCATTGGCCTCAACTTAAAGTCACCAGTTTCATTCGCCCCTAAAAGAGAATCGGCCTGTGCCTTGAAGCTTTGAAACCATGCATTGACTTCTCTCCAAATATAAAAGTCCTGGATGGCATCTTCTTCCAATAAAAAAGGCTATCTTGTCTACAGTGAAAATCTGTTGTTTCGTGTAGCCACCTTCATCAATTGTCTCAGCTACATCTTCTGGATAACTTGCTGCTGCTGGATAACAGCACTTGCGTTTCACCTTGCACTTTTATGTTATGGAGATGGCTTCTTTCCTTAAACTTCATGAACCAACCTGTTAGCTTCAAGCTTTTCTTCTGCAACTTACTTACCTCTCTCAGCCATCACTGAATTGCAGAGAGAGGGGGCCTTGTCCTGGATTAGGCTTTGGCTTAAGGAAAGGCTATGGCTGCTTTGATCTTCTATCCAGACCGCTAAAACCTTCTCCATATTAGCAATCAGCCTGTTTTGCTTCCTCATCATTTGTATGTTCACTGGAGTAGCAGTCTTAATTTTCTTTGAGAATATATACACGTATTTTGCATTCGCAACTTGTCTGACTGGTGCAAGAAGTCTAGCCTTCCTCACAATCATTTTTAGCTTTTAGTTTAAAGTGAGAGGCTTCTGACTCTTCCTTTCACTAGGACACTTAGAAACCCTCATAGACTTATTAGATGGCCTACTTTCAATATTGTTGTGTCTTAGGAAGTAGGGAGGCCCAAGAGGGAGAGAGATGGGGAATGGCTACTCAGTGGAGCAGTCAAAAAACACATGGCATTGATTAAGGTAGTTGTCCTATAGGACATGGTTCATGTTCATGTTGCCCCCAAACAATTACAATAGTAACGTCAACGATCACTGATCACAGATCATAAGAGATAGGATAACAACAAAAAAGTTTAAAATAGGCTGGGCGTGGTGGCTCACGCCTGTAATCCCACCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCATTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATACATATATGCTAAGCAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGCGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCTATTTATCTGGGCCAGTGCTAAGGATTAAAAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGGAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCATATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAAGACTTTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAGTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCGAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATGAGGAACCTGGCTTTGCTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAGAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGCTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTTTTAATGTTCCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCCGCTGAGGACTTCACAGTTACCCGAGAGGTTAAACACCCAAGCAGACGCCGAAGATAGAGCGACAGGCAAGTAGCAATACGTG";
        let md_tag = b"1053A46^AAG684^TATG16A316T122A0T124A380A138A243C1^G2T0T1^TA2A6T170^C1108^T297G462C1T330T25";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"TACTGTGAGATTTTCCTTAAGACTCTGTGTTTGTGATGGCAGTGGGGCTGGTGGGGATCTTCTGCTTACCTTCACTCTGCAATGAGAAGTCCTTCCTGTCTCTGGGCCAGTTCAACCCAGGTGGGGGAGAAGGGACTGCAGAGGCTGAGTGCTTCCATGCTGTGGGGCCTTCCAGTCACCACAGGTGTGTCTCCTCCACTCCTTTGCTGCGCTCCAGGACTATCCCTTTGATACTCTAATCAGACCTTAGATGTGTATTTGTTCCTGGGTTTTCTTGCAGGGGAGATGAGCACAAGGCATCTCTAGTCACCCATCTTGCTTGATTTAGCATAACTCGTAAGGGCCCTAGGATTTTCAGAACAGTCAATGATCATTGGCCTCAACTTAAAGTCACCAGTTTCATTCGCCCCTAAAAGAGAATCGGCCTGTGCCTTGAAGCTTTGAAACCATGCATTGACTTCTCTCCAAATATAAAAGTCCTGGATGGCATCTTCTTCCAATAAAAAAGGCTATCTTGTCTACAGTGAAAATCTGTTGTTTCGTGTAGCCACCTTCATCAATTGTCTCAGCTACATCTTCTGGATAACTTGCTGCTGCTGGATAACAGCACTTGCGTTTCACCTTGCACTTTTATGTTATGGAGATGGCTTCTTTCCTTAAACTTCATGAACCAACCTGTTAGCTTCAAGCTTTTCTTCTGCAACTTACTTACCTCTCTCAGCCATCACTGAATTGCAGAGAGAGGGGGCCTTGTCCTGGATTAGGCTTTGGCTTAAGGAAAGGCTATGGCTGCTTTGATCTTCTATCCAGACCGCTAAAACCTTCTCCATATTAGCAATCAGCCTGTTTTGCTTCCTCATCATTTGTATGTTCACTGGAGTAGCAGTCTTAATTTTCTTTGAGAATATATACACGTATTTTGCATTCGCAACTTGTCTGACTGGTGCAAGAAGTCTAGCCTTCCTCACAATCATTTTTAGCTTTTAGTTTAAAGTGAGAGGCTTCTGACTCTTCCTTTCACTAGGACACTTAGAAACCCTCATAGACTTATTAaATGGCCTACTTTCAATATTGTTGTGTCTTAGGAAGTAGGGAGGCCCAAGAAGAGGGAGAGAGATGGGGAATGGCTACTCAGTGGAGCAGTCAAAAAACACATGGCATTGATTAAGGTAGTTGTCCTATAGGACATGGTTCATGTTCATGTTGCCCCCAAACAATTACAATAGTAACGTCAACGATCACTGATCACAGATCATAAGAGATAGGATAACAACAAAAAAGTTTAAAATAGGCTGGGCGTGGTGGCTCACGCCTGTAATCCCACCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCATTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATGTATACATATATGCTAAaCAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGtGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCatTTTATCTGGGCCAGTGCTAAGGATTAAAAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGaAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCATATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAaTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCaAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATcAGGGttCTACTaGCTTTGtTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGgTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGcTtCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCtGCTGAGGACTTCACAGTTACCCGAG";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 58), (0, 237), (1, 1), (0, 3), (2, 2), (0, 144), (1, 3), (0, 575), (2, 3), (0, 818), (2, 4), (0, 88), (2, 2), (0, 17), (2, 2), (0, 12), (2, 2), (0, 154), (1, 4), (0, 486), (2, 1), (0, 93), (1, 1), (0, 35), (2, 2), (0, 105), (2, 2), (0, 1143), (2, 1), (0, 1336), (2, 1), (0, 6), (1, 1), (0, 4), (1, 1), (0, 223), (2, 1), (0, 775), (2, 1), (0, 87), (2, 1), (0, 29), (2, 1), (0, 40), (1, 1), (0, 252), (4, 88)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"ACCACGTATTGCTACTTGCCTGTCGCTCTATCTTCGGCGTCTGCTTGGGTGTTTAACCTGAGATTTTCCTTAAGACTCTGTGTTTGTGATGGCAGTGGGGCTGGTGGGGATCTTCTGCTTACCTTCACTCTGCAATGAGAAGTCCTTCCTGTCTCTGGGCCAGTTCAACCCAGGTGGGGGAGAAGGGACTGCAGAGGCTGAGTGCTTCCATGCTGTGGGGCCTTCCAGTCACCACAGGTGTGTCTCCTCCACTCCTTTGCTGCGCTCCAGGACTATCCCTTTGATACTCTAATCAGGACTAGATGTGTATTTGTTCCGGGGTTCTCTTGCAGGGGAGATGAGCACAAGGCATCTCTAGCTACCCATCTTGCTTGATTTAGCATAACTCGTAAGGGCCCTAGGATTTTCAGAACAGTCAATGATCATTGGCCTCAACTTAAAGTCCCCACCAGTTTCATTCGCCCCTAAAAGAGAATCGGCCTGTGCCTTGAAGCTTTGAAACCATGCATTCACTTCTCTCCAAATATAAAAGTCCTGGATGGCATCTTCTTCCAATAAAAAAGGCTATCTTGTCTACAGCTAAAATCTGTTGTTTCGTGTAGCCACCTTCATCAATTGTCTCAGCTACATCTTCTGGATAACTTGCTGCTGCTGGATAACAGCACTTGCGTTTCACCTTGCACTTTTATGTTATGGAGATGGCTTCTTTCCTTAAACTTCATGAACCAACCTGTTAGCTTCAAGCCTTTCTTCTGCAACTTACTTACCTCTCTCAGCCATCACTGAATTGCAGAGAGAGGGGGCCTTGTCCTGGATTAGGCTTTGGCTTAAGGAAAGGCTATGGCTGCTTTGATCTTCTATCCAGACCGCTAAAACCTTCTCCATATTAGCAATCAGCCTGTTTTGCTTCCTCATCATTTGTATGTTCACTGGAGTAGCAGTCTTAATTTTCTTTGAGAATATATACACGTATTTTGCATTCGCAACTTGTCTGACTGGTGCAAGAAGTCTAGCCTTCCTCATCATTTTTAGCTTTTAGTTTAAAGTGAGAGGCTTCTGACTCTTCCTTTCACTAGGACACTTAGAAACCCTCATAGACTTATTAAATGGCCTACTTTCAATATTGTTGTGTCTTAGGAAGTAGGGAGGCCCAAGAAGAGGGAGAGAGATGGGGAATGGCTACTCAGTGGAGCAGTCAAAAAACACATGGCATTGATTAAGGTAGTTGTCCTATAGGACATGGTTCATGTTCATGTTGCCCCCAAACAATTACAATAGTAACGTCAACGATCACTGATCACAGATCATAAGAGATAGGATAACAACAAAAAAGTTTAAAATAGGCTGGGCGTGGTGGCTCACGCCTGTAATCCCACCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCATTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATACATGTACTGATGTTATTATTTGGTCAGTGCTATAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAGAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAATACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATTCCCTCTTTTCTTTCTCCCCATATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATAATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAGTTATTTATTTGAAACCTGGTATTTAACTTTTCTGTAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGCAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACCCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATCACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGATACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAGTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGTATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCGCCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGCTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATACACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAAATGGTTAATTTCTAACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAACGTGTGACAGCCTTCGTGGTGGCAGTTATTTCAACTTATTCTCTCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGCTCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAGTCTCTTAGCCCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGCACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTACCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGCCATGCATCCAGATTAAGTAACTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGGGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATCGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAGAGAGAAATTTTACAGCTGGGCCTCCAGGCTCACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAAGCCGCAAGACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGTATCAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAATAAAAGATCACAAGGGAAGACAGGCAGAGCAAGATCACAAGGCCAGGGCGAAATTAGAATTATTGATGAGGTTCCATGTCCCACTGGGCACCCATTGTCACTGATAAACATCTTAACAGGAAACAGAGTTCCAGAGCAGACAACAGGTCCGACTAGAGGTTAAACACCCAAGCAGACGCCGAAGATAGAGCGACAGGCAAGTAGCAATACGTAACTGAACGAAGGTACAAACTAACATAAACACC";
        let md_tag = b"240^CT18T5T34T0C147G68T0G164T275^ACA818^TATG84A1A1^AC14A2^GG12^AG362A226G50^C128^TC69A33C1^AG147T60T318G352C262^A3T72A359A361G144T197C12G149T13T17^A4C1A172T0C52^A11T343T48T204G113A51^G87^A23G5^G4G43A44G198";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"TGAGATTTTCCTTAAGACTCTGTGTTTGTGATGGCAGTGGGGCTGGTGGGGATCTTCTGCTTACCTTCACTCTGCAATGAGAAGTCCTTCCTGTCTCTGGGCCAGTTCAACCCAGGTGGGGGAGAAGGGACTGCAGAGGCTGAGTGCTTCCATGCTGTGGGGCCTTCCAGTCACCACAGGTGTGTCTCCTCCACTCCTTTGCTGCGCTCCAGGACTATCCCTTTGATACTCTAATCAGACCTTAGATGTGTATTTGTTCCtGGGTTtTCTTGCAGGGGAGATGAGCACAAGGCATCTCTAGtcACCCATCTTGCTTGATTTAGCATAACTCGTAAGGGCCCTAGGATTTTCAGAACAGTCAATGATCATTGGCCTCAACTTAAAGTCACCAGTTTCATTCGCCCCTAAAAGAGAATCGGCCTGTGCCTTGAAGCTTTGAAACCATGCATTgACTTCTCTCCAAATATAAAAGTCCTGGATGGCATCTTCTTCCAATAAAAAAGGCTATCTTGTCTACAGtgAAAATCTGTTGTTTCGTGTAGCCACCTTCATCAATTGTCTCAGCTACATCTTCTGGATAACTTGCTGCTGCTGGATAACAGCACTTGCGTTTCACCTTGCACTTTTATGTTATGGAGATGGCTTCTTTCCTTAAACTTCATGAACCAACCTGTTAGCTTCAAGCtTTTCTTCTGCAACTTACTTACCTCTCTCAGCCATCACTGAATTGCAGAGAGAGGGGGCCTTGTCCTGGATTAGGCTTTGGCTTAAGGAAAGGCTATGGCTGCTTTGATCTTCTATCCAGACCGCTAAAACCTTCTCCATATTAGCAATCAGCCTGTTTTGCTTCCTCATCATTTGTATGTTCACTGGAGTAGCAGTCTTAATTTTCTTTGAGAATATATACACGTATTTTGCATTCGCAACTTGTCTGACTGGTGCAAGAAGTCTAGCCTTCCTCACAATCATTTTTAGCTTTTAGTTTAAAGTGAGAGGCTTCTGACTCTTCCTTTCACTAGGACACTTAGAAACCCTCATAGACTTATTAAATGGCCTACTTTCAATATTGTTGTGTCTTAGGAAGTAGGGAGGCCCAAGAAGAGGGAGAGAGATGGGGAATGGCTACTCAGTGGAGCAGTCAAAAAACACATGGCATTGATTAAGGTAGTTGTCCTATAGGACATGGTTCATGTTCATGTTGCCCCCAAACAATTACAATAGTAACGTCAACGATCACTGATCACAGATCATAAGAGATAGGATAACAACAAAAAAGTTTAAAATAGGCTGGGCGTGGTGGCTCACGCCTGTAATCCCACCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCATTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATGTATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAaAaGACTACTGATGTTATTAaTTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAaAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAgTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCATATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAaTTATTTATTTGAAACCTGGTATTTAACTTTTCTcTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGtAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACtCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATgACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGAcACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAAtTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGaATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCaCCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGgTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATAtACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAAcTGGTTAATTTCTgACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAAtGTGTGACAGCCTTtGTGGTGGCAGTTATTTCAAACTcAaCTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGtcCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCtCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGtACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTtCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGCgATGCATCCAGATTAAGTAACTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGaGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATCGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAAGAGAGAAATTTTACAGCTGGGgCTCCAGGGCTgACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAaACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGTATgAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAATAAAAGATCACAAGGGAAGACAGGCAGAGCAAGATCACAAGGCCAGGGCGAAATTAGAATTATTGATGAGGTTCCATGTCCCACTGGGCACCCATTGTCACTGATAAACATCTTAACAGGAAACAGAGTTCCAGAGCAGACAACAGGTCCGACTAGA";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 83), (0, 130), (2, 1), (0, 15), (2, 1), (0, 45), (1, 1), (0, 1), (1, 1), (0, 643), (2, 4), (0, 277), (1, 2), (0, 48), (2, 2), (0, 52), (1, 1), (0, 1), (1, 1), (0, 16), (2, 1), (0, 491), (2, 3), (0, 100), (1, 2), (0, 246), (2, 1), (0, 199), (2, 2), (0, 842), (2, 1), (0, 669), (1, 2), (0, 2), (2, 1), (0, 781), (2, 1), (0, 622), (1, 1), (0, 46), (4, 58)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"TAGCTATGTTATGTTGACCTTAAACTTAAACTTGGTCATTTGTTCTGTTGGTGCTGATATTGACCTGTTTGGGTGTCTGACCTAGTCTAGCCTTCCTCACAATCATTTTTAGCTTTTAGTTTAAAGTGAGAGGCTTCTGACTCTTCCTTTCACTAGGACACTTAGAAACCCTCATAGACTTATTAAATGGCCTACTTTCAATATTGTTGTGTCTAGGAAGTAGGGAGGCCAAGAAGAGGGAGAGAGATGGGGAATGGCTACTCAGTGGAGCAGATGCAAAAAACACATGGCATTGATTAAGGTAGTTGTCCTATAGGACATGGTTCATGTTCATGTTGCCCCCAAACAATTACAATAGTAACGTCAACGATCACTGATCACAGATCATAAGAGATAGGATAACAACAAAAAAGTTTAAAATAGGCCGGGCGTGGTGGCTCACGCCTGTAATCCCACCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCATTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTGAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATACAAGTTGTCTTCAATGACGCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAAAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCATATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTCTGACACAAATGAGTAATAATTGCTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAGTTATTTATTTGAAACCTGGTATTTAACTTTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGCTCCTGAGGAAGAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGGTGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTGGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGTCCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTCTGCTGAGGACTTCACAGTTACCCGAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACGTCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTAACTGCATCTTCGCCTTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACAGGTTAAACACCAAAGCAGACGCCGAAGATAGAGCGACAGGCAAGAGCAATACATTAT";
        let md_tag = b"130^T15^C195T373C119^TATG325^TG69^A0C490^TTT24A47A273^C199^CT842^T661G9^A93A203C408G0T73^T338A280A48";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"AGTCTAGCCTTCCTCACAATCATTTTTAGCTTTTAGTTTAAAGTGAGAGGCTTCTGACTCTTCCTTTCACTAGGACACTTAGAAACCCTCATAGACTTATTAAATGGCCTACTTTCAATATTGTTGTGTCTTAGGAAGTAGGGAGGCCCAAGAAGAGGGAGAGAGATGGGGAATGGCTACTCAGTGGAGCAGTCAAAAAACACATGGCATTGATTAAGGTAGTTGTCCTATAGGACATGGTTCATGTTCATGTTGCCCCCAAACAATTACAATAGTAACGTCAACGATCACTGATCACAGATCATAAGAGATAGGATAACAACAAAAAAGTTTAAAATAGGCtGGGCGTGGTGGCTCACGCCTGTAATCCCACCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCATTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTcAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATGTATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACAcCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAAAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCATATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTaCTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAaTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGgTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGaTGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTcGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGgtCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACaTCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTAACTGCATCTTCaCCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAAC";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 55), (0, 463), (2, 4), (0, 277), (1, 2), (0, 940), (1, 3), (0, 13), (2, 2), (0, 6), (1, 1), (0, 31), (1, 1), (0, 173), (2, 1), (0, 86), (1, 2), (0, 5), (1, 1), (0, 24), (2, 1), (0, 1338), (2, 3), (0, 749), (1, 1), (0, 1045), (1, 1), (0, 294), (2, 1), (0, 1), (2, 1), (0, 738), (2, 1), (0, 73), (2, 2), (0, 450), (1, 1), (0, 36), (4, 71)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"ATCGTATCACTCTGCCTGTCGCTCTATCTTCGGCGTCTGCTTGGGTGTTTAACCTCACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCACTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAATATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAAAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCACATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAGTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATAATCATGGATGTCTTCTGCTCCCTCAGATACGGTTAAGTAACTGCATACCTAAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTACGGCACCTGTTTTCTCTTTTATCCAACTAGATCCAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACCCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCGGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTCTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGCTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGGTGATATAAGGTCCGGAGCACCGGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTCTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTATACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTAACTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATCGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAAGAGAGAAATTTTACAGCTGGGCCTCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAAACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGGTATGAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAATAAAAGATCACAAGGGAAGACAGGCAGAGCAAGATCACAAGGCCAGGGCGAAATTAGAATTATTGATGAGGTTCCATGTCCCACTGGGCACCCATTGTCACTGATAAACATCTTAACAGGAAACAGAGTTCCAGAGCAGACAACAGGTCCGACTAGAATTCGCCAGGCTGGAATTTCCTAATCCTAGCAAGCCTGAGGGCACTGCAGGAGACCAGGGCGTATTTCATCCCTTATCTTCAACTGGAAGACAGACACTCCCAGAGCGGCCATTTATAGACCTCTCCCTGGGAATGCATTCCTTTCCCAGGGTTATTCCTTGCTGGGAAAAGAATTCAGTGATACTTCTCCTACTTGCTTTCTCCAAGAAGAAGAATAATACTCTGTTCTGCCCGGCCCCGCAGGCAGTCAGACCTTATAGTTATCTCCTTTGTTCCCTGAAAATCGCTGTTAGCCTGTTCTTTTCTAGGATGCCAAGATTTCATATTGTTCAAACACACATGTTTTACAAACAATTTGCAAAGATAATGCAATCATCACAGGACCCTGGGGCGACCTACCTTCTCAGCTTACGAAGATGACGGGATTAAGAGATTAAAGTAAATACAGGCATAGGAAATTATAAGAGTATTGACTAGGGAAGTGATAAATGTCCATGAAATCTTCATAATTTATGTTCAGAGACTGCAGTAAAGACAGGCAGAATAAATTATAAAAGTATTAATTTGGGGAACTAATAAATGTCCATGAAATCTTCATAACTTATGTTCTTCTGCCACAGCTTCAACCAGTCCCTCCACTCAGGGTCCCTGATTTCCAGCAACAAGTTGTCTTTCCTGCCCAAGAGTTACGGCCAAGTAGCTGCAGGCCTCAAGCAAGGGCTTCATCCCATGTGATGGTGGGTATGGGCCAAGGGTGTCAGCCTCACATTATGGCAGAGCTCTGGAGGGTGTGAAATATTTCCAGGGTTATTAGCTCTGTTCTCTTTTTTGCTGGTGTTTCTCTGCTCTTTGTCCACTCCTATCACTTTCTAACTCCAAGGAGGATTTGCTTCTCTCGGTCCACACTACTGGAACAGTCTCATTCCCACAGAAAGGAAAGGTGACAATCACTCACATAATAAGCCTCCCTCTATGAGTTCTTCCCCATTTTCCTACCTCCATTCAAAAAGCAAAAGACTATTCCGTACAAAGCTCATCCTGAGAGGGCTCTGGAAATAGTTTCAAGGGAACTCCTTGTTCCCATGGATGCTCCTGACAGTGAATAGCCCAAATCGCAATCTCTCTAGTGTATATGAATGCATGGATGTGTGTGTTTGTATATGCATATATACATATACAATTGTATTCATGTCACATGTTGCAATGGACAGCCTATTAGAAGATACTTTCAGTTCACACACAGCGCAGTGCTAAAGAACTTAAAACGTTTATAGTAAGACAACATGTAACTTCCTACACTTAACATTTTCAGTTCAATGTAGTTTGCATTAATTTTCTTTTTCTTTATTTTTTTTTTTTCTTTCGAGACAGAGTCTCACTCTGTAGGTTAAACACCCAAGCAGACGCCGAAGATAGAGACAAATACGTAACTGAACGAAGTGCATACATAAACAT";
        let md_tag = b"211T251^TATG71G713T177A266^TT32G177^T84T8C2T18^A464T421T47T403^AGA43G728A125T271G532G384^C1^T125G0A5G536A68^T73^TT486";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"CACTGTGGGAGGCCAAGGCAGGCGGATCACGAGGTCAGGAGTTTGAGACCAGCCTGACGAACATGGTGAAACCCCGTCTCTACTAAGAATATAAAAATTAGCCGGGCGTGGTGGCAGGTGCCTGTAATCCCAGCTACTCAGGAGGCTGTGGCAGAGAATCGCTTGAACCCGGGAGGCTGAGGTTGCAGTGAGCTGAGATCGTGCCACTGCAtTCCAGCCTGGGTGTCAGAGCGAGAGTCTGTCTCAAAAAAAGAGTTTAAGATATTGCAAGAGTTACTAAAATGTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTCAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATGTATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAgTATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAAAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCAtATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAaTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAATGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAgACCTAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTAtGACCTGTTcCTtTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACtCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCtGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTtTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGgTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCaGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTtTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTgTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTAACTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATCGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAAGAGAGAAATTTTACAGCTGGGgCTCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAAACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGTATGAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAATAAAAGATCACAAGGGAAGACAGGCAGAGCAAGATCACAAGGCCAGGGCGAAATTAGAATTATTGATGAGGTTCCATGTCCCACTGGGCACCCATTGTCACTGATAAACATCTTAACAGGAAACAGAGTTCCAGAGCAGACAACAGGTCCGACTAGAATTCGCCAGGCTGGAATTTCCTAATCCTAGCAAGCCTGAGGGCACTGCAGGAGACCAGGGCGTATTTCATCCCTTATCTTCAACTGCGTAAGACAGACACTCCCAGAGCGGCCATTTATAGACCTCTCCCTGGGAATGCATTCCTTTCCCAGGGTTATTCCTTGCTGGGAAAAGAATTCAGTGATACTTCTCCTACTTGCTTTCTCCAAGAAGAgaAATAAgACTCTGTTCTGCCCGGCCCCGCAGGCAGTCAGACCTTATAGTTATCTCCTTTGTTCCCTGAAAATCGCTGTTAGCCTGTTCTTTTCTAGGATGCCAAGATTTCATATTGTTCAAACACACATGTTTTACAAACAATTTGCAAAGATAATGCAATCATCACAGGACCCTGGGGCGACCTACCTTCTCAGCTTACGAAGATGACGGGATTAAGAGATTAAAGTAAATACAGGCATAGGAAATTATAAGAGTATTGACTAGGGAAGTGATAAATGTCCATGAAATCTTCATAATTTATGTTCAGAGACTGCAGTAAAGACAGGCAGAATAAATTATAAAAGTATTAATTTGGGGAACTAATAAATGTCCATGAAATCTTCATAACTTATGTTCTTCTGCCACAGCTTCAACCAGTCCCTCCACTCAGGGTCCCTGATTTCCAGCAACAAGTTGTCTTTCCTGCCCAAGAGTTACGGCCAAGTAGCTGCAGGCCTCAAGCAAGGGCTTCATCCCATGTGATGGTGGGTATGGGCCAAGGGaGTCAGCCTCACATTATGGCAGAGCTCTGGAGGGTGTGAAATATTTCCAGGGTTATTAGCTCTGTTCTCTTTTTTTGCTGGTGTTTCTCTGCTCTTTGTCCACTCCTATCACTTTCTAACTCCAAGGAGGATTTGCTTCTCTCTTGGTCCACACTACTGGAACAGTCTCATTCCCACAGAAAGGAAAGGTGACAATCACTCACATAATAAGCCTCCCTCTATGAGTTCTTCCCCATTTTCCTACCTCCATTCAAAAAGCAAAAGACTATTCCGTACAAAGCTCATCCTGAGAGGGCTCTGGAAATAGTTTCAAGGGAACTCCTTGTTCCCATGGATGCTCCTGACAGTGAATAGCCCAAATCGCAATCTCTCTAGTGTATATGAATGCATGGATGTGTGTGTTTGTATATGCATATATACATATACAATTGTATTCATGTCACATGTTGCAATGGACAGCCTATTAGAAGATACTTTCAGTTCACACACAGCGCAGTGCTAAAGAACTTAAAACGTTTATAGTAAGACAACATGTAACTTCCTACACTTAACATTTTCAGTTCAATGTAGTTTGCATTAATTTTCTTTTTCTTTATTTTTTTTTTTCTTTCGAGACAGAGTCTCACTCTGT";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 80), (0, 91), (2, 1), (0, 87), (2, 4), (0, 277), (1, 6), (0, 46), (2, 2), (0, 956), (1, 1), (0, 800), (1, 1), (0, 2), (2, 1), (0, 664), (2, 1), (0, 6), (1, 1), (0, 275), (1, 2), (0, 222), (1, 1), (0, 222), (2, 1), (0, 101), (2, 1), (0, 7), (2, 1), (0, 6), (2, 2), (0, 119), (1, 1), (0, 631), (1, 1), (0, 1360), (2, 1), (0, 25), (2, 1), (0, 64), (2, 1), (0, 605), (2, 1), (0, 4), (1, 1), (0, 999), (4, 56)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"TGTTATGTTCTGTACTTCGTTCAGTTACGTATTGCTCTTGCCTGTCGCTCTATCTTCGGCGTCTGCTTGGGTGTTTAACCTGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTGAGTTTATAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATATATATACACACATACATATATTCACGTGTGTGTGTGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAAAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCATATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAGTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAGTGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTCCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTACTCACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTCCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCCAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAATTAGGAGACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGCTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTCAGGGAAAAACTAGCTCTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATTCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGACAGTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGCATGTCCCCAGCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTCATTTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATCGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAAGAGAGAAATTTTACAGCTGGGCCTCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAAACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGTATGAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAATAAAAGATCACAAGGGAAGACAGGCAGAGCAAGATCACAAGGCCAGGGCGAAATTAGAATTATTGATGAGGTTCCATGTACCACTGGGCACCCATTGTCACTGATAAACATCTTAACAGGAAACAGAGTTCCAGAGCAGACAACAGGTCCGACTAGAATTCGCCAGGCTGGAATTTCCTAATCCTAGCAAGCCTGAGGGCACTGCAGGAGACCAGGGCGTATTTCATCCCTTATCTTCAACTGCGTAAGACAGACACTCCCAGAGCGGCCATTTATAGACCTCTCCCTGGGAATGCATTCCTTTCCCAGGGTTATTCCTTGCTGGGAAAAGAATTCAGTGATACTTCTCCTACTTGCTTTCTCCAAGAAGAGAAATAAGACTCTGTTCTGCCCGGCCCCGCAGGCAGTCAGACCTTATAGTTATCTCCTTTGTTCCCTGAAAATCGCTGTTAGCCTGTTCTTTTCTAGGATGCCAAGATTTCATATTGTTCAAACACACATGTTTTACAAACAATTTGCAAAGATAATGCAATCATCACAGGACCCTGGGGCGACCTACCTTCTCAGCTTACGAAGATGACGGGATTAAGAGATTAAAGTAAATACAGGCATAGGAAATTATAAGAGTATTGACTAGGGAAGTGATAAATGTCCATGAAATCTTCATAATTTATGTTCAGAGACTGCAGTAAAGACAGGCAGAATAAATTATAAAAGTATTAATTTGGGGAACTAATAAATGTCCATGAAATCTTCATAACTTATGTTCTTCTGCCACAGCTTCAACCAGTCCCTCCACTCAGGGTCCCTGATTTCCAGCAACAAGTTGTCTTTCCTGCCCAAGAGTTACGGCCAAGTAGCTGCAGGCCTCAAGCAAGGGCTTCATCCCATGTGATGGTGGGTATGGGCCAAGGGAGTCAGCCTCACATTATGGCAGAGCTCTGGAGGGTGTGAAATATTCCAGGGTTATTAGCTCTGTTCTCTTTTTTGCTGGTGTTTCTCTGCTCTTTGTCCACTCCTATCACTTTCTAACTCCAAGGAGAATTTCTTCTCTCTTGGTCCACACTACTGGAACAGTCTCATTCCCACAGAAAGGAAAGGTGACAATCACTCACATAATAAGCCTCCCTCTATGAGTTCTTCCCCATTTTCCTACCTCCATTCAAAAAGCAAAAGACTATTCCGTACAAAGCTCATCCTGAGAGGGCTCTGGAAATAGTTTCAAGGGAACTCCTTGTTCCCATGGATGCTCCTGACAGTGAATAGCCCAAATCGCAATCTCTCTAGTGTATATGAATGCATGGATGTGTGTGTTTGTATATGCATATATACATATACAATTGTATTCATGTCACATGTTGCAATGGACAGCCTATTAGAAGATACTTTCAGTTCACACACAGCGCAGTGCTAAAGAACTTAAAACGTTTATAGTAAGACAACATGTAACTTCCTACACTTAACATTTTCAGTTCAATGTAGTTTGCATTAATTTTCTTTTTCTTTATTTTTTTTTTTCTTTCGAGACAGAGTCTCACTCTGTCACCCACACTGGAGTCCGGTGGCATGATCTCGACTAACTACAAACTCTGCCTCCCAGGTTCAAGTAATTCTCCTGCCTCAGCCTCCCAAGTAGCCGGGACTACAGGCGCTGCCACCACGCCTGGCTAATTTTTGTATTGTTAGTAAAGACAGGGTTTCACCATGTTGATCAGGCTAGTCTTGAACTCCTGACCTCGTGATCCACCCGACTTGACCGCCCAAAGCACTGGGACCACAGGTATCAGCCAACGCGCCTGGCCTTCCAATTACTTTTCAGCCTGTAAACTGCTCTACCCTACAGAATTCACGTGTGTATTTGGTTCTTTGGAATCAGCCATGCCATTTACCCTACACATCATACTTTCTATGATTACAGCACACATTTTACACCTATCCAAGACTCACTGACTCAAGAGTTAACCAAAACTCTCTACCACCGGAGCTGACCAATGCATGACAGCCTCTTGTGGGTGAATTTCTTTTAAATTCAACTCCACTGTGTAAGCAGTCATCCCTCTAATCCCATCTATGGCCCCCTTCCTGGATCTCTCCATCACGTCTTTCCTTTCACCCATCATGCTAGTCTTTCCCGTCCATGAGACACAGGCAACTAGCTGCGAACCTAAGGAAGAGCAGCATCCAATGAGGTGCGTGCCAGTGTTTGTTTTTTAGCACCAACCTCATATGTTGACAGAGCTCAGGAGTGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCTTTGCCCAGGGCTTCACACTCCTTGTCGGCTGCTATCTCCTCCATGCTGCAAATAGGATTCTCTCCTCCCTTGTGTCAATCTGGTAGGCTCTCAAACTCAGGAAAACTGAGTCCTTTAAGAAGCTGCTTAGTAGGAACTAAGGGTGACAAATACACCAACAAGAAGTCGCAGCTCTCCTATGACTGGCTCCTTTTTATACAACTACATTAGAAAAGGATGCAGTGTTACCTACAAAGGCTACCATGAGAAGAATTGCAAACCCTGTTAAAAACCACTCTCTGTGGCCATGCAGGCTTCTAGAAGAAACTCCACATATGTCCATCCGACTCCTTATTTTATGTTTCCATTGTGTGCATGGGTTTGTTAGTGGACGAGGTTAAACACCCAAGCAGACGCCGAAGATAGAGCGACAGGCAAGAAGCAATACGT";
        let md_tag = b"83C7^A87^TATG323^TA638A60A1058^T338T107T217^C1C192G322T138G64G3^T101^G3G1C1^C6^CC751A0C257G220C878^T25^T59G4^G583G6A14^A1003";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"TGACACAGAGACACAAAGTGAGCACGTGTTGTTGGAAAAATGAAACCAAGAGACTTGCTCAATGCAAGGTTGCTACAAACCTTcAGTTTATAAAAAAGTAATATCTGAGGGGCAATAAAGTAAAGCACAATAACATAAGGTATACATGTATACACACACACGTGTACATATCTACTGTATATGTATACATATATGCTAAACAATATATCTACATATACGTATCTATATATATACATGCATGTCTATATGCTAAAGTATATATACATAAAAGACTACTGATGTTATTAATTGGGGTCAGTGCTATAGAATATATCTATATAGATACATATATGTACATAAAGATATGTATATATCTATATATGCACATATAGATATATGTATATAGATATGTTCATACATATACACACATACATGTATGTAGGTATCTATATATACTAAATATATATATAAAGGATATACCTATATATATATATATATACACACATACATATATTCACGTGTGTGTGTATGTGTGTGTGTGTGTATAGATATATTCTTTAGCAATGACCCAGATTAATAACATCAGTTGTCTTCAATGACACCACCATTGAAAGGTGAGTTACAGCAGATATATCAAAGTAGTGAAAGACTACTGATGTCATTTTATCTGGGCCAGTGCTAAGGATTAAAAAGGTACATGCCAGATCAAGCGACTTCAGAGACTTGCTTTTCTGGTTGAAGTTATGTACAATTGTTCTACTGCATTTATAAACTGGTTGAACTCTGAAAACGTATATGTCAATTAATTACTTGGAATCTAGTATTAATATATCATTTTTTTTCCCTAAAAGAAAGCCTTACATAATGTAAAAAAGCTTTTGGGCCTTAACAAACAGCCTAGTGAATCAAAATTTAAGTACAATTCTTTATTACTTCATTTGTGAGTGAATCTTACCAATCTCATTATCTCCCTCTTTTCTTTCTCCCCATATATTGATGACTCTACCACAGACTACTCTTTTTAGTTCAATGGGGTAATAGCTAAATGATTAAAAATGAATATAGGAAGACTCCACATGTCTTCCTAGACTATTTTTCTGACACAAATGAGTAATAATTACTCTACAATTTATAATCTAGTTACCCTAAAAATTGACATCTTAATCAaTTATTTATTTGAAACCTGGTATTTAACTTTTCTCTAGAAACGAGGTTTCAGATGTTTGCAaTGAAATTCTTAGGCCTGTGGAAAAAAACTTAATGAGTGAAGTTTATATACATTTCTTTACTCCTAACCCTAATCCCCAAGTTACCATTTCTGGCAAAATTTACATAAGTTCAACTTCAGTTTGTAAACACCCACCTTGCTTGCACTTGTAAGTTTCTGCCTTCTCAAGTTCAAATACCAATAACTCTTCACCAATCATGGATGTCTTTTCTGCCCCTCAGATACGGTTAAGTAACTGCAGACCTAAGGCAGGGGCTCCATCCAATCTGATGGTGGGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTACTAGCTTTGTTCTTTTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATGAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTtCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCtAAAGAAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTcAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGgTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTCGAAATTTACTTTTCGGGAAAAACTAGCTtTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATgCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGTGgCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCgTcTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCGAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATCTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTAacTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATCGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAAGAGAGAAATTTTACAGCTGGGgCTCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAAACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGTATGAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAATAAAAGATCACAAGGGAAGACAGGCAGAGCAAGATCACAAGGCCAGGGCGAAATTAGAATTATTGATGAGGTTCCATGTcCCACTGGGCACCCATTGTCACTGATAAACATCTTAACAGGAAACAGAGTTCCAGAGCAGACAACAGGTCCGACTAGAATTCGCCAGGCTGGAATTTCCTAATCCTAGCAAGCCTGAGGGCACTGCAGGAGACCAGGGCGTATTTCATCCCTTATCTTCAACTGCGTAAGACAGACACTCCCAGAGCGGCCATTTATAGACCTCTCCCTGGGAATGCATTCCTTTCCCAGGGTTATTCCTTGCTGGGAAAAGAATTCAGTGATACTTCTCCTACTTGCTTTCTCCAAGAAGAGAAATAAGACTCTGTTCTGCCCGGCCCCGCAGGCAGTCAGACCTTATAGTTATCTCCTTTGTTCCCTGAAAATCGCTGTTAGCCTGTTCTTTTCTAGGATGCCAAGATTTCATATTGTTCAAACACACATGTTTTACAAACAATTTGCAAAGATAATGCAATCATCACAGGACCCTGGGGCGACCTACCTTCTCAGCTTACGAAGATGACGGGATTAAGAGATTAAAGTAAATACAGGCATAGGAAATTATAAGAGTATTGACTAGGGAAGTGATAAATGTCCATGAAATCTTCATAATTTATGTTCAGAGACTGCAGTAAAGACAGGCAGAATAAATTATAAAAGTATTAATTTGGGGAACTAATAAATGTCCATGAAATCTTCATAACTTATGTTCTTCTGCCACAGCTTCAACCAGTCCCTCCACTCAGGGTCCCTGATTTCCAGCAACAAGTTGTCTTTCCTGCCCAAGAGTTACGGCCAAGTAGCTGCAGGCCTCAAGCAAGGGCTTCATCCCATGTGATGGTGGGTATGGGCCAAGGGAGTCAGCCTCACATTATGGCAGAGCTCTGGAGGGTGTGAAATATTTCCAGGGTTATTAGCTCTGTTCTCTTTTTTTGCTGGTGTTTCTCTGCTCTTTGTCCACTCCTATCACTTTCTAACTCCAAGGAGgATTTGCTTCTCTCTTGGTCCACACTACTGGAACAGTCTCATTCCCACAGAAAGGAAAGGTGACAATCACTCACATAATAAGCCTCCCTCTATGAGTTCTTCCCCATTTTCCTACCTCCATTCAAAAAGCAAAAGACTATTCCGTACAAAGCTCATCCTGAGAGGGCTCTGGAAATAGTTTCAAGGGAACTCCTTGTTCCCATGGATGCTCCTGACAGTGAATAGCCCAAATCGCAATCTCTCTAGTGTATATGAATGCATGGATGTGTGTGTTTGTATATGCATATATACATATACAATTGTATTCATGTCACATGTTGCAATGGACAGCCTATTAGAAGATACTTTCAGTTCACACACAGCGCAGTGCTAAAGAACTTAAAACGTTTATAGTAAGACAACATGTAACTTCCTACACTTAACATTTTCAGTTCAATGTAGTTTGCATTAATTTTCTTTTTCTTTATTTTTTTTTTTCTTTCGAGACAGAGTCTCACTCTGTCACCCACACTGGAGTCCGGTGGCATGATCTCGACTAACTACAAACTCTGCCTCCCAGGTTCAAGTAATTCTCCTGCCTCAGCCTCCCgAGTAGCaGGGACTACAGGCGCATGCCCCACGCCTGGCTAATTTTTGTATTGTTAGTAAAGACAGGGTTTCACCATGTTGATCAGGCTAGTCTTGAACTCCTGACCTCGTGATCCACCCGACTTGACCGCCCAAAGCACTGGGACCACAGGTATCAGCCAACGCGCCTGGCCTTCCAATTACTTTTCAGCCTGTAAACTGCTCTACCCTACAGAATTCACGTGTGTATTTGGTTCTTTGGAATCAGCCATGCCATTTACCCTACACATCATACTTTCTATGATTACAGCACACATTTTACACCTATCCAAGACTCACTGACTCAAGAGTTAACCAAAACTCTCTACCACCGGAGCTGACCAATGCATGACAGCCTCTTGTGGGTGAATTTCTTTTAAATTCAACTCCACTGTGTAAGCAGTCATCCCTCTAATCCCATCTATGGCCCCCTTCCTGGATCTCTCCATCACGTCTTTCCTTTCACCCATCATGCTAGTCTTTCCCGTCCATGAGACACAGGCAACTAGCTGCGAACCTAAGGAAGAGCAGCATCCAATGAGGTGCGTGCCAGTGTTTGTTTTTTAGCACCAACCTCATATGTTGACAGAGCTCAGGAGTGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCTTTGCCCAGGGCTTCACACTCCTTGTCGGCTGCTATCTCCTCCATGCTGCAAATAGGATTCTCTCCTCCCTTGTGTCAATCTGGTAGGCTCTCAAACTCAGGAAAACTGAGTCCTTTAAGAAGCTGCTTAGTAGGAACTAAGGGTGACAAATACACCAACAAGAAGTCGCAGCTCTCCTATGACTGGCTCCTTTTTATACAACTACATTAGAAAAGGATGCAGTGTTACCTACAAAGGCTACCATGAGAAGAATTGCAAACCCTGTTAAAAACCACTCTCTGTGGCCATGCAGGCTTCTAGAAGAAACTCCACATATGTCCATCCGACTCCTTATTTTATGTTTCCATTGTGTGCATGGGTTTGTTAGTGGACG";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 84), (0, 62), (2, 3), (0, 550), (1, 1), (0, 7), (1, 2), (0, 865), (2, 1), (0, 104), (2, 1), (0, 691), (2, 1), (0, 994), (2, 1), (0, 1174), (1, 1), (0, 60), (1, 2), (0, 1069), (1, 2), (0, 71), (2, 1), (0, 357), (1, 3), (0, 219), (2, 1), (0, 235), (1, 2), (0, 428), (4, 60)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"GTGTTATGCTGATCTCCTTGGGGCAGAGACGTATTGCTACTTGCCTGTCGCTCTATCTTCGGCGTCTGCTTGGGTGTTTAACCTGTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTGCTGGCTTTCTTCTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTGAGTTTTTCCCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATCAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTTTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAACGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAACGCTAGACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGCTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTTGAAATTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGCGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCCAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCGACAGTGAATATCTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTAACTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATAGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAGAGAGAAATTTTACAGCTGGGCCACCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAAACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGTATGAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAATAAAAGATCACAAGGGAAGACAGGCAGAGCAAGATCACAAGGCCAGGGCGAAATTAGAATTATTGATGAGGTTCCATGTCCCACTGGGCACCCATTGTCACTGATAAACATCTTAACAGGAAACAGAGTTCCAGAGCAGACAACAGGTCCGACTAGAATTCGCCAGGCTGGAATTTCCTAATCCTAGCAAGCCTGAGGGCACTGCAGGAGACCAGGGCGTATTTCATCCCTTATCTTCAACTGCGTAAGACAGACACTCCCAGAGCGGCCATTTATAGACCTCTCCCTGGGAATGCATTCCTTTCCCAGGGTTATTCCTTGCTGGGAAAAGAATTCAGTGATACTTCTCCTACTTGCTTTCTCCAAGAAGAGAAATAAGACTCTGTTCTGCCCGGCCCCGCAGGCAGTCAGACCTTATAGTTATCTCCTTTGTTCCCTGAAAATCGCTGTTAGCCTGTTCTTTTCTAGGATGCCAAGATTTCATATTGTTCAAACACACATGTTTTACAAACAATTTGCAAAGATAATGCAATCATCACAGGACCCTGGGGCGACCTACCTTCTCAGCTTACGAAGATGACGGGATTAAGAGATTAAAGTAAATACAGGCATAGGAAATTATAAGAGTATTGACTAGGGAAGTGATAAATGTCCATGAAATCTTCATAATTTATGTTCAGAGACTGCAGTAAAGACAGGCAGAATAAATTATAAAAGTATTAATTTGGGGAACTAATAAATGTCCATGAAATCTTCATAACTTATGTTCTTCTGCCACAGCTTCAACCAGTCCCTCCACTCAGGGTCCCTGATTTCCAGCAACAAGTTGTCTTTCCTGCCCAAGAGTTACGGCCAAGTAGCTGCAGGCCTCAAGCAAGGGCTTCATCCCATGCGATGGTGGGTATGGGCCAAGGGAGTCAGCCTCACATTATGGCAGAGCTCTGGAGGGTGTGAAATATTTCCAGGGTTATTAGCTCTGTTCTCTTTTTTTGCTGGTGTTTCTCTGCTCTTTTGTCCACTCCTATCACTTTCTAACTCCAAGGAGGATTTGCTTCTCTCTTGGTCCACACTATACTGGAACAGTCTCATTCCCACAGAAAGGAAAGGTGACAATCACTCACATAATAAGCCTCCCTCTATGAGTTCTTCCCCATTTTCCTACCTCCATTCAAAAAGCAAAAGACTATTCCGTACAAAGCTCATCCGGAGAGGGCTCTGGAAATAGTTTCAAGGGAACTCCTTGTTCCCATGGATGCTCCTGACAGTGAATAGCCCAAATCGCAATCTCTCTAGTGTATATGAATGCATGGATGTGTGTGTTTGTATATGCATATATACATATACAATTGTATTCATGTCACATGTTGCAATGGACAGCCTATTAGAAGATACTTTCAGTTCACACACAGCGCAGTGCTAAAGAACTTAAAACGTTTATAGTAAGACAACATGTAACTTCCTACACTTAACATTTTCAGTTCAATGTAGTTTGCATTAATTTTCTTTTTCTTTATTTTTATTTTATTTTCGAGACAGAGTCCCACTCTGTCACCCACACTGGAGTCCGGTGGCATGATCTCGACTAACTACAAACTCTGCCTCCCAGGTTCAAGTAATTCTCCTGCCTCAGCCTCCCGAGTAGCAGGGACTACAGGCGCATGCCCCACGCCTGGCTAATTTTTGTATTGTTAGTAAAGACAGGGTTTCACCATGTTGATCAGGCTAGTCTTGAACTCCTGACCTCGTGATCCACCCGACTTGACCGCCCAAAGCACTGGGACCACAGGTATGAGCCAACGCGCCTGGCCTTCCAATTACTTTTCAGCCTGTAAACTGCTCTACCCTACAGAATTCACGTGTGTATTTGGTTCTTTGGAATCAGCCATGCCATTTACCCTACACATCATACTTTCTATGATTACAGCACACATTTTACACCTATCCAAGACTCACTGACTCAAGAGTTAACCAAAACTCTCTACCACCGGAGCTGACCAATGCATGACAGCCTCTTGTGGGTGAATTTCTTTTAAATTCAACTCCACTGTGTAAGCAGTCATCCCTCTAATCCCATCTATGGCCCCCTTCCTGGATCTCTCCATCACGTCTTTCCTTTCACCCATCATGCTAGTCTTTCCCATCCCCATGAGACACAGGCAACTAGCTGCGAACCTAAGGAAGAGCAGCATCCAATGAGGTGCGTGCCAGTGTTTGTTTTTAGCACCAACCTCATATGTTGACAGAGCTCAGGAGTGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCTTTGCCCAGGGCTTCACACTCCTTGTCGGCTGCTATCTCCTCCATGCTGCAAATAGGATTCTCTCCTCCCTTGTGTCAATCTGGTAGGCTCTCAAACTCAGGAAAACTGAGTCCTTTAAGAAGCTGCTTAGTAGGAACTAAGGGTGACAAATACACCAACAAGAAGTCGCAGCTCTCCTATGACTGGCTCCTTTTTATACAACTACATTAGAAAAGGATGCAGTGTTACCTACAAAGGCTCCCATGAGAAGAATTGAGAACCCTCTTAAAAACCACTCTCTGTGATGGCCATGCAGGCTTCTAGAAGAAACTCCACATATGTCCATCCGACTCCTTATTTTATGTTTCCATTGTGTGCATGGGTTTGTTAGTGGACGTGTCTTGAGAGAAGTGGGGATGGATGGATGGATGCATGGATGGATTGATGGATATATTTGTTTCTGTGCCTGTGTGTGTGTGACTACTTCTATGCACAGTTATGCATACGTGTGTCTTTATTTGTCTGATCAAGTTCACCTTTCATTGATACGTGCCTCTGGGTTTGCTGTTGTCAATTCATTCGTGGGTAGACTAGAGGATGATAAAAAGATACAGTCAGACAGGAGACACAATTTGAAATTGACTTTTCTGGAGAAAATAGCTTTCATTACTCTGCACATAAACCAGTGAAGTTCTGACTATGCATGTGTATAGCACTATTTGGTTCTGAAATGTCATTTTTTTTTCATTAGAAAGGAGGGTTTTTGTATGATGGCAACCAGCATTTTAGGCTGGGGAAAACAGTAATGCTTCCAGATTTAAATAAACTTCTTTATTATGACCTCAGTCCCAATGCGTGTCAGTCTTTGTGGTGGCAGTTATTTCAAACTCAACTTTTGTTTGGAGGTAGTCAGCCCTCTTATTTCATTGAATGTCCTGTTTCAAAGGCCCTCCATTACCTCCATACTTTCAACCAGCATGGCTGTCTCCCCAGCCCCACAAACAGAGTCAAGTCACTAGAGGTCTGACATAGGGGCACAATCCAATACGATGGTGACAGGAGGTTTGCAGCACCAGCCTCCATGGTGCCAGAGCTCTGGAGGCTGGAGAACACCTCCTAGGCTCTTAGCTTTGTTCCTTTCTGCTGAGGGCTTCACAGTTCCCTGAGTACTCATACCTTCTCCTTGCTCAGGTTAAACACCCAAGCAGACGCCGAAGATAGAGCGACAGGCAAGTGAGCAATACGTGGA";
        let md_tag = b"55A2A3^TTG5T838G281C58T236^A99T4^C0A49G293C230T115^C177G531A195C88^A23G1T1031T309T312T4T0C15T249C348G72^T315A15C0A6G236^A0A2A0T1A0C74G0A151A22C404";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"GTTTGCCTTTGGGACAGTAACCTCATGTGGTAATAGAGGAAGAGATCAGGGTTCTaCTaGCTTTGTTCTTtTCTGTTAAGAGCTCCTCAGCTCCCTGCCCACTGCTACCTCCTCCATGCTAAAAATAGGACTGTCTCCTCTCTTGGCTCACTCATATTTTCCTAAGGTTTTTGGCTACTGCTAAATAAGAGTGACCAATGCAATAGGAATAAGAAGCATTGCTCTTATGACCTGTTCCTTTTTTATCCAACTAGATCCAAAAAGGATAGAAAGGTAACTGCAGATACTAAAGTAATTAGACTTTAGAAAACAGTGTCTAAGACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTGCCAAATCTGAATCTCTCTTCTGTACATGTATGCTTTTGTGTGTGTGTGTCTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTAATTTATTTAGACCTCACTTTTTGTTGAAGGATGCTGCATGAGTTATTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAAAAAGGTACAGAGCCATATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAAAACAATTTTTGCACCAAAAGTAGTTTTTCCATAAAAATAAAATATCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACGTAAGTCCAGTAAGTAAATTCCAACAATTAACAAAGACTCATTTTTCAATGGCCAATTAATTAAACAATTAAATGACTATATAAAAATTAAACCTATCTACATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATGAATAGAGAAAATTTTTATTTACTGGCTTAAGAAATCAATACTACATATTCAACTGATGTACTACTGCTTAATgAGCAATTACAGAATTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATACATAGCACAAAGAACCTCTTTAAAAATTAGTCACTGATCAAACACAATCTTTTTATAGCTTCTGTAACATCTAATTAATTTTGTAAATTTTCTTGAGCATTACACTATAATTAACTTGCATCAGGTTTTCCCTGTAACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGCCTGTGAAAACCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTcTTTCCTGTCCATGAGACACAGGCAAGTAGCTGTGGGCCTAAAGAAGGGCTGCATCCAAtGAGCTGGTGGGTTTGGGGTTGGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACGGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCGCCTCCCCTGGGCCACTCTGGTAGGCTCTCGAACTCAGGAACTCTGAGTCCCTTAAAAAACTGCTTACTAAGAACTAAAGGTGATGAATACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATCAAAAAAGGATGCAAAGTTACCTACAAAtGCTACaACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGgTCCTGAGAGAAAATTCCACAAATGTCCATCTGTCTCCTTGTTTTAAGATTCAATTGTGTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGATAGATTTGTTTACATGCCTGTGCATGTGTATATACTTCTATGCACACTTACGCATACATGTGTCTTTATTTATTTATAACTCACTTTTCGTAGATACGTGCCTCCTTGGTTCCTCCTGTCAGTTCACTTGTGTGGGGACTAGAGGATGATAAAAGAATACAGTCAGACAGGTGACATAATTcGAAATTTACTTTTCGGGAAAAACTAGCTTTCATTACTCTAAACATAAACTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAATCAGGATTTATATGATGGCTACCACCATCTTAGGATGGAGAAAATAGTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTCATCCCAATGTGTGACAGCCTTTGTGGtGGCAGTTATTTCAAACTCAACTCCAATTTTTAGGTAGTCAGCCCTCTTAATCCATTGAATGTCCTGCTTCAAAAATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGTCTCCCCCAGCCCCAAAAAATGCATCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTGGTGATATAAGGTCCGGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCTTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTACCCgAGTACTCCTACCTCTTCCTTGCTCCAAATAGGATTTTCTCTTCCCTCGGCCCACTCTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAAAGACAGTAACAGCCGGATACTCTGTTCCTCTTTTATCCAAAACAATTGTAAAAATGTGGATGTTTAACTATAGGTGCTACAGTGAGAAGATGAACTGTGTCCAAGACACTCTGTATTGCCATGGACACTCCTGAGAGCAAATCCCACTAATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACTTCCACCAGGTCAACAGGATGGGGACTGTGGGAAAAACATGATCAGAGTCAGACAGGACACATTATTCTAATGTTTTGAATCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTATTGAACTGGGATGAAATTTTTTCTTATTAGAAATGAGGTTTTATGTGCCaACAGTGAATATCTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTAACTGCATCTTCACCTCAATGTGTAACAACCTTTTTCGAGGGCAGTTAATTCCAGTTCAACTTTAGTTTTAAGTCCCCTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATcGGGGGAACCTGCCCCCAGTATGTCAACATAGATTCTTTTCTATTTTCCCTAAGTGTCAGCCAGTCTGAGAAATAAAGAGAAAGAGTACAAAGAGAGAAATTTTACAGCTGGGgCtCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAAAACCAGCAAGTTTTTATTAGGGATTTCAAAAGGGGAGGGGGGTATGAACAGGGAGTAAGTTACAAGATCACATGCTTGAAAGGCCAATAAAAGATCACAAGGGAAGACAGGCAGAGCAAGATCACAAGGCCAGGGCGAAATTAGAATTATTGATGAGGTTCCATGTCCCACTGGGCACCCATTGTCACTGATAAACATCTTAACAGGAAACAGAGTTCCAGAGCAGACAACAGGTCCGACTAGAATTCGCCAGGCTGGAATTTCCTAATCCTAGCAAGCCTGAGGGCACTGCAGGAGACCAGGGCGTATTTCATCCCTTATCTTCAACTGCGTAAGACAGACACTCCCAGAGCGGCCATTTATAGACCTCTCCCTGGGAATGCATTCCTTTCCCAGGGTTATTCCTTGCTGGGAAAAGAATTCAGTGATACTTCTCCTACTTGCTTTCTCCAAGAAGAGAAATAAGACTCTGTTCTGCCCGGCCCCGCAGGCAGTCAGACCTTATAGTTATCTCCTTTGTTCCCTGAAAATCGCTGTTAGCCTGTTCTTTTCTAGGATGCCAAGATTTCATATTGTTCAAACACACATGTTTTACAAACAATTTGCAAAGATAATGCAATCATCACAGGACCCTGGGGCGACCTACCTTCTCAGCTTACGAAGATGACGGGATTAAGAGATTAAAGTAAATACAGGCATAGGAAATTATAAGAGTATTGACTAGGGAAGTGATAAATGTCCATGAAATCTTCATAATTTATGTTCAGAGACTGCAGTAAAGACAGGCAGAATAAATTATAAAAGTATTAATTTGGGGAACTAATAAATGTCCATGAAATCTTCATAACTTATGTTCTTCTGCCACAGCTTCAACCAGTCCCTCCACTCAGGGTCCCTGATTTCCAGCAACAAGTTGTCTTTCCTGCCCAAGAGTTACGGCCAAGTAGCTGCAGGCCTCAAGCAAGGGCTTCATCCCATGtGATGGTGGGTATGGGCCAAGGGAGTCAGCCTCACATTATGGCAGAGCTCTGGAGGGTGTGAAATATTTCCAGGGTTATTAGCTCTGTTCTCTTTTTTTGCTGGTGTTTCTCTGCTCTTTGTCCACTCCTATCACTTTCTAACTCCAAGGAGGATTTGCTTCTCTCTTGGTCCACACTACTGGAACAGTCTCATTCCCACAGAAAGGAAAGGTGACAATCACTCACATAATAAGCCTCCCTCTATGAGTTCTTCCCCATTTTCCTACCTCCATTCAAAAAGCAAAAGACTATTCCGTACAAAGCTCATCCtGAGAGGGCTCTGGAAATAGTTTCAAGGGAACTCCTTGTTCCCATGGATGCTCCTGACAGTGAATAGCCCAAATCGCAATCTCTCTAGTGTATATGAATGCATGGATGTGTGTGTTTGTATATGCATATATACATATACAATTGTATTCATGTCACATGTTGCAATGGACAGCCTATTAGAAGATACTTTCAGTTCACACACAGCGCAGTGCTAAAGAACTTAAAACGTTTATAGTAAGACAACATGTAACTTCCTACACTTAACATTTTCAGTTCAATGTAGTTTGCATTAATTTTCTTTTTCTTTATTTTTtTTTTtcTTTCGAGACAGAGTCtCACTCTGTCACCCACACTGGAGTCCGGTGGCATGATCTCGACTAACTACAAACTCTGCCTCCCAGGTTCAAGTAATTCTCCTGCCTCAGCCTCCCGAGTAGCAGGGACTACAGGCGCATGCCCCACGCCTGGCTAATTTTTGTATTGTTAGTAAAGACAGGGTTTCACCATGTTGATCAGGCTAGTCTTGAACTCCTGACCTCGTGATCCACCCGACTTGACCGCCCAAAGCACTGGGACCACAGGTATcAGCCAACGCGCCTGGCCTTCCAATTACTTTTCAGCCTGTAAACTGCTCTACCCTACAGAATTCACGTGTGTATTTGGTTCTTTGGAATCAGCCATGCCATTTACCCTACACATCATACTTTCTATGATTACAGCACACATTTTACACCTATCCAAGACTCACTGACTCAAGAGTTAACCAAAACTCTCTACCACCGGAGCTGACCAATGCATGACAGCCTCTTGTGGGTGAATTTCTTTTAAATTCAACTCCACTGTGTAAGCAGTCATCCCTCTAATCCCATCTATGGCCCCCTTCCTGGATCTCTCCATCACGTCTTTCCTTTCACCCATCATGCTAGTCTTTCCCgTCCATGAGACACAGGCAACTAGCTGCGAACCTAAGGAAGAGCAGCATCCAATGAGGTGCGTGCCAGTGTTTGTTTTTTAGCACCAACCTCATATGTTGACAGAGCTCAGGAGTGTGAGGGACAAGTCCATGTTCCATTAGCTCCATTCTTTGCCCAGGGCTTCACACTCCTTGTCGGCTGCTATCTCCTCCATGCTGCAAATAGGATTCTCTCCTCCCTTGTGTCAATCTGGTAGGCTCTCAAACTCAGGAAAACTGAGTCCTTTAAGAAGCTGCTTAGTAGGAACTAAGGGTGACAAATACACCAACAAGAAGTCGCAGCTCTCCTATGACTGGCTCCTTTTTATACAACTACATTAGAAAAGGATGCAGTGTTACCTACAAAGGCTaCCATGAGAAGAATTGcaAACCCTgTTAAAAACCACTCTCTGTGGCCATGCAGGCTTCTAGAAGAAACTCCACATATGTCCATCCGACTCCTTATTTTATGTTTCCATTGTGTGCATGGGTTTGTTAGTGGACGTGTCTTGAGAGAAGTGGGGATGGATGGATGGATGCATGGATGGATTGATGGATATATTTGTTTCTGTGCCTGTGTGTGTGTGACTACTTCTATGCACAGTTATGCATACGTGTGTCTTTATTTGTCTAaATatAacTCACCTTTCATTGATACGTGCCTCTGGGTTTGCTGTTGTCAATTCATTCGTGGGTAGACTAGAGGATGATAAAAgaATACAGTCAGACAGGAGACACAATTTGAAATTGACTTTTCTGGAGAAAATAGCTTTCATTACTCTGCACATAAACCAGTGAAGTTCTGACTATGCATGTGTATAGCACTATTTGGTTCTGAAATGTCATTTTTTTTTCATTAGAAAGGAGGaTTTGTATGATGGCAACCAGCATcTTAGGCTGGGGAAAACAGTAATGCTTCCAGATTTAAATAAACTTCTTTATTATGACCTCAGTCCCAATGCGTGTCAGTCTTTGTGGTGGCAGTTATTTCAAACTCAACTTTTGTTTGGAGGTAGTCAGCCCTCTTATTTCATTGAATGTCCTGTTTCAAAGGCCCTCCATTACCTCCATACTTTCAACCAGCATGGCTGTCTCCCCAGCCCCACAAACAGAGTCAAGTCACTAGAGGTCTGACATAGGGGCACAATCCAATACGATGGTGACAGGAGGTTTGCAGCACCAGCCTCCATGGTGCCAGAGCTCTGGAGGCTGGAGAACACCTCCTAGGCTCTTAGCTTTGTTCCTTTCTGCTGAGGGCTTCACAGTTCCCTGAGTACTCATACCTTCTCCTTGCTC";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 80), (0, 17), (1, 1), (0, 5), (1, 1), (0, 7), (2, 1), (0, 9), (2, 1), (0, 16), (2, 3), (0, 24), (1, 1), (0, 3), (1, 1), (0, 27), (1, 2), (0, 4), (2, 2), (0, 27), (2, 1), (0, 6), (1, 2), (0, 1), (2, 1), (0, 7), (2, 2), (0, 27), (1, 1), (0, 32), (2, 1), (0, 7), (2, 1), (0, 3), (2, 1), (0, 8), (1, 1), (0, 12), (2, 1), (0, 12), (1, 1), (0, 7), (1, 1), (0, 3), (2, 2), (0, 1), (2, 1), (0, 7), (2, 2), (0, 31), (1, 1), (0, 5), (1, 1), (0, 51), (1, 3), (0, 2), (2, 4), (0, 26), (2, 1), (0, 3), (1, 1), (0, 2), (1, 1), (0, 12), (1, 1), (0, 9), (1, 1), (0, 20), (2, 1), (0, 25), (1, 1), (0, 7), (2, 1), (0, 30), (1, 1), (0, 10), (2, 3), (0, 4), (2, 1), (0, 9), (2, 1), (0, 32), (1, 1), (0, 18), (2, 1), (0, 45), (1, 1), (0, 10), (1, 1), (0, 5), (2, 1), (0, 3), (2, 1), (0, 4), (1, 2), (0, 17), (1, 1), (0, 16), (2, 2), (0, 8), (2, 1), (0, 18), (2, 1), (0, 20), (1, 2), (0, 12), (2, 2), (0, 2), (1, 3), (0, 9), (2, 3), (0, 25), (1, 4), (0, 11), (2, 1), (0, 9), (1, 1), (0, 7), (1, 1), (0, 6), (2, 1), (0, 7), (2, 1), (0, 61), (1, 4), (0, 12), (2, 1), (0, 24), (1, 1), (0, 15), (1, 5), (0, 47), (1, 1), (0, 27), (1, 3), (0, 4), (2, 1), (0, 43), (1, 3), (0, 26), (2, 1), (0, 10), (2, 2), (0, 6), (2, 1), (0, 11), (2, 1), (0, 17), (2, 4), (0, 15), (2, 1), (0, 31), (1, 1), (0, 2), (2, 2), (0, 2), (2, 3), (0, 7), (2, 1), (0, 23), (1, 1), (0, 12), (2, 1), (0, 4), (1, 1), (0, 8), (1, 2), (0, 12), (1, 1), (0, 4), (2, 2), (0, 21), (1, 2), (0, 22), (1, 1), (0, 6), (2, 4), (0, 8), (1, 2), (0, 6), (1, 1), (0, 3), (2, 2), (0, 14), (1, 1), (0, 41), (2, 5), (0, 3), (2, 1), (0, 15), (1, 2), (0, 80), (1, 1), (0, 6), (1, 1), (0, 5), (1, 1), (0, 12), (2, 1), (0, 32), (2, 1), (0, 6), (1, 1), (0, 5), (1, 2), (0, 1), (1, 2), (0, 3), (1, 1), (0, 1), (1, 2), (0, 5), (2, 1), (0, 13), (2, 1), (0, 15), (2, 2), (0, 7), (2, 2), (0, 20), (2, 1), (0, 2), (1, 2), (0, 2), (1, 2), (0, 1), (1, 1), (0, 3), (1, 2), (0, 12), (1, 1), (0, 13), (1, 1), (0, 5), (2, 4), (0, 24), (1, 3), (0, 13), (1, 2), (0, 3), (2, 1), (0, 36), (2, 3), (0, 11), (2, 1), (0, 18), (1, 1), (0, 54), (2, 2), (0, 1), (1, 1), (0, 41), (1, 1), (0, 17), (1, 1), (0, 26), (2, 2), (0, 12), (1, 1), (0, 97), (1, 2), (0, 3), (1, 1), (0, 2), (1, 1), (0, 7), (2, 1), (0, 12), (2, 1), (0, 3), (2, 1), (0, 24), (1, 4), (0, 2), (1, 1), (0, 32), (1, 2), (0, 12), (2, 1), (0, 15), (2, 1), (0, 2), (2, 1), (0, 8), (2, 1), (0, 8), (1, 2), (0, 24), (2, 2), (0, 6), (2, 2), (0, 26), (1, 1), (0, 1), (1, 1), (0, 5), (1, 1), (0, 19), (2, 2), (0, 7), (2, 1), (0, 46), (1, 2), (0, 1), (2, 1), (0, 10), (1, 1), (0, 9), (2, 1), (0, 6), (2, 1), (0, 2), (2, 1), (0, 11), (2, 1), (0, 19), (1, 1), (0, 4), (2, 1), (0, 11), (2, 2), (0, 2), (2, 1), (0, 100), (2, 1), (0, 11), (2, 1), (0, 40), (2, 1), (0, 11), (2, 1), (0, 7), (1, 1), (0, 56), (1, 1), (0, 5), (1, 2), (0, 8), (2, 2), (0, 8), (2, 3), (0, 4), (1, 1), (0, 3), (2, 1), (0, 2), (2, 1), (0, 38), (1, 1), (0, 4), (1, 1), (0, 1), (2, 2), (0, 2), (2, 1), (0, 21), (1, 1), (0, 41), (2, 1), (0, 17), (1, 1), (0, 2), (1, 1), (0, 4), (2, 3), (0, 25), (2, 1), (0, 33), (2, 1), (0, 6), (2, 1), (0, 37), (1, 1), (0, 29), (2, 1), (0, 8), (2, 3), (0, 19), (1, 2), (0, 55), (1, 7), (0, 7), (2, 1), (0, 32), (1, 1), (0, 8), (1, 1), (0, 5), (1, 1), (0, 8), (1, 1), (0, 4), (1, 5), (0, 50), (1, 1), (0, 2), (2, 1), (0, 13), (2, 2), (0, 6), (1, 3), (0, 18), (1, 1), (0, 39), (2, 1), (0, 12), (2, 1), (0, 19), (2, 1), (0, 3), (1, 1), (0, 5), (2, 2), (0, 42), (1, 1), (0, 1), (2, 2), (0, 13), (2, 1), (0, 13), (1, 2), (0, 18), (1, 1), (0, 5), (1, 2), (0, 2), (2, 1), (0, 17), (1, 1), (0, 23), (1, 1), (0, 51), (1, 1), (0, 6), (2, 1), (0, 13), (1, 1), (0, 30), (2, 1), (0, 6), (2, 1), (0, 5), (2, 2), (0, 4), (2, 1), (0, 6), (1, 1), (0, 4), (2, 1), (0, 18), (1, 1), (0, 13), (1, 1), (0, 20), (2, 1), (0, 6), (2, 1), (0, 13), (1, 1), (0, 4), (2, 1), (0, 11), (2, 1), (0, 8), (2, 1), (0, 4), (1, 1), (0, 10), (1, 1), (0, 9), (1, 1), (0, 10), (2, 1), (0, 5), (1, 1), (0, 12), (1, 1), (0, 6), (2, 1), (0, 5), (2, 1), (0, 19), (2, 1), (0, 21), (1, 1), (0, 17), (2, 1), (0, 10), (1, 1), (0, 5), (2, 1), (0, 36), (1, 1), (0, 21), (1, 1), (0, 12), (1, 1), (0, 10), (1, 1), (0, 39), (2, 1), (0, 7), (2, 1), (0, 1), (2, 1), (0, 20), (2, 1), (0, 19), (1, 5), (0, 17), (2, 1), (0, 33), (1, 1), (0, 14), (1, 2), (0, 5), (1, 1), (0, 7), (2, 2), (0, 2), (1, 1), (0, 80), (2, 2), (0, 26), (1, 1), (0, 15), (2, 3), (0, 14), (2, 3), (0, 14), (2, 1), (0, 3), (1, 1), (0, 5), (2, 3), (0, 4), (2, 2), (0, 11), (2, 1), (0, 25), (2, 1), (0, 81), (2, 2), (0, 43), (1, 1), (0, 88), (2, 1), (0, 3), (1, 1), (0, 13), (1, 5), (0, 4), (1, 1), (0, 24), (1, 1), (0, 42), (1, 4), (0, 10), (2, 2), (0, 2), (1, 1), (0, 8), (1, 1), (0, 14), (1, 3), (0, 5), (1, 1), (0, 7), (1, 1), (0, 10), (1, 1), (0, 21), (2, 1), (0, 3), (2, 1), (0, 45), (2, 1), (0, 197), (1, 1), (0, 30), (1, 4), (0, 38), (1, 1), (0, 24), (2, 2), (0, 17), (2, 1), (0, 4), (2, 2), (0, 43), (1, 4), (0, 26), (2, 1), (0, 13), (2, 1), (0, 3), (1, 2), (0, 41), (1, 1), (0, 5), (1, 2), (0, 42), (2, 2), (0, 3), (2, 1), (0, 5), (2, 3), (0, 9), (1, 1), (0, 31), (1, 1), (0, 9), (1, 1), (0, 5), (2, 1), (0, 10), (1, 1), (0, 37), (1, 3), (0, 1), (1, 1), (0, 4), (1, 2), (0, 12), (1, 1), (0, 7), (1, 1), (0, 18), (1, 1), (0, 14), (2, 1), (0, 34), (2, 1), (0, 4), (1, 1), (0, 4), (1, 1), (0, 7), (2, 2), (0, 7), (2, 1), (0, 97), (2, 1), (0, 3), (2, 1), (0, 1), (2, 1), (0, 3), (2, 2), (0, 22), (1, 1), (0, 3), (1, 2), (0, 3), (2, 2), (0, 9), (1, 1), (0, 29), (1, 1), (0, 4), (1, 1), (0, 15), (2, 1), (0, 31), (1, 3), (0, 16), (1, 1), (0, 2), (2, 2), (0, 6), (2, 2), (0, 9), (1, 2), (0, 16), (1, 3), (0, 12), (1, 1), (0, 20), (1, 3), (0, 11), (2, 1), (0, 21), (1, 1), (0, 22), (1, 1), (0, 7), (1, 1), (0, 9), (1, 1), (0, 2), (2, 1), (0, 3), (2, 2), (0, 114), (1, 3), (0, 6), (1, 1), (0, 25), (1, 2), (0, 30), (1, 1), (0, 5), (2, 1), (0, 12), (2, 1), (0, 18), (2, 1), (0, 15), (2, 2), (0, 1), (2, 1), (0, 8), (2, 2), (0, 34), (1, 1), (0, 6), (1, 1), (0, 2), (1, 1), (0, 4), (1, 2), (0, 36), (2, 1), (0, 23), (2, 1), (0, 18), (1, 2), (0, 6), (1, 2), (0, 3), (1, 1), (0, 9), (2, 1), (0, 4), (1, 2), (0, 3), (1, 1), (0, 23), (2, 1), (0, 4), (2, 2), (0, 17), (1, 3), (0, 4), (2, 1), (0, 4), (1, 1), (0, 24), (2, 1), (0, 6), (1, 1), (0, 17), (1, 3), (0, 58), (2, 1), (0, 5), (1, 1), (0, 19), (1, 1), (0, 9), (1, 2), (0, 5), (1, 1), (0, 3), (2, 1), (0, 39), (1, 1), (0, 24), (2, 1), (0, 50), (1, 1), (0, 14), (1, 1), (0, 36), (1, 1), (0, 27), (1, 2), (0, 4), (2, 1), (0, 13), (1, 1), (0, 29), (2, 1), (0, 23), (2, 1), (0, 4), (1, 1), (0, 3), (1, 1), (0, 10), (2, 1), (0, 2), (1, 1), (0, 2), (1, 2), (0, 4), (2, 1), (0, 13), (1, 3), (0, 27), (1, 1), (0, 4), (2, 2), (0, 32), (2, 2), (0, 6), (1, 1), (0, 2), (1, 2), (0, 45), (1, 1), (0, 4), (1, 2), (0, 7), (2, 1), (0, 19), (1, 1), (0, 4), (1, 1), (0, 32), (2, 1), (0, 8), (1, 1), (0, 191), (2, 1), (0, 31), (2, 1), (0, 74), (1, 2), (0, 3), (1, 1), (0, 15), (1, 1), (0, 33), (2, 1), (0, 12), (2, 2), (0, 2), (2, 1), (0, 13), (2, 4), (0, 16), (2, 2), (0, 6), (1, 4), (0, 2), (1, 2), (0, 13), (2, 1), (0, 8), (1, 1), (0, 5), (1, 2), (0, 1), (2, 1), (0, 10), (2, 4), (0, 56), (1, 1), (0, 7), (2, 1), (0, 21), (1, 2), (0, 8), (2, 1), (0, 8), (2, 1), (0, 3), (1, 1), (0, 1), (1, 1), (0, 6), (2, 1), (0, 11), (1, 1), (0, 4), (1, 1), (0, 4), (1, 1), (0, 21), (2, 2), (0, 4), (2, 1), (0, 58), (2, 1), (0, 14), (2, 1), (0, 15), (1, 1), (0, 65), (2, 1), (0, 3), (1, 1), (0, 5), (1, 1), (0, 4), (1, 1), (0, 11), (2, 2), (0, 4), (1, 1), (0, 2), (1, 1), (0, 31), (2, 2), (0, 2), (1, 1), (0, 5), (2, 1), (0, 34), (2, 1), (0, 6), (2, 2), (0, 20), (2, 1), (0, 72), (1, 1), (0, 6), (1, 1), (0, 43), (1, 3), (0, 23), (2, 1), (0, 7), (1, 1), (0, 6), (2, 1), (0, 39), (1, 1), (0, 5), (2, 2), (0, 77), (2, 1), (0, 3), (1, 2), (0, 10), (2, 2), (0, 33), (2, 2), (0, 74), (1, 1), (0, 2), (1, 1), (0, 32), (1, 1), (0, 7), (1, 2), (0, 7), (2, 1), (0, 4), (1, 1), (0, 8), (1, 1), (0, 9), (1, 1), (0, 2), (2, 1), (0, 13), (1, 1), (0, 4), (2, 1), (0, 4), (2, 1), (0, 10), (2, 1), (0, 15), (2, 13), (0, 18), (1, 1), (0, 82), (1, 1), (0, 227), (1, 1), (0, 6), (1, 1), (0, 11), (2, 2), (0, 18), (1, 1), (0, 5), (1, 1), (0, 20), (1, 2), (0, 58), (1, 1), (0, 4), (1, 3), (0, 3), (1, 1), (0, 27), (1, 1), (0, 1), (2, 2), (0, 14), (2, 1), (0, 42), (1, 1), (0, 10), (1, 1), (0, 44), (2, 1), (0, 19), (1, 1), (0, 24), (1, 2), (0, 40), (1, 1), (0, 6), (2, 1), (0, 6), (1, 1), (0, 5), (1, 1), (0, 43), (2, 1), (0, 40), (2, 3), (0, 4), (1, 1), (0, 4), (1, 3), (0, 82), (2, 1), (0, 4), (2, 2), (0, 1), (2, 1), (0, 10), (1, 1), (0, 6), (2, 1), (0, 4), (1, 1), (0, 2), (2, 2), (0, 18), (2, 1), (0, 13), (1, 1), (0, 37), (1, 2), (0, 1), (2, 1), (0, 8), (1, 1), (0, 11), (1, 1), (0, 5), (2, 1), (0, 17), (2, 2), (0, 41), (2, 1), (0, 2), (2, 1), (0, 4), (1, 1), (0, 4), (1, 1), (0, 71), (1, 1), (0, 8), (2, 6), (0, 30), (2, 1), (0, 34), (2, 1), (0, 28), (2, 1), (0, 6), (2, 1), (0, 17), (2, 1), (0, 16), (1, 3), (0, 41), (2, 1), (0, 2), (2, 1), (0, 5), (1, 3), (0, 16), (1, 2), (0, 7), (1, 1), (0, 16), (2, 1), (0, 9), (2, 2), (0, 30), (2, 1), (0, 26), (2, 1), (0, 3), (2, 3), (0, 17), (2, 6), (0, 17), (2, 2), (0, 19), (1, 2), (0, 3), (1, 1), (0, 24), (1, 1), (0, 8), (2, 2), (0, 8), (2, 1), (0, 17), (1, 2), (0, 14), (2, 1), (0, 5), (2, 1), (0, 12), (2, 1), (0, 28), (1, 1), (0, 8), (2, 3), (0, 7), (1, 3), (0, 2), (1, 1), (0, 2), (2, 1), (0, 29), (1, 1), (0, 22), (1, 1), (0, 3), (2, 1), (0, 6), (1, 1), (0, 5), (2, 1), (0, 16), (1, 1), (0, 31), (1, 1), (0, 21), (2, 1), (0, 3), (1, 1), (0, 36), (1, 1), (0, 3), (1, 1), (0, 10), (2, 2), (0, 8), (2, 2), (0, 14), (2, 1), (0, 3), (2, 1), (0, 4), (2, 1), (0, 8), (2, 1), (0, 4), (2, 1), (0, 1), (1, 3), (0, 11), (2, 1), (0, 23), (1, 2), (0, 5), (1, 1), (0, 6), (1, 1), (0, 11), (1, 1), (0, 4), (2, 1), (0, 13), (2, 1), (0, 2), (2, 1), (0, 10), (1, 1), (0, 23), (2, 1), (0, 13), (2, 1), (0, 1), (2, 2), (0, 4), (2, 2), (0, 1), (2, 1), (0, 19), (1, 1), (0, 18), (2, 1), (0, 39), (1, 1), (0, 38), (2, 1), (0, 15), (1, 1), (0, 9), (1, 2), (0, 30), (2, 2), (0, 37), (2, 1), (0, 82), (2, 1), (0, 231), (2, 1), (0, 8), (2, 1), (0, 12), (2, 1), (0, 19), (2, 1), (0, 60), (2, 1), (0, 4), (1, 1), (0, 24), (2, 2), (0, 34), (1, 2), (0, 4), (2, 1), (0, 18), (2, 1), (0, 20), (2, 2), (0, 16), (1, 3), (0, 2), (2, 1), (0, 2), (1, 1), (0, 18), (2, 2), (0, 3), (2, 1), (0, 41), (1, 1), (0, 12), (2, 1), (0, 35), (2, 1), (0, 25), (4, 61)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"ATGTGTCTGGTTACAGTTGGTGCTTGCTACTTGCCTGTCACTATTTGTTCAGCATACTGCTTCAGATCTAACCTTTTGAGACAGTGTCTACAACACTACTGTACTTCTTATGGTGCTCCAACAATGAACTTACCAATAAATATCTCTTCTGTACATGTATATCTATTTATATATATGTGTGTATGTGTATATTATTACATATATATGTTTACATCTAGCTTTTGATTATTTGCAACCTAGCTTTCTTGAAGGATGCTGCATGAGTTTTCTGTCATTTCACTGAAGGCAGTGCTCATGGAGTGAGAGGTTAGGCAGTATAGAACACCACATGATTTCTCTTTTTTCTGACACTGAGTCAGAACAATTTGCACAGAAGGTGACTTTTCCATAGAAATCAGAGTGCTCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACATACATAGGTAAGTAAATTCCAACAATTGACAAAACTACATTTTTTAAATAGCAAAATTGATTAAAACAATTAAATGACTATATAAAATTAAACCTGTGTTCATATTAGAAAATATATTTAATTCACCCATGCCTCCACTGTACTATCAAAATAAATAAAAATATTATTTATCACTTAAGAAATCAATACTACATATTCAACTAAATGTTTTACTGCTTAATAACGATCCCAGATTTCACAATCTTGATGTTTCCTTGCAAAAGAAGTTGGTAATATAGTAGCAAAAAACCTGTTTTTAAAAATAATTTAGCTAGTAGAACACAATCTTTATAGCTCTGTAACATCTGATTGATTTATAAATCTCTTTGAGCATCTTGCTATATAATACATTTTGCATAAGTTCCCTCTGACTTACATTTTTTATGAATTATTTATTTCCATAGAAGTCTGGAACATGTGCTGTGAAGCCATCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAGATCTTCAGCTGTGGGCCTGAGGAGGGCTGCATCCAATGAGCTGGGTGGGTTTGGGGTTCCTTCCGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACAGTGAGAGGACGAGTCCATGTTCCATTAGCTCCAGACTTCCTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTGCCCCATGTTGCAAATAGGATTCTCACCTCCCTGGGCCACTAGTAGCTCTAGAACTAGGAACTATGAGTCCCTAAACTGCTTACTAAGACTGAAGGTGATGATTACACTAACAGGAAGACAGACCTATGACTGCCCCTCTTTATATAACTCCATAGAAAGAAATAAGAAAGTTTGCCTACAGAAATGCTACAACAGAGAAACTGGAAACCCTGTTAAAAAAGACACTGTATGGCCATGCAGGCTACCTGAGAATTCCACAGAAATATCCCATGTCTCCTTGTTTTAAAGATTGGATTATTTGCATGGGTTTGTTTGTGTACGTGTGTGAATGTGTATAGATGGATGGGGACAAATTTTTTTACATGCCTGTGCATGTGTATCTATTTATATGCACACTTACGCATACATGTGTCTTTATTTATTTATCCATTTAGCTTTTGAGTAGATACGTGTTTGTTGGTTCCTCCTGTCAGTTCACTTGTGTGCAACTAAGAGGGAATTTGAGTCCAAAAGATTGAGTCAGACAGTGACATCATTCGAAGTACTTTAGGAAAAACTAGCTTTCATTCTTGCTTGAGAACGTCTGAGTTGGTTAGATTTCTGACTATTACATGTAGCACTATGTGGACCTGAAATATCATTATTTTTTATCAGAGGAGTTGGATTTATATAATGGCTTCCACCATCTCAGGATGGAAGTCATAATGTTCCAGATTTAAGTAAACACTTTTCATTACAACCTAGTCCCAACATGTGACAGCACTTGTGGTGGCAGTTATTAGAACTCGACTCCAATTTTTGGGTCATAAGCACTCTTAATCCAGTTGAATGTCCTACTTCAGAAGATCCTCCATTACCTTGTTACTTTCCCAGCATGGGCAGCCTTCCCCAGCCCCAAAAAATGCACCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTCGTAATATGAGGTTCCGAGCACCAGCCTCCCCAGTGAGTGGCAGGTTCTGGAGGCTGAGACACCTCCAAGTCTCCTAGCTCTGCTTTTTCCCTTTCTGCTGAGGACTTCACAGTTCCCAGAGACTACTCCTACCTCTCCTTGCTCCAATTAGATTTCTCTTCCTCAGCCAGCTCTTTAGTATGTCTCATACACAGAATGAGTTCAAGGCATTTGGCTGCTACTAACAGAGATGGTGGAAGAGACAGTGACAGCAGTACTCTGTCCTCTTTATTCCAAAACAATTGTGAAGATGTCAATGTCTGACTATCCAGTGCTCAAGTGGTGGAGGTGACTGTGCCAGACACTCTGTGTCCCATGGACACTCCTGAGGAGCAATCCCACTCATGATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACCTCCACCAGGTCAACAAGATAGGACTGTAGGAAGACATGATCAGAGTCAGACAGGACACATTCTTCTAATGTTTGGTTCAAAAGCTACACCTAACTCTATGAGTGATTCCTTGCATACACTTGTTGAACTAGGATGAAAATTTTTCTTTATTTGAGAAATGATTTTATGTAACAGGTGATTGTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTGAGCTAGTCTCACCTCAATGTGTGACAACCTTTTTTCAAGGGCAGTTAATTCCAGTTCATTTTCAGACTTCATCCCTTTCTTATAATTTGCATTCTGACCTCATTCTCTATCACCTCCTCAGTTTCACCCATCCTAGTTGTTATTAGGGAGACCTACACCATATGTCAACATAGATTTCTTTCTATTTTCCCTTAGTGTTGAGCCAGTCTGAGAAATAAAGGGGGAGACACAAAGAAAAATTTACAGCTGGGCCTGCCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAGAACCCAGGGGCAGAAAGTATTATTGCAGATTTCAAAAGGAGAGAGGGGTGCTGAACAGAGGAGTAAAGTTACAGAGATGTAGGCAGTTGTTTGAAAGGCCAATAAAGAATCACCAAGGAAGACAGGCAGAGCAGAGTAGCAAGGCCACGGAGGTTCAGAGAATTATTGATGAGGTTACCATGTCCCACTGGGCGCCCATTGTCACTGATAAACATCCATCCAGAAACAAGTTCCAGAGCAGACAACAGTCAGGACTCATTGGCCAGGCTGGAATTTCCTAATCCTAGCAAGCCTGAGGAGCTCCAGGAGACCAGGCGTATTCCATCTTGCTTATCTTCAACTGCGTGAGGACCAAGCACTTGCAGAGCAGCCATTTTCTGGGCACCTCCCTGGGAATACCATTCCTTTTCCAGGGTTGTTCCTTGCTGGGAGAAGAATTCAGTGATTCTCTCTCCTCTTGCTTTCTCCAGAGAAGAGAAATAAGACTCTGTTCTGCCCAGCCCGCAGCAGTGACCTATGGTCTGTCCCTTGATCCCCTGGGAGTACGCTGTTAGCCTGTTTCTTTTCTAGGATGCCAAGGTTCATCTGTTAGAACACAGCATGCCTACAAACAATTCCAAAGTGGTCCCAATGATCACCAGGACCCTCAGGACAACCTGCTTCTTCAGCTTAGGAAGGATGACAGATTAGAGATTAAAGTAGATGCAGCATCAGAAATTATGAGATTATTTGACTAGGAAAATGATGATGCCCATGAAAATCTCATCATTTATGTTCAGAGACTGCAGTAAAGACAGGACAGAAATAATTATAAAAGTCTCTAATTTTGAGAAGCTGAACAATGTTCCATGAAATCTTCATAACTTATGTTCTTCTGCCACAGCTGAACCATCCTCCACTCAGGGTCCCTGATTCCAGCAACAAGTTTTCTGCTACTTCCTGCCCAAGACTTAGGCCAAGTAGCTGCAGGCCTCAAGCAAGGGCTTGGAGTCCATGTGATGGAGTGGGTTGTGGGCAGAGGAGTCAGCCTAGCATTATGGCAGAGCTCTGGAGGGTGTGAAATATGTGCAGGGTTGTTGGCTCTGCTCTCTTTTTGTCCGTGTTTCTCTGCTCTTTGTCCACTCCTTCTCACTTTCTGACTAGGAGGATTTGCTTTCTTGGTCCACACTCTGTGAACATGGTCCACAGAAAGGAAGGTGACAATCACTAGCATAATAACCTGCCTCTATACGTACTTCCCCATTTTCCTACCTCCATTCAAAAAGCAAAAAACTATCCATTACAAAGCTCATCCTAAAAGCTATGGAAATAGTTTCAAGGGAACTCCTTGTTCCCATGGATGCCTCCTGACAGTGGATGGCCCAAATCACAATCTCTCTAGTGTATATCGATGCATGGATGTGTGTGTTTGTATATGCATATATACATATAAATATATATTCATGTTAACCTTCATGTTTGCAATGGACAGCCTGTTCAAAGAATCCTTTCAGTTCACACACAGCACCATGCTAAAGAACTTAAACATTACATTTCTAGAGAACAACATGTTGACTTCCTACACTTACTAACATTTTTCAGTTTCAATGTAGTTTTACATTAATTTTCTTTTTCTTATTTTTTTTTTCTTTCGAGACAGAGTCTCACTCTGTCACCCAGCCTAAGTACAGTGGCATGATCTCGACTAACTACAAATTCTGCCTCCCAGGTTCAAGTAATTCTCCTGCCTCAGCCTCCCGAGTAGCAGGGACTACAGGCGCATGCCCCACGCCTGGCTAATTTTTGTATTCTTAGTAAAGACAGGGTTTCACCATGTTGATCAGGCTAGTCTTGAACTCCTGACCTCGTGATCCACCCGAGCTTGACCACCCAGAGCACTGGGACCACAGGTGCCTGTGAGCCAACGCGCCTGGCCTTCCAATTACTTTTCAGCCCTGTAAACTGCTCTACCCTCCAGTTCACATGTGCATCTCGTCTTCAATCAGCCATGCCATTTACCCTACACATCATACTTTCTATGAATTTTTGCAGCACACATTCTACACCTGTCCAGACTTAGCAACTAAGAAAGTTAACCAAAACTCTCTACCACCGGAGCTGACCAATGCATCAACAGTCCCTCTTGTGGGTGAATTTCTTTTAAATTCAACTCCACTGTGTGCATAGTCCTAATACCAGTCTATGGCCCCCTTCCTGGATCTCTCCATCATCTTCTTTCCTTTTCACCATCATGCTAATTGTTTCCCGTCCATGAGACACAGGCAACTAGCTGCAGAGAAACCTGTAAGGAAGAGCAGGCATCCAGATGAGGTGCGTGCCAGTGTTTTGTTTTTTAGCACGACCTCATATGTTGACAGAGCTCAGGAGTGTGAGGACAAAGTGCCATCTTAGTCACTCATTCTTTGCCCAGGGCTTCACGCTCCTTGTCGGCTGCTATCTCCTCCATGCTGCAAATAGGGTCTTCTCCTCCCTTGTGTGAATCTGGTAGGCTCTAGATAGGAACTAAGTCCTTTAAGAAGCTGCCTTCCATTGAACTAAGGTGAGACAATTACACCAACAAGAAGTCACAGCCTCTCCCTATGACTGGCTCCTTTTATACAACTACATTAGAAAAGGATGAAGTATTGTTACCTCTAAAGGCTTAGTGAGAAATTGCAAAATTCCTGTTAAAAACCACCCTTCTCTGTGGCCATTGCAGGCTTCTAGAAGAAGCATTTCCACATATGTCATCCAACTCCTTATTTTATGTTTTCCATACTGTGCATGGGTTTACTTCATGCGACATATCTCTGGAGGTAGGAATGAACAGATGGATGCATGGATGGATTGATGGATATATTTTCTTCTGTGCCTGTGTGTGTGTGACTACTTCTATGCACAGTTATGCATAAGTATGTATTTATTTGTTTCAGAAATATCAACTAGCCTTTCATTGATACATGCCCTTCTGGGTTTTATATTATCAATTCATTCATGAGGTAGCTAGAGGACAATAAAGAATACAGTCAGACAGAAACACATTTTGAATACATTTCTAGAAAATCCTTTTCATTACTCTGCACATAAACCAGGTGAAGATTACTGAGGCTCTGCATGTGTATAGCACTATTTGGTTCTGAAATGCATTTTTTTTTCATTAGAAAGGAGACTTGTATGATAGCGACTACAACATTACTTCAGGCTGGGGAAACTGGGTGATTTCTTCCAGATTCCAATAAACTCTTTTATGACCTCAGTCCAATACATGCTGTCAAGTCTTTGTGGTGGCAGTTATTTCAATTCAGACTTTTGTTTGGAGGTAGATGTCAGCCCTCTTATTTCATTGAATGTCCTGTTTCAAAGGCCCTCCATTACCTGCATACTTCAACCCAGCATGGCTGTCTCCCCAAGCCCCACAGGGACAGAAGTAAGTCACTAGAGGTCTGACATAGGGGCACAATAGGATGCAGTGGGTCGCAGGAGTTTTGCAGCACAGCCTCCATGGTGCCAAGGCTCTGGAGGCTGGAGGACACCTCCTAGGCTCCTTAGCTTTGTTCCTTTTCTGCTGAGGGCTTCACAGTTCCCTGAGTACTCAGTCCCTTCTCCTTGCTCCAAATAGGGGTACCTCTTTCCTGCAGCCCAGCTCCAGTATGCCTCATACACAGAGAGTGTAGACTGCAAGGCATCTCACTGTTCTAACCCACAGGTGGAGAAACGAGGATGACGATGGAGGCCTCTCTCCTCTATGTCTGTTTCTCTTTCTAAACAGTCTGGAAGTATGGAAGTTACCTATACATGATACAGTGACAATCAAGGAGATATGGTGTCAAAGATATGCTGTGATACCGTCAACACTCCTGAGGAGTAAAAATGCCGTAATCTCAGCCTCCTGCTCCTGTCCACTGATTTGTGTGTGTGTTTGTGTGTACATCTTTTCTTGTTTTTTGAGACAGAGTTTCGCTCCTGCTGCCCAGGCTGGAGTGCAGTGGCACGGTCTCGGCTCACTGCAACGTCCACCTCCCAGGTACAAGCGATTCTCCTGCCTCAGCCTCCTGAGTAGCTGGGATTAGAGGCACCCGCCACCATGCTGGGCTAACTATTGTATTTTTAGTAGAGACAGGGTTTCACCATGTTGCCAGCCTGGTCTCGAACTCCTGACCTCAGGTATCTGCCCGCCTTGGTCTCCCAGAGTGCTAGGATTCCAGGTGTGAACCACTGCTCCTGGCCATCTGTACATTGTTTATTATAACGTACCTGTGCATAATTGGATTCATAGATGATGACTGCCACCATTCATTCAGATAGATAAAGGACAAATACATAGTTGGGCACGATGTCATTAGTGCTTAAATGTTTTCCAACAATGGCTACCAGTTCCATCTATGAGTTCCCAGTTACACTTTGATTATTCACATGCATGTATATCACTAGATGGACCTGAGATGAGCATTTCTTTTTTTGTTAGAAATAAAGATTTTTATATGCAACAGTGGTACTACTCAGGTATGAACATATGGGATGCCATCCCAGACAAAGTAACTTTATCTGCCTGATTTCTGACGGCATTTTTGGAGGGCATTTAATTCCAGTTGAACTTTGGTTTGTAGGTCCCCTTCTTAAAATCCATCTCATCTCCTCTCCTCTCTATCACCTCCAATCACCTCCTCAATTTGCACCCATCCTAGTAGTCTTTCCTCCGAAGAGTACAGGCCAGAGTAGGCTGAAGGCCTGGCAGAGAGGCTTCATCCCATGTGATGGTGGGTATGAGCAGAGGAGTGGCCTCACATTATGGCAGAGCTCTGCAGGGTGGGAAAACTAAGGTTTATTAGCTCTGTTCTTTTTTTTGCCAGGGTATCTCTGCTCTTTGTCCCCTCCTATCTCCTTCTCGCTCCAAGCAGGATTCACTGCTGCAACTGAGTCCACACTACAAGTACACTCATTCCCACAGAAAGGAAAAATGACTACAAACACTCACATAATAAGCATCCCCTGTGGATTCCTGCCCATGTTCCCACCTCCATCCAAAAAGCATAAGAATATTTCCTAAAAGCTAGTCCTGAGAAGACTCTGGAAATAGTTTCACAGCAACTCCTTGTAGGCATGGATGCTCCTGACAGTGAATGCCTACCCATCTCTTGCTGGTATGTGTATGAATGCATGGATGTATGTGTGTATATGCATATATACATGTATAATTGTATTCCTTTATATCACATGTTGCAATGGACAGCCTACTGGAAGACACCTACTCAGTTCACTCACAGCACAGTGCCCAAGAACTTTAAAACGTGTACACATAGGTAGGACATGCCAACTTCCTCAGCTTAACATTTTCAGGTTCAACTGGTCCAATTATTTTCTTTTGCTTGGTATTTTTTTTTAGAGAGAGGTCTCGCTCTGTCGCCCAGGCTGGAGTACAGTGGTGTGATCTCAGCTCATTGCAACCTCGGCCTCCCAGGTTCAAGTGATTCTTCCAGCCTCATCCCCCCGAGTAGCAGGGACTACAGGCACGCAACACCATGCCTGGTTAATTTTTGTATTTTTAGCAGAGATGACAATTTCACCATGTTGGCCAGGCTGGTCTTGAACTCCTGATCTCATGATCCACCTCCATTGCCCTCCCAAAGCACTGGGACCACAGGCGTGAGCCAATGTGACCAGCCTTCCAATTCTTTTTCAACCTGTGGACTCCTGGATCCTTACGGAGACTCACATGTGTCTCATTATTTGGATGCTCCCCATCCCCACTTACCCTACACATCATTTTCTTTCTCAGATTCCAGCACACATTTTACACCTATCCAAGACTCACTGACTAGAGAGTTTAAGACCCAGAACTCTGTACCACCAGAGCTGACCAATAGTGACAGCCCTTTGTGGTAAATTTTTTTTAAATTAGACTTCACTGTGTCAGCAGTGATTCCTTTTAATACCCATCTATGGCCCCCCTCCTGGATCTCTCCATCATGTCTTTCCTTCACCCATCATGTTAGTCTTTTCCCATCCATGAGACACAGGCAGGACTCACTTCAGGCCTGAGGAAGAGCAGCATCCAATGAGGTCAGAGTATTCTGTCTTGTGTTTTTAGCACCAACCTCACATGTTGACAGAGCTTAGGAGTGTGAGGACAAGTCCATGCTTCATTAGCTCCATTTCTTGCCCAGGTCTCTTACTTTCCCTTGTCAGCTGCCATCTCCTTCATGTTGCAAATAGGTTTCTCTCCTCCCTTGTGTCAACCTGGTAGGCTCTCAAACTCAGGAGACAACCTTACAGAGAGAAGCTAGTCAGAGTAAGGGTGACAAATGCCTGAGCAGAAGTCAGCAGCTCTCCTATGACTGGTTCCTTTTTATACAACTAAGCTCAGAAAAGGGGTGGGATGTGTACCTCAAAGGCTACCATGAGAAACTGCAAACCCTGTTAGAAACCACTCTCTGTGGCAGTACAGCTCTAGGAAGAAAAGTCCACATATGTCCATCCGTCCTACTGTATGTTTCCATTGTGTGCATGGGTTTGTTAGTGGATATCTCTTTGAGTGAGATAGACAGATGCATGGGTGAATGGATGGATTATTTGTTTCTGTGCCTGTGTGTGTGTGACTACTCTATGCACAGTTATGCATACGTGTGTCTTCTTTTCTAAAATTGACTCACCTTCATTGATGAGTGCCAGTTCTGGGTCTGCTGTTGTCAATTCATTTGTGGGTGGACTTTAGAGATAAATGAAGGATACGGTGGGACAGAGGAGACAACAATTTGAAATTGACTTTCTGGAGACTAGCTTCCATTACTCTGCACATAAACCATGAACTTATGACTATGCATGTGTATACACTTGGTTCTGAAATATCATTTTAATTAGGAAGGGGCTGCTGTGGTAGCAACCAATTTATACTTAGGCTGAGGAAGATAGTAATGTGTTCCAGCTCAGAAAATTCGTTATGACCTATGTACCCCAATGCATATCATCTATACGTGACAGGTACCCAAACTCAACTTTGTCTGGAGGTAGTAGAGCCCTCTCTCATTTGGGATATTCCTGTTTCAAAGGCCCTCCATTACCCCCTTTACTTTAGACCAGCATGGGTGTTCTCCCAGTCCCCACAACAGAGTCAAGTCACTTGGAATTATGACACAGGGGCACAATACAATATCGACAGTGATGACAGGTGTCGGCACCCAGCCTAGATGGTGCCAGAGTTCTGGAGGCTGGAGAAACTCCCTCCTAGGCTTGGCTTTACCCTTCTACTGAGGCTCACGTTTACCGGTATCATTGTTCCTTCTCCTGCTCCAATGAGGAGTCTATATTTGCAATTAGTCCCAGCTCCAGTATGTGCTCAACACAGAAAATATAGTTCCAAGGCAGCTTCACTGTTCCTAACCAAGGGTAGAAGACAGTAACGTGAACCTATCTATGATGTGCTTCTCCTTTCTTGCTAAACAACTTGGAGATTTAAAAGTTGCCTATACATGATACAGTGAGACAGATAAAGGAATGGTGTCAAAGATATGCTGTGATACCATAACACTCCTGAGAGTGGAATTGCAGGATGATCTCAGCCTCCTGCTCTGTCACTGATTTGTGTGTGTTTGTGTGTACATCTTTTTCTCGTTTTAGAAGGAGGGTTTCGCTCCTGCTGCCCAGGCTGGAATGCAGTGGCACCATCTTGGCTCACTGCAACATCCACCTCCCAGGTACAGCAATTCTCCTGCCTCAGCCTCCCGAGTAGCTGGGATTAGAGGCACCTGCCAACACACCTGGCTAATTGTTGTATTTTTATTAGAGACAGGGTTTCACCATGTTTGCCAGCCTGGTTTCGAACTCCTGACCTCAGGGTATCTGCCCGCCTTGGTCTCCCCAAGTGCTCGGATTCCAGGTGTGAACCACTGCTCCTGGCCATGTGTCCATCTTTTATAATGCACATGTTCTCATGGATCGTGGATAGTGACTCCACCAGGTCATTCAGATAGGATTAAAGGACAAATATGGTAAAAATAAGACACGATGCATCATTCTAATGTTTTGCAACAATAGGCTACAGTTAGTCTATGAGTTATTCCAGTTGCACTTTGATTATTCACACGTATTATATCACACTAAAGGACCTGAGATGGCACGTTTTTTATTAGAAATGAGACTATGTGCCAACAGTCTGGATACTGTTGGGCTGTGAACATAGATCATCCAGACTAAGTAACTGTATCTACACCTCAATGTATAACAAGCATTTTTCGAGGCACTTCATTCCAGTTCAACTTTAGTTTGTAGGTCCCTTCTTAAAATACCATGCAAGTTTCCTACACGTTAAACACCCAAGCAACACAAAGATAGACAACAGCAGAGTCACAATACATAT";
        let md_tag = b"10A0G17^G9^G11G4^ATC1G3C18G5G1G1G1G5C15^AT25A1^T7^G4C0A1^TT3G21A33^A0A1A4^A0C2^A2C0A16^T22^AG1^A1A5^TT7C0A2A2A0G0T9A5A1A1A47G4^TCCA20A5^G10C3G2C4A23^A12A1C1A15^T35G4^GAG4^T1T7^C1G0G29G4A0C11^G1G1A2T0A4A40C6^C3^G5C9T1G1C3G0A1C0A9^TT8^T12A3A1^T3G5T1T0C9A1A0C6^TA8C2^GTT6G1A27^G16C5^A2A4^C58A0G13^A0A2A35G40G9A26^T65G3^C10^CT2G3^G5C5^C7C9^TAAA15^A3A10A18^CA2^CTT7^T23C3A2G0G1T1^C6A21^AG41G7^AGAA12G4^TC20C0A3G1G27^TGAGA3^G18T1G4G23A2C2C39A1C1C6C11^C0C1C0C28^G1G0G18^A2A0C9^G7A7^AT0T6^TC0G19^A8A1A1A0C24^GTAC38A1^C0A10G6A9T7^AGA2A1A0G5^C34C0A6T0G9C0T17^TC6A12A3A0G1C3C22G7A23^CA13T2C20T43G2G4A4C1G24^A12^G3^A15T35A2C15^T12A2^G2^T8^C4G4A2C19^AA6^TT35A7A5C1^GA7^T8T0A14A2A4G0G4T1A6^G5A0C4A1A2A2^A6^T2^A11^A0T1G20^A9A1^TC2^A80T15G3^G0G7G2^A2A28A8^T4A0A5^G39A7G8T19^GG8^GCC7^A2^A1C37A3^CA2^T14A12G21A0C2T2T0T2A1^G4C18^AGT25^T22C0G3G0A4^G0C1C3^G16C0T14A5C20A1A0A3^G0T7^GAG3T12G53A6C3^T1T5A0G0G12G3G5A26C0A2C16A0G5A1G19^A1C0A8G1^GC2A0A36A22^T0T1A1A0G6^G19^G3C4^AG0A3C38^CA2G10^G8T4C18A5^A4C0C6G7A1A1A1C0T12G9C8A13A14A8^A41G1^C6^G5^CA4^T3A3A2^T4T0G1T5A0A1A34^A0T5^A0T4C0A10^T0T0T9^T2G5^A1A0A1G4C13G2G1G4^A0C12C9^G0G4^A13A2A2^G4A0G8A3G11G2G4^A0A3T10^T4A36T0A12A7G1G5A1T0A43^T1C5^G1^C20^T15G16G3^C33C1T0C16A5^CA13C0A33T1C7A2A6T9T1G1^TG27A9A3^CCA14^CTC14^A8^GTC1C0A1^TC11^A15C0A8^G3C7G0A2T36G5T1C0G15G1G1^GG3C51A2A9G19G0A41^C4G9C21A2A0G5A19G1A0G18G3A3^TA11A37G17^T3^T41C0A2^G0G3C1G26C93G76G4A18A1C54A3^AA5G4T2T1G1^T4^TG0G44A11T7A3^A5C1C0T0G3^C44G46^AA3^G1C0A2^CCT5C35G12^C11G1C89^C1A32^G12G2^CC1T1A0G2^C22A39A1T0C15C15^C1A1^C1^C3^AA4G21G1^AG10T5A17G22^T28C10A0C8^CC0A5^AG9C45A12^C5G22T0G13G2A0G5G1G5^A3^AA2G2G3G1T0G33G0T47C2G3C9C11C0A14G13G0C1G2G11G7^A8T0G2^A18^G2G5A6^AT1^G2T5^GG7A0G0C38A33^T23^G2T9G2A4G15^A4A3A1G10T0A7^T4^TA15C5^G28^A2C72C5^T33A7^C32C0C0A2A2A0T3G0A6G9^C17G0A16A65A22A6^A36A1A1C1^G2T0T1C8T8^A10G0T5^G5A2^A6A33A3^TA0A2A18G9^GA3G7A21A23G3C2^A16T38^T47A51A99^T16A14^G25A6C0A32G7C20C0G0G6G2C0A2A9G3^T7G4^GG2^T13^TGGT2G3C0A1A3A2^GC16G4^A3A10^G3G6^ACTC3G37A21^T7A8G1G10^C8^A6T3^C1G23G8G5^AC0A3^C0A4A1A2A29C6A10^C14^C6G2T6T63^T23^CA0A33G2^CA7^C0A33^A6^TC0G3G15^T10A0G4T52T3C0G0T41G0G23C1^C5A3G3^T1C42^TA7C0A28A0G38^A4A0A7^TC0T2C2G21G4^TT93G28^G2A2C0A5T10^A6G3A6^A4^A1T8^C15^TGTTTTTTTTTTT1T83G9A86G2G52G0A0C32C25A0C12A0A3G15T9^TA1T1G0G2C6A0T2G4G3T16A6A0C4G37C0A25G13^CA8T5^G3G5C9C0A12A0G5C55^T46A0G2G1G30G1T3^C2G40C10^G13T1C13C0T9^GAT5G9G29A44^A1A2^TG1^G0T4T0A5C0T2^T6^GA1C14A1^A1C0A1C0A26C18^A1T8A2C0A0G8^A17^AG17A17C0A1G2^G2^T11C62G1G7A2^AGTGGG3G2T0G22^C0A33^T28^T2A3^G6T0A1A7^T9A0C12T30A0G1^G2^T8A4A2C0A26^T9^AA30^G4G2C18^G3^TAT13G3^TTTTTT4C5A6^GA0T1T0A1A2A1G8C11G4A9C7^TT1A1A4^C13C11G1G3^G3T1^G0T0G3G3T2^T0T0T9T16C7^TTA1T9^G36C0A16^C11^A17A3G1C17C15A0G5T1G1^A10C0C30A9^CT3A4^TG1T2T4G4^G3^T4^G3C0C3^A4^G3A8^T8A0T8C5C0C0G2G19^G11C1^G2^T10T9A12^G13^A1^AC0G3^GC1^T2C5A0G4T11T10^G1A0A1A2A1G6A7G21A16C5A5G1^C19C0C4A28^TG35C1^A3A2A38A0G3C31^G80C2A15G16C51A54C3C0A2^G0G3G3^A1C0A3A5^C14G4^G21C1G1G1C6A4G20^A28^CC38^A18^T0T0T18^GT0T0T16^G3C2A13^TC3^C50G2^G4T2A27^C13C4C1G4";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"ACAGTGTCTAagACACTCTGTATTCTTATGGGTGCTCCAGACAATGAACTTgCCAAATCTgAATcTCTCTTCTGTACATGTATgCTTTTgTgTgTgTGTGTcTATGTGTATATTTACATATATATATGTTTACATCTAGCTTTTaATTTATTTAGACCTcaCTTTTTgTTGAAGGATGCTGCATGAGTTaTTGTCATTTCACTGAAGGCAGTGCTCATGGAGTAaAaAGGTAcAGAGCcaTATAAACACCACATGATTTTCTCTTTTTTTGACACTAGTAGCAAaAACAATTTTTGCACcaAAaGTagtTTTTCCATAaAAATAaAaTaTCAACTATACACATATGTAATGGGTGAGGGTATGCAAATCCTACCACgTAAGTCCAGTAAGTAAATTCCAACAATTaACAAAGACTCATTTTTcAATgGCcAATTaATTAAACAATTAAATGACTATATAAAAATTAAACCTaTcTaCATATTAGAAATATATTTTAATTCACCCATGCCTCCACTGTACTATAAAATgAATAGAGAAAATTtTTATTTACTggCTTAAGAAATCAATACTACATATTCAACTgATGTacTACTGCTTAATGAgCaATtaCAGAaTTCACAATCTTGATGTTTCCTTGCAAAAGAAGTGGTAATAcATAGCACAAAGAACCTcTTTAAAAATtAgTcACTgaTcaAACACAATCTTTTTATAGCTTCTGTAACATCTaATTaATTTTgTAAATtTtcTTGAGCATTaCacTATAATTAACTTGCATcAGGTTTTCCCTgTaACTTACATTTTTTATGATTTATTTCCAGTAGAAGTCTGAACATGcCTGTGAAAaCCATCCGTCATGTCCTTCCTTTCACCCATCATGCTAGTCTTTCCTGTCCATGAGACACAGGCAagTAGCTGTGGGCCTAaAGaAGGGCTGCATCCAATGAGCTGGTGGGTTTGGGGTTgGGTCTGGGCACCAACCTCACATGTTGACAGAGGTCTGGACgGTGAGGGACaAGTCCATGTTCCATTAGCTCCATTCCTTTCTGCCCAGGGCTCCACTGCTCCTTGTCTGCTGCTATCTCCTCCATGTTGCAAATAGGATTCTCgCCTCCCCTGGGCCACTCTgGTAGGCTCTcGAACTCAGGAACTcTGAGTCCCTTAAAAAACTGCTTACTAAGAACTaAAGGTGATGAaTACACTAACAGGAAGAAGCAACCTTCTATGACTTGCCCCTCTTTATATAACTCCATcAAAaAAggAtGCAAAGTTaCCTACAAATGCTACAACGAGAAGAACTGGAAACCCTGTTAAAAAACACTGTATGGCCATGCAGGgTCCTGAGAGAAAATTCCACAAATgTCCATCTGTCTCCTTGTTTTAAGATTcaATTgTgTGCATGGGTTTGTTTGTGTACGTGTGTTGAGAGAAGTGTGTATAGATGGATGGAtAgATTTgTTTACATGCCTGTGCATGTGTATaTAcTTcTATGCACACTTACGCATACATGTGTCTTTATTTATTTATaAcTcACTTTTcGTAGATACGTGCcTccTTGGTTCCTCCTGTCAGTTCACTTGTGTGGggACTAGAGGATGATAAAAGAATacAGTCAGACAGGTGACATaATTCGAAATtTACTTTTCgGGAAAAACTAGCTTTCATTACTCTAAACaTaAacTGGTTAATTTCTGACTATTCATGTGTACAGCACTATGTGGACCTGAAATATCATTTTTTATCAGAAaTCaGGATTTATATgATGGCTaCCACCATCTtAGGATGGAGAAAaTagTAATGCTTCCAGATTTAAGTAAACCTTTTCATTACAACCTcaTCCCAAtgTGTGACAGCctTTGTGGTGGCAGTTATTTCAAACTCaACTCCAATTTTTaGGTagTcAGCcCTCTTAATCCATTGAATGTCCTgCTTCAAAaATCCTCCATTACCTTGTTACTTTCACCCAGCATGGGCGtCTcCCCCAGCCCCAAAAAATGCAtCAAGTAACTAAAGGTTTTTCACAGGCGCACAACCTGATATGGTgGTgATATaAGGTcCgGAGCACCAGCCTCCATGGTGGCAGAGTTCTGGAGGCTGGAGAACACCTCCAAGTCTCtTAGCTCTGTTCCTTTCTGCTGAGGACTTCACAGTTaCCcGAGTACTCCTACCTCTTCCTTGCTCCAAaTAGGATTTTCTCTTCCCTCgGCCCaCTcTAGTATGTCTCATACACAGAAAATGAGTTTTCAAGGCATTTGGCTGCTACTAACAAATGGTGAAaAGACAGTaACAGCcGGATACTCTGTTCCTCTTTtaTCCAAAACAATTGTaAAaATGTggATGTtTaACTATAGGTGCTacAGTGaGaAGaTGAACTGTGTCCAAGACACTCTGTAtTgCCATGGACACTCCTGAGAGCAAATCCCACTaATCTGAATATCTTTCTCTGTCACTGTTTTGTTTGTGTGTACATCCTTTATAACTCACATGTCATGGGTGTCTCTATCAATGATGACtTCCACCAGGTCAACAgGATGgGGACTGTgGGAAAaACATGATCAGAGTCAGACAGGACACATTaTTCTAATGTTTTGaaTCAAAGAGCTACACTAACTCTATGAGTGATTCCTTGCATACACTTaTTGAACTgGGATGAAAtTTTTTCTTATTAGAAATGAGGTTTTATGTGCCAACAGTGAATATcTTGGGCTGTGAACATAGCGATGCATCCAGATTAAGTAaCTGCATCTTCACCTCAATGTGTaACAACCTTTTTCgAGGGCAGTTAATTCCAGTTCAacTTtAGttTTaAGTCCCcTTCTTATAATTTCATCTGAGTACCTCATTCTCTATCACCTCCTCAGTTTTCACCCATCCTAGTTGTTATcgGGGgaACCTGcCcCCAGTATGTCAACATAGATTctTTTCTATTTTCCCTaAGTGTcAGCCAGTCTGAGAAATAAAGaGaaAGAGtACAAAGAGAGAAAtTTTACAGCTGGGgCTCCAGGGCTGACATCACATGTTGGCAGGTTCAGTGATGCCTCCTGAGCCGCAaAACCAGcAAGTTtTTATTaggGATTTCAAAAGGgGAGgGGGGTaTGAACAGGGAGTAAGTTACAAGATCAcaTGcTTGAAAGGCCAATAAAagATCACaAgGGAAGACAGGCAGAGCAAGATcaCAAGGCCAgGGCGAaaTTAGAATTATTGATGAGGTTCCATGTCCCACTGGGCaCCCATTGTCACTGATAAACATCTtAaCagGAAACAGAGTTCCAGAGCAGACAACAGGTCcGACTAGaATTcGCCAGGCTGGAATTTCCTAATCCTAGCAAGCCTGAGGGCACTgCAGGAGACCAGGGCGTATTtCATCcCTTATCTTCAACTGCGTAaGACAGACACTccCAGAGCgGCCATTTaTaGaCctCTCCCTGGGAATgCATTCCTTTcCCAGGGTTaTTCCTTGCTGGGAaAAGAATTCAGTGATaCTTCTCCTACTTGCTTTCTCCAAGAAGAGAAATAAGACTCTGTTCTGCCCgGCCCCGCAGGCAGTCAGACCTTATaGTTaTCTCCTTtgTtCCCTGaaAaTCGCTGTTAGCCTGTTCTTTTCTAGGATGCCAAGAtTTCATAtTGTTcaAACACACATGTttTACAAACAATTTgCAAAGATaaTgCAATcATCACAGGACCCTgGGgCgACCTAcCTTCTCAGCTTAcGAAGATGACGgGATTAAGAGATTAAAGTAaATaCAGGCATagGAAATTATaAGAgTATTGACTAGGgAAgTGATAaATGtCCATGAAATCTTCATaATTTATGTTCAGAGACTGCAGTAAAGACAGGCAGAAtaAATTATAAAAGTaTTAATTTgGgGAACTaAtaAATGTCCATGAAATCTTCATAACTTATGTTCTTCTGCCACAGCTTcAACCAGTCCCTCCACTCAGGGTCCCTGATTTCCAGCAACAAGTTgTCTTTCCTGCCCAAGAgTTACGGCCAAGTAGCTGCAGGCCTCAAGCAAGGGCTTcAtcCCATGTGATGGTGGGTaTGGGCCAAGGGAGTCAGCCTcaCATTATGGCAGAGCTCTGGAGGGTGTGAAATATtTcCAGGGTTaTTaGCTCTGtTCTCTTTTTtTgCTGGTGTTTCTCTGCTCTTTGTCCACTCCTaTCACTTTCTaACTCCAAGGAGGATTTGCTTCTCTCTTGGTCCACACTACTGGAACAGTCTcaTTCCCACAGAAAGGAAAGGTGACAATCACTcaCATAATAAGCCTcCCTCTATgaGTtCTTCCCCATTTTCCTACCTCCATTCAAAAAGCAAAAgACTATtCcgTACAAAGCTCATCCTgAgAGGGCTcTGGAAATAGTTTCAAGGGAACTCCTTGTTCCCATGGATGCTCCTGACAGTGaATaGCCCAAATCgCAATCTCTCTAGTGTATATgaATGCATGGATGTGTGTGTTTGTATATGCATATATACATATACAATTgTATTCATGTcACATGTTGCAATGGACAGCCTaTTagAAGATaCTTTCAGTTCACACACAGCgCagTGCTAAAGAACTTAAAACgTTTaTAGTAAGACAACATGTaACTTCCTACACTTAACATTTTCAGTTCAATGTAGTTTgCATTAATTTTCTTTTTCTTTATTTTTTTTTTTCTTTCGAGACAGAGTCTCACTCTGTCACCCAcaCTGgAGTcCgGTGGCATGATCTCGACTAACTACAAAcTCTGCCTCCCAGGTTCAAGTAATTCTCCTGCCTCAGCCTCCCGAGTAGCAGGGACTACAGGCGCATGCCCCACGCCTGGCTAATTTTTGTATTgTTAGTAAAGACAGGGTTTCACCATGTTGATCAGGCTAGTCTTGAACTCCTGACCTCGTGATCCACCCGACTTGACCgCCCAaAGCACTGGGACCACAGGTaTcAGCCAACGCGCCTGGCCTTCCAATTACTTTTCAGCCTGTAAACTGCTCTACCCTaCAGAATTCACgTGTGtATtTgGTTCTTTGgAATCAGCCATGCCATTTACCCTACACATCATACTTTCTATGATTaCAGCACACATTtTACACCTaTCCAAGACTcActgACTCAAGAGTTAACCAAAACTCTCTACCACCGGAGCTGACCAATGCATgACAGCCTCTTGTGGGTGAATTTCTTTTAAATTCAACTCCACTGTGTAAGCAGTcaTCCCTCTAATcCCATCTATGGCCCCCTTCCTGGATCTCTCCATCACgTCTTTCCTTTCACCCATCATGCTAgTcTTTCCCGTCCATGAGACACAGGCAACTAGCTGCGAACCTAAGGAAGAGCAGCATCCAATGAGGTGCGTGCCAGTGTTTGTTTTTTAGCACCaACCTCATATGTTGACAGAGCTCAGGAGTGTGAGGGACAAGTCCATgTTCCAtTagCTCCATTCTTTGCCCAGGGCTTCACaCTCCTTGTCGGCTGCTATCTCCTCCATGCTGCAAATAGGaTtcTCTCCTCCCTTGTGTcAATCTGGTAGGCTCTCAaACTCAGGAAAACTgAGTCCTTTAAGAAGCTGCTTAgTAGGAACTAAGGGtGACAAaTACACCAACAAGAAGTCgCAGCTCTCCTATGACTGGCTCCTTTTTATACAACTACATTAGAAAAGGATGcAGTGTTACCTacAAAGGCTACCaTGAGAAGAATTGCAAAcCCTGTTAAAAACCACTCTCTGTGGCCATGCAGGCTTCTAGAAGAAaCTCCACATATGTCCATCCgACTCCTTATTTTATGTTTCCATtgTGTGCATGGGTTTgTTagTGGACgTgTCTTGAGAGAAGTgGGgATGgAtgGATGGATGCATGGATGGATTGATGGATATATTTgtTTCTGTGCCTGTGTGTGTGTGACTACTTCTATGCACAGTTATGCATAcGTgTGTcTTTATTTGTcTAAATATAACTcaCCTTTCATTGATACgTGCCTCTGGGTTTgcTgTTgTCAATTCATTCgTGGGTAGACTAGAGGAtgATAAAAGAATACAGTCAGACAGGAgACACAaTTTGAAATTGACtTTTCTGGAGAAAATagcTTTCATTACTCTGCACATAAACCAGTGAAGTTCTGACTaTGCATGTGTATAGCACTATTTGGTTCTGAAATGTCATTTTTTTTTCATTAGAAAGGAGGAtTTGTATGATgGCaACCAgCATCTTAGGCTGGGGAAAACaGTAaTgCTTCCAGATTtaAATAAACTTCTTTATTATGACCTCAGTCCcAATGCGTGTCAGTCTTTGTGGTGGCAGTTATTTCAAAcTCAACTTTTGTTTGGAGGTAGTCAGCCCTCTTATTTCATTGAATGTCCTGTTTCAAAGGCCCTCCATTACCTcCATACTTTCAACCAGCATGGCTGTCTCCCCAGCCCCACAaACAGAGTCAAGTCACTAGAGGTCTGACATAGGGGCACAATccaATaCGatGGTgaCAGGAGgTTTGCAGCACCAGCCTCCATGGTGCCAgaGCTCTGGAGGCTGGAGaACACCTCCTAGGCTCTTAGCTTTGTTCCTTTCTGCTGAGGGCTTCACAGTTCCCTGAGTACTCATaCCTTCTCCTTGCTCCAAATAGGaGTCTCTATTCCTGCAGCCCACTCCAGTATGCCTCATACACAGAaAaTcTGAGttTcCAAGGCATtTCACTGTTACTAACCAAGGgtGAGAAGACAGTaACAGATGGAaGCCTCTCTCTATGTCTGTTTCTCTTTCTAAACAaCTGTAaAAaTATGGAAGTTACCTATACgTGATACAGTGAGACgATAAAGAaATGGTGTCAAAGATATGCTGTaATACCGTCAACACTCCTGAGAGTgAATcCCAGTAATCTCAGCCTCCTtCTCTGTCACTGATTTGTGTGTGTGTTTGTGTGTACATCTTTTTCTTGTTTTTGAGACAGAGTTTCGCTCCTGCTGCCCAGGCTGGAaTGCAGTGGCACGGTCTCGGCTCACTGCAACGTCCACCTCCCAGGTACAAGCaATTCTCCTGCCTCAGCCTCCTGAGTAGCTGGGATTAGAGGCACCCGCCACCATGCTGGGCTAACTATTGTATTTTTAGTAGAGACAGGGTTTCACCATGTTTGCCAGCCTGGTCTCaAACTCCTGACCTCAGGGTATCTGCCCGCCTTGGTCTCCCAaAGTGCTcaGATTCCAGGTGTGAACCACTGCTCCTGGCCATgTGTACATcTTTTATAACGTACCTGTCATcggTGGATTgATcaATaATGACTGCCgCCATTTCATTCgGATAGGGATTAAAGGACAAATATGGTCAgAGTcaGaCACaATGCGTCATTCTAATGTTTTgCAACAAATaGCTACAGTTAGTCTgTGAGTTACTCCCCgGTTACACTTTGATTATTCACATGCATGTATATCACTAaATGGACCTGAGATGGCATTTCTTTTTTTTaTTAGAAATgAgGATTTATATGCCAACAGTGAGTATCTtAGGCTgTGAACATATGGATCCATCCAGACgAAGTAACTgTATCTACaCCTCaATTTaTaACaGCATTTTTGGAGGGCATTTAATTCCAGTTcAACTTTaGTTTGTAGGTCCCCCTTCTTAAAATCCCATCTgATtTCCTCCtTCTCTATCACCTCCAATCACCTCCTCAATTTGCACCCATCCTAGTAGTCTTTCCTCCGAAGAGTTACGGCCAAGTAGCTGAAGGCCTCAaGCAAGGGCTTCATCCCATGTGATGGTGGGTATGgGCCAAGGGAGTCaGCCTCACATTATGGCAGAGCTCTGCAGGGTGGGAAAAACTTCgAGGgTTATTAGCTCTGTTCTTTTTTTTTGCagGGGTtTCTCTGCTCTTTGTCCCCTCCTATCTCCTTCTCGCTCCAAGCAGGATTCACTtCTCcgtTGGTCCACACTACAAGTACACTCATTCCCACAGAAAGGAAAggTGACAAACACTCACATAATAAGCcTCCCCCTaTGAgTCCTTcCCCATGTTCCCACCTCCATCCAAAAAGCATAAGAATATTCCTTAAAAAGCTcaTCCTGAGAAGACTCTGGAAATAGTTTCAagGCAACTCCTTGTAGGCATGGATGCTCCTGACAGTGAATAGCCCaaATCTCTTTCtCTcGTgTGTGTATGAATGCATGGATGTgTGTGTTTGTATATGCATATATACATGTATAATTGTATTCCTTTATATCACATGTTGCAATGGACAGCCTACTGGAAGACACTCTCAGTTCACTCACAGCgCAGTGCCCAAGAACTTAAAACGTACACAGTAaGAcaACATGtAACTTCCTAGACTTAACgTTTaCAGTTCAAACTAGtTCCAATTACTTTTCTTTTGCTTGGTGTTTTTTTTTTTTtTTTTTTTTTAGAGAGAGTCTCGCTCTGTCGCCCAGGCTGGAGTACAGTGGTGTGATCTCAGCTCATTGCAACCTCGGCCTCCCgGGTTCAAGTaATTCTCCAGCCTCATCCCCCCGAGTAGCAGGGACTACAGGCACGCAACACCATGCCTGGTTAATTTTTGTATTTTTAGCAGAGATGgCAgTTTCACCATGTTGGCCAGGCTGGTCTTGAACTCCTGATCTCATGATCCACCTgacTTGCCCTCCCAAAGCACTGGGACCACAGGCGTcAGCCAATGTGACCAGCCTTCCAATTacTTTTCAACCTGTaaACTgCTGGATCCTACGGAAtTCACATGTGTATtTggTTcTTTGGAatCTgCCATgCCAtTTACCCTACACATCATaCTTTCTacGATTgCAGCACACATTTTACACCTATCCAAGACTCACTGACTcaAGAGTTAACCAAACTCTGTACCACCgGAGCTGACCAATGCATGACAGCCtTTTGTGGGTgAATTTcTTTTAAATTcaACTTCACTGTGTagGCAGTcATCCTTTTAATCCCATCTATGGCCCCCCTCCTGGATCTCTCCATCATGTCTTTCCTTTCACCCATCATGTTAGTCTTTCCCATCCATGAGACACAGGCAACTagCTgCgGGCCTGAGGAAGAGCAGCATCCAATGAGGTgGtGTACTTgTGTTTGTGTTTTAGCACCAACCTCACATGTTGACAGAGCTcAGGAGTGTGAGGGACAAGTCCATGtTcCATTAGCTCCATTctTTGCCCAGGGATTCTCTgCTCCTTGTCgGCTGCCATCTCCTTCATGTTGCAAATAGGaTTCTCTCCTCCCTTGTGTCAACCTGGTAGGCTCTCAAACTCAGGAAaACTGAGtCCTTtaAGAAGctGCTTAGTAGGAAcTAAGGGTGACAAATaCACcaAcaAGAAGTCGCAGCTCTCCTATGACTGGcTCCTTTTTATACAACTACATtAGAAAAGGaTGcagTGTTACCTACAAAGGCTACCATGAGAAGAACTGCAAACCCTGTTAaAAACCACTCTCTGTGGCcaTgCAGGCTTCTAGAAGAAAcTCCACATATGTCCATCCGTCCTACTGTATGTTTCCATTGTGTGCATGGGTTTGTTAGTGGATgTgTCTTGAGaGAAGTGGGGATgGAtgGATGCATGGGTGAATGGATGGACaTATTTGTTTCTGTGCCTGTGTGTGTGTGACTACTTCTATGCACAGTTATGCATACGTGTGTCTTTaTTTGTCTAAAtaTaACTCACCTTTCATTGATacGTGCCTCTGGGTtTGCTGTTGTCAATTCATTTGTGGGTGGACTagAGGATGATAAAAGaATACaGTcaGACAGGAGACACAATTTGAAATTGACTTTTCTGGAGAAACTAGCTTCCATTACTCTGCACATAAACCAGTGAAgTTcTGACTATGCATGTGTATAGCACTATTTGGTTCTGAAATgTCATTTTTTTTTTcATTAGaAAGGGGGAtTtaTaTGaTgGCAACCAAcATCTTAGGCTGgGGAAaATAGTAATGcTTCCAGCTTTaAaAAAACTTCGTTATGACCTcTGTCCCAATGCgTgTCAGTCTtTGtgGTGgCAGtTATttCAAACTCAAtTTTGTCTGGAGGTAGTcAGCCCTCTTATtTCATTGAATGTCCTGTTTCAAAGGCCCTCCATTACCCCCTTACTTTcaACCAGCATGGGTGTCTCCCCAGTCCCACAAACAGAGTCAAGTCACTaGAAgTcTGACACAGGGGCACAATcCAATACGACAGTGATagCAGGTtTgGAGCACCAGCCTccATGGTGCCAGAGTTCTGGAGGCTGGAGAACaCCTCCTAGGCTCTTaGCTTTGTtCCtTTCTgCTGAGGGCTTCACGGTTccCCGAGTATGCGTaCCTTCTCCTTGCTCCAAatAGGAGTCTcTATTCccgTGgCCCACTCCAGTATGTCTCAGACACAGAAAATcTGAGTTTCCAAGGCAtTTCACTGTTaCTAACCAAGGGTGAGAAGACAGTAACAGACgGAAGCCTCTcTCTATagTGTGtTTCTCTTTCTTtCTAAACAACTGTaaAaATaTgAAAGTTaCCTATACgTGATACAGTGAGACGATAAAGaAATGGTGTCAAAGATAcGCTGTaATACCgTCAACACTCCTGAGAGTGAATccCAGTaATCTCAGCCTCCTGCTCTGTCACTGATTTGTGTGTGTGTTTGTGTGTACATCTTTTTCTCGTTTTcGAAAGaAGaGTTTCGCTCCTGCTGCCCAGGCTGGAATGCAGTGGCACagTCTcGGCTCACTGCAACATCCACCTCCCAGGTACAGGCAATTCTCCTGCCTCAGCCTCCCGAGTAGCTGGGATTAGAGGCACCTGCCAACACACCTGGCTAATTGTTGTATTTTTAcTAaAGACAGGGTTTCACCgTGTTTGCCAGCCTGGTcTCGAACTCCTGACCTCAGGGTATCTGCCCGCCTTGGTCTCCCCAAGTGCTCaGATTCCAGGTGTGAACCACTGCTCCTGGCCATGTGTCCATCTTTTATAATGCACcTGTcaTCGgTGGgTCGATcaATAaTGACTCCCACCAGGTCATTCgGATAGGGATTAAAGGACAAATATGGTcAgAgTcAGACACaATGCgTCATTCTAATGTTTTGCAACAAATAGCTACAGTTAGTCTATGAGTTATTCCCCAGTTGCACTTTGATTATTCACACGTATTATATCACTAAAAGGACCTGAGATGGCATttTTTTTTATTAGAAATGAGGTttTATGTGCCAACAGTGAGTATcTTaGGCTGTGAACATATCGATCCATCCAGACTAAGTAACTGTATCTACACCTCAATGTATAACAGCATTTTTgGAGGGCAtTTaATTCCAGTTCAACTTTAGTTTGTAGGTCCCCTTCTTAAAATcCCATcCgAGTT";
        assert_eq!(reference, expected.to_vec());
    }

    #[test]
    fn test_rna_examples() {
        let cigar_raw = vec![(4, 1), (0, 103), (1, 1), (0, 12), (2, 1), (0, 9), (2, 3), (0, 13), (1, 1), (0, 16), (2, 5), (0, 16), (2, 2), (0, 26), (2, 1), (0, 4), (4, 4)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"GGCAAGTTATCAGCTTGCGTTGATTACGTCCCTGCCCTTTGTACACACCGCCCGTCGTTAGTTTTGATTGAATGGCTTAGTGAGGCCTCAGGATCTGTTAAGAGAAAGGGGGCAACTCATCTCAGAAAGAATTTGTGCATCACTTGGTCATTTCGATAAAAGTCGTATCCAGTTCCGTAAGCGATCCTGCGGAAGGATATTAAAAT";
        let md_tag = b"6C49C4A0C0C32C1T16^C9^GCG0G8G0A2A12A2^GGAAC11A1A2^GT7G1T1T14^C4";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"GCAAGTcATCAGCTTGCGTTGATTACGTCCCTGCCCTTTGTACACACCGCCCGTCGcTAGTaccGATTGAATGGCTTAGTGAGGCCTCAGGATCTGcTtAGAGAAGGGGGCAACTCCATCTCAGAGCGgAGAATTTGgaCAaACTTGGTCATTTaGAGGAACTAAAAGTCGTAaCaAGGTTTCCGTAgGtGtTCCTGCGGAAGGATCATTA";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 1), (0, 9), (2, 1), (0, 66), (2, 3), (0, 26), (2, 10), (0, 10), (2, 1), (0, 1), (2, 1), (0, 24), (2, 2), (0, 4), (2, 1), (0, 8), (2, 3), (0, 12), (4, 7)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"CCCATTGAAAGGAATCCGGTTAAGATTCCGGAACCTGGATATGGATTCTTCACGGTAACGTAACTGAATGTGGAGACGGCGCGAGCCCTGGGAGGTGTTATCAACAGCTTATACCCGGAATTGGTTTATCCGGAGATGGTCTATGGCTGGAGGCCAGCACCAAAATCC";
        let md_tag = b"4C4^G66^CGT19A6^TTTTCTTCTT10^C1^C24^GG4^T8^GAA12";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"CCATcGAAAGGGAATCCGGTTAAGATTCCGGAACCTGGATATGGATTCTTCACGGTAACGTAACTGAATGTGGAGACGTCGGCGCGAGCCCTGGGAGGaGTTATCTTTTCTTCTTAACAGCTTATCACCCCGGAATTGGTTTATCCGGAGATGGGGTCTTATGGCTGGAAGAGGCCAGCACC";
        assert_eq!(reference, expected.to_vec());

        let cigar_raw = vec![(4, 1), (0, 19), (2, 1), (0, 6), (2, 1), (0, 2), (2, 2), (0, 13), (1, 1), (0, 6), (2, 6), (0, 2), (2, 4), (0, 5), (2, 3), (0, 5), (1, 1), (0, 8), (2, 2), (0, 1), (2, 1), (0, 9), (1, 1), (0, 5), (2, 2), (0, 24), (2, 2), (0, 39), (1, 1), (0, 23), (1, 1), (0, 1), (2, 2), (0, 60), (1, 1), (0, 2), (1, 1), (0, 9), (4, 7)];
        let cigar = tuplevec_to_cigarvec(cigar_raw);
        let query = b"ACCATGCTAGAACGTGGTGATTCTTTATACACAATATTGATAGGATACGGCTTGTTTTGCTTGAACCATCGATTAGCAACCGGTGCTTGGCGGAAAGGCCTTGGGTGCTCTGGCGAATTGCAATGTCATTTTGCGTGGGGATAAATCATTTTGTATATGTATTAGATGTTTATAGGGTATTGTAAGCAGTAGAGTAGCCTTGTTGTTACGATCTGCTGAGATTAAGCCTTTGTTGGTCCTGATTTGTAAAAATC";
        let md_tag = b"13C5^T6^G0C1^CC9A9^GAATAA2^CGTC5^GGC0G1C10^AG1^A1G0C11^CA24^TG47C1A0C8A0C2^CG71";
        let reference = build_reference_sequence(query, &cigar, md_tag).unwrap();
        let expected = b"CCATGCTAGAACGcGGTGATTTCTTTGcTCCACACAATATaGATGGATACGAATAAGGCGTCCTTGTGGCgTcGCTGAACCATAGCAGgcTAGCAACGGTGCACTTGGCGGAAAGGCCTTGGGTGCTTGCTGGCGAATTGCAATGTCATTTTGCGTGGGGATAAATCATTTGTATAcGacTTAGATGTacAACGGGGTATTGTAAGCAGTAGAGTAGCCTTGTTGTTACGATCTGCTGAGATTAAGCCTTTGTTGTCTGATTTGT";
        assert_eq!(reference, expected.to_vec());
    }
}