#!/bin/bash
set -euo pipefail

BASEDIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

DATADIR="${BASEDIR}/../01_data/dna_giab"
SUBSETDIR="${DATADIR}/subsets/medium_100000"
bam_path="${SUBSETDIR}/subset.bam"
fastq_path="${SUBSETDIR}/subset.fastq"
pod5_path="${SUBSETDIR}/subset.pod5"
blow5_path="${SUBSETDIR}/subset.blow5"
kmer_table_path="${DATADIR}/levels.txt"
ref_path="${DATADIR}/ref.fa"

BINDIR="${BASEDIR}/../bin"
REMORA="${BINDIR}/remora"
FISHNET="${BINDIR}/fishnet"
F5C="${BINDIR}/f5c"

OUTDIR="${BASEDIR}/alignments"

source "${BASEDIR}/../venv/bin/activate"

echo "Running Remora query alignment..."
$REMORA \
    --bam "$bam_path" \
    --pod5 "$pod5_path" \
    --kmer "$kmer_table_path" \
    --out "${OUTDIR}/remora_query.tsv"

echo "Running Remora reference alignment..."
$REMORA \
    --bam "$bam_path" \
    --pod5 "$pod5_path" \
    --kmer "$kmer_table_path" \
    --out "${OUTDIR}/remora_reference.tsv" \
    --alignment_type "reference"

echo "Running Fishnet query alignment..."
$FISHNET align \
    --bam "$bam_path" \
    --pod5 "$pod5_path" \
    --kmer-table "$kmer_table_path" \
    --out "${OUTDIR}/fishnet_query.parquet" \
    --threads 24 \
    --output-level 1

echo "Running Fishnet reference alignment..."
$FISHNET align \
    --bam "$bam_path" \
    --pod5 "$pod5_path" \
    --kmer-table "$kmer_table_path" \
    --out "${OUTDIR}/fishnet_reference.parquet" \
    --threads 24 \
    --alignment-type "reference" \
    --output-level 1

echo "Running Uncalled4 query alignment..."
uncalled4 align \
    --bam-in "$bam_path" \
    --reads "$pod5_path" \
    --flowcell "FLO-PRO114M" \
    --kit "SQK-LSK114" \
    --tsv-out "${OUTDIR}/uncalled4_query.tsv" \
    --processes 24 \
    --self

echo "Running Uncalled4 reference alignment..."
uncalled4 align \
    --bam-in "$bam_path" \
    --reads "$pod5_path" \
    --flowcell "FLO-PRO114M" \
    --kit "SQK-LSK114" \
    --tsv-out "${OUTDIR}/uncalled4_reference.tsv" \
    --processes 24 \
    --ref "$ref_path"

echo "Running f5c query alignment..."
$F5C resquiggle \
    "$fastq_path" \
    "$blow5_path" \
    -t 24 \
    > "${OUTDIR}/f5c_query.tsv"

echo "Running f5c reference alignment..."
$F5C eventalign \
    -r "$fastq_path" \
    -b "$bam_path" \
    -g "$ref_path" \
    --slow5 "$blow5_path" \
    -t 24 \
    -c \
    > "${OUTDIR}/f5c_reference.tsv"