function sigma_step_2 = sort_adj_matrix_step_2(unsorted_adj_matrix, estimate_no_of_comm)
% SORT_ADJ_MATRIX_STEP_2 performs refined sorting of the adjacency matrix
% based on its spectral properties (eigenvectors and eigenvalues).
% This is used as the second step to our already introduced 1st analytical step for communitry detection 
% — to better align nodes that belong to the same community.
%
% INPUTS:
%   unsorted_adj_matrix   - adjacency matrix (after some random permutation)
%   estimate_no_of_comm   - estimated number of communities
%
% OUTPUT:
%   sigma_step_2          - permutation (ordering of nodes) that gives a
%                           better community structure after step 2
%
% -------------------------------------------------------------------------
% Step A: Compute eigen-decomposition and sort eigenvectors by importance
% -------------------------------------------------------------------------
N_total = size(unsorted_adj_matrix, 1);         % total number of nodes
[V, D] = eig(unsorted_adj_matrix);              % eigen-decomposition of adjacency matrix
[~, ind] = sort(abs(diag(D)), 'descend');       % sort eigenvalues by magnitude (largest first)
v = V(:, ind);                                  % reorder eigenvectors accordingly
percentile = 2;

% -------------------------------------------------------------------------
% Step B: Compute a node-level sorting variable using top eigenvectors
% -------------------------------------------------------------------------
% The sum of the first 'estimate_no_of_comm' eigenvectors (column-wise)
% highlights nodes that are within the same community.
x_scramble = sum(v(:, 1:estimate_no_of_comm), 2); 

% Sort nodes based on this spectral feature (descending order)
[~, sigma] = sort(x_scramble, 'descend');       
s = sigma;  % current node order (initial sorting)

% Step two of sorting 
% -------------------------------------------------------------------------
% Step C: Detect boundaries between communities based on spectral gaps
% -------------------------------------------------------------------------
% Find large jumps (differences) in the sorted spectral values.
% Small gaps (below the 2nd percentile) indicate internal community structure,
% while large gaps suggest community boundaries.
initial_boundaries = [1; 1 + find(diff(x_scramble(s)) < prctile(diff(x_scramble(s)), percentile)); N_total];

% -------------------------------------------------------------------------
% Step D: Measure variation of the first eigenvector within each segment
% -------------------------------------------------------------------------
% Compute the standard deviation of the first eigenvector's entries
% within each detected segment.
sd = zeros(estimate_no_of_comm, 1);
for ii = 1:length(initial_boundaries) - 1
    sd(ii) = std(v(s(initial_boundaries(ii):initial_boundaries(ii+1)), 1));
end

% -------------------------------------------------------------------------
% Step E: Identify segments that might need re-sorting
% -------------------------------------------------------------------------
% Compare standard deviations to identify segments that differ significantly
% from others — possibly indicating misplaced nodes.
check = zeros(1, estimate_no_of_comm);
tmp = sd;
tmp2 = 2 * tmp;

for ii = 1:length(tmp)
    % If a segment has much higher variation than others, mark it for re-sorting
    if any(tmp(ii) > tmp2(tmp ~= tmp(ii)))
        check(ii) = 1;
    end
end

check = find(check);  % indices of segments to refine

% -------------------------------------------------------------------------
% Step F: Locally re-sort problematic segments using the first eigenvector
% -------------------------------------------------------------------------
if length(check) > 1
    for ii = 1:length(check) - 1
        % Get indices for the segment to refine
        indices = initial_boundaries(check(ii)):initial_boundaries(check(ii) + 1);

        % Locally sort nodes within that segment based on the first eigenvector
        [~, resort] = sort(v(s(indices), 1), 'descend');
        
        % Update global node order
        tmp = s(indices);
        sigma_step_2 = s;
        sigma_step_2(indices) = tmp(resort);
        s = sigma_step_2;
    end
else
    % If no problematic segments found, keep the initial sorting
    sigma_step_2 = sigma;
end

end