function epsilon = sort_error(N, M, random_perm, sort_perm)
% SORT_ERROR computes how well the community structure was recovered 
% after scrambling and resorting a graph's adjacency matrix.
%
% INPUTS:
%   N           - array of community sizes in the ground truth partition
%   M           - number of communities
%   random_perm - permutation used to randomly scramble the ground truth adjacency matrix
%   sort_perm   - permutation used to reorder (sort) the scrambled matrix 
%                 in an attempt to reveal the original community structure
%
% OUTPUT:
%   epsilon     - sorting error (fraction of nodes misplaced from their communities after sorting)
%
% The function compares the sorted adjacency matrix’s node order with 
% the true community order and computes the fraction of misplaced nodes.

% -------------------------------------------------------------------------
% Step A: Compute node positions before and after scrambling/sorting
% -------------------------------------------------------------------------
N_total = sum(N);                         % Total number of nodes
position_adj_matrix = 1:N_total;          % Original node positions
position_unsorted_adj_matrix = position_adj_matrix(random_perm); % Node order after scrambling
position_sorted_adj_matrix = position_unsorted_adj_matrix(sort_perm); % Node order after sorting

% -------------------------------------------------------------------------
% Step B: Identify how nodes are grouped after sorting
% -------------------------------------------------------------------------
% Sort positions to recover the order of nodes in the sorted adjacency matrix
[sorted_position_sorted_adj_matrix, indices_position_sorted_adj_matrix] = sort(position_sorted_adj_matrix); 

% Divide sorted nodes into M parts according to the true community sizes
divided_parts = cell(1, M);
boundaries = [0, cumsum(N)];
for i = 1:M
    % Each cell contains the indices (positions) of nodes assigned to that community
    divided_parts{i} = sort(indices_position_sorted_adj_matrix(boundaries(i) + 1:boundaries(i + 1)));
end

% -------------------------------------------------------------------------
% Step C: Reorder communities by their approximate positions
% -------------------------------------------------------------------------
% Find the mean position of each community (roughly indicates its location)
first_elements = cellfun(@(x) mean(x), divided_parts);

% Sort communities based on their mean positions (so that lower mean = earlier community)
[~, sorted_indices] = sort(first_elements);
sorted_divided_parts = divided_parts(sorted_indices);

% -------------------------------------------------------------------------
% Step D: Define the ideal ("perfect") community partitions for comparison
% -------------------------------------------------------------------------
actual_divided_parts = cell(1, M); % actual level of nodes in case of perfect sort
actual_divided_parts{1} = 1:length(sorted_divided_parts{1});
for i = 2:M
    actual_divided_parts{i} = actual_divided_parts{i-1}(end)+1 : ...
                              actual_divided_parts{i-1}(end)+length(sorted_divided_parts{i});
end

% -------------------------------------------------------------------------
% Step E: Compute the sorting accuracy and final error
% -------------------------------------------------------------------------
acc = 0;
for i = 1:M
    % Count correctly placed nodes in each community
    acc = acc + numel(intersect(sorted_divided_parts{i}, actual_divided_parts{i}));
end

% Compute error = 1 - accuracy fraction, rounded to 2 decimal places
epsilon = round(1 - (acc / N_total), 2);

end