function adj_matrix = generate_graph_with_communities(M, N, alpha, beta, noise)
% GENERATE_GRAPH_WITH_COMMUNITIES creates an adjacency matrix for a 
% block-structured graph with M communities.
%
% Each community has a specified number of nodes, strong intra-community 
% connections, weak inter-community connections, and optional additive noise.
%
% INPUTS:
%   M     - number of communities
%   N     - array of community sizes (length M)
%   alpha - array of within-community edge weights (one per community)
%   beta  - scalar weight for all between-community edges
%   noise - noise level; scales the standard normal random noise
%
% OUTPUT:
%   adj_matrix - (sum(N) x sum(N)) adjacency matrix representing the graph
%
% The output graph exhibits: 
%   - Uniformly strong within-community connections
%   - Uniformly weak between-community connections
%   - Optional noise that perturbs all edge weights

% -------------------------------------------------------------------------
% Step A: Initialize adjacency matrix
% -------------------------------------------------------------------------
N_total = sum(N);                 % total number of nodes in the graph
adj_matrix = zeros(N_total);      % initialize empty adjacency matrix

% Precompute community boundaries in the adjacency matrix indexing
boundaries = [0, cumsum(N)];

% -------------------------------------------------------------------------
% Step B: Fill in block structure
% -------------------------------------------------------------------------
for ii = 1:M
    % Assign within-community edge weights (square block on diagonal)
    adj_matrix(boundaries(ii)+1 : boundaries(ii+1), ...
               boundaries(ii)+1 : boundaries(ii+1)) = alpha(ii);
    
    % Assign between-community edge weights (off-diagonal blocks)
    for jj = (ii+1):M
        adj_matrix(boundaries(ii)+1 : boundaries(ii+1), ...
                   boundaries(jj)+1 : boundaries(jj+1)) = beta;
        adj_matrix(boundaries(jj)+1 : boundaries(jj+1), ...
                   boundaries(ii)+1 : boundaries(ii+1)) = beta;
    end
end

% -------------------------------------------------------------------------
% Step C: Remove self-loops (set diagonal to zero)
% -------------------------------------------------------------------------
for ii = 1:N_total
    adj_matrix(ii, ii) = 0;
end

% -------------------------------------------------------------------------
% Step D: Add Gaussian noise 
% -------------------------------------------------------------------------
% noise * randn(N_total) adds standard normal noise scaled by `noise`
adj_matrix = adj_matrix + noise * randn(N_total);

end
