function sigma = sort_adj_matrix(unsorted_adj_matrix, estimate_no_of_comm)

% SORT_ADJ_MATRIX sorts (relabels) the nodes of an adjacency matrix so that
% the community block structure becomes visible.
%
% INPUTS:
%   unsorted_adj_matrix   - adjacency matrix of a graph whose nodes have been
%                           permuted (scrambled)
%   estimate_no_of_comm   - estimated number of communities in the graph
%
% OUTPUT:
%   sigma                 - permutation vector; applying A(sigma, sigma)
%                           reorders the matrix to reveal community blocks
%
% METHOD:
%     - Compute eigenvalues and eigenvectors of the adjacency matrix.
%     - Select the top(largest magnitude) 'estimate_no_of_comm' eigenvectors.
%     - Sum these eigenvectors row-wise to form a sorting variable.
%     - Sort nodes by this value to group similar nodes together.

% -------------------------------------------------------------------------
% Step A: Eigen-decomposition of adjacency matrix
% -------------------------------------------------------------------------
[V, D] = eig(unsorted_adj_matrix);          % eigenvectors + eigenvalues of A
[d, ind] = sort(abs(diag(D)), 'descend');   % sort eigenvalues by magnitude

% Reorder eigenvectors so that the most informative ones come first
v = V(:, ind);

% -------------------------------------------------------------------------
% Step B: Compute sorting feature using top eigenvectors
% -------------------------------------------------------------------------
% Each node's "spectral signature" is the sum of its components across the
% first 'estimate_no_of_comm' eigenvectors.
%
% Nodes from the same community tend to have similar signatures.
x_scramble = sum( v(:, 1:estimate_no_of_comm), 2 );

% -------------------------------------------------------------------------
% Step C: Sort nodes based on spectral signatures
% -------------------------------------------------------------------------
% Sorting in descending order clusters nodes with similar structure together.
[~, sigma] = sort( x_scramble, 'descend' );

end
