function M_estimate = M_estimate_inv_eigengap(X)

% M_ESTIMATE_INV_EIGENGAP estimates the number of communities in a graph 
% using an inverse eigengap–based heuristic.
%
% The idea:
%   - Compute all eigenvalues of X (sorted by magnitude).
%   - Form a matrix of pairwise inverse differences of eigenvalues.
%   - Sum each row to create a feature vector.
%   - Look for the largest jump in this vector, which indicates the 
%     transition from "signal" to "noise" eigenvalues.
%
% OUTPUT:
%   M_estimate - index where the largest jump occurs (community count estimate)

% -------------------------------------------------------------------------
% Step A: Compute sorted eigenvalues of X
% -------------------------------------------------------------------------
X_eig = sort(abs(eig(X)), 'descend');      % sorted by magnitude (largest first)

% Initialize matrix to store inverse eigengap values
M_estimate_diff = zeros(size(X));          % same size as X (square)

% -------------------------------------------------------------------------
% Step B: Build matrix of inverse eigenvalue differences
% -------------------------------------------------------------------------
% For eigenvalues λ_i and λ_j, compute:
%       1 / ( |λ_i − λ_j| + 1 )
% Small eigengap → large value    (indicates "cluster" of informative eigenvalues)
% Large eigengap → small value    (indicates separation)
for i = 1:length(X_eig)
    for j = 1:length(X_eig)
        M_estimate_diff(i,j) = 1 / (abs(X_eig(i) - X_eig(j)) + 1);
    end
end

% -------------------------------------------------------------------------
% Step C: Summarize each row to get spectral signatures
% -------------------------------------------------------------------------
M_estimate_sum = sum(M_estimate_diff, 2);  % sum of each row
M_estimate_sum_diff = abs(diff(M_estimate_sum));  % forward differences

% -------------------------------------------------------------------------
% Step D: Detect largest jump in the first half of spectrum
% -------------------------------------------------------------------------
% Only the first half is used because:
%   - informative eigenvalues lie in the top part,
%   - community count is always ≤ half of total eigenvalues when min size ≥ 2.
max_jump = max(M_estimate_sum_diff(1 : ceil(length(M_estimate_sum_diff) / 2)));

% -------------------------------------------------------------------------
% Step E: Estimate community count as location of max eigengap
% -------------------------------------------------------------------------
M_estimate = find(M_estimate_sum_diff == max_jump);

end
