%       ███╗   ██╗██████╗ ███████╗███████╗
%       ████╗  ██║██╔══██╗██╔════╝██╔════╝
%       ██╔██╗ ██║██████╔╝███████╗█████╗
%       ██║╚██╗██║██╔═══╝ ╚════██║██╔══╝
%       ██║ ╚████║██║     ███████║███████╗
%       ╚═╝  ╚═══╝╚═╝     ╚══════╝╚══════╝
%
%    Copyright (C) 2024 by Future Flow Group
%        Beijing Institute of Technology
%            All rights reserved

restoredefaultpath;
clc; clear; close all;
addpath(genpath('./'));
for iter =2901:3000
    fprintf('第 %d 次执行：\n', iter);
    rng(iter);
    currentDateTime = datetime('now');
    % 1 -- Case definition & Key parameters
    Case.folder  = './';
    Case.info    = 'secondary_Ma0';
    Case.method  = 'Standard';          % Standard / IterFmn / allmodes / twin
    Case.flow    = '2DBL';              % 2DBL / 3DBL / Channel
    Case.fluid   = 'Ideal_Air';         % Ideal_Air / Nonideal_CO2
    Case.name    = [Case.info,'_',Case.method,'_',Case.flow,'_',Case.fluid];
    Case.file    = [Case.folder, Case.name,'.mat'];
    Case.dfile   = [Case.folder, Case.name,'.txt'];
    Case.overwrite_res = true;  % false true
    if exist(Case.dfile, 'file') ; delete(Case.dfile); end
    diary(Case.dfile)
    fprintf('The code runs at: %s\n', currentDateTime);

    switch Case.fluid
        case 'Ideal_Air'
            Case.Ma       = 0.01;
            Case.gama     = 1.4;  % specific heat ratio
            Case.Pr       = 0.72; % Prandtl number
            Case.Ec       = Case.Ma^2*(Case.gama-1);
            Case.Te       = 300;
            Case.Rg       = 1/Case.gama/Case.Ma^2;
            Case.Re0      = 535/sqrt(2); % 430;  % Starting Reynolds number == x0
            Case.x0       = Case.Re0;
            Case.xmax     = 2500; % 3000
            Case.alpha    = 0;


        case 'Nonideal_CO2'
            addpath('./Tables/Data1D');
            Refprop = load('CO2_80.dat');   %  1  2  3  4  5  6  7  8
            %  h  T  p  D  mu Cp ka c
            Case.Ec   = 0.2;
            Case.Tr   = 280;
            Case.pr   = interp1(Refprop(:,2),Refprop(:,3),Case.Tr,'spline');
            Case.rhor = interp1(Refprop(:,2),Refprop(:,4),Case.Tr,'spline');
            Case.mur  = interp1(Refprop(:,2),Refprop(:,5),Case.Tr,'spline');
            Case.kar  = interp1(Refprop(:,2),Refprop(:,7),Case.Tr,'spline');
            Case.Cpr  = interp1(Refprop(:,2),Refprop(:,6),Case.Tr,'spline');
            Case.ur   = sqrt(Case.Ec*Case.Cpr*Case.Tr);
            Case.Pr   = Case.mur*Case.Cpr/Case.kar;
            clear Refprop
    end

    % 2 -- Numerical & Mesh & Derivative Matrix
    Mesh.cheby = false;
    Mesh.N     = 101;
    Mesh.ymax  = 100;
    Mesh.yi    = 20;
    Mesh       = Set_Grid(Mesh,Case);
    Mesh       = Set_Derivative(Mesh);

    Mesh.Nx    = 201; % 301
    Mesh.dx    = (Case.xmax-Case.x0)/(Mesh.Nx-1);
    Mesh.x     = linspace(Case.x0,Case.xmax,Mesh.Nx);
    Mesh.Reall = sqrt(Case.Re0*Mesh.x);

    % 3 -- The base flow
    switch [Case.flow,'_',Case.fluid]
        case '2DBL_Ideal_Air'
            Baseflow = NPSE_BaseFlow_BL(Mesh,Case);
        case '2DBL_Blasius'
            error('to be added')
        case '2DBL_Nonideal_CO2'
            error('to be added')
    end

    % 4 -- LST
    Case.space     = 'PSE';
    Case.Spatial   = true;
    Case.Spatialf  = true;    % Spatial problem, full matrix
    Case.Matrix    = 'Non_ideal_no_curvature';
    Case.NN        = 1;       % numbers of eigenvalues to be solved
    Case.Trans     = 'Sutherland';   % 'FP'  'Power'  'Sutherland'
    Case.EOS       = 'IG';

    Baseflow0      = NaN(Mesh.N,9);
    Baseflow0(:,1) = Baseflow.U(:,1);
    Baseflow0(:,4) = Baseflow.T(:,1);
    Baseflow0(:,7) = Baseflow.Den(:,1);


    % 5 -- Initial PSE
    Case.nonlinear = true;
    Case.mmax     = 5; % 9;
    Case.nmax     = 5; % 9;
    Case.res_a    = 1e-12;
    Case.res_f    = 1e-10;
    Case.start    = 1e-16; % 1e-12 -20
    Fai           = zeros(5*Mesh.N,Mesh.Nx,(1+Case.mmax)*(2*Case.nmax+1));
    alf           = zeros(         Mesh.Nx,(1+Case.mmax)*(2*Case.nmax+1));
    modes         = zeros (1,              (1+Case.mmax)*(2*Case.nmax+1))';
    Case.MN       = mn2mn(Case,Case.mmax,Case.nmax);
    Case.Nz       = 2^3; % 2^5
    Case.Nt       = 2^3; % 2^5
    Case.itermax  = 1000;
    Case.phase    = [0 0];
    Case.m0       = 2;
    Case.n0       = 0;
    Case.mn0      = mn2mn(Case,Case.m0,Case.n0);
    Case.Sigma    = NaN(Case.MN,1);
    Case.amp      = ones(Case.MN,1)*0e-16;
    Case.phase    = zeros(Case.MN,1);
    Case.F        = 40e-6 + (80e-6 - 40e-6) * rand;
    Case.B        = 2e-4 + (9e-4 - 2e-4) * rand; %4.1888e-04; %3.3e-4;

    fprintf('\n');
    fprintf('NPSE: initialization \n');
    switch Case.method
        case {'Standard','twin'}
            fprintf('NPSE strategy: %s \n', Case.method);
            Case.iterFmn    = false; % true \ false
            m = 2; n = -1; mn = mn2mn(Case,m,n);
            Case.amp(mn) =  0.15*1.26e-5+(0.7*1.26e-5-0.15*1.26e-5)*rand; modes(mn) = 1;Case.phase(mn) = 0;
            m = 2; n = +1; mn = mn2mn(Case,m,n);
            Case.amp(mn) = Case.amp(mn2mn(Case,2,-1)); modes(mn) = 1;Case.phase(mn) = 0;
            m = 1; n = -1; mn = mn2mn(Case,m,n);
            Case.amp(mn) =  0.15*1.26e-5+(0.7*1.26e-5-0.15*1.26e-5)*rand; modes(mn) = 1;Case.phase(mn) = 0;
            m = 1; n = +1; mn = mn2mn(Case,m,n);
            Case.amp(mn) = Case.amp(mn2mn(Case,1,-1)); modes(mn) = 1;Case.phase(mn) = 0;
            m = 2; n = 0; mn = mn2mn(Case,m,n);
            Case.amp(mn) = 0.001+(0.01-0.001)*rand; modes(mn) = 1;Case.phase(mn) = 0;
        case 'allmodes'
            fprintf('NPSE strategy: %s \n', Case.method);
            Case.iterFmn    = false; % true \ false
            for m=0:Case.mmax
                for n=-Case.nmax:Case.nmax
                    mn = mn2mn(Case,m,n);
                    Case.amp(mn) = 1e-10; modes(mn) = 1;
                end
            end

            m = 1; n = -1; mn = mn2mn(Case,m,n);
            Case.amp(mn) = 0.001; modes(mn) = 1;
            m = 1; n = +1; mn = mn2mn(Case,m,n);
            Case.amp(mn) = 0.001; modes(mn) = 1;
    end
    load('Table_Ma0.mat');

    for m=0:Case.mmax
        for n=-Case.nmax:Case.nmax
            mn = mn2mn(Case,m,n);
            if m==0 && n==0
                Fai(:,1,mn) = 0;
                alf(  1,mn) = 0;
            else
                Case.Sigma(mn) =  interp2(Table.m_all, Table.n_all, Table.Eval,...
                    m, abs(n), 'cubic', NaN);
                if Case.amp(mn) > 0.00
                    Case.omega  = m*Case.F*Case.Re0;
                    Case.beta   = n*Case.B*Case.Re0;
                    Case.sigma  = Case.Sigma(mn);
                    [Eval,Evec] = LST_Calc(Case,Mesh,Baseflow0);
                    amp0        = max(abs(Evec(2:5:end/2)));
                    Fai(:,1,mn) = Evec(1:end/2)/amp0...
                        *Case.amp(mn)*exp(1i*Case.phase(mn))...
                        *(sqrt(2));
                    alf(  1,mn) = Eval;
                    fprintf(['Mode (%d,%+d), amp=%2.1e, ' ...
                        'a= %.3e %+.3e error=%2.1e \n'],...
                        m,n,Case.amp(mn),real(Eval),imag(Eval),...
                        abs(Eval-Case.Sigma(mn)));
                else
                    alf(  1,mn) = Case.Sigma(mn);
                end
            end

        end
    end


    clear Baseflow0 Table
    clear amp0 ans Eval Evec m n mn
    clear beta

    % PSE
    Case.omega    = Case.F*Case.Re0;
    Case.beta     = Case.B*Case.Re0;
    Case.showFmn    = 9;
    Case.showFai    = 9;
    Case.showRes    = 3;
    Case.showline   = 3;
    %% 7 -- NPSE marching
    try
        fprintf('\n');
        fprintf('NPSE: Marching start. \n');
        if Case.iterFmn
            fprintf('...Strategy: Fmn is iterated with alpha. \n');
        else
            fprintf('...Strategy: Fmn is NOT iterated. \n');
        end


        for xi=2:Mesh.Nx

            fprintf('NPSE: station %d/%d, Re=%.1f/%.1f (%.2f %% active)\n',...
                xi,Mesh.Nx,Mesh.Reall(xi),Mesh.Reall(Mesh.Nx),...
                100*nnz(modes)/length(modes));
            % (a) Give value to live modes
            for i=1:Case.MN
                if xi==2
                    alf(xi,i) = alf(xi-1,i);
                    if modes(i)
                        modes(i) = modes(i)+1; % track its age
                        Fai(:,xi,i) = Fai(:, xi-1,i);
                    end
                else
                    alf(xi,i) = 2*alf(xi-1,i)-alf(xi-2,i);
                    if modes(i)
                        modes(i) = modes(i)+1; % track its age
                        Fai(:,xi,i) = 2*Fai(:, xi-1,i)-Fai(:, xi-2,i);
                    end
                end
            end

            % (b) Calculate Fmn(5*Ny,MN) contributed by live modes
            if Case.nonlinear
                if ~Case.iterFmn
                    [Fmn] = NPSE_Fmn(xi,Case,Mesh,Baseflow,alf,Fai,modes);
                end
                show_Fmn;
            else
                Fmn=zeros(5*Ny,Case.MN);
            end

            % (c) Iteration on the wavenumber & [Fmn]
            for j=1:Case.itermax % maximum iteration number
                if Case.iterFmn
                    Fmn1 = Fmn;
                    [Fmn] = NPSE_Fmn(xi,Case,Mesh,Baseflow,alf,Fai,modes);
                    Fmn2 = Fmn;
                    if xi > 2
                        Fmn=(1-Case.factor)*Fmn2+Case.factor*Fmn1;
                    end
                    Residual_Fmn=max(max(abs(Fmn-Fmn1)));
                else
                    Residual_Fmn=0;
                end

                [gamma,A,A2,B,C,D,Vxx,Vyy,Vzz,Vxy,Vyz,Vxz] =...
                    NPSE_matrix(Baseflow,xi,Case,Mesh); % checked

                % prescribe wavenumber for other modes and calculate JF
                [alf,JF] = update_alpha_harmonics(Case,alf,Mesh,xi);

                % Solve the linear system
                Residual=zeros(Case.MN,1);
                for m=0:Case.mmax
                    for n=-Case.nmax:Case.nmax
                        mn = mn2mn(Case,m,n);
                        if modes(mn)
                            [Residual(mn),alf(xi,mn),Fai(:,xi,mn)]=...
                                NPSE_solve_Chang(Baseflow, xi, Case, Mesh, m, n,...
                                alf(xi,mn),Fai(:,xi-1,mn),Fmn(:,mn),JF(mn),...
                                gamma,A,A2,B,C,D,Vxx,Vyy,Vzz,Vxy,Vxz,Vyz,modes(mn));

                        end
                    end
                end


                [Residual_max,mn] = max(abs(Residual));
                [m,n] = mn2mn(Case,mn);
                fprintf('  res_a(%d,%+d)=%.2e, res_n=%.2e, a=%.7e %.7e i \n',...
                    m,n,Residual_max,  Residual_Fmn,...
                    real(alf(xi,Case.mn0)),imag(alf(xi,Case.mn0)));

                if isnan(Fmn((end+1)/2,1))
                    % break;
                    error('The transition has begun');
                end

                if Residual_max<Case.res_a && Residual_Fmn<Case.res_f
                    break;
                end

                if j==Case.itermax
                    error('itermax = %d reached! \n',j);
                end



            end

            % show amplitude of Fai
            % show_Fai;



            if Case.nonlinear
                for m=0:Case.mmax
                    for n=-Case.nmax:Case.nmax
                        mn = mn2mn(Case,m,n);
                        if modes(mn)==0
                            if max(abs(Fmn(:,mn)))>Case.start
                                modes(mn)=1;
                                fprintf('  Mode(%d ,%d) Started \n',m,n);
                            end
                        end
                    end
                end
                fprintf('\n')
            else
            end
        end
    end
    %%
    % x1 = Mesh.x;
    % alf = reshape(alf,[Mesh.Nx,2*Case.nmax+1,Case.mmax+1]);
    % fai = reshape(Fai,[5*Mesh.N,Mesh.Nx,2*Case.nmax+1,Case.mmax+1]);
    % JF = cumtrapz(x1,alf);
    % rho = fai(1:5:end,:,:,:).*reshape(exp(1i*JF),[1,size(JF)]);
    % T   = fai(5:5:end,:,:,:).*reshape(exp(1i*JF),[1,size(JF)]);
    % Nt = 50;  t = linspace(0,2*pi/Case.omega,Nt+1); t(end)=[];
    % Nz = 100; z = linspace(0,4*2*pi/Case.beta,Nz);
    % rho_phy = zeros(size(rho,1),size(rho,2),Nz,Nt);
    % T_phy = zeros(size(T,1),size(T,2),Nz,Nt);
    % for m=0:Case.mmax
    %     for n=-Case.nmax:Case.nmax
    %         fprintf('mode(%d,%+d) \n',m,n);
    % 
    %         exp_z = exp(1i*n*Case.beta*z);
    %         exp_t = exp(-1i*m*Case.omega*t);
    %         exp_z = reshape(exp_z,[1,1,Nz,1]);
    %         exp_t = reshape(exp_t,[1,1,1,Nt]);
    %         if m==0
    %             rho_phy = rho_phy + real(rho(:,:,n+1+Case.nmax,m+1).*exp_z.*exp_t);
    %             T_phy = T_phy + real(T(:,:,n+1+Case.nmax,m+1).*exp_z.*exp_t);
    %         else
    %             rho_phy = rho_phy + 2*real(rho(:,:,n+1+Case.nmax,m+1).*exp_z.*exp_t);
    %             T_phy = T_phy + 2*real(T(:,:,n+1+Case.nmax,m+1).*exp_z.*exp_t);
    %         end
    %     end
    % end
    % rho_avee=squeeze(mean(rho_phy,3));
    % rho_ave=rho_avee(1,:,:);
    % rho_ave=reshape(rho_ave,[201,50]);
    % T_avee=squeeze(mean(T_phy,3));
    % T_ave=T_avee(1,:,:);
    % T_ave=reshape(T_ave,[201,50]);
    % bst=Baseflow.T(1,:)';
    % bsd=Baseflow.Den(1,:)';
    % P=bst.*rho_ave.*Case.Rg+bsd.*T_ave.*Case.Rg;
    %%
    clear A A2 B C D Fmn gamma i j JF m n mn mn2
    clear Residual_max Residual xi kk index
    clear Vxx Vyy Vzz Vxy Vxz Vyz Residual_Fmn
    diary off
    % 8 -- save resultss
    endTime = datetime('now');
    fprintf('第 %d 次执行结束，开始时间：%s，结束时间：%s\n', iter, currentDateTime, endTime);
    save(['result_0', num2str(iter), '.mat']);
end
fprintf('所有 %d 次执行已完成。\n', numIterations);