

function [BaseFlow]=NPSE_BaseFlow_BL(Mesh,Case)

% extract parameters
Ny    = Mesh.N;
ym    = Mesh.y;
Nx    = Mesh.Nx;
xall  = Mesh.x;


Re0   = Case.Re0;
Pr    = Case.Pr;              
r     = Case.gama;               
Ma    = Case.Ma;               
Te    = Case.Te;  

Reall = sqrt(Re0*xall);

a1    = 110.4/Te;                
a2    = 194/Te;                  
 


% initialization
BaseFlow.U      = NaN(Ny,Nx);
BaseFlow.Ux     = NaN(Ny,Nx);
BaseFlow.Uy     = NaN(Ny,Nx);
BaseFlow.Uxy    = NaN(Ny,Nx);
BaseFlow.Uyy    = NaN(Ny,Nx);

BaseFlow.V      = NaN(Ny,Nx);
BaseFlow.Vx     = NaN(Ny,Nx);
BaseFlow.Vy     = NaN(Ny,Nx);
BaseFlow.Vxy    = NaN(Ny,Nx);
BaseFlow.Vyy    = NaN(Ny,Nx);

BaseFlow.T      = NaN(Ny,Nx);
BaseFlow.Tx     = NaN(Ny,Nx);
BaseFlow.Ty     = NaN(Ny,Nx);
BaseFlow.Tyy    = NaN(Ny,Nx);

BaseFlow.Den    = NaN(Ny,Nx);
BaseFlow.Denx   = NaN(Ny,Nx);
BaseFlow.Deny   = NaN(Ny,Nx);
BaseFlow.Denyy  = NaN(Ny,Nx);


% solve
tmax = 10;
solinit = bvpinit(linspace(0,tmax,100),@NPSE_BaseFlow_BL_guess);
sol=bvp4c(@(x,y) NPSE_BaseFlow_BL_eq(x,y,Case),...
          @NPSE_BaseFlow_BL_bc, solinit);

t0 = linspace(0,tmax,20000);                    
y  = deval(sol,t0);                                 

fprintf(['Baseflow: solved with bvp4c' ...
    ' @ eta=[%3.1f, %7.4f], y=[%3.1f, %7.4f] \n'], 0, tmax,0,y(6,end));


% post-processing
f  = y(1,:);
f1 = y(2,:);   %first derivative of f
f2 = y(3,:);   %second derivative of f
g  = y(4,:);
g1 = y(5,:);   %first derivative of g
G  = y(6,:);    %Integral of g, η

%U0=f1;
%T0=g;
%plot(G,U0,'c')
%plot(G,T0,'m')
%hold on

C1=sqrt(g)*(1+a1)./(g+a1);                      %C1
C2=sqrt(g)*(1+a2)./(g+a2);                      %C2
C11=(1+a1)*g1.*(0.5*g.^(-0.5).*(g+a1)-sqrt(g))./((g+a1).^2); %first derivative of C1
f3=(-0.5*f.*f2-C11.*f2)./C1;                                                 %third derivative of f
g2=(-Pr*(r-1)*Ma^2*C1.*f2.^2 -0.5*Pr*f.*g1-C11.*g1)./C2;       %second derivative of g


U       = f1;
Uy      = f2./g;
Uyy     = (f3.*g-f2.*g1)./g.^3;
T       = g;
Ty      = g1./g;
Tyy     = (g2.*g-g1.^2)./g.^3;
Den     = 1./g;
Deny    = -g1./g.^3;
Denyy   = (3*g1.^2.*g.^2-g2.*g.^3)./g.^7;

Y=G;
[~,n]=size(Y);
ex=10000;
Y_e=linspace(y(6,end),300,ex+1);

for i=1:Nx
    R=Reall(i);

    %The component of the velocity U      
    Ux=-0.5/R*f2.*G./g;
    Uxy=0.5/R*(f2.*g1.*G./g.^3-f2./g-f3.*G./g.^2);
    
    %The component of the velocity V
    V=0.5/R*(G.*f1-f.*g);
    Vx=0.25/R^2*(f.*g1.*G./g-G.^2.*f2./g+f.*g-G.*f1);
    Vy=0.5/R*(G.*f2-f.*g1)./g;
    Vxy=0.25/R^2*((f1.*g1.*G+f.*g2.*G+f.*g1.*g-2*f2.*G.*g-G.^2.*f3)./g+(f2.*g1.*G.^2-f.*g1.^2.*G)./g.^2+f.*g1-G.*f2)./g;
    Vyy=0.5/R*(f2.*g.^2+f3.*g.*G-f1.*g1.*g-f.*g2.*g-f2.*g1.*G+f.*g1.^2);
  
    %The component of the temperature T
    Tx=-0.5/R*(g1.*G./g);

  
    %The component of the density ρ
    Denx=0.5/R*(G.*g1./g.^3);

  
    %Expand the physical fields
    Y0=[Y Y_e(2:end)]*R/Re0;

    U_e=U(n)*ones(1,ex);
    Ux_e=Ux(n)*ones(1,ex);
    Uy_e=Uy(n)*ones(1,ex);
    Uxy_e=Uxy(n)*ones(1,ex);
    Uyy_e=Uyy(n)*ones(1,ex);

    V_e=V(n)*ones(1,ex);
    Vx_e=Vx(n)*ones(1,ex);
    Vy_e=Vy(n)*ones(1,ex);
    Vxy_e=Vxy(n)*ones(1,ex);
    Vyy_e=Vyy(n)*ones(1,ex);

    T_e=T(n)*ones(1,ex);
    Tx_e=Tx(n)*ones(1,ex);
    Ty_e=Ty(n)*ones(1,ex);
    Tyy_e=Tyy(n)*ones(1,ex);
    
    Den_e=Den(n)*ones(1,ex);
    Denx_e=Denx(n)*ones(1,ex);
    Deny_e=Deny(n)*ones(1,ex);
    Denyy_e=Denyy(n)*ones(1,ex);
    
    U2=[U U_e];
    Ux2=[Ux Ux_e]*Re0/R;
    Uy2=[Uy Uy_e]*Re0/R;
    Uxy2=[Uxy Uxy_e]*(Re0/R)^2;
    Uyy2=[Uyy Uyy_e]*(Re0/R)^2;

    V2=[V V_e];
    Vx2=[Vx Vx_e]*Re0/R;
    Vy2=[Vy Vy_e]*Re0/R;
    Vxy2=[Vxy Vxy_e]*(Re0/R)^2;
    Vyy2=[Vyy Vyy_e]*(Re0/R)^2;
    
    T2=[T T_e];
    Tx2=[Tx Tx_e]*Re0/R;
    Ty2=[Ty Ty_e]*Re0/R;
    Tyy2=[Tyy Tyy_e]*(Re0/R)^2;
    
    Den2=[Den Den_e];
    Denx2=[Denx Denx_e]*Re0/R;
    Deny2=[Deny Deny_e]*Re0/R;
    Denyy2=[Denyy Denyy_e]*(Re0/R)^2;

    %interpolation
    BaseFlow.U(:,i)=interp1(Y0,U2,ym,'spline');
    BaseFlow.Ux(:,i)=interp1(Y0,Ux2,ym,'spline');
    BaseFlow.Uy(:,i)=interp1(Y0,Uy2,ym,'spline');
    BaseFlow.Uxy(:,i)=interp1(Y0,Uxy2,ym,'spline');
    BaseFlow.Uyy(:,i)=interp1(Y0,Uyy2,ym,'spline');
    
    BaseFlow.V(:,i)=interp1(Y0,V2,ym,'spline');
    BaseFlow.Vx(:,i)=interp1(Y0,Vx2,ym,'spline');
    BaseFlow.Vy(:,i)=interp1(Y0,Vy2,ym,'spline');
    BaseFlow.Vxy(:,i)=interp1(Y0,Vxy2,ym,'spline');
    BaseFlow.Vyy(:,i)=interp1(Y0,Vyy2,ym,'spline');
    
    BaseFlow.T(:,i)=interp1(Y0,T2,ym,'spline');
    BaseFlow.Tx(:,i)=interp1(Y0,Tx2,ym,'spline');
    BaseFlow.Ty(:,i)=interp1(Y0,Ty2,ym,'spline');
    BaseFlow.Tyy(:,i)=interp1(Y0,Tyy2,ym,'spline');
    
    BaseFlow.Den(:,i)=interp1(Y0,Den2,ym,'spline');
    BaseFlow.Denx(:,i)=interp1(Y0,Denx2,ym,'spline');
    BaseFlow.Deny(:,i)=interp1(Y0,Deny2,ym,'spline');
    BaseFlow.Denyy(:,i)=interp1(Y0,Denyy2,ym,'spline');

    if mod(i,10)==1
        fprintf('Baseflow: extracted at i=%4d, Re=%7.2f, x=%7.2f \n',...
            i,R,xall(i));
    end

end

end


