function [Value,Vector] = LST_Calc(Case,Mesh,Baseflow)

% Parameters
N       = Mesh.N;
beta    = Case.beta;
NN      = Case.NN;
alpha   = Case.alpha;
sigma   = Case.sigma;
if strcmp(Case.fluid, 'Blasius')
    switch Case.space 
        case 'Single'
            omega       = Case.omega;
            Re          = Case.Re;
            [AAA,BBB]   = LST_build_incompressible(Case,Mesh,Baseflow,Re,alpha,beta,omega);
            AAA         = full(AAA);
            BBB         = full(BBB);
            [Vector,Value] = eig(AAA,BBB);
            Value=diag(Value);
            [~,is]=sort(imag(Value));
            Value=Value(is);
            Vector=Vector(:,is);

        case 'Re_F' 
            Re_all    = Case.Re_all;
            F_all     = Case.F_all;
            N_F       = Case.N_F;
            N_Re      = Case.N_Re;
            i_left    = 1;
            i_right   = N_Re;
            j_up      = N_F;
            j_down    = 1;
            istart    = find_closest(Case.Re_all,Case.Re0);
            jstart    = find_closest(Case.F_all, Case.F0);
            
            Value     = zeros(N_Re, N_F, NN);
            if Case.Spatialf
                dim = 8;
            else
                dim = 4;
            end
            Vector    = zeros(dim*N,N_Re,N_F,NN);
            
            % Step1: Iteration on Re at fixed F/omega
            fprintf('Step1: Iteration on Re at fixed F \n');
            j  = jstart;
            F  = Case.F_all(j);
            for i = [istart:i_right,istart-1:-1:i_left]
                Re = Case.Re_all(i);
                omega = F*Re;
                [AAA,BBB] = LST_build_incompressible(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if mod(i,50)==0
                    fprintf('\n');
                    fprintf(['i = %d/%d, Re=%4.3f, omega=%+4.3f, '...
                            'beta = %+3.2f, ', 'F = %4.2e \n'],...
                            i, Case.N_Re, Re, omega, beta, F);
                end
    
                for k=1:NN
                    [Vector(:,i,j,k),Value(i,j,k)]=eigs(AAA,BBB,1,sigma(k));
                    if mod(i,50)==0
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value(i,j,k)),imag(Value(i,j,k)),...
                        abs(Value(i,j,k)-sigma(k)));
                    end
                end

                sigma = LST_update_sigma_Re_F_step1(Value,i,j,istart,i_right);
            end

            
            
            % Step2: Iteration on F/omega now!
            fprintf('\n \n \n');
            fprintf('Step2: Iteration on F now! \n');
            sigma0_step2 = squeeze(Value(:,jstart,:));

            parfor i = i_left:i_right
                sigma = sigma0_step2(i,:);
                Vector_local = zeros(dim*N, N_F, NN);
                Value_local  = zeros(       N_F, NN);
                Value_local(jstart,:) = sigma0_step2(i,:);

            for j = [jstart+1:j_up,jstart-1:-1:j_down]

                Re = Re_all(i);
                F  = F_all(j);
                omega = F*Re;
                [AAA,BBB] = LST_build_incompressible(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if j==jstart-1 
                    % fprintf('\n');
                    fprintf(['i = %3d/%3d, Re=%8.3f, j = %3d/%3d, F=%4.2e, omega=%+4.3f, '...
                            'beta = %+3.2f'],...
                            i, Case.N_Re, Re, j, Case.N_F, F, omega, beta);
                end

                for k=1:NN
                    [ Vector_local(:,j,k),Value_local(j,k)]=eigs(AAA,BBB,1,sigma(k));
                   
                    if j==jstart-1
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value_local(j,k)),imag(Value_local(j,k)),...
                        abs(Value_local(j,k)-sigma(k)));
                    end
                    
                end
                

                Value1 = Value_local(j,:);
                if j>j_down
                    Value2 = Value_local(j-1,:);
                else
                    Value2 = NaN;
                end
                Value3 = Value_local(jstart+1,:);
                Value4 = Value_local(jstart,:);
                if j<j_up
                    Value5 = Value_local(j+1,:);
                else
                    Value5 = NaN;
                end

                sigma = LST_update_sigma_Re_F_step2(Value1,...
                             Value2,Value3,Value4,Value5,j,jstart,j_up);
            end

            Value (i,:,:)= Value_local;
            Vector(:,i,:,:) = Vector_local;
            end


        case 'G_beta'
            G_all     = Case.G_all;
            Be_all    = Case.Be_all;
            N_G       = Case.N_G;
            N_Be      = Case.N_Be;
%             i_left    = 1;
%             i_right   = N_Be;
%             j_up      = N_G;
%             j_down    = 1;
%             istart    = find_closest(Case.Be_all,Case.beta);
%             jstart    = find_closest(Case.G_all, Case.G0);
            i_left    = 1;
            i_right   = N_G;
            j_up      = N_Be;
            j_down    = 1;
            istart    = find_closest(Case.G_all,Case.G0);
            jstart    = find_closest(Case.Be_all, Case.beta);
            
%             R   = Case.R;
%             niu = Case.niu;           %dynamic viscosity
%             Ue  = Case.Ue;                      %inflow velocity
            
%             Value     = zeros(N_Be, N_G, NN);
            Value     = zeros(N_G, N_Be, NN);
            if Case.Spatialf
                dim = 8;
            else
                dim = 4;
            end
%             Vector    = zeros(dim*N,N_Be,N_G,NN);
            Vector    = zeros(dim*N,N_G,N_Be,NN);
            
            % Step1
%             fprintf('Step1: Iteration on Be at fixed G \n');
            fprintf('Step1: Iteration on G at fixed Be \n');
            j  = jstart;
%             G  = G_all(j);
            beta = Be_all(j);
            for i = [istart:i_right,istart-1:-1:i_left]
%                 beta = Be_all(i);
                G  = G_all(i);
                
%                 X0=(G^2*R*sqrt(niu/Ue))^(2/3);
%                 delta0=sqrt(niu*X0/Ue);
%                 Re=delta0*Ue/niu;
%                 K=-delta0/R;
                Re=G/sqrt(-Case.K);
                omega = Case.F0*Re;
                
                Case.Re=Re;
                switch [Case.flow,'_',Case.fluid]
                    case '2DBL_Ideal_Air'
                        Baseflow = BaseFlow_BL(Mesh,Case);
                    case '2DBL_Blasius'
                        Baseflow = BaseFlow_Blasius(Mesh,Case);
                end
                
                [AAA,BBB] = LST_build_incompressible_K(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if mod(i,50)==0
                    fprintf('\n');
%                     fprintf('i = %d/%d, G=%4.3f, beta=%+4.3f, ',...
%                         i, Case.N_Be, G, beta);
                    fprintf('i = %d/%d, G=%4.3f, beta=%+4.3f, ',...
                        i, Case.N_G, G, beta);
                end
                
                for k=1:NN
                    [Vector(:,i,j,k),Value(i,j,k)]=eigs(AAA,BBB,1,sigma(k));
                    if mod(i,50)==0
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                            k, real(Value(i,j,k)),imag(Value(i,j,k)),...
                            abs(Value(i,j,k)-sigma(k)));
                    end
                end
                
                sigma = LST_update_sigma_Re_F_step1(Value,i,j,istart,i_right);
            end
        
            % Step2
            fprintf('\n \n \n');
            fprintf('Step2: Iteration on G now! \n');
            sigma0_step2 = squeeze(Value(:,jstart,:));
            
            for i = i_left:i_right
                sigma = sigma0_step2(i,:);
%                 Vector_local = zeros(dim*N, N_G, NN);
%                 Value_local  = zeros(       N_G, NN);
                Vector_local = zeros(dim*N, N_Be, NN);
                Value_local  = zeros(       N_Be, NN);
                
                Value_local(jstart,:) = sigma0_step2(i,:);
                
                for j = [jstart+1:j_up,jstart-1:-1:j_down]
                    
%                     beta = Be_all(i);
%                     G  = G_all(j);   
                    beta = Be_all(j);
                    G  = G_all(i); 
                    
%                     X0=(G^2*R*sqrt(niu/Ue))^(2/3);
%                     delta0=sqrt(niu*X0/Ue);
%                     Re=delta0*Ue/niu;
%                     K=-delta0/R;
%                     omega = Case.F0*Re;
                     Re=G/sqrt(-Case.K);
                     omega = Case.F0*Re;
                     
                    Case.Re=Re;
                    switch [Case.flow,'_',Case.fluid]
                        case '2DBL_Ideal_Air'
                            Baseflow = BaseFlow_BL(Mesh,Case);
                        case '2DBL_Blasius'
                            Baseflow = BaseFlow_Blasius(Mesh,Case);
                    end
                    
                    [AAA,BBB] = LST_build_incompressible_K(Case,Mesh,Baseflow,Re,alpha,beta,omega,K);
                     % [AAA,BBB] = LST_build_incompressible(Case,Mesh,Baseflow,Re,alpha,beta,omega);

                    if j==jstart-1
                        % fprintf('\n');
                        %                         fprintf('i = %3d/%3d, beta=%8.3f, j = %3d/%3d, G=%4.2e',...
%                             i, Case.N_Be, beta, j, Case.N_G, G);
                        fprintf('i = %3d/%3d, beta=%8.3f, j = %3d/%3d, G=%4.2e',...
                            i, Case.N_G, G, j, Case.N_Be, beta);
                    end
                    
                    for k=1:NN
                        [ Vector_local(:,j,k),Value_local(j,k)]=eigs(AAA,BBB,1,sigma(k));
                        
                        if j==jstart-1
                            fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                                k, real(Value_local(j,k)),imag(Value_local(j,k)),...
                                abs(Value_local(j,k)-sigma(k)));
                        end
                        
                    end
                    
                    
                    Value1 = Value_local(j,:);
                    if j>j_down
                        Value2 = Value_local(j-1,:);
                    else
                        Value2 = NaN;
                    end
                    if j==j_up
                        Value3 = Value_local(jstart+1,:);
                        Value4 = Value_local(jstart,:);
                    else
                        Value3 = NaN;
                        Value4 = NaN;
                    end
                    if j<j_up
                        Value5 = Value_local(j+1,:);
                    else
                        Value5 = NaN;
                    end
                    
                    sigma = LST_update_sigma_Re_F_step2(Value1,...
                        Value2,Value3,Value4,Value5,j,jstart,j_up);
                end
                
                Value (i,:,:)= Value_local;
                Vector(:,i,:,:) = Vector_local;
            end
        

        case 'x_omega' 
            Re          = Case.Re;
            xall        = Case.xall;
            omega_all   = Case.omega_all;
            N_F         = Case.N_F;
            Nx          = Case.Nx;
            i_left      = 1;
            i_right     = Nx;
            j_up        = N_F;
            j_down      = 1;
            istart      = find_closest(xall,Case.x_start);
            jstart      = find_closest(omega_all, Case.omega_start);
            
            Value       = zeros(Nx, N_F, NN);
            if Case.Spatialf
                dim = 8;
            else
                dim = 4;
            end
            Vector      = zeros(dim*N,Nx,N_F,NN);
            
            % Step1: Iteration on Re at fixed F/omega
            fprintf('Step1: Iteration on Re at fixed F \n');
            j  = jstart;
            omega =omega_all(j);
            for i = [istart:i_right,istart-1:-1:i_left]
                
                bf = [Baseflow.U(:,i),Baseflow.Uy(:,i)];
                [AAA,BBB] = LST_build_incompressible(Case,Mesh,bf,...
                                Re,alpha,beta,omega);
                
                if mod(i,50)==0
                    fprintf('\n');
                    fprintf(['i = %d/%d, omega=%+4.3f, '...
                            'beta = %+3.2f, ', '\n'],...
                            i, Case.Nx, omega, beta);
                end
    
                for k=1:NN
                    [Vector(:,i,j,k),Value(i,j,k)]=eigs(AAA,BBB,1,sigma(k));
                    if mod(i,50)==0
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value(i,j,k)),imag(Value(i,j,k)),...
                        abs(Value(i,j,k)-sigma(k)));
                    end
                end

                sigma = LST_update_sigma_Re_F_step1(Value,i,j,istart,i_right);
            end

            
            % Step2: Iteration on omega now!
            fprintf('\n \n \n');
            fprintf('Step2: Iteration on omega now! \n');
            sigma0_step2 = squeeze(Value(:,jstart,:));

            parfor i = i_left:i_right
                sigma = sigma0_step2(i,:);
                Vector_local = zeros(dim*N, N_F, NN);
                Value_local  = zeros(       N_F, NN);
                Value_local(jstart,:) = sigma0_step2(i,:);
                bf = [Baseflow.U(:,i),Baseflow.Uy(:,i)];

            for j = [jstart+1:j_up,jstart-1:-1:j_down]

                omega =omega_all(j);
                [AAA,BBB] = LST_build_incompressible(Case,Mesh,bf,Re,...
                                    alpha,beta,omega);
                
                if j==jstart-1 
                    % fprintf('\n');
                    fprintf(['i = %3d/%3d, Re=%8.0f, j=%3d/%3d, omega=%+4.3f, '...
                            'beta = %+3.2f'],...
                            i, Case.Nx, Re, j, Case.N_F, omega, beta);
                end

                for k=1:NN
                    [ Vector_local(:,j,k),Value_local(j,k)]=eigs(AAA,BBB,1,sigma(k));
                   
                    if j==jstart-1
                        fprintf('mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value_local(j,k)),imag(Value_local(j,k)),...
                        abs(Value_local(j,k)-sigma(k)));
                    end
                    
                end
                

                Value1 = Value_local(j,:);
                if j>j_down
                    Value2 = Value_local(j-1,:);
                else
                    Value2 = NaN;
                end
                Value3 = Value_local(jstart+1,:);
                Value4 = Value_local(jstart,:);
                if j<j_up
                    Value5 = Value_local(j+1,:);
                else
                    Value5 = NaN;
                end

                sigma = LST_update_sigma_Re_F_step2(Value1,...
                             Value2,Value3,Value4,Value5,j,jstart,j_up);
            end

            Value (i,:,:)= Value_local;
            Vector(:,i,:,:) = Vector_local;
            end 

    end
    
% -----------------------------------------------------------------------             
% -----------------------------------------------------------------------            
else   % Non-Blasius
    switch Case.space 
        case 'Single'
            omega       = Case.omega;
            Re          = Case.Re;
            [AAA,BBB]   = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
            AAA         = full(AAA);
            BBB         = full(BBB);
            [Vector,Value] = eig(AAA,BBB);
            Value=diag(Value);
            [~,is]=sort(imag(Value));
            Value=Value(is);
            Vector=Vector(:,is);

        case 'omega_Re'
            Re_all    = Case.Re_all;
            omega_all = Case.omega_all;
            i_left    = 1;
            i_right   = Case.N_Re;
            N_omega   = Case.N_omega;
            j_up      = N_omega;
            j_down    = 1;
            istart    = find_closest(Case.Re_all,Case.Re0);
            jstart    = find_closest(Case.omega_all, Case.omega0);

            if Case.Spatialf
                dim = 10;
            else
                dim = 5;
            end

            Value     = zeros(Case.N_Re, N_omega, NN);
            Vector    = zeros(dim*Mesh.N,Case.N_Re,N_omega,NN);
 

            % Step1: Iteration on Re at fixed F/omega
            j  = jstart;
            omega  = Case.omega_all(j);
            for i = [istart:i_right,istart-1:-1:i_left]
                Re = Case.Re_all(i);
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);

                if mod(i,20)==0
                    fprintf('\n');
                    fprintf(['i = %d/%d, Re=%4.3f, omega=%+4.3f, '...
                        'beta = %+3.2f, '],...
                        i, Case.N_Re, Re, omega, beta);
                end

                for k=1:NN
                    [Vector(:,i,j,k),Value(i,j,k)]=eigs(AAA,BBB,1,sigma(k));
                    if mod(i,20)==0
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                            k, real(Value(i,j,k)),imag(Value(i,j,k)),...
                            abs(Value(i,j,k)-sigma(k)));
                    end
                end
                sigma = LST_update_sigma_Re_F_step1(Value,...
                    i,j,istart,i_right);
            end



            % Step2: Iteration on F/omega now!
            fprintf('\n \n \n');
            fprintf('Step2: Iteration on F now! \n');
            sigma0_step2 = squeeze(Value(:,jstart,:));

            parfor i = i_left:i_right
                sigma = sigma0_step2(i,:);
                Vector_local = zeros(dim*N, N_omega, NN);
                Value_local  = zeros(       N_omega, NN);
                Value_local(jstart,:) = sigma0_step2(i,:);

            for j = [jstart+1:j_up,jstart-1:-1:j_down]

                Re = Re_all(i);
                omega  = omega_all(j);
                F = omega/Re;
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if j==jstart-1 
                    % fprintf('\n');
                    fprintf(['i = %3d/%3d, Re=%8.3f, j = %3d/%3d, omega=%+4.3f, F=%4.2e, '...
                            'beta = %+3.2f'],...
                            i, Case.N_Re, Re, j, N_omega, omega, F, beta);
                end

                for k=1:NN
                    [ Vector_local(:,j,k),Value_local(j,k)]=eigs(AAA,BBB,1,sigma(k));
                   
                    if j==jstart-1
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value_local(j,k)),imag(Value_local(j,k)),...
                        abs(Value_local(j,k)-sigma(k)));
                    end
                    
                end
                

                Value1 = Value_local(j,:);
                if j>j_down
                    Value2 = Value_local(j-1,:);
                else
                    Value2 = NaN;
                end
                Value3 = Value_local(jstart+1,:);
                Value4 = Value_local(jstart,:);
                if j<j_up
                    Value5 = Value_local(j+1,:);
                else
                    Value5 = NaN;
                end

                sigma = LST_update_sigma_Re_F_step2(Value1,...
                             Value2,Value3,Value4,Value5,j,jstart,j_up);
            end

            Value (i,:,:)= Value_local;
            Vector(:,i,:,:) = Vector_local;
            end 

% -----------------------------------------------------------------------            
        case 'x_omega' 
            Re          = Case.Re;
            xall        = Case.xall;
            omega_all   = Case.omega_all;
            N_F         = Case.N_F;
            Nx          = Case.Nx;
            i_left      = 1;
            i_right     = Nx;
            j_up        = N_F;
            j_down      = 1;
            istart      = find_closest(xall,Case.x_start);
            jstart      = find_closest(omega_all, Case.omega_start);
            
            Value       = zeros(Nx, N_F, NN);
            if Case.Spatialf
                dim = 10;
            else
                dim = 5;
            end
            Vector      = zeros(dim*N,Nx,N_F,NN);
            
            % Step1: Iteration on Re at fixed F/omega
            fprintf('Step1: Iteration on x at fixed omega \n');
            j  = jstart;
            omega =omega_all(j);
            for i = [istart:i_right,istart-1:-1:i_left]

                bf = [Baseflow.U(:,i),Baseflow.Uy(:,i),Baseflow.Uyy(:,i),Baseflow.T(:,i),Baseflow.Ty(:,i),Baseflow.Tyy(:,i),Baseflow.Den(:,i),Baseflow.Deny(:,i)];
                [AAA,BBB] = LST_build(Case,Mesh,bf,...
                                Re,alpha,beta,omega);
                
                if mod(i,50)==0
                    fprintf('\n');
                    fprintf(['i = %d/%d, omega=%+4.3f, '...
                            'beta = %+3.2f, ', '\n'],...
                            i, Case.Nx, omega, beta);
                end
    
                for k=1:NN
                    [Vector(:,i,j,k),Value(i,j,k)]=eigs(AAA,BBB,1,sigma(k));
                    if mod(i,50)==0
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value(i,j,k)),imag(Value(i,j,k)),...
                        abs(Value(i,j,k)-sigma(k)));
                    end
                end

                sigma = LST_update_sigma_Re_F_step1(Value,i,j,istart,i_right);
            end

            
            % Step2: Iteration on omega now!
            fprintf('\n \n \n');
            fprintf('Step2: Iteration on omega now! \n');
            sigma0_step2 = squeeze(Value(:,jstart,:));

            parfor i = i_left:i_right
                sigma = sigma0_step2(i,:);
                Vector_local = zeros(dim*N, N_F, NN);
                Value_local  = zeros(       N_F, NN);
                Value_local(jstart,:) = sigma0_step2(i,:);
                
            for j = [jstart+1:j_up,jstart-1:-1:j_down]

                omega =omega_all(j);
                bf = [Baseflow.U(:,i),Baseflow.Uy(:,i),Baseflow.Uyy(:,i),Baseflow.T(:,i),Baseflow.Ty(:,i),Baseflow.Tyy(:,i),Baseflow.Den(:,i),Baseflow.Deny(:,i)];
                [AAA,BBB] = LST_build(Case,Mesh,bf,Re,...
                                    alpha,beta,omega);
                
                if j==jstart-1 
                    % fprintf('\n');
                    fprintf(['i = %3d/%3d, Re=%8.3f, j = %3d/%3d, omega=%+4.3f, '...
                            'beta = %+3.2f'],...
                            i, Case.Nx, Re, j, Case.N_F, omega, beta);
                end

                for k=1:NN
                    [ Vector_local(:,j,k),Value_local(j,k)]=eigs(AAA,BBB,1,sigma(k));
                   
                    if j==jstart-1
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value_local(j,k)),imag(Value_local(j,k)),...
                        abs(Value_local(j,k)-sigma(k)));
                    end
                    
                end
                

                Value1 = Value_local(j,:);
                if j>j_down
                    Value2 = Value_local(j-1,:);
                else
                    Value2 = NaN;
                end
                Value3 = Value_local(jstart+1,:);
                Value4 = Value_local(jstart,:);
                if j<j_up
                    Value5 = Value_local(j+1,:);
                else
                    Value5 = NaN;
                end

                sigma = LST_update_sigma_Re_F_step2(Value1,...
                             Value2,Value3,Value4,Value5,j,jstart,j_up);
            end

            Value (i,:,:)= Value_local;
            Vector(:,i,:,:) = Vector_local;
            end 



% -----------------------------------------------------------------------            
        case 'Re_F'

            Re_all    = Case.Re_all;
            F_all     = Case.F_all;
            N_F       = Case.N_F;
            N_Re      = Case.N_Re;
            i_left    = 1;
            i_right   = N_Re;
            j_up      = N_F;
            j_down    = 1;
            istart    = find_closest(Case.Re_all,Case.Re0);
            jstart    = find_closest(Case.F_all, Case.F0);
            
            Value     = zeros(N_Re, N_F, NN);
            if Case.Spatialf
                dim = 10;
            else
                dim = 5;
            end
            Vector    = zeros(dim*N,N_Re,N_F,NN);
            
            % Step1: Iteration on Re at fixed F/omega
            fprintf('Step1: Iteration on Re at fixed F \n');
            j  = jstart;
            F  = Case.F_all(j);
            for i = [istart:i_right,istart-1:-1:i_left]
                Re = Case.Re_all(i);
                omega = F*Re;
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if mod(i,50)==0
                    fprintf('\n');
                    fprintf(['i = %d/%d, Re=%4.3f, omega=%+4.3f, '...
                            'beta = %+3.2f, ', 'F = %4.2e \n'],...
                            i, Case.N_Re, Re, omega, beta, F);
                end
    
                for k=1:NN
                    [Vector(:,i,j,k),Value(i,j,k)]=eigs(AAA,BBB,1,sigma(k));
                    if mod(i,50)==0
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value(i,j,k)),imag(Value(i,j,k)),...
                        abs(Value(i,j,k)-sigma(k)));
                    end
                end

                sigma = LST_update_sigma_Re_F_step1(Value,i,j,istart,i_right);
            end

            
            
            % Step2: Iteration on F/omega now!
            fprintf('\n \n \n');
            fprintf('Step2: Iteration on F now! \n');
            sigma0_step2 = squeeze(Value(:,jstart,:));

            parfor i = i_left:i_right
                sigma = sigma0_step2(i,:);
                Vector_local = zeros(dim*N, N_F, NN);
                Value_local  = zeros(       N_F, NN);
                Value_local(jstart,:) = sigma0_step2(i,:);

            for j = [jstart+1:j_up,jstart-1:-1:j_down]

                Re = Re_all(i);
                F  = F_all(j);
                omega = F*Re;
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if j==jstart-1 
                    % fprintf('\n');
                    fprintf(['i = %3d/%3d, Re=%8.3f, j = %3d/%3d, F=%4.2e, omega=%+4.3f, '...
                            'beta = %+3.2f'],...
                            i, Case.N_Re, Re, j, Case.N_F, F, omega, beta);
                end

                for k=1:NN
                    [ Vector_local(:,j,k),Value_local(j,k)]=eigs(AAA,BBB,1,sigma(k));
                   
                    if j==jstart-1
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value_local(j,k)),imag(Value_local(j,k)),...
                        abs(Value_local(j,k)-sigma(k)));
                    end
                    
                end
                

                Value1 = Value_local(j,:);
                if j>j_down
                    Value2 = Value_local(j-1,:);
                else
                    Value2 = NaN;
                end
                Value3 = Value_local(jstart+1,:);
                Value4 = Value_local(jstart,:);
                if j<j_up
                    Value5 = Value_local(j+1,:);
                else
                    Value5 = NaN;
                end

                sigma = LST_update_sigma_Re_F_step2(Value1,...
                             Value2,Value3,Value4,Value5,j,jstart,j_up);
            end

            Value (i,:,:)= Value_local;
            Vector(:,i,:,:) = Vector_local;
            end 



        case 'Re_F_temporal'
            Re_all    = Case.Re_all;
            F_all     = Case.F_all; % F is alpha
            N_F       = Case.N_F;
            N_Re      = Case.N_Re;
            i_left    = 1;
            i_right   = N_Re;
            j_up      = N_F;
            j_down    = 1;
            istart    = find_closest(Case.Re_all,Case.Re0);
            jstart    = find_closest(Case.F_all, Case.F0);
            phi       = Case.phi;
            
            Value     = zeros(N_Re, N_F, NN);
            if Case.Spatialf
                dim = 10;
            else
                dim = 5;
            end
            Vector    = zeros(dim*N,N_Re,N_F,NN);
            
            % Step1: Iteration on Re at fixed F/omega
            fprintf('Step1: Iteration on Re at fixed F \n');
            j  = jstart;
            F  = Case.F_all(j);
            for i = [istart:i_right,istart-1:-1:i_left]
                Re = Case.Re_all(i);
                omega = NaN;
                alpha = F;
                beta  = tan(phi)*alpha;
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if mod(i,50)==0
                    fprintf('\n');
                    fprintf(['i = %d/%d, Re=%4.3f, omega=%+4.3f, '...
                            'beta = %+3.2f, ', 'F = %4.2e \n'],...
                            i, Case.N_Re, Re, omega, beta, F);
                end
    
                for k=1:NN
                    [Vector(:,i,j,k),Value(i,j,k)]=eigs(AAA,BBB,1,sigma(k));
                    if mod(i,50)==0
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value(i,j,k)),imag(Value(i,j,k)),...
                        abs(Value(i,j,k)-sigma(k)));
                    end
                end

                sigma = LST_update_sigma_Re_F_step1(Value,i,j,istart,i_right);
            end

            
            
            % Step2: Iteration on F/omega now!
            fprintf('\n \n \n');
            fprintf('Step2: Iteration on F now! \n');
            sigma0_step2 = squeeze(Value(:,jstart,:));

            parfor i = i_left:i_right
                sigma = sigma0_step2(i,:);
                Vector_local = zeros(dim*N, N_F, NN);
                Value_local  = zeros(       N_F, NN);
                Value_local(jstart,:) = sigma0_step2(i,:);

            for j = [jstart+1:j_up,jstart-1:-1:j_down]

                Re = Re_all(i);
                F  = F_all(j);
                omega = NaN;
                alpha = F;
                beta  = tan(phi)*alpha;
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if j==jstart-1 
                    % fprintf('\n');
                    fprintf(['i = %3d/%3d, Re=%8.3f, j = %3d/%3d, F=%4.2e, omega=%+4.3f, '...
                            'beta = %+3.2f'],...
                            i, Case.N_Re, Re, j, Case.N_F, F, omega, beta);
                end

                for k=1:NN
                    [ Vector_local(:,j,k),Value_local(j,k)]=eigs(AAA,BBB,1,sigma(k));
                   
                    if j==jstart-1
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value_local(j,k)),imag(Value_local(j,k)),...
                        abs(Value_local(j,k)-sigma(k)));
                    end
                    
                end
                

                Value1 = Value_local(j,:);
                if j>j_down
                    Value2 = Value_local(j-1,:);
                else
                    Value2 = NaN;
                end
                Value3 = Value_local(jstart+1,:);
                Value4 = Value_local(jstart,:);
                if j<j_up
                    Value5 = Value_local(j+1,:);
                else
                    Value5 = NaN;
                end

                sigma = LST_update_sigma_Re_F_step2(Value1,...
                             Value2,Value3,Value4,Value5,j,jstart,j_up);
            end

            Value (i,:,:)= Value_local;
            Vector(:,i,:,:) = Vector_local;
            end 

        case 'F_B'
            F_all     = Case.F_all;
            B_all     = Case.B_all;
            N_F       = Case.N_F;
            N_B       = Case.N_B;
            i_left    = 1;
            i_right   = N_B;
            j_up      = N_F;
            j_down    = 1;
            istart    = find_closest(B_all, Case.B0);
            jstart    = find_closest(F_all, Case.F0);
            Re        = Case.Re;
            Value     = zeros(N_B, N_F, NN);
            if Case.Spatialf
                dim = 10;
            else
                dim = 5;
            end
            Vector    = zeros(dim*N,N_B,N_F,NN);
            
            % Step1: Iteration on Re at fixed F/omega
            fprintf('Step1: Iteration on B at fixed F \n');
            j  = jstart;
            F  = Case.F_all(j);
            for i = [istart:i_right,istart-1:-1:i_left]
                B = B_all(i);
                beta = B*Re;
                omega = F*Re;
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if mod(i,50)==0
                    fprintf('\n');
                    fprintf(['i = %d/%d, B=%4.3f, omega=%+4.3f, '...
                            'beta = %+3.2f, ', 'F = %4.2e \n'],...
                            i, Case.N_B, B, omega, beta, F);
                end
    
                for k=1:NN
                    [Vector(:,i,j,k),Value(i,j,k)]=eigs(AAA,BBB,1,sigma(k));
                    if mod(i,50)==0
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value(i,j,k)),imag(Value(i,j,k)),...
                        abs(Value(i,j,k)-sigma(k)));
                    end
                end

                sigma = LST_update_sigma_Re_F_step1(Value,i,j,istart,i_right);
            end

            
            
            % Step2: Iteration on F/omega now!
            fprintf('\n \n \n');
            fprintf('Step2: Iteration on F now! \n');
            sigma0_step2 = squeeze(Value(:,jstart,:));

            parfor i = i_left:i_right
                sigma = sigma0_step2(i,:);
                Vector_local = zeros(dim*N, N_F, NN);
                Value_local  = zeros(       N_F, NN);
                Value_local(jstart,:) = sigma0_step2(i,:);

            for j = [jstart+1:j_up,jstart-1:-1:j_down]

                B  = B_all(i);
                F  = F_all(j);
                omega = F*Re;
                beta  = B*Re;
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                
                if j==jstart-1 
                    % fprintf('\n');
                    fprintf(['i = %3d/%3d, B=%8.3f, j = %3d/%3d, F=%4.2e, omega=%+4.3f, '...
                            'beta = %+3.2f'],...
                            i, Case.N_B, B, j, Case.N_F, F, omega, beta);
                end

                for k=1:NN
                    [ Vector_local(:,j,k),Value_local(j,k)]=eigs(AAA,BBB,1,sigma(k));
                   
                    if j==jstart-1
                        fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                        k, real(Value_local(j,k)),imag(Value_local(j,k)),...
                        abs(Value_local(j,k)-sigma(k)));
                    end
                    
                end
                

                Value1 = Value_local(j,:);
                if j>j_down
                    Value2 = Value_local(j-1,:);
                else
                    Value2 = NaN;
                end
                Value3 = Value_local(jstart+1,:);
                Value4 = Value_local(jstart,:);
                if j<j_up
                    Value5 = Value_local(j+1,:);
                else
                    Value5 = NaN;
                end

                sigma = LST_update_sigma_Re_F_step2(Value1,...
                             Value2,Value3,Value4,Value5,j,jstart,j_up);
            end

            Value (i,:,:)= Value_local;
            Vector(:,i,:,:) = Vector_local;
            end 


        case 'PSE'
            omega       = Case.omega;
            Re          = Case.Re0;
            [AAA,BBB]   = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
            AAA         = full(AAA);
            BBB         = full(BBB);
            Value     = zeros(NN,1);
            if Case.Spatialf
                    Vector=zeros(10*Mesh.N,NN);
                else
                    Vector=zeros(5*Mesh.N,NN);
            end
            for k=1:NN
                [Vector(:,k),Value(k)]=eigs(AAA,BBB,1,sigma(k));

                % fprintf(['Initial: mode%d, ' ...
                %     'alpha=(%+3.1e,%+3.1e), error=+%3.1e \n'],...
                % k, real(Value(k)),imag(Value(k)),...
                % abs(Value(k)-sigma(k)));
            end
            
        case 'Re' 
            % baseflow does not change along Re (scaled locally)
            omega_all = Case.omega;
            Re_all    = Case.Re;
            Re0       = Case.Re0;
            istart    = find_closest(Re_all,Re0);
            i_left    = Case.i_left;
            i_right   = Case.i_right;
            Value     = zeros(Case.N_Re, NN);
            if Case.Spatialf
                    Vector=zeros(10*Mesh.N,Case.N_Re,NN);
                else
                    Vector=zeros(5*Mesh.N,Case.N_Re,NN);
            end
    
            for i = [istart:i_right,istart-1:-1:i_left]
                Re = Re_all(i);
                omega = omega_all(i);
                [AAA,BBB] = LST_build(Case,Mesh,Baseflow,Re,alpha,beta,omega);
                fprintf('\n');
                fprintf(['i = %d/%d, Re=%4.3f, omega=%+4.3f, '...
                        'beta = %+3.2f, ', 'F = %4.2e \n'],...
                        i, Case.N_Re, Re, omega, beta, Case.F);
    
                for k=1:NN
                    [Vector(:,i,k),Value(i,k)]=eigs(AAA,BBB,1,sigma(k));
    
                    fprintf(' mode%d, alpha=(%+3.1e,%+3.1e), error=+%3.1e \n',...
                    k, real(Value(i,k)),imag(Value(i,k)),...
                    abs(Value(i,k)-sigma(k)));
                end
    
                sigma = LST_update_sigma(Case,Value,i,istart,i_left,i_right);
    
                figure(1);
    
                subplot(2,2,1);hold on;
                xlim([0 max(Case.Re)])
                ylim([1-1/Case.Ma-0.4,1+1/Case.Ma+0.4])
                plot(Re,omega/real(Value(i,1)),'r.')
                plot(Re,omega/real(Value(i,2)),'b.')
                xlabel('$Re$')
                ylabel('$c$')
    
                subplot(2,2,2);hold on;
                plot(Re,imag(Value(i,1)),'r.')
                plot(Re,imag(Value(i,2)),'b.')
                xlabel('$Re$')
                ylabel('$\alpha_i$')
    
                subplot(2,2,3);
                tmp=max(abs(Vector(2:5:end/2,i,1)));
                plot(abs(Vector(1:5:end/2,i,1))/tmp,Mesh.y)
                hold on;
                plot(abs(Vector(2:5:end/2,i,1))/tmp,Mesh.y)
                plot(abs(Vector(3:5:end/2,i,1))/tmp,Mesh.y)
                plot(abs(Vector(4:5:end/2,i,1))/tmp,Mesh.y)
                plot(abs(Vector(5:5:end/2,i,1))/tmp,Mesh.y)
                hold off
                legend('$\rho$','$u$','$v$','$w$','$T$')
                xlabel('$|q|$')
                ylabel('$y$')
    
                subplot(2,2,4);
                tmp=max(abs(Vector(2:5:end/2,i,2)));
                plot(abs(Vector(1:5:end/2,i,1))/tmp,Mesh.y)
                hold on;
                plot(abs(Vector(2:5:end/2,i,1))/tmp,Mesh.y)
                plot(abs(Vector(3:5:end/2,i,1))/tmp,Mesh.y)
                plot(abs(Vector(4:5:end/2,i,1))/tmp,Mesh.y)
                plot(abs(Vector(5:5:end/2,i,1))/tmp,Mesh.y)
                hold off
                legend('$\rho$','$u$','$v$','$w$','$T$')
                xlabel('$|q|$')
                ylabel('$y$')
    
    
            end

    end
end










end


