library(tidyverse)
library(tuneR)

semi_random_permute <- function(..., n = 1, exclude = NULL){
  # Creates a rnadom permutation of the provided arguments, with some conditions
  # ...: named list of arguments that define the permutations
  # n: number of arguments contrasined to remain the same between successive states
  # exclude: permutations to exclude
  df <- cross_df(.l = list(...), .filter = exclude)
  
  if (n > ncol(df) - 1) stop("n is ", n, " but max allowable value is ", ncol(df) - 1, " (i.e. number of passed arguments other than n - 1).")
  
  cols <- colnames(df)
  samp <- slice(df, sample.int(nrow(df), 1))
  df <- anti_join(df, samp, by = cols)
  
  for (i in 1:(nrow(df))){
    dataf <- filter(df, 
                    map2(.x = df, .y = samp[nrow(samp), ], 
                         .f = `%in%`) %>% 
                      transpose() %>% 
                      map_lgl(.f = ~ reduce(.x, .f = `+`) == n))
    if (nrow(dataf) > 0) samp <- bind_rows(samp, slice(dataf, sample.int(nrow(dataf), 1))) 
    else next
    df <- anti_join(df, samp, by = cols)
  }
  return(samp)
}


generate <- function(x = NULL, bpm = 48, samp.rate = 48000, p = .1){
  if (!is.null(x)) { samp.rate <- x@samp.rate}
  
  if (is.null(x)) x <- tuneR::noise(duration = round(60 / bpm * samp.rate), samp.rate = samp.rate)
  else if (length(x) < round(samp.rate * 60 / bpm)){
    stop("bpm is ", bpm, ". x should then be of length at least ", 
         60 / bpm, " s but is", length(x) / samp.rate,
         ". Please provide x at least this long.")
  } else { 
    x@left <- x@left[1:round(60 / bpm * samp.rate)]}
  
  x_list <- map(.x = p, 
                .f = ~ {
                  c(x@left[1:round(60 / bpm * samp.rate * (1 - .x))], 
                    rep(0, round(60 / bpm * samp.rate * .x)))
                })
  
  return(Wave(left = unlist(x_list), samp.rate = samp.rate, bit = x@bit, pcm = x@pcm))
}

repeat_wav <- function(x, total=15){
  times <- total * x@samp.rate / length(x@left)
  x@left <- rep(x@left, times)
  return(x)
}


create_stimulus <- function(x, bpm, rhythm = c('isochronous', "binary", "ternary"), total = 15, silence = 5, center=TRUE, scale=TRUE, filename="seq", path=NULL){
  p <- switch(match.arg(rhythm),
              isochronous = c(.2),
              binary = c(.2, .5),
              ternary = c(.2, .5, .5))
  
  seq <- generate(x, bpm = bpm, p = p) 
  seq <- repeat_wav(seq, total = total)

  if (center) seq@left <- seq@left - mean(seq@left)
  if (scale) seq@left <- (seq@left - min(seq@left)) / (max(seq@left) - min(seq@left)) * (32767 + 32768) - 32768
  
  seq@left <- c(seq@left, rep(0, seq@samp.rate * silence))
  
  filename <- sprintf("%s_bpm%s_%s.wav", filename, bpm, rhythm)
  if (!is.null(path)) filename <- file.path(path, filename)
  writeWave(seq, filename = filename)
}

