library(tidyverse)
library(tuneR)

setwd(dirname(rstudioapi::getActiveDocumentContext()$path))

source("stimulus_utils.R")

# Combinations of tempo and rhythm
bpm <- c(36, 60, 84, 120) 
rhythm <- c("iso", "bin", "ter")

# Parameters (in seconds)
duration <- 15 # Length of rhythmic sequence
silence <- 5 # Length of silent interval between consecutive sequences
total_duration <- 120 # Length of final output

n_sequences <- as.integer(total_duration / (duration + silence))

# Paths
in_path <- "original_files"
seq_path <- 'rhythmic_sequences'
out_path <- 'final_outputs'

# Read the stimulus files
stimuli_files <- dir(in_path, pattern = "stimulus.wav")
stimuli_sounds <- map(.x = stimuli_files,
                      .f = ~ readWave(file.path(in_path, .x)) %>% 
                        (function(x) { 
                          Wave(left = x@left, samp.rate = x@samp.rate, bit = x@bit, pcm = x@pcm)
                          }))

# Creates the rhythmic sequences from the original 'stimulus' files
map2(.x = stimuli_sounds, 
     .y = stimuli_files,
     .f = ~ cross_df(.l = list(bpm = bpm, rhythm = rhythm)) %>% 
       pmap(.f = create_stimulus,
            x = .x,
            total = duration,
            silence = silence,
            scale=F, center=F,
            path = seq_path,
            filename = .y %>% str_remove("_stimulus.wav")))

# Generate N different stimuli (one stimulus = combinations of rhythmic sequences) for the final outputs
N <- 30
seq <- map(.x = 1:N, .f = ~ {
  semi_random_permute(bpm = bpm, rhythm = rhythm, n = 1) %>%
    slice(1:n_sequences)
})
while(diff(range(table(bind_rows(seq)))) > 2){
  seq <- map(.x = 1:N, .f = ~ {
    semi_random_permute(bpm = bpm, rhythm = rhythm, n = 1) %>%
      slice(1:n_sequences)
  })
  print(range(table(bind_rows(seq))))
}

# Create the final outputs and save them
for (name in stimuli_files){
  name <- str_remove(name, '_stimulus.wav')
  
  for (sequence in seq){
    filename <- pmap(sequence, .f = paste0) %>% unlist() %>% paste(collapse="_")
    filename <- sprintf("%s_%s.wav", name, filename)
    
    wav <- sequence %>% mutate(file = sprintf("%s_bpm%s_%s.wav",
                                              name %>% str_remove("_stimulus.wav"),
                                              bpm,
                                              rhythm))
    wav <- map(.x = wav$file, .f = ~ readWave(file.path(seq_path, .x)))
    samp.rate <- wav[[1]]@samp.rate
    bit <- wav[[1]]@bit
    pcm <- wav[[1]]@pcm
    wav <- map(.x = wav, .f = ~ .x@left) %>% unlist()
    wav <- Wave(left = wav, samp.rate = samp.rate, bit = bit, pcm = pcm)
    
    # rites out the final stimulus sound file
    writeWave(wav, filename=file.path(out_path, filename))
    
    timings <- rle(wav@left == 0 & c(wav@left[2:length(wav@left)], 0) == 0)
    runs <- c(0, cumsum(timings$lengths)[-length(timings$lengths)])
    timings <- data.frame(start = runs[seq(1, length(runs), 2)] / samp.rate,
                          end = runs[seq(2, length(runs), 2)] / samp.rate)

    # writes out the timings for the starts/ends of the stimuli to later calculate latencies
    write.table(x=timings, file=file.path(out_path, filename) %>% str_replace('wav', 'txt'), quote=F, sep='\t', col.names = F, row.names = F)

  }
}
