suppressPackageStartupMessages({
  required_packages <- c(
    "tidyverse", "ggplot2"
  )
  
  for (pkg in required_packages) {
    if (!requireNamespace(pkg, quietly = TRUE)) {
      install.packages(pkg)
    }
    library(pkg, character.only = TRUE)
  }
})

# Source the regional allocation algorithm and define "not in" operator
source("regional_allocation_algorithm.R")
'%!in%' <- function(x,y)!('%in%'(x,y))

# Load historical solar PV data by region (disaggregated and aggregated)
global_solar_decomp <- read_csv("data/input/solarpv_historical_by_region_disagg.csv") 
regional_solar <- read_csv("data/input/solarpv_historical_by_region_agg.csv")

# Load initialization data for solar PV
initialisation_solar <- read_csv("data/input/solarpv_initialisation.csv")

# Set deployment ceiling values for each region
regional_ceilings <- regional_solar %>% select(Region) %>% distinct()%>% 
  mutate(Ceiling=case_when(Region=="East Asia"~0.5,
                           Region=="North America"~0.45,
                           Region=="South Asia"~0.55,
                           Region=="Europe"~0.45,
                           Region=="Asia-Pacific Developed"~0.3,
                           Region=="South-East Asia and developing Pacific"~0.55,
                           Region=="Africa"~0.55,
                           Region=="Eurasia"~0.4,
                           Region=="Latin America and Caribbean"~0.45,
                           Region=="Middle East"~0.55)
  )

# Prepare baseline data for regional additions
regional_add_baseline <- regional_solar %>% filter(Year==2023) %>% 
  select(Region, Rate=Region.Add, R3=Region.R3, Weight=weight, Region.Share) %>% distinct() %>% 
  mutate(Rate=case_when(Rate<0~0, TRUE~Rate), R3=case_when(R3<0~0, TRUE~R3)) %>% left_join(regional_ceilings)

# Load counterfactual trajectory data
counterfactual_trajectories <- read_csv("data/input/counterfactual_trajectories_w_baseline.csv") %>% 
  filter(Technology=="Solar PV") %>% 
  rename(Share=Deployment) %>% filter(Year>2023)

# Get unique scenario names
counterfactual_scenarios <- unique(counterfactual_trajectories$Scenario)

# Initialize empty dataframes for results
regional_rates <- data.frame()
regional_deployment <- data.frame()
global_add <- data.frame()
global_summary <- data.frame()

# Process each scenario
for (scenario in counterfactual_scenarios) {
  
  # Prepare trajectory data for current scenario
  traj_long <-counterfactual_trajectories %>% filter(Scenario==scenario) %>% 
    select(Year, Share) %>% 
    rbind(initialisation_solar %>% select(Year, Share=Global.Share)) %>% arrange(Year) %>% 
    mutate(Rate=Share-lag(Share)) %>% drop_na() %>% 
    rbind(initialisation_solar %>% select(Year, Share=Global.Share, Rate=Global.Rate)) %>% arrange(Year)
  
  # Calculate regional trajectory acceleration
  result_scenario <- calculate_trajectory_acceleration(
    current_rates = regional_add_baseline$R3,
    current_values =  regional_add_baseline$Region.Share,
    weights = regional_add_baseline$Weight,
    region_names = regional_add_baseline$Region,
    target_trajectory = traj_long %>% select(Year, Target_Value=Share),
    max_rate = 0.03,
    max_yearly_accel = 0.01,
    max_deployment = regional_add_baseline$Ceiling
  )
  
  # Format regional rates results
  scenario_regional_rates <- result_scenario$rates %>% 
    pivot_longer(
      cols = -Year,           
      names_to = "Region",    
      values_to = "Rate"       
    ) %>% arrange(Region, Year) %>% mutate(Scenario=scenario)
  
  # Format regional deployment results
  scenario_regional_deployment <- result_scenario$values %>% 
    pivot_longer(
      cols = -Year,           
      names_to = "Region",    
      values_to = "Deployment"       
    ) %>% arrange(Region, Year) %>% 
    group_by(Region) %>% 
    mutate(Add=Deployment-lag(Deployment)) %>% ungroup() %>% left_join(result_scenario$region_info) %>% 
    mutate(Contribution=Add*Weight) %>% mutate(Scenario=scenario)
  
  # Format global additions
  scenario_global_add<- result_scenario$diagnostics %>% select(Year, Add=Achieved_Growth) %>% 
    rbind(initialisation_solar %>% select(Year, Add=Global.Rate)) %>% 
    mutate(Scenario=scenario)
  
  # Combine results from all scenarios
  regional_rates <- regional_rates %>% rbind(scenario_regional_rates)
  regional_deployment <- regional_deployment %>% rbind(scenario_regional_deployment)
  global_add <- global_add %>% rbind(scenario_global_add)
  global_summary <- global_summary %>% rbind(result_scenario$diagnostics %>% mutate(Scenario=scenario))
}

# Output files 
# write_csv(regional_rates, "data/output/counterfactuals/solar/regional_rates.csv")
# write_csv(regional_deployment, "data/output/counterfactuals/solar/regional_deployment.csv")
# write_csv(global_add, "data/output/counterfactuals/solar/global_add.csv")
# write_csv(global_summary, "data/output/counterfactuals/solar/global_summary.csv")

# Set factor levels for scenarios
regional_deployment$Scenario <- factor(regional_deployment$Scenario, 
                                       levels = c('Baseline', 'Late acceleration', 
                                                  'Early acceleration'))

# Define major regions for analysis
major_regions <- c('East Asia', 'North America', 'Europe', 'South Asia', 'Latin America and Caribbean')

# Filter for regions not in major regions
others_data <- regional_deployment %>% 
  filter(Region %!in% major_regions)

# Aggregate non-major regions into "Others" category
others_deployment <- others_data %>%
  group_by(Year, Scenario) %>%
  summarize(
    WeightedSum = sum(Deployment * Weight),
    TotalOthersWeight = sum(Weight),
    Deployment = WeightedSum / TotalOthersWeight,
    Contribution = sum(Contribution),
    Weight = TotalOthersWeight,
    Cat = "Others",
    .groups = 'drop'
  ) %>% ungroup() %>% 
  select(Year, Cat, Deployment, Weight, Contribution, Scenario) %>% arrange(Scenario, Year) %>% 
  group_by(Scenario, Cat) %>% 
  mutate(Add=Deployment-lag(Deployment)) %>% ungroup() %>% distinct()

# Combine major regions with "Others" category
regional_deployment_big5_solar <- regional_deployment %>% select(-Max_Deployment) %>% 
  filter(Region %in% major_regions) %>% 
  rename(Cat=Region) %>% 
  rbind(others_deployment) %>% distinct() 

# Output file is commented out
# write_csv(regional_deployment_big5_solar, 'data/output/counterfactuals/solar/regional_grouped.csv')

# Create historical data with same regional grouping 
regional_solar_big5 <- regional_solar %>% 
  filter(Region %!in% major_regions) %>% 
  group_by(Year) %>%
  summarize(
    WeightedSum = sum(Region.Share * weight),
    TotalOthersWeight = sum(weight),
    Deployment = WeightedSum / TotalOthersWeight,
    Contribution = sum(weight*Region.Add),
    Weight = TotalOthersWeight,
    Cat = "Others",
    .groups = 'drop'
  ) %>% ungroup() %>% 
  select(Year, Cat, Deployment, Weight, Contribution) %>% arrange(Year) %>% 
  mutate(Add=Deployment-lag(Deployment)) %>% 
  rbind(regional_solar %>% 
          filter(Region %in% major_regions) %>% mutate(Contribution=weight*Region.Add) %>% 
          select(Year, Cat=Region, Deployment=Region.Share, Weight=weight, Add=Region.Add, Contribution)
  ) %>% distinct()


# Set factor levels for region categories
regional_deployment_big5_solar$Cat <- factor(regional_deployment_big5_solar$Cat, 
                                             levels = c('East Asia', 'North America', 'Europe', 'South Asia', 'Latin America and Caribbean',
                                                        'Others'))

regional_solar_big5$Cat <- factor(regional_solar_big5$Cat, 
                                  levels = c('East Asia', 'North America', 'Europe', 'South Asia', 'Latin America and Caribbean',
                                             'Others'))

# Set factor levels for scenarios (order matters for visualization)
regional_deployment_big5_solar$Scenario <- factor(regional_deployment_big5_solar$Scenario, 
                                                  levels = c('Baseline', 'Late acceleration', 
                                                             'Early acceleration'))

# Format historical data with same categories
global_solar_decomp_big5 <- global_solar_decomp %>% 
  mutate(Cat=case_when(Region %in% major_regions~Region, TRUE~"Others"))

global_solar_decomp_big5$Cat <- factor(global_solar_decomp_big5$Cat, 
                                       levels = c('East Asia', 'North America', 'Europe', 'South Asia', 'Latin America and Caribbean',
                                                  'Others'))

# Define color palette for regions
reg_clrs <- c('East Asia' = '#fb8072', 'North America' = '#8dd3c7', 'Europe' = '#80b1d3',
              'South Asia' = '#b3de69', 'Latin America and Caribbean' = '#fdb462',
              'Others' = '#BFD8E5')

# Create stacked bar plot of regional contributions
global_add_plot_big5_solar <- ggplot()+
  geom_col(data = regional_deployment_big5_solar, aes(x=Year, y=Contribution, fill=Cat))+
  geom_col(data = global_solar_decomp_big5, aes(x=Year, y=Contribution, fill = Cat))+
  geom_vline(xintercept = 2023, linetype=2, alpha=0.5)+
  geom_vline(xintercept = 2030, linetype=4)+
  facet_wrap(~Scenario, labeller = label_wrap_gen(width = 15))+
  scale_y_continuous(labels = scales::percent)+
  scale_fill_manual(values = reg_clrs)+
  theme_bw()+
  theme(panel.grid.minor = element_blank(),
        panel.grid.major.x = element_blank(), 
        title = element_text(face="bold", color = '#0C0C0C', size=10),
        plot.title = element_text(hjust = 0.5, size=10),
        strip.text = element_text(face="bold"),
        panel.spacing = unit(16, "pt"),
        legend.position = "bottom",
        strip.background = element_blank())+
  labs(y="Annual additions", x="Year", fill="Region", title = "Solar PV")

# Display the plot
global_add_plot_big5_solar
