suppressPackageStartupMessages({
  required_packages <- c(
    "tidyverse", "ggplot2", "dplyr", "minpack.lm", "rstatix", "Rbeast", "zoo", "scales"
  )
  
  for (pkg in required_packages) {
    if (!requireNamespace(pkg, quietly = TRUE)) {
      install.packages(pkg)
    }
    library(pkg, character.only = TRUE)
  }
})

# Define "not in" operator
'%!in%' <- function(x,y)!('%in%'(x,y))

source("fitting_functions.R")

# Read and prepare technology data
# Data should be structured with columns: Country, Year, Value, Total, Share
# Value = Electricity generation from source e.g. solar PV/onshore wind (or equivalent measure of deployment in absolute terms for other technologies)
# Total = Total electricity generation (or equivalent measure of market size in absolute terms for other technologies)
# Share = Value/Total

y0 <- read_csv('data/input/national_solarpv.csv') #path to the input file
# y0 <- read_csv('data/input/national_onshore_wind.csv') #for onshore wind

# Find first year with positive adoption for each country
min_year <- y0 %>% group_by(Country) %>% filter(Value>0) %>% 
  filter(Year==min(Year)) %>% ungroup() %>% 
  select(Country, Min.Year=Year) %>% distinct()

# Filter data to include only countries with at least 5 years of data
tech_rates <- y0 %>% 
  select(Country, Year, Value, Share, Total) %>% 
  left_join(min_year) %>% 
  group_by(Country) %>% 
  filter(Year>=Min.Year) %>% 
  filter(length(Year)>=5) %>% ungroup()

cases <- unique(tech_rates$Country)

# Create empty dataframe for takeoff years
takeoff <- data.frame(Country = character(), TO.Year = numeric(), Probability.TO = numeric(),
                      TO.Year.Share = numeric(), Probability.TO.Share = numeric())

# Identify takeoff points for each country using Bayesian change point detection
for (t in cases) {
  tryCatch({
    # Filter data for the current country
    abs_dep <- tech_rates %>% filter(Country == t)
    
    # Detect breakpoints in absolute values and shares
    out_dep <- beast(abs_dep$Value, start=abs_dep$Year[1], season = "none", tseg.leftmargin=5, print.progress=FALSE)
    out_dep_share <- beast(abs_dep$Share, start=abs_dep$Year[1], season = "none", tseg.leftmargin=5, print.progress=FALSE)
    
    # Process absolute value breakpoints
    threshold_dep0 <- out_dep$trend$inc_cp
    probability_dep0 <- out_dep$trend$inc_cpPr
    dep0_table <- data.frame(Threshold=threshold_dep0, Probability=probability_dep0) %>% filter(Threshold!=Inf, Probability>0.1)
    threshold_dep <- min(dep0_table$Threshold, na.rm = TRUE)
    probability_dep1 <- dep0_table %>% filter(Threshold==threshold_dep)
    probability_dep <- probability_dep1$Probability[1]
    
    # Process share value breakpoints
    threshold_dep0_share <- out_dep_share$trend$inc_cp
    probability_dep0_share <- out_dep_share$trend$inc_cpPr
    dep0_table_share <- data.frame(Threshold=threshold_dep0_share, Probability=probability_dep0_share) %>% 
      filter(Threshold!=Inf, Probability>0.1)
    threshold_dep_share <- min(dep0_table_share$Threshold, na.rm = TRUE)
    probability_dep1_share <- dep0_table_share %>% filter(Threshold==threshold_dep_share)
    probability_dep_share <- probability_dep1_share$Probability[1]
    
    # Store results
    to <- data.frame(Country = t, TO.Year = threshold_dep, Probability.TO = round(probability_dep, digits = 2),
                     TO.Year.Share = threshold_dep_share, Probability.TO.Share = round(probability_dep_share, digits = 2))
    
    takeoff <- takeoff %>% bind_rows(to)
    
    cat("Successfully processed", t, "\n")
    
  })
}

#NOTE: It is normal for there to be errors or warnings after this finishes executing: these are
#     cases where the algorithm fails to identify takeoff



# Filter valid takeoff years and save
takeoff_final <- takeoff %>% filter(TO.Year!=Inf | TO.Year.Share != Inf)

# write_csv(takeoff_final, 'data/output/national_solar_takeoff_years.csv') #export identified takeoff years

# Get threshold values at takeoff points
takeoff_thresholds <- takeoff_final %>% left_join(y0 %>% select(Country, TO.Year=Year, TO.Threshold=Share)) %>% 
  left_join(y0 %>% select(Country, TO.Year.Share=Year, TO.Threshold.Share=Share))

# write_csv(takeoff_thresholds, 'data/output/national_solar_takeoff_shares.csv') #export identified takeoff thresholds

# Filter data for curve fitting (absolute values)
y1 <- y0 %>% left_join(takeoff_final %>% select(Country, TO.Year)) %>% 
  drop_na() %>% 
  group_by(Country) %>% 
  filter(Year>=TO.Year, TO.Year!=Inf)%>% 
  mutate(MinMax5=TO.Year+4, Max.Year=max(Year)) %>% 
  ungroup() %>% filter(!is.na(MinMax5), MinMax5 <= Max.Year) %>%
  select(Country, TO.Year, MinMax5, Max.Year) %>% distinct() %>% drop_na()

y.to <- y1 %>% select(Country, TO.Year) 
cnts <- unique(y1$Country) 
s.abs <- y0 %>% select(Country, Year, Value)

# Fit growth curves for absolute values
df.fits.abs <- data.frame()

for (cnt in cnts) {
  yt <- y1 %>% filter(Country == cnt)
  to.yr <- yt$TO.Year[1]
  min.year <- yt$MinMax5[1]
  max.year <- yt$Max.Year[1]
  for (y in min.year:max.year) {
    rt <- s.abs %>% filter(Country == cnt, Year <= y, Year>=to.yr) %>%
      mutate(Country = str_c(Country, as.integer(y))) %>%
      select(Country, Year, Value)
    fit <- fit_curve(rt, fit = c("S", "G", "E", "L")) %>%
      mutate(Country = cnt, Max.Year = y)
    df.fits.abs <- df.fits.abs %>% rbind(fit)
  }}

fit1 <- df.fits.abs %>% filter(Good==1) 

# Calculate and save growth parameters for absolute values
sample_parameters.abs <- fit1 %>% left_join(y.to) %>% 
  select(Max.Year, Fit, Country, Maturity, K, L, G, TMax) %>% 
  mutate(TMax=round(TMax)) %>% mutate(Year=TMax) %>% 
  left_join(y0 %>% select(Country, Year, Total)) %>% 
  mutate(G.Size=G/Total, L.Size=L/Total) %>% select(-c(Year, Total)) %>% 
  mutate(Model=case_when(Fit=="S"~"Logistic", Fit=="G"~"Gompertz", Fit=="E"~"Exponential", Fit=="L"~"Logistic-linear")) %>% 
  mutate(dT=case_when(Fit=="S"~log(81)/K, Fit=="G"~log(log(0.1)/log(0.9))/K, TRUE~0)) %>% 
  rename(Year=Max.Year) %>% left_join(takeoff_final %>% select(Country, TO.Year)) %>% 
  filter(TMax>TO.Year)

# write_csv(sample_parameters.abs, 'NESub/data/output/national_solar_parameters_abs.csv') #export fitted parameters

# Filter data for curve fitting (share values)
y2 <- y0 %>% left_join(takeoff_final %>% select(Country, TO.Year=TO.Year.Share)) %>% 
  drop_na() %>% 
  group_by(Country) %>% 
  filter(Year>=TO.Year, TO.Year!=Inf)%>% 
  mutate(MinMax5=TO.Year+4, Max.Year=max(Year)) %>% 
  ungroup() %>% filter(!is.na(MinMax5), MinMax5 <= Max.Year) %>%
  select(Country, TO.Year, MinMax5, Max.Year) %>% distinct() %>% drop_na()

y.to <- y2 %>% select(Country, TO.Year) 
cnts <- unique(y2$Country) 
s.share <- y0 %>% select(Country, Year, Value=Share)

# Fit growth curves for share values
df.fits.share <- data.frame()

for (cnt in cnts) {
  yt <- y2 %>% filter(Country == cnt)
  to.yr <- yt$TO.Year[1]
  min.year <- yt$MinMax5[1]
  max.year <- yt$Max.Year[1]
  for (y in min.year:max.year) {
    rt <- s.share %>% filter(Country == cnt, Year <= y, Year>=to.yr) %>%
      mutate(Country = str_c(Country, as.integer(y))) %>%
      select(Country, Year, Value)
    fit <- fit_curve(rt, fit = c("S", "G", "E", "L")) %>%
      mutate(Country = cnt, Max.Year = y)
    df.fits.share <- df.fits.share %>% rbind(fit)
  }}

fit2 <- df.fits.share %>% filter(Good==1) 

# Calculate and save growth parameters for share values
sample_parameters.share <- fit2 %>% left_join(y.to) %>% 
  select(Max.Year, Fit, Country, Maturity, K, L, G, TMax) %>% 
  mutate(TMax=round(TMax)) %>% 
  mutate(Model=case_when(Fit=="S"~"Logistic", Fit=="G"~"Gompertz", Fit=="E"~"Exponential", Fit=="L"~"Logistic-linear")) %>% 
  mutate(dT=case_when(Fit=="S"~log(81)/K, Fit=="G"~log(log(0.1)/log(0.9))/K, TRUE~0)) %>% 
  rename(Year=Max.Year)%>% left_join(takeoff_final %>% select(Country, TO.Year=TO.Year.Share)) %>% 
  filter(TMax>TO.Year)

# write_csv(sample_parameters.share, 'NESub/data/output/national_solar_parameters_share.csv') #export fitted parameters
