calculate_trajectory_acceleration <- function(
    current_rates,      # Vector of current growth rates (absolute change in deployment share)
    current_values,     # Vector of current deployment values (between 0 and 1)
    weights,            # Vector of national weights (should sum to approximately 1)
    region_names,       # Vector of region names
    target_trajectory,  # Data frame with Year and Target_Value columns (values between 0 and 1)
    max_deployment,     # Vector of maximum allowed deployment in each region (between 0 and 1)
    max_rate = 0.02,    # Maximum allowed growth rate (absolute change in deployment)
    max_yearly_accel = 0.003,  # Maximum allowed yearly acceleration in growth rate
    tolerance = 1e-6    # Tolerance for target achievement
) {
  # Input validation
  if(length(current_rates) != length(weights) || 
     length(current_rates) != length(current_values) || 
     length(current_rates) != length(max_deployment)) {
    stop("Current rates, values, weights, and max_deployment must have same length")
  }
  
  # Check if global trajectory is theoretically achievable given max_deployment
  max_possible_deployment <- sum(weights * max_deployment)
  if(any(target_trajectory$Target_Value > max_possible_deployment)) {
    stop(sprintf("Global target exceeds maximum possible deployment (%.3f) given regional max_deployment constraints", 
                 max_possible_deployment))
  }
  if(abs(sum(weights) - 1) > 0.001) {
    warning("Weights sum to ", sum(weights), ", which deviates from 1 by more than 0.001")
    weights <- weights / sum(weights)
  }
  if(any(max_deployment > 1) || any(max_deployment <= 0)) {
    stop("All values in max_deployment must be between 0 and 1")
  }
  if(any(current_values > max_deployment)) {
    stop("Some current values exceed their region's max_deployment")
  }
  
  # Initialize variables
  n_countries <- length(weights)
  n_years <- nrow(target_trajectory) - 1  # Excluding the starting year
  
  # Create matrices to store results
  rates_matrix <- matrix(0, nrow = n_years, ncol = n_countries)
  values_matrix <- matrix(0, nrow = n_years + 1, ncol = n_countries)
  values_matrix[1,] <- current_values  # Set initial values
  
  # We'll initialize rates during the loop instead of here
  
  # Iterative optimization for each year
  for(year in 1:n_years) {
    # Calculate target increase for this year
    current_total <- sum(values_matrix[year,] * weights)
    target_total <- target_trajectory$Target_Value[year + 1]
    target_increase <- target_total - current_total
    
    # Initialize rates for this year
    # For year 1, use the provided current_rates
    # For subsequent years, use the previous year's calculated rates
    current_year_rates <- if(year == 1) current_rates else rates_matrix[year-1,]
    new_rates <- current_year_rates
    
    # Iterative adjustment process for this year
    max_iterations <- 100
    iteration <- 0
    converged <- FALSE
    
    while(!converged && iteration < max_iterations) {
      iteration <- iteration + 1
      previous_rates <- new_rates
      
      # Calculate remaining increase needed
      projected_values <- values_matrix[year,] + new_rates
      projected_total <- sum(projected_values * weights)
      remaining_increase <- target_total - projected_total
      
      # Handle overshooting by allowing rates to decrease to zero but not below
      need_decrease <- remaining_increase < 0
      
      # Calculate available countries
      available <- rep(TRUE, n_countries)
      
      if(need_decrease) {
        # For decreasing, check minimum rate constraints (can't go below 0)
        at_min_rate <- new_rates <= 0
        available[at_min_rate] <- FALSE
        
        # Check deceleration constraints
        rate_acceleration <- new_rates - current_year_rates
        at_max_decel <- rate_acceleration <= -max_yearly_accel
        available[at_max_decel] <- FALSE
      } else {
        # For increasing, check maximum rate constraints
        at_max_rate <- new_rates >= max_rate
        available[at_max_rate] <- FALSE
        
        # Check acceleration constraints
        rate_acceleration <- new_rates - current_year_rates
        at_max_accel <- rate_acceleration >= max_yearly_accel
        available[at_max_accel] <- FALSE
        
        # Check region-specific max_deployment constraint
        # Current values plus new rates should never exceed region-specific max_deployment
        would_exceed_max <- (values_matrix[year,] + new_rates) >= (max_deployment - tolerance)
        available[would_exceed_max] <- FALSE
        
        # Also check if any future increase would exceed region-specific max_deployment
        approaching_max <- (values_matrix[year,] + new_rates) >= (max_deployment - max_rate)
        available[approaching_max] <- FALSE  # Prevent increases that would lead to exceeding max in next step
      }
      
      # Calculate distribution factors for available countries
      factors <- rep(0, n_countries)
      
      if(need_decrease) {
        # For decreasing, calculate how much we can decrease (to zero)
        headroom <- pmin(
          new_rates,  # Can only decrease to zero
          rate_acceleration - (-max_yearly_accel)
        )
      } else {
        # For increasing, calculate how much we can increase
        headroom <- pmin(
          max_rate - new_rates,
          max_yearly_accel - rate_acceleration,
          max_deployment - (values_matrix[year,] + new_rates)  # Using region-specific max_deployment
        )
        
        # Add a safety factor when close to target to avoid overshooting
        if(abs(remaining_increase) < 0.01) {
          headroom <- headroom * 0.5  # More conservative increases when close to target
        }
      }
      
      # Weight factors by available headroom
      factors[available] <-  headroom[available]
      
      if(sum(factors) > 0) {
        factors <- factors / sum(factors)
      } else {
        break
      }
      
      # Calculate new increment
      increment <- remaining_increase * factors
      
      # Apply increment while respecting constraints
      for(i in which(available)) {
        if(need_decrease) {
          min_rate_increment <- -new_rates[i]  # Can only decrease to zero
          min_accel_increment <- -max_yearly_accel - rate_acceleration[i]
          max_increment <- max(min_rate_increment, min_accel_increment)
          increment[i] <- max(increment[i], max_increment)
        } else {
          max_rate_increment <- max_rate - new_rates[i]
          max_accel_increment <- max_yearly_accel - rate_acceleration[i]
          max_value_increment <- max_deployment[i] - (values_matrix[year,i] + new_rates[i])
          max_increment <- min(max_rate_increment, max_accel_increment, max_value_increment)
          increment[i] <- min(increment[i], max_increment)
        }
      }
      
      # Update rates
      new_rates <- new_rates + increment
      
      # Check convergence
      change <- max(abs(new_rates - previous_rates))
      converged <- change < tolerance
    }
    
    # Update rates and values matrices with strict max_deployment enforcement
    rates_matrix[year,] <- new_rates
    new_values <- values_matrix[year,] + new_rates
    
    # Strictly enforce region-specific max_deployment
    exceeded <- new_values > max_deployment
    if(any(exceeded)) {
      new_values[exceeded] <- max_deployment[exceeded]
      rates_matrix[year, exceeded] <- new_values[exceeded] - values_matrix[year, exceeded]
      
      warning(sprintf("Year %d: Some regions hit max_deployment constraint. Global target may not be achievable.", 
                      target_trajectory$Year[year + 1]))
    }
    values_matrix[year + 1,] <- new_values
  }
  
  # Convert matrices to data frames with region names
  rates_df <- as.data.frame(rates_matrix)
  names(rates_df) <- region_names
  rates_df$Year <- target_trajectory$Year[-1]
  
  values_df <- as.data.frame(values_matrix)
  names(values_df) <- region_names
  values_df$Year <- target_trajectory$Year
  
  # Create weight and max_deployment mapping
  mapping <- data.frame(
    Region = region_names,
    Weight = weights,
    Max_Deployment = max_deployment
  )
  
  # Prepare results
  results <- list(
    rates = rates_df,
    values = values_df,
    years = target_trajectory$Year,
    target_trajectory = target_trajectory$Target_Value,
    achieved_trajectory = apply(values_matrix, 1, function(x) sum(x * weights)),
    region_info = mapping
  )
  
  # Add diagnostics
  results$diagnostics <- data.frame(
    Year = target_trajectory$Year[-1],
    Target_Growth = diff(target_trajectory$Target_Value),
    Achieved_Growth = diff(results$achieved_trajectory),
    Target_Value = tail(target_trajectory$Target_Value, -1),
    Achieved_Value = tail(results$achieved_trajectory, -1),
    Error = tail(results$achieved_trajectory, -1) - tail(target_trajectory$Target_Value, -1)
  )
  
  return(results)
}
