suppressPackageStartupMessages({
  required_packages <- c(
    "tidyverse", "ggplot2", "scales", "patchwork", "cowplot"
  )
  
  for (pkg in required_packages) {
    if (!requireNamespace(pkg, quietly = TRUE)) {
      install.packages(pkg)
    }
    library(pkg, character.only = TRUE)
  }
})


hindcast_data <- read_csv('data/input/hindcast_data_1.csv')

proj_ranges <- read_csv('data/input/fig4_ranges.csv')

model_colours <- c(c("Logistic" = "#B4BA39",
                     'PROLONG' = '#379A8B',
                     'Exponential' = '#DB444B'))

m1 <- ggplot(hindcast_data %>% filter(Technology=="Mobiles") %>% filter(Max.Year %in% c(2002)), aes(x=Year))+
  geom_line(aes(y=Value, colour = Model))+
  geom_ribbon(data = proj_ranges %>% filter(Technology=="Mobiles"), aes(ymin=Q1, ymax=Q3, fill = Model), alpha=0.2, show.legend = FALSE)+
  geom_ribbon(data = proj_ranges%>% filter(Technology=="Mobiles"), aes(ymin=Min, ymax=Max, fill = Model), alpha=0.1, show.legend = FALSE)+
  geom_point(data = hindcast_data %>% filter(Technology=="Mobiles", Max.Year %in% c(2002)), aes(y=Global, shape = Data), size=0.5)+
  scale_shape_manual(values = c('In-sample data'=16, 'Out-of-sample data'=4), name=NULL)+
  scale_colour_manual(values = model_colours)+
  scale_fill_manual(values = model_colours)+
  facet_wrap(~Label, scales = 'free', 
             labeller = labeller(Label = label_wrap_gen(width = 10)))+
  scale_x_continuous(breaks = c(1990, 2005, 2020))+
  scale_y_continuous(labels = scales::percent)+
  coord_cartesian(ylim = c(0, 1.5))+
  theme_bw()+
  theme(panel.grid.minor = element_blank(),
        panel.grid.major.x = element_blank(), 
        title = element_text(face="bold", color = '#0C0C0C', size=10),
        strip.text = element_text(face="bold"),
        panel.spacing = unit(16, "pt"),
        legend.position = "none",
        strip.background = element_blank())+
  labs(x=NULL, y=NULL)


c1 <- ggplot(hindcast_data %>% filter(Technology=="CCGT", Max.Year %in% c(1998)), aes(x=Year))+
  geom_line(aes(y=Value, colour = Model))+
  geom_ribbon(data = proj_ranges %>% filter(Technology=="CCGT"), aes(ymin=Q1, ymax=Q3, fill = Model), alpha=0.2, show.legend = FALSE)+
  geom_ribbon(data = proj_ranges%>% filter(Technology=="CCGT"), aes(ymin=Min, ymax=Max, fill = Model), alpha=0.1, show.legend = FALSE)+
  geom_point(data = hindcast_data %>% filter(Technology=="CCGT", Max.Year %in% c(1998)), aes(y=Global, shape = Data), size=0.5)+
  scale_shape_manual(values = c('In-sample data'=16, 'Out-of-sample data'=4), name=NULL)+
  scale_colour_manual(values = model_colours)+
  scale_fill_manual(values = model_colours)+
  facet_wrap(~Label, scales = 'free', 
             labeller = labeller(Label = label_wrap_gen(width = 10)))+
  scale_x_continuous(breaks = c(1990, 2005, 2020))+
  scale_y_continuous(labels = scales::percent)+
  coord_cartesian(ylim = c(0, 1), xlim = c(1990, 2020))+
  theme_bw()+
  theme(panel.grid.minor = element_blank(),
        panel.grid.major.x = element_blank(), 
        title = element_text(face="bold", color = '#0C0C0C', size=10),
        strip.text = element_text(face="bold"),
        panel.spacing = unit(16, "pt"),
        legend.position = "none",
        strip.background = element_blank())+
  labs(x=NULL, y="Global deployment")



w1 <- ggplot(hindcast_data %>% filter(Technology=="Onshore wind", Max.Year %in% c(2010)), aes(x=Year))+
  geom_line(aes(y=Value, colour = Model))+
  geom_ribbon(data = proj_ranges %>% filter(Technology=="Onshore wind"), aes(ymin=Q1, ymax=Q3, fill = Model), alpha=0.2, show.legend = FALSE)+
  geom_ribbon(data = proj_ranges%>% filter(Technology=="Onshore wind"), aes(ymin=Min, ymax=Max, fill = Model), alpha=0.1, show.legend = FALSE)+
  geom_point(data = hindcast_data %>% filter(Technology=="Onshore wind", Max.Year %in% c(2010)), aes(y=Global, shape = Data), size=0.5)+
  scale_shape_manual(values = c('In-sample data'=16, 'Out-of-sample data'=4), name=NULL)+
  scale_colour_manual(values = model_colours)+
  scale_fill_manual(values = model_colours)+
  facet_wrap(~Label, scales = 'free', 
             labeller = labeller(Label = label_wrap_gen(width = 15)))+
  scale_x_continuous(breaks = c(2000, 2010, 2020))+
  scale_y_continuous(labels = scales::percent)+
  coord_cartesian(ylim = c(0, 0.2))+
  theme_bw()+
  theme(panel.grid.minor = element_blank(),
        panel.grid.major.x = element_blank(), 
        title = element_text(face="bold", color = '#0C0C0C', size=10),
        strip.text = element_text(face="bold"),
        panel.spacing = unit(16, "pt"),
        legend.position = "none",
        strip.background = element_blank())+
  labs(x="Year", y=NULL)

s1 <- ggplot(hindcast_data %>% filter(Technology=="Solar PV", Max.Year %in% c(2015)), aes(x=Year))+
  geom_line(aes(y=Value, colour = Model))+
  geom_ribbon(data = proj_ranges %>% filter(Technology=="Solar PV"), aes(ymin=Q1, ymax=Q3, fill = Model), alpha=0.2, show.legend = FALSE)+
  geom_ribbon(data = proj_ranges%>% filter(Technology=="Solar PV"), aes(ymin=Min, ymax=Max, fill = Model), alpha=0.1, show.legend = FALSE)+
  geom_point(data = hindcast_data %>% filter(Technology=="Solar PV", Max.Year %in% c(2015)), aes(y=Global, shape = Data), size=0.5)+
  scale_shape_manual(values = c('In-sample data'=16, 'Out-of-sample data'=4), name=NULL)+
  scale_colour_manual(values = model_colours)+
  scale_fill_manual(values = model_colours)+
  facet_wrap(~Label, scales = 'free', 
             labeller = labeller(Label = label_wrap_gen(width = 10)))+
  scale_x_continuous(breaks = c(2010, 2017, 2024))+
  scale_y_continuous(labels = scales::percent)+
  coord_cartesian(ylim = c(0, 0.2))+
  theme_bw()+
  theme(panel.grid.minor = element_blank(),
        panel.grid.major.x = element_blank(), 
        title = element_text(face="bold", color = '#0C0C0C', size=10),
        strip.text = element_text(face="bold"),
        panel.spacing = unit(16, "pt"),
        legend.position = "bottom",
        strip.background = element_blank())+
  labs(x=NULL, y=NULL)



future_proj <- hindcast_data %>% 
  filter(Year>Max.Year) %>% drop_na()

hindcast_cap <- hindcast_data %>% 
  group_by(Technology) %>% 
  filter(Global==max(Global)) %>%
  select(Technology, Max.Hindcast=Year) %>% distinct()

errors <- future_proj %>% filter(Value>=0) %>% 
  left_join(hindcast_cap) %>% 
  filter(Max.Year<=Max.Hindcast-3, Year<=Max.Hindcast) %>%  
  mutate(State = case_when(Year<=Max.Year~"Past", TRUE~"Future"),
         Horizon=Year-Max.Year) %>%
  filter(Horizon>0) %>% 
  group_by(Technology, Model, Max.Year, State) %>% 
  mutate(n=length(Value)) %>% 
  mutate(sMPE=sum((Value-Global)/((Value+Global)/2))/n,
         sMAPE=sum(abs(Value-Global)/((Value+Global)/2))/n,
         RMSE=sqrt(sum((Value-Global)^2)/n)) %>% 
  ungroup() %>% 
  select(Technology, Max.Year, Model, State, n, sMPE, sMAPE, RMSE) %>% distinct() 

errors$Technology <- factor(errors$Technology, levels = c("CCGT", "Mobiles", "Onshore wind",
                                                          "Solar PV"))

smape_evolution <- ggplot(errors, aes(x=Max.Year, y=sMAPE, colour = Model))+
  geom_line()+
  facet_wrap(~Technology, scales = 'free_x', nrow = 1)+
  scale_x_continuous(breaks = extended_breaks(n=4))+
  scale_colour_manual(values = model_colours)+
  theme_bw()+
  coord_cartesian(ylim = c(0,NA))+
  theme(panel.grid.minor = element_blank(),
        panel.grid.major.x = element_blank(), 
        title = element_text(face="bold", color = '#0C0C0C', size=10),
        strip.text = element_text(face="bold"),
        axis.text.x = element_blank(),
        panel.spacing = unit(16, "pt"),
        legend.position = "bottom",
        strip.background = element_blank())+
  labs(x = NULL)


smpe_evolution <- ggplot(errors, aes(x=Max.Year, y=sMPE, colour = Model))+
  geom_hline(yintercept = 0, linetype=5, alpha=0.5)+
  geom_line()+
  facet_wrap(~Technology, scales = 'free_x', nrow=1)+
  scale_x_continuous(breaks = extended_breaks(n=4))+
  scale_colour_manual(values = model_colours)+
  theme_bw()+
  theme(panel.grid.minor = element_blank(),
        panel.grid.major.x = element_blank(), 
        title = element_text(face="bold", color = '#0C0C0C', size=10),
        strip.text = element_text(face="bold", colour='white'),
        panel.spacing = unit(16, "pt"),
        legend.position = "bottom",
        strip.background = element_blank())+
  labs(x = "Last year in fitted chunk")

smpe_evolution


modelling_perf <- (c1|m1|w1|s1) + 
  plot_layout(guides = "collect") &
  theme(legend.position = "bottom")

errors_fig <- smape_evolution/smpe_evolution + 
  plot_layout(guides="collect") &
  theme(legend.position = "none")


legend_modelling <- get_legend(
  modelling_perf + 
    theme(legend.position = "bottom", 
          legend.box = "horizontal")
)

legend_errors <- get_legend(
  errors_fig + 
    theme(legend.position = "none", 
          legend.box = "horizontal")
)

combined_legend <- plot_grid(
  legend_modelling, legend_errors,
  ncol = 1
)

# Remove legends from original plots
modelling_perf_no_legend <- modelling_perf & theme(legend.position = "none")
errors_no_legend <- errors_fig & theme(legend.position = "none")

# Combine everything with the legend at the bottom
figure4 <- plot_grid(
  modelling_perf_no_legend, errors_no_legend, combined_legend,
  ncol = 1,
  labels = c("a", "b", ""),  # Empty label for the legend row
  label_size = 12,
  rel_heights = c(0.4, 0.6, 0.1)  # Adjust this ratio as needed
)

figure4

# ggsave("output/figure4.pdf", plot = figure4,
#        width = 7.2, height = 8, units = "in",
#        dpi = 300)
