suppressPackageStartupMessages({
  required_packages <- c(
    "tidyverse", "ggplot2", "ggstar", "patchwork", "scales"
  )
  
  for (pkg in required_packages) {
    if (!requireNamespace(pkg, quietly = TRUE)) {
      install.packages(pkg)
    }
    library(pkg, character.only = TRUE)
  }
})

custom_labeller <- function(labels) {
  # Extract only country names
  country_only <- labels
  if ("Label" %in% names(country_only)) {
    country_only$Label <- NULL  # Remove Technology labels
  }
  return(country_only)
}



deployment_data <- read_csv('data/input/f2_deployment_data.csv')
fitted_curves1 <- read_csv('data/input/f2_fitcurves1.csv')
fitted_curves2 <- read_csv('data/input/f2_fitcurves2.csv')
fitted_curves_acc <- read_csv('data/input/f2_fitcurves_acc.csv')
turning_points <- read_csv('data/input/f2_turning_points.csv')
tmax_single <- read_csv('data/input/f2_tmax_single.csv')
tmax_multiple <- read_csv('data/input/f2_tmax_multiple.csv')

country_order <- read_csv('data/input/f2_label_order.csv')


deployment_data$Label <- factor(deployment_data$Label, levels = unique(country_order$Label))
turning_points$Label <- factor(turning_points$Label, levels = unique(country_order$Label))
fitted_curves1$Label <- factor(fitted_curves1$Label, levels = unique(country_order$Label)) 
fitted_curves2$Label <- factor(fitted_curves2$Label, levels = unique(country_order$Label))
fitted_curves_acc$Label <- factor(fitted_curves_acc$Label, levels = unique(country_order$Label))
tmax_multiple$Label <- factor(tmax_multiple$Label, levels = unique(country_order$Label))
tmax_single$Label <- factor(tmax_single$Label, levels = unique(country_order$Label))

tech_colour <- c('Onshore wind'='#3EBCD2', 'Solar PV'='#EBB434')



acc <-  c('Hungary Solar PV', 'Estonia Solar PV', 'Cyprus Solar PV', 'Bangladesh Solar PV')

single_peak <-  c('United Kingdom Onshore wind', 'Croatia Onshore wind', 'South Korea Solar PV', 'Netherlands Solar PV')

similar_peak <-  c('United States Onshore wind', 'Mexico Onshore wind', 'Japan Solar PV', 'India Solar PV')

higher_peak <-  c('Netherlands Onshore wind', 'Greece Onshore wind',  'China Solar PV', 'Australia Solar PV')


accelerating <- ggplot(deployment_data %>% filter(Label %in% acc) , aes(x=Year, y=Add, colour=Technology))+
  geom_col(aes(y=R3, fill=Technology), colour='transparent', alpha=0.3)+
  geom_line(data = fitted_curves_acc %>% filter(Label %in% acc), linetype=1, show.legend = FALSE)+
  guides(color=guide_legend(override.aes = list(alpha=1)))+
  geom_star(data = turning_points %>% filter(Label %in% acc), aes(x=TO.Year.Share, y=TO.Add.R3), fill='white', size=2, show.legend = FALSE)+
  facet_wrap(Label~Country, scales = 'free_y', ncol = 4, labeller = custom_labeller)+
  scale_colour_manual(values = tech_colour, guide="none")+
  scale_fill_manual(values = tech_colour, guide="none")+
  scale_y_continuous(labels = scales::percent, breaks = breaks_extended(n=3))+
  scale_x_continuous(breaks = c(2000, 2010, 2020))+
  coord_cartesian(ylim = c(0, NA), xlim = c(1998,2025))+
  theme_minimal()+
  theme(legend.position = "bottom", panel.grid.minor = element_blank(), 
        panel.grid.major.x = element_blank(),
        strip.text = element_text(face = "bold"),
        axis.text.x = element_blank(),
        plot.title = element_text(hjust = 0.5, face="italic", size=10),
        title = element_text(face = "bold"))+
  labs(y=NULL, x=NULL, title = "Accelerating growth")

accelerating



one_pulse <- ggplot(deployment_data %>% filter(Label %in% single_peak) , aes(x=Year, y=Add, colour=Technology))+
  geom_col(aes(y=R3, fill=Technology), colour='transparent', alpha=0.3)+
  geom_line(data = fitted_curves2 %>% filter(Label %in% single_peak), linetype=1, show.legend = FALSE)+
  geom_point(data = fitted_curves2 %>% filter(Label %in% single_peak), aes(x=TMax, y=G), shape=16, size=2, show.legend = FALSE)+
  geom_text(data = fitted_curves2 %>% filter(Label %in% single_peak), aes(x=TMax-3, y=G, label=TMax), size=2, show.legend = FALSE)+
  geom_star(data = turning_points %>% filter(Label %in% single_peak), aes(x=TO.Year.Share, y=TO.Add.R3), fill='white', size=2, show.legend = FALSE)+
  geom_segment(data = tmax_single %>% filter(Label %in% single_peak), aes(x=TMax2, xend=2025, y=Avg, yend=Avg, colour=NULL), colour='#758D99', linetype=2)+
  facet_wrap(Label~Country, scales = 'free_y', ncol = 4, labeller = custom_labeller)+
  scale_colour_manual(values = tech_colour)+
  scale_fill_manual(values = tech_colour)+
  guides(color=guide_legend(override.aes = list(alpha=1)))+
  scale_y_continuous(labels = scales::percent, breaks = breaks_extended(n=3))+
  scale_x_continuous(breaks = c(2000, 2010, 2020))+
  coord_cartesian(ylim = c(-0.002, 0.031), xlim = c(1998,2025))+
  theme_minimal()+
  theme(legend.position = "none", panel.grid.minor = element_blank(), 
        panel.grid.major.x = element_blank(),
        strip.text = element_text(face = "bold"),
        axis.text.x = element_blank(),
        plot.title = element_text(hjust = 0.5, face="italic", size=10),
        title = element_text(face = "bold"))+
  labs(y=NULL, x=NULL, title = "Single pulse")

one_pulse


similar_pulse <- ggplot(deployment_data %>% filter(Label %in% similar_peak), aes(x=Year, y=Add, colour=Technology))+
  geom_col(aes(y=R3, fill=Technology), colour='transparent', alpha=0.3)+
  geom_line(data = fitted_curves1 %>% filter(Label %in% similar_peak), linetype=5, show.legend = FALSE)+
  geom_point(data = fitted_curves1 %>% filter(Label %in% similar_peak), aes(x=TMax, y=G), shape=17, size=2, show.legend = FALSE)+
  geom_line(data = fitted_curves2 %>% filter(Label %in% similar_peak), linetype=1, show.legend = FALSE)+
  geom_point(data = fitted_curves2 %>% filter(Label %in% similar_peak), aes(x=TMax, y=G), shape=16, size=2, show.legend = FALSE)+
  geom_text(data = fitted_curves1 %>% filter(Label %in% similar_peak), aes(x=TMax-3, y=1.1*G, label=TMax), size=2, show.legend = FALSE)+
  geom_text(data = fitted_curves2 %>% filter(Label %in% similar_peak), aes(x=TMax+3, y=1.1*G, label=TMax), size=2, show.legend = FALSE)+
  geom_star(data = turning_points %>% filter(Label %in% similar_peak), aes(x=TO.Year.Share, y=TO.Add.R3), fill='white', size=2, show.legend = FALSE)+
  geom_segment(data = tmax_multiple %>% filter(Label %in% similar_peak), aes(x=TMax1, xend=2025, y=Avg, yend=Avg, colour=NULL), colour='#758D99', linetype=2)+
  facet_wrap(Label~Country, scales = 'free_y', ncol = 4, labeller = custom_labeller)+
  scale_colour_manual(values = tech_colour)+
  scale_fill_manual(values = tech_colour)+
  guides(color=guide_legend(override.aes = list(alpha=1)))+
  scale_y_continuous(labels = scales::percent, breaks = breaks_extended(n=3))+
  scale_x_continuous(breaks = c(2000, 2010, 2020))+
  coord_cartesian(ylim = c(-0.002, 0.012), xlim = c(1998,2025))+
  theme_minimal()+
  theme(legend.position = "none", panel.grid.minor = element_blank(), 
        panel.grid.major.x = element_blank(),
        strip.text = element_text(face = "bold"),
        axis.text.x = element_blank(),
        plot.title = element_text(hjust = 0.5, face="italic", size=10),
        title = element_text(face = "bold"))+
  labs(y=NULL, x=NULL, title = "Multiple pulses: Overall growth similar or slower than the first pulse")

similar_pulse


faster_pulse <- ggplot(deployment_data %>% filter(Label %in% higher_peak) , aes(x=Year, y=Add, colour=Technology))+
  geom_col(aes(y=R3, fill=Technology), colour='transparent', alpha=0.3)+
  geom_line(data = fitted_curves1 %>% filter(Label %in% higher_peak), linetype=5, show.legend = FALSE)+
  geom_point(data = fitted_curves1 %>% filter(Label %in% higher_peak), aes(x=TMax, y=G), shape=17, size=2, show.legend = FALSE)+
  geom_line(data = fitted_curves2 %>% filter(Label %in% higher_peak), linetype=1, show.legend = FALSE)+
  geom_point(data = fitted_curves2 %>% filter(Label %in% higher_peak), aes(x=TMax, y=G), shape=16, size=2, show.legend = FALSE)+
  geom_text(data = fitted_curves1 %>% filter(Label %in% higher_peak), aes(x=TMax-3, y=1.2*G, label=TMax), size=2, show.legend = FALSE)+
  geom_text(data = fitted_curves2 %>% filter(Label %in% higher_peak, TMax<=2024), aes(x=TMax-1, y=1.1*G, label=TMax), size=2, show.legend = FALSE)+
  geom_star(data = turning_points %>% filter(Label %in% higher_peak), aes(x=TO.Year.Share, y=TO.Add.R3), fill='white', size=2, show.legend = FALSE)+
  geom_segment(data = tmax_multiple %>% filter(Label %in% higher_peak), aes(x=TMax1, xend=2025, y=Avg, yend=Avg, colour=NULL), colour='#758D99', linetype=2)+
  facet_wrap(Label~Country, scales = 'free_y', ncol = 4, labeller = custom_labeller)+
  scale_colour_manual(values = tech_colour)+
  scale_fill_manual(values = tech_colour)+
  guides(color=guide_legend(override.aes = list(alpha=1)))+
  scale_y_continuous(labels = scales::percent, breaks = breaks_extended(n=3))+
  scale_x_continuous(breaks = c(2000, 2010, 2020))+
  coord_cartesian(ylim = c(-0.002, 0.03), xlim = c(1998,2025))+
  theme_minimal()+
  theme(legend.position = "bottom", panel.grid.minor = element_blank(), 
        panel.grid.major.x = element_blank(),
        strip.text = element_text(face = "bold"),
        plot.title = element_text(hjust = 0.5, face="italic", size=10),
        title = element_text(face = "bold"))+
  labs(y=NULL, x=NULL, title = "Multiple pulses: Overall growth faster than the first pulse")

faster_pulse


pulse_plot <- accelerating/one_pulse/similar_pulse/faster_pulse + plot_layout(guides = "collect") & theme(legend.position = "bottom")

pulse_plot
