import pandas as pd
import geopandas as gpd
import glob
import cpi
import os
import numpy as np
import bw2calc as bc
import matplotlib.colors  as mcolors
import matplotlib.pyplot  as plt
import matplotlib.ticker  as ticker
import matplotlib.patches as mpatches
import matplotlib.image   as mpimg
from matplotlib import lines
import calc_utils as cu
import importlib
importlib.reload(cu)
from calc_utils import get_EAF_from_GEM_gdp 

def get_material_stocks_and_demands(cur_path):    
    stock_data = pd.read_excel(cur_path+'/Results/fossil_infrastructure_material_stocks_distribution.xlsx', index_col=0)
    stock_data.columns = [col.replace(' [kg]','') for col in stock_data.columns]
    stock_data_to_Mt=stock_data/10**9 #to Mt
    
    translation_dict = {
        'Al': 'Al',
        'Steel':'Steel',
        'Cement':'Concrete',
        'Si':'Si',
        'Cu':'Cu',
        'Ni':'Ni'
    }
    
    # Specify the folder path
    folder_path = cur_path+'/Databases/Wang2023_data'
    
    # Use glob to get all CSV files in the folder
    csv_files = glob.glob(os.path.join(folder_path, "*.csv"))
    
    # Create an empty list to hold dataframes
    dataframes = []
    
    # Loop through the list of CSV files and read each one into a dataframe
    i=0
    for file in csv_files:
        df = pd.read_csv(file, index_col=0)
        df['ID'] = len(df)*[i]
        dataframes.append(df)
        i+=1
        
    merged = pd.concat(dataframes).reset_index()
    materials = list(set(merged['Material']))
    distr_df = {}
    for mat in materials:
        distr = merged[merged['Material']==mat].groupby('ID')['Total Demand'].sum()
        distr_df[mat] = list(distr)
    
    def cement_to_concrete(df):
        #add cement to concrete
        if 'Cement' in list(df.columns):
            #kg concrete/ kg cement uncertainty for concrete in ecoinvent 3.9
            median, scale = np.log(6.779069767441861), np.log(1.688874005951699)
            factors = [np.random.lognormal(mean=median,  sigma =scale) for _ in df['Cement']]
            
            concrete = [value*factor for value, factor in zip(df['Cement'], factors)]
            df['Concrete'] = concrete
            del df['Cement']
        return df
    
    def align_columns(df1, df2):
        for col in df2.columns:
            if col not in df1.columns:
                df1[col] = [np.nan]*len(df1)
        return df1[df2.columns]
    
    def clean_WANG_data(WANG_data):
        #WANG_data_cropped = WANG_data[[col for col in WANG_data.columns if col in translation_dict.keys()]]
        WANG_data_cropped = WANG_data
        WANG_data_cropped_plus_concrete = cement_to_concrete(WANG_data_cropped)
        demand_data = WANG_data_cropped_plus_concrete/10**5 #to Megatons
        return demand_data
        
    WANG_data = pd.DataFrame(distr_df)
    
    demand_data = clean_WANG_data(WANG_data)
    #demand_data = align_columns(demand_data, stock_data)
    
    stacked_stock = stock_data_to_Mt.stack().reset_index().iloc[:,1:]
    stacked_stock.columns=['Material', 'Demand/Stock [Mt]']
    stacked_stock['Label'] = ['stock']*len(stacked_stock)
    
    stacked_demand = demand_data.stack().reset_index().iloc[:,1:]
    stacked_demand.columns=['Material', 'Demand/Stock [Mt]']
    stacked_demand['Label'] = ['demand']*len(stacked_demand)
    demand_and_stock = pd.concat([stacked_stock, stacked_demand])
    return demand_and_stock

def get_production_data(cur_path):
    path = cur_path+'/Databases/agg_material_production.xlsx'
    data = pd.read_excel(path, index_col=0)
    return data

def get_production_data_median_stocks_and_demands(prod_data, demand_and_stock, materials):
    prod_data_cropped = prod_data.T[materials].T
    stocks     = demand_and_stock[demand_and_stock.Label=='stock'].set_index('Material').T[materials].T.reset_index().groupby('Material')['Demand/Stock [Mt]'].median().reindex(materials)
    try:
        demands    = demand_and_stock[demand_and_stock.Label=='demand'].set_index('Material').T[materials].T.reset_index().groupby('Material')['Demand/Stock [Mt]'].median().reindex(materials)
    except:
        return prod_data_cropped, stocks
    return prod_data_cropped, stocks, demands

def set_second_axis(ax,y):
    ax.set_xlim([-0.02,1.25])
    ax.set_xlabel('[-] / [a]', fontsize=fontsize, labelpad=10)
    ax.set_xticklabels(ax0.get_xticklabels(), fontsize=fontsize)
    ax.tick_params(axis='x', which='major', pad=0.08, length=3)
    ax.xaxis.set_label_coords(1.04, y)

def capacity_to_category(cap):
    if cap <1:
        return '< 1   Mt/a'
    elif cap >=1 and cap <9:
        return '1 - 9 Mt/a'
    elif cap >=9:
        return '> 9   Mt/a'
    else:
        return np.nan

def capacity_to_color(cap):
    if cap <1:
        return 'g'
    elif cap >=1 and cap <9:
        return 'y'
    elif cap >=9:
        return 'r'
    else:
        return np.nan


def custom_formatter(x, pos):
    if x < 1:
        if x < 0.1:
            if x < 0.01:
                if x < 0.001:
                    return f'{x:.4f}'
                return f'{x:.3f}'
            return f'{x:.2f}'
        return f'{x:.1f}'
    else:
        return f'{int(x)}'

def add_bar_to_map(fig, bar_data, continent, coords_dict, fontsize, bbox_to_anchor, only_idle):
    if only_idle:
        col_name = 'Idle Capacity [Mt/a]'
    else:
        col_name = 'Capacity [Mt/a]'
        
    lat, lon = coords_dict[continent]
    ax_bar = fig.add_axes([0.5*(1+lon/180) , 0.22*(1+lat/90) , 0.02, 0.08])
    ax_bar.set_title(continent, fontsize=fontsize, pad=-0.1)
    bars = ax_bar.bar(height=bar_data.T[[col_name, 'Total [Mt]']].T[continent], x=[1,2],color=['magenta', 'g'], width=0.5)
    ax_bar.set_xlim([0.5,2.5])
    ax_bar.set_xticklabels(['', ''])
    ax_bar.set_yticklabels(ax_bar.get_yticklabels(), fontsize=fontsize)
    ax_bar.spines[['right', 'top']].set_visible(False)
    ax_bar.tick_params(bottom=False, length=2, width=0.5)
    ax_bar.spines[['left','bottom']].set_linewidth(0.5)
    if continent == 'North America':
        ax_bar.legend(handles= bars.patches, labels=[col_name, 'Stock [Mt]'],
                      fontsize=fontsize, bbox_to_anchor=bbox_to_anchor, frameon=False)
    return ax_bar

def plot_boxplot2(ax,y,demand_and_stock, material, colors, width, dy):
    X1 = np.array(demand_and_stock[(demand_and_stock['Material']==material) & (demand_and_stock['Label']=='stock')]['Demand/Stock [Mt]'])
    X2 = np.array(demand_and_stock[(demand_and_stock['Material']==material) & (demand_and_stock['Label']=='demand')]['Demand/Stock [Mt]'])
    props = dict(linestyle='-', linewidth=0.2, color='k')
    if material=='Steel':
        ax.fill_between(x=[10**-5, 10**5], y1=[y,y], facecolor='gainsboro',
                   y2=[y+2,y+2], edgecolor='none',lw=0.2, zorder=-10) 

    for X, color, j in zip([X1,X2], colors, [dy, -dy]):
        if len(X)>0:
            bplot1 = ax.boxplot(x=X, widths=[width], vert=False, positions=[1+y+j],
                               patch_artist=True, medianprops=props,whis=[5, 95],
                              capwidths=[width], whiskerprops=props, capprops=props,showfliers=False)
            # fill with colors
            for patch in bplot1['boxes']:
                patch.set_facecolor(color)
                patch.set_edgecolor('none')
                patch.set_alpha(1)
                patch.set_linewidth(0.2)
        else:
            continue

def plot_boxplot3(ax,y,demand_and_stock, material, colors):
    X1 = np.array(demand_and_stock[(demand_and_stock['Material']==material) & (demand_and_stock['Label']=='stock')]['Demand/Stock [Mt]'])
    X2 = np.array(demand_and_stock[(demand_and_stock['Material']==material) & (demand_and_stock['Label']=='demand')]['Demand/Stock [Mt]'])
    props = dict(linestyle='-', linewidth=0.4, color='k')

    for X, color, j in zip([X1,X2], colors, [0.3, -0.3]):
        if len(X)>0:
            bplot1 = ax.boxplot(x=X, widths=[0.5], vert=False, positions=[1+y+j],
                               patch_artist=True, medianprops=props,whis=[5, 95],
                              capwidths=[0.4], whiskerprops=props, capprops=props,showfliers=False)
            # fill with colors
            for patch in bplot1['boxes']:
                patch.set_facecolor(color)
                patch.set_edgecolor('k')
                patch.set_alpha(1)
                patch.set_linewidth(0.2)
        else:
            continue
        
    if material in ['Co', 'Ceramic']:
        ax1.plot([10**-5, 10**5], [j,j], lw=0.5, color='k', zorder=30-i*2+1)

def get_EAF_production_data(cur_path):
    filepath = cur_path + '/Databases/EAF_production_2024.xlsx'
    data = pd.read_excel(filepath, index_col=0, sheet_name='EAF_prod_countries')
    return data

def get_map_data(cur_path):
    country_name_translation = pd.read_excel(cur_path+'/Databases/geography_translation.xlsx', index_col=0, sheet_name='country_dict').to_dict()['country name']
    country_continent_dict   = pd.read_excel(cur_path+'/Databases/geography_translation.xlsx', index_col=0, sheet_name='continent_dict').to_dict()['continent']
    EAF_production           = pd.read_excel(cur_path+'/Databases/EAF_production_2024.xlsx', index_col=0, sheet_name='EAF_prod_countries')[['Mt EAF']]
    
    filepath = cur_path+'/Results/fossil_infrastructure_material_stocks_per_country.xlsx'
    country_data = pd.read_excel(filepath, index_col=0).set_index('Country')
    
    url = "https://naciscdn.org/naturalearth/110m/cultural/ne_110m_admin_0_countries.zip"
    material = 'Steel'
    countries = gpd.read_file(url)
    shape_country_dict = countries[['NAME', 'geometry', 'CONTINENT']].set_index('NAME').to_dict()
    country_data['country_gpd'] = [name if name not in country_name_translation.keys() else country_name_translation[name]
                               for name in country_data.index]
    translated_data = country_data.groupby('country_gpd')[['{} median [kg]'.format(material)]].sum()
    translated_data['geometry'] = [shape_country_dict['geometry'][name]
                                   if name in shape_country_dict['geometry'].keys() else np.nan
                                           for name in translated_data.index]
    
    translated_data['Continent'] = [shape_country_dict['CONTINENT'][name]
                                   if name in shape_country_dict['CONTINENT'].keys() else country_continent_dict[name]
                                           for name in translated_data.index]
    
    geo_translated_data = gpd.GeoDataFrame(translated_data)
    geo_translated_data['Total [Mt]'] = geo_translated_data['{} median [kg]'.format(material)]/10**9
    
    #define missing data countries
    grays = countries[countries['NAME'].isin(set(countries.NAME).difference(set(geo_translated_data.index)))][['NAME', 'geometry', 'CONTINENT']]
    grays['Total'] = [np.nan]*len(grays)
    grays['Total [Mt]'] = [np.nan]*len(grays)
    grays = grays.set_index('NAME')
    grays = grays.rename(columns={'geometry':'geometry', 'CONTINENT':'Continent'})
    
    geo_translated_data_plus_missing_data = pd.concat([geo_translated_data, grays])
    geo_translated_data_plus_missing_data = geo_translated_data_plus_missing_data[geo_translated_data_plus_missing_data.index!='Antarctica']
    geo_translated_data_plus_missing_data.sort_values(by='Total [Mt]',ascending=False)
    
    stocks_per_continent = geo_translated_data_plus_missing_data.groupby('Continent')['Total [Mt]'].sum()
    
    #GET RECYCLING PLANTS
    EAFs = get_EAF_from_GEM_gdp()
    EAFs['Capacity [Mt/a]'] = [(eaf_val+if_val)/1000 for eaf_val,if_val in zip(EAFs['Nominal EAF steel capacity (ttpa)'], EAFs['Other/unspecified steel capacity (ttpa)'])]
    EAFs['size']            = [np.sqrt((eaf_val+if_val))/10 for eaf_val,if_val in zip(EAFs['Nominal EAF steel capacity (ttpa)'], EAFs['Other/unspecified steel capacity (ttpa)'])]
    EAFs['cap_cat']= [capacity_to_category(cap) for cap in EAFs['Capacity [Mt/a]']]
    EAFs['color']  = [capacity_to_color(cap) for cap in EAFs['Capacity [Mt/a]']]
    EAFs['Country']   = [name if name not in country_name_translation.keys() else country_name_translation[name]
                                 for name in EAFs['Country/Area']]
    EAFs['Continent'] = [shape_country_dict['CONTINENT'][name]
                           if name in shape_country_dict['CONTINENT'].keys() else country_continent_dict[name]
                               for name in EAFs['Country']]
    capacity_per_contigent   = EAFs.groupby('Continent')['Capacity [Mt/a]'].sum()
    production_per_continent = EAF_production[EAF_production.index.isin(capacity_per_contigent.index)]['Mt EAF']
    idle_capacity_per_continent = capacity_per_contigent-production_per_continent.sort_index()
    idle_capacity_per_continent.name = 'Idle Capacity [Mt/a]'
    bar_data = pd.concat([pd.DataFrame(capacity_per_contigent),pd.DataFrame(idle_capacity_per_continent), pd.DataFrame(stocks_per_continent)], axis=1).dropna().T
    
    indicator_agg_dict = {
     'climate change'            : 'Global warming potential (100a)',
     'freshwater eutrophication' : 'Freshwater eutrophication',
     'marine eutrophication'     : 'Marine eutrophication',
     'metal depletion'           : 'Metal depletion',
     'resource'                  : 'Land use',
     'water depletion'           : 'Water depletion' }
    
    return EAFs, bar_data, geo_translated_data_plus_missing_data

def truncate_colormap(cmap, minval=0.0, maxval=1.0, n=100):
    new_cmap = mcolors.LinearSegmentedColormap.from_list(
        'trunc({n},{a:.2f},{b:.2f})'.format(n=cmap, a=minval, b=maxval),
        plt.get_cmap(cmap)(np.linspace(minval, maxval, n)))
    return new_cmap

def get_GDP_projection(cur_path, scenario_raw):
    scenario, ssp    = scenario_raw[1], scenario_raw[0]
    start_year, now_year = 2010, 2023
    filepath_gdp     = cur_path + '/LCIA_data/monetarization_factors.xlsx'
    GDP_2010_cap     = pd.read_excel(filepath_gdp, sheet_name='GDP_per_capita', header=[1,2], index_col=0).iloc[:,:6]
    years            = [2025, 2030, 2035, 2040, 2045, 2050]
    GDP_2023         = GDP_2010_cap.loc[years] * cpi.inflate(1, start_year, now_year)/1000 #to trillion USD 2023    
    GDP_2023_per_year= GDP_2023['Remind 2.1 {}'.format(scenario)][ssp]
    return GDP_2023_per_year

def plot_clustered_stacked(axe, xticks, dfall, labels=None, H="///", fontsize=6,y_icon=55,ymax=10, **kwargs):
    """Given a list of dataframes, with identical columns and index, create a clustered stacked bar plot. 
labels is a list of the names of the dataframe, used for the legend
title is a string for the title of the plot
H is the hatch used for identification of the different dataframe"""

    n_df = len(dfall)
    n_col = len(dfall[0].columns) 
    n_ind = len(dfall[0].index)
    #axe = plt.subplot(111)

    line_bottom0 = dfall[0].loc['low-alloyed steel'].sum()
    line_bottom1 = dfall[1].loc['low-alloyed steel'].sum()

    axe.add_line(lines.Line2D([1.2,1.2], [line_bottom1,6.2], lw=0.5, c='k', label='_nolegend_'))
    axe.add_line(lines.Line2D([0.9,0.6], [line_bottom0,6.2], lw=0.5, c='k', label='_nolegend_'))

    for df in dfall : # for each data frame
        axe = df.plot(kind="bar",
                      linewidth=0,
                      stacked=True,
                      ax=axe,
                      color = ['#d7191c', plt.get_cmap('viridis')(0.3)],
                      legend=False,
                      grid=False,
                      **kwargs)  # make bar plots

    h,l = axe.get_legend_handles_labels() # get the handles we want to modify
    for i in range(0, n_df * n_col, n_col): # len(h) = n_col * n_df
        for j, pa in enumerate(h[i:i+n_col]):
            for rect in pa.patches: # for each index
                rect.set_x(rect.get_x() + 1 / float(n_df + 1) * i / float(n_col))
                rect.set_hatch(H * int(i / n_col)) #edited part
                rect.set_edgecolor("k")
                #rect.set_width(1 / float(n_df + 1))
                rect.set_width(0.3)
                #if i==0:
                 #   rect.set_alpha(0.5)

    axe.annotate('primary',
                        xy=(1, 1),
                        xytext=(-30, y_icon),  # 4 points vertical offset.
                        textcoords='offset points',
                        ha='center', va='bottom')

    axe.annotate('secondary',
                        xy=(1, 1),
                        xytext=(10, y_icon),  # 4 points vertical offset.
                        textcoords='offset points',
                        ha='center', va='bottom')

    #axe.set_xticks((np.arange(0, 2 * n_ind, 2) + 1 / float(n_df + 1)) / 2.)
    axe.set_xticks(np.array(range(3))+0.07)
    axe.set_xticklabels(xticks, rotation=0, fontsize=fontsize)
    #axe.set_xticklabels(df.index, rotation = 0)
    axe.set_ylabel('total costs [USD$_{2023}$ / kg]', fontsize=fontsize)
    axe.set_xlabel('steel', fontsize=fontsize)
    axe.set_ylim(ymax=ymax)

    # Add invisible data to add another legend
    n=[]        
    for i in range(n_df):
        n.append(axe.bar(0, 0, color="gray", hatch=H * i))
    if labels!=None:
        l1 = axe.legend(labels, loc='upper left', frameon=False, fontsize=fontsize)
        axe.add_artist(l1)
    return axe

def plot_icon(cur_path, filename, fig, ax, pos):
    # Load the image
    img = mpimg.imread(cur_path+'/Figures/icons/{}.png'.format(filename))
    # Add the image to the figure
    position= [pos[0], pos[1], 0.07, 0.07]
    newax = fig.add_axes(position, anchor='NE', zorder=10)
    newax.imshow(img)
    newax.axis('off')

def split_data(data, idx_dict):
    dfs = []
    for i, idx in enumerate(data.index):
        df = data.iloc[i:i+1,:]
        df.index = [idx_dict[label] for label in df.index]
        dfs.append(df)
    return dfs

def plot_clustered_stacked2(axe, xticks, dfall, labels=None, H="\\", ylabel=None, fontsize=None,color_dict=None, **kwargs):
    """Given a list of dataframes, with identical columns and index, create a clustered stacked bar plot. 
labels is a list of the names of the dataframe, used for the legend
title is a string for the title of the plot
H is the hatch used for identification of the different dataframe"""

    n_df = len(dfall)
    n_col = len(dfall[0].columns) 
    n_ind = len(dfall[0].index)
    
    for i, df in enumerate(dfall): # for each data frame
        axe = df.plot(kind="bar",
                      linewidth=0,
                      stacked=True,
                      ax=axe,
                      color = [color_dict[mat] for mat in df.columns],
                      legend=False,
                      grid=False,
                      **kwargs)  # make bar plots

    h,l = axe.get_legend_handles_labels() # get the handles we want to modify
    for i in range(0, n_df * n_col, n_col): # len(h) = n_col * n_df
        for j, pa in enumerate(h[i:i+n_col]):
            for rect in pa.patches: # for each index
                rect.set_x(rect.get_x() + 1 / float(n_df + 1) * i / float(n_col))
                rect.set_hatch(H * int(i / n_col)) #edited part
                rect.set_edgecolor("k")
                #rect.set_width(1 / float(n_df + 1))
                rect.set_width(0.3)
                #if i==0:
                 #   rect.set_alpha(0.5)
    
    axe.set_xticks(np.array(range(4))+0.07)
    axe.set_xticklabels(xticks, rotation=0, fontsize=fontsize, ha='center')
    axe.set_ylabel(ylabel, fontsize=fontsize)

    # Add invisible data to add another legend
    n=[]        
    for i in range(n_df):
        n.append(axe.bar(0, 0, color="gray", hatch=H * i))
    return axe

def get_renewable_LCIA_data(cur_path):
    filepath1 = cur_path+'/Results/renewables_LCA_gwp.xlsx'
    filepath2 = cur_path+'/Results/renewables_LCA_ext.xlsx'
    agg_normalized_data_gwp = pd.read_excel(filepath1, index_col = 0).T
    agg_normalized_data_ext = pd.read_excel(filepath2, index_col = 0).T
    
    order = ['Aluminium',
     'Steel',
     'Cell',
     'Concrete',
     'Copper',
     'Electricity',
     'Electronics',
     'Glass',
     'Iron',
     'Lead',
     'Other',
     'Plastic',
     'Road',
     'Transport']
    order.reverse()
    
    idx_dict = {
     'photovoltaic open ground installation, 570 kWp, single-Si, on open ground, default' : 'Photovoltaic, open ground',
     'photovoltaic open ground installation, 570 kWp, single-Si, on open ground, steel, secondary' : 'Photovoltaic, open ground',
     'photovoltaic slanted-roof installation, 3kWp, single-Si, panel, mounted, on roof, default' : 'Photovoltaic, slanted roof',
     'photovoltaic slanted-roof installation, 3kWp, single-Si, panel, mounted, on roof, steel, secondary' : 'Photovoltaic, slanted roof',
     'wind turbine construction, 2MW, offshore, primary, optimistic' : 'Wind, offshore',
     'wind turbine construction, 2MW, offshore, secondary, optimistic' : 'Wind, offshore',
     'wind turbine construction, 4.5MW, onshore, primary, optimistic' : 'Wind, onshore',
     'wind turbine construction, 4.5MW, onshore, secondary, optimistic' : 'Wind, onshore'
    }    
    dfs_gwp = split_data(agg_normalized_data_gwp.reindex(idx_dict.keys()), idx_dict)
    dfs_ext = split_data(agg_normalized_data_gwp.reindex(idx_dict.keys()), idx_dict)
    return [dfs_gwp, dfs_ext], order, idx_dict

def adjust_unit(unit, power):
    if 'kg' in unit:
        if power==3:
            unit = unit.replace('kg','t')
        elif power==6:
            unit = unit.replace('kg','kt')
        elif power==9:
            unit = unit.replace('kg','Mt')
        elif power==12:
            unit = unit.replace('kg','Gt')
        else:
            unit = unit.replace('kg','Mt')
            power = 9
    elif 'species' in unit:
        if power==3:
            unit = unit.replace('species','thousand\u00A0species')
        elif power==6:
            unit = unit.replace('species','million\u00A0species')
    elif 'Bq' in unit:
        if power==9:
            unit = unit.replace('kBq','10$^9$ kBq')
        elif power==10:
            unit = unit.replace('kBq','10$^9$ kBq')
            power = 9
        
    else:
        if power==3:
            unit = unit.replace('m','thousand\u00A0m')
        if power==6:
            unit = unit.replace('m','million\u00A0m')
        if power==9:
            unit = unit.replace('m','billion\u00A0m')
        if power==10:
            unit = unit.replace('m','billion\u00A0m')
            power = 9
        if power==12:
            unit = unit.replace('m','trillion\u00A0m')
    return unit, power

def limit_contribution(df, n):
    if len(df)<n:
        return df
        
    df = df.sort_values(by=df.columns[0], ascending=False)
    
    others_row = pd.concat([df.iloc[n:].T, df.loc['Other']], axis=1).T.sum().to_frame().transpose()
    others_row.index = ['Other']
    
    cropped_df = pd.concat([df.iloc[:n][df.iloc[:n].index!='Other'], others_row], axis=0)
    return cropped_df

def put_others_last(df):
    row_z = df.loc['Other']
    df = df.drop('Other')
    df = pd.concat([df.T, row_z], axis=1).T
    return df

def adjust_magnitude(data, unit, met):
    magnitude = int(np.log10(data.sum(axis=0).min()))

    if magnitude<=-3 and magnitude>-6:
        f = 10**3
        unit = unit.replace('kg\u00A0', 'g\u00A0').replace('$species', 'milli $species')
        data = data*f

    if magnitude<=-6 and magnitude>-9:
        f = 10**6
        unit = unit.replace('kg\u00A0', 'mg\u00A0').replace('$species', '\u03BC $species')
        data = data*f

    if magnitude<=-9 and magnitude>-12:
        f = 10**9
        unit = unit.replace('kg\u00A0', '\u03BCg\u00A0').replace('$species', 'nano $species')
        data = data*f

    return data, unit


def get_indicator_dicts():
    ind_0 = 'ReCiPe 2016 v1.03, midpoint (H)'
    show_indicators_dict = {
    ('IPCC 2021', 'global warming potential (GWP100)')                      :{'name': 'Global warming potential 100a'  , 'unit': 'kg\u00A0CO$_2$-eq'},
     (ind_0, 'agricultural land occupation (LOP)')                          :{'name': 'Agricultural land occupation'   , 'unit': 'm$^2$a'},
     (ind_0, 'fossil fuel potential (FFP)')                                 :{'name': 'Fossil depletion'               , 'unit': 'kg\u00A0oil-eq'},
     (ind_0, 'freshwater eutrophication potential (FEP)')                   :{'name': 'Freshwater eutrophication'      , 'unit': 'kg\u00A0P-eq'},
     (ind_0, 'freshwater ecotoxicity potential (FETP)')                     :{'name': 'Freshwater ecotoxicity'         , 'unit': 'kg\u00A01,4-DCB-eq'},
     #(ind_0, 'global warming potential (GWP100)')                          :{'name': 'Global warming potential 100a'  , 'unit': 'kg\u00A0CO$_2$-eq'},
     (ind_0, 'human toxicity potential (HTPc)')                             :{'name': 'Human toxicity, carcinogenic'   , 'unit': 'kg\u00A01,4-DCB-eq'},
     (ind_0, 'human toxicity potential (HTPnc)')                            :{'name': 'Human toxicity, non-carcinogenic', 'unit': 'kg\u00A01,4-DCB-eq'},
     (ind_0, 'ionising radiation potential (IRP)')                          :{'name': 'Ionising radiation'             , 'unit': 'kBq\u00A0Co$_{60}$-eq'},
     (ind_0, 'surplus ore potential (SOP)')                                 :{'name': 'Metal depletion'                , 'unit': 'kg\u00A0Cu-eq'},
     (ind_0, 'marine eutrophication potential (MEP)')                       :{'name': 'Marine eutrophication'          , 'unit': 'kg\u00A0N-eq'},
     (ind_0, 'marine ecotoxicity potential (METP)')                         :{'name': 'Marine ecotoxicity'             , 'unit': 'kg\u00A01,4-DCB-eq'},
     #(ind_0, 'NLTP')                                                       :{'name': 'Natural land transformation'    , 'unit': 'm$^2$'},
     (ind_0, 'ozone depletion potential (ODPinfinite)')                     :{'name': 'Ozone depletion'                , 'unit': 'kg\u00A0CFC$_{11}$-eq'},
     (ind_0, 'particulate matter formation potential (PMFP)')               :{'name': 'Particulate matter'             , 'unit': 'kg\u00A0PM$_{2.5}$-eq'},
     (ind_0, 'photochemical oxidant formation potential: ecosystems (EOFP)'):{'name': 'Photochemical oxidants,\necosystems', 'unit': 'kg\u00A0NO$_x$-eq'},
     (ind_0, 'photochemical oxidant formation potential: humans (HOFP)')    :{'name': 'Photochemical oxidants,\nhumans' , 'unit': 'kg\u00A0NO$_x$-eq'},
     (ind_0, 'terrestrial acidification potential (TAP)')                   :{'name': 'Terrestrial acidification'      , 'unit': 'kg\u00A0SO$_{2}$-eq'},
     (ind_0, 'terrestrial ecotoxicity potential (TETP)')                    :{'name': 'Terrestrial ecotoxicity'        , 'unit': 'kg\u00A01,4-DCB-eq'},
     #(ind_0, 'ULOP')                                                       :{'name': 'Urban land occupation'          , 'unit': 'm$^2$a'},
     (ind_0, 'water consumption potential (WCP)')                           :{'name': 'Water use'                      , 'unit': 'm$^3$'},
     ('selected LCI results', 'land occupation')                            :{'name': 'Land use'                       , 'unit': 'm$^2$a'},
     ('ReCiPe 2016 v1.03, endpoint (H)', 'ecosystem quality')               :{'name': 'Biodiversity loss'              , 'unit': '$species \cdot a$'}
    }
    
    biodiv_loss = [
    ('ReCiPe 2016 v1.03, endpoint (H)', 'acidification: terrestrial'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'climate change: freshwater ecosystems'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'climate change: terrestrial ecosystems'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'ecotoxicity: freshwater'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'ecotoxicity: marine'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'ecotoxicity: terrestrial'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'eutrophication: freshwater'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'eutrophication: marine'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'land use'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'photochemical oxidant formation: terrestrial ecosystems'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'water use: aquatic ecosystems'),
    ('ReCiPe 2016 v1.03, endpoint (H)', 'water use: terrestrial ecosystems')
    ]
    return show_indicators_dict, biodiv_loss