clear all; clc;


rng(1);  % fixed random seed to ensure reproducible simulation results

% Parameters related to the OCT system used to acquire the experimental data
lam_min = 1170.5e-9; lam_max = 1407.8e-9;  % minmum and maximum wavelengths (meters)
f_min = 1./lam_max; f_max = 1./lam_min; f0 = (f_min + f_max)/2; Df = f_max - f_min; K0 = 2*pi*f0;

% parameters pf the OCT A-scans in z-space
zmin_s = -5e-6; zmin = +50e-6; zmax = 450e-6; DZ = zmax-zmin; DZ_long = zmax-zmin_s;
dz_tmp = 0.7*sqrt(2)*1e-8; Nz = ceil(DZ/dz_tmp/2)*2+1; zbar = linspace(zmin,zmax,Nz).';  ; 


% scatterers definition (Unloaded case)
dens_scat = 7;
zmax_s = zmax + 20e-6; DZ_s = zmax_s - zmin_s; 
Ns_long = ceil(dens_scat*DZ_s*1e6); rho_max = 3e-4; rho_long = rho_max.*rand(Ns_long,1);
z_scatlongU = zmin_s + DZ_s.*sort(rand(1,Ns_long));  
izs_min = min(find(z_scatlongU > zmin-5e-6));
z_scatU = z_scatlongU(izs_min:Ns_long); Ns = length(z_scatU);
rho = rho_long(izs_min:Ns_long);

% scatterers definition (Loaded case)
alpha = -2e-3; Disp = 1e-6; %-2e-7*sqrt(2);
z_scatL = Disp + z_scatU.*(1+alpha); zL = zbar-z_scatL;

cut_off = 1e-3; % Frequencies cut-off at the edge of the spectrum
Wk = 2*pi*(f_max-f0)/sqrt(log(1/cut_off)); T = 1./2./Wk.^2; % Spectral width of the light-source

% % OCT generalized A-scan definition,  Eq. (5)
An = @(z,zs,f0,W,rho,Nd)  (z-zs).^Nd.*exp(-1i.*zs.*4.*pi.*f0-W.^2.*(z-zs).^2)*rho;

zlim = [150,250]*1e-6; % Region of interest (ROI) chosen for the plot
IZ = find( (zbar >= zlim(1)) & (zbar <= zlim(2))); % Axial indices related to the ROI

AU = An(zbar,z_scatU,f0,Wk,rho,0); % Unloaded OCT A-scan
AL = An(zbar,z_scatL,f0,Wk,rho,0); % Loaded OCT A-scan

epsz = 1e-10; % Parameter approximating the infinitesimal increment used for the derivative of AU and |AU|

% Definition of the terms appearing in Eq.(10)
pU = abs(AU); % AU magnitude
AU_p = An(zbar+epsz,z_scatU,f0,Wk,rho,0); % AU(z+dz) 
AU_m = An(zbar-epsz,z_scatU,f0,Wk,rho,0); % AU(z-dz) 
DAU = (AU_p - AU_m)./2./epsz; % Approximate derivative of AU, evaluated at z = zbar
d_phiU = imag(DAU./AU); % Approximate derivative of the phase of AU, evaluated at z = zbar  (see Eq. (7))
dpU = pU.*real(DAU./AU); %(abs(AU_p)- abs(AU_m))./2./epsz; % Approximate derivative of pU, evaluated at z = zbar (see Eq. (7))

Dphi = unwrap(angle(AU.*conj(AL))); % Unwrapped phase difference
Dphi_est_PHDE = 2.*K0.*alpha.*zbar; % Linear estimation of the phase difference (PHDE, see Eq. (1))
Dphi_est_acc = 2.*K0.*alpha.*zbar + K0.*alpha./(Wk.^2).*dpU./pU + (alpha.*zbar+Disp).*d_phiU; % Accurate estimation of the phase difference ( see Eq. (10))

iz0 = IZ(1); 

% Translating the phase differences to let them start from the same value at z = z0
Dphi = Dphi - Dphi(iz0); 
Dphi_est_PHDE = Dphi_est_PHDE- Dphi_est_PHDE(iz0); 
Dphi_est_acc = Dphi_est_acc - Dphi_est_acc(iz0); 


% Fig. (3)
figure(1); clf;  NL = 2; NF = 15; MU = max(abs(AL));  
subplot(2,1,1); semilogy(zbar*1e6, (abs(AL./MU)),'Linewidth',NL); xlabel('z(\mum)');  set(gca,'FontSize',NF);xlim(zlim*1e6)
subplot(2,1,2); hold on; plot(zbar*1e6, Dphi,'k','Linewidth',NL); xlabel('z(\mum)'); plot(zbar*1e6, Dphi_est_PHDE,'b--','Linewidth',NL); plot(zbar*1e6, Dphi_est_acc,'r','Linewidth',NL);  set(gca,'FontSize',NF);
legend('\Delta\phi','PDE','Accurate'); xlim(zlim*1e6)

