% This script generates Fig. (4), including a video created by sequentially plotting data over time

clear all; clc;


% Update the path with the folder containing the scripts.
path_string = [''];

% Add to the path the folder containing all the nested functions employed in the following lines of code
% addpath(genpath([path_string,'handle_functions_to_add']));

rng(5); % fixed random seed to ensure reproducible simulation results

%%%%%%%%%%%%%%%%%%%     Parameters of the simulation   %%%%%%%%%%%%%%%%%%%%%%%%
% Scatterers region
minZ = 0e-6;  % minimum scatterer position
maxZ = 300e-6;  % maximum scatterer position
DZ = maxZ - minZ; Nv = 2000;
z_U = sort(minZ + rand(1,Nv)*DZ);  % (z coordinates of all scatterers)
rhoMax = 3e-4; rho = rand(1,Nv).*rhoMax; rho = rho(:); % Array of the reflectivity profiles of all scatterers (It is set random)

% Parameters related to the OCT system used to acquire the experimental data
lam_min = 1170.5e-9;  lam_max = 1407.8e-9;  f_min = 1./lam_max; f_max = 1./lam_min; f0 = ( f_max + f_min)/2; Df = f_max - f_min; a = 4*pi*f0; % frequencies spectrum width
Nk_nyq = 4*max(abs(z_U))*Df; % Nyquist frequency threshold
Nk = ceil(Nk_nyq*1.5); % Number of monochromatic components of the simulation
f_vec = linspace(f_min,f_max,Nk);  % Spatial-frequencies array

cut_off = 1e-3; % Frequencies cut-off at the edge of the spectrum
Wk = 2*pi*(max(f_vec)-f0)/sqrt(log(1/cut_off)); % Spectral width of the light-source
Sk =  1/sqrt(pi)./Wk.*exp(-(2*pi*(f_vec(:)-f0)./Wk).^2); % Spectral density function
zmin = z_U(1) + 4e-6; zmax = z_U(Nv) - 4e-6; % max/min axial sampling of the OCT signals
Nz = 21149; % Number of axial points of the OCT signals in the interval [zmin,zmax]
zbar = linspace(zmin,zmax,Nz).'; dz = zbar(2)-zbar(1); % Array of z coordinates wher the OCT signals have been calculated

% OCT signals generation
An = @(z,a,W,r,n) ((z.^n).*exp(1i.*a*z-W.^2.*(z.^2)))*r./2./pi; % OCT A-scan definition,  Eq. (3)
Phi_Term = exp(-1i.*a.*zbar); % %Phase term removed form the A-scan  
zbU = zbar-z_U; AU = An(zbU,a,Wk,rho,0).*Phi_Term;  % Demodulated unloaded OCT A-scan (Eqs. (4)) 
Disp = 0e-7; alpha = -2.3e-4; z_L = z_U.*(1+alpha)+ Disp; zbL = zbar-z_L; AL = An(zbL,a,Wk,rho,0).*Phi_Term; % Demodulated "first" loaded OCT A-scan (Eqs. (4))
Disp2 = 0e-7; alpha2 = alpha - 7e-5; z_L2 = z_U.*(1+alpha2)+ Disp2; zb2L = zbar-z_L2; AL2 = An(zb2L,a,Wk,rho,0).*Phi_Term; % Demodulated "second" loaded OCT A-scan (Eqs. (4))

% Unwrapped phase differences 
DPhi = unwrap(angle(AU.*conj(AL))); DPhi2 = unwrap(angle(AU.*conj(AL2)));  

%Linear estimation of the phase differences: PHDE assumption
Estimation_standard_tmp = alpha.*a.*zbar; Estimation_standard = Estimation_standard_tmp - Estimation_standard_tmp(1);
Estimation_standard_tmp2 = alpha2.*a.*zbar; Estimation_standard2 = Estimation_standard_tmp2 - Estimation_standard_tmp2(1);

% Indices near the dark speckle creating a high-oscillation peak and 2pi-jump, for the first and the second loaded signal, respectively.
IND1 = [8650:11350]; IND2 = [9865:9882];  INDs = [1:11:Nz]; 


% Plots of Fig.(4)
r1 = zbar(IND1([1,end]))*1e6; r2 = zbar(IND2([1,end]))*1e6;
MAU = max(abs(AU)); AU = AU./MAU; AL = AL./MAU; AL2 = AL2./MAU;  
figure(1); clf; NF = 13; NL = 2;  dxlim = 3e-3; dxlim_large = 5e-1;
subplot(3,2,1); semilogy(zbar(INDs).*1e6, (abs([AL(INDs)])),'LineWidth',NL);   set(gca, 'FontSize',NF);
subplot(3,2,3); hold on; plot(zbar(INDs).*1e6,[DPhi(INDs),DPhi2(INDs)]/2/pi,'LineWidth',NL);      set(gca, 'FontSize',NF);
subplot(3,2,5); hold on; plot(zbar(INDs).*1e6,(DPhi(INDs) - Estimation_standard(INDs))/2/pi,'LineWidth',NL); plot(zbar(INDs).*1e6,(DPhi2(INDs) - Estimation_standard(INDs))/2/pi,'LineWidth',NL); plot(zbar*1e6,zbar.*0 + [0:1],'k--');  set(gca, 'FontSize',NF);
subplot(2,2,2); hold on;  plot(real(AU(IND1 )),imag(AU(IND1 )),'LineWidth',NL); plot(real(AL(IND1 )),imag(AL(IND1 )),'LineWidth',NL);  plot(real(AL2(IND1 )),imag(AL2(IND1 )),'LineWidth',NL);  title(['z = ',num2str(r1(1)),'\mum. ',num2str(r1(2)),'\mum']);    xlim([-1,1]*dxlim_large); line([dxlim,-dxlim], [-dxlim,dxlim]); ylim([-1,1]*dxlim_large);  scatter(0,0,'ko','filled'); set(gca, 'FontSize',NF);
subplot(2,2,4); hold on;  plot(real(AU(IND2 )),imag(AU(IND2 )),'LineWidth',NL);  plot(real(AL(IND2 )),imag(AL(IND2 )),'LineWidth',NL);   plot(real(AL2(IND2 )),imag(AL2(IND2 )),'LineWidth',NL);  title(['z = ',num2str(r2(1)),'\mum. ',num2str(r2(2)),'\mum']);  scatter(0,0,'ko','filled'); xlim([-1,1]*dxlim); ylim([-1,1]*dxlim);  set(gca, 'FontSize',NF);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Video frames that better show the different behaviour of the phase difference when oscillation peaks and 2pi-jumps are created

PhiL = unwrap(angle(AL)); IND2 = [9865:9882]; PhiL = PhiL - PhiL(IND2(1)); PhiL2 = unwrap(angle(AL2)); PhiL2 = PhiL2 - PhiL2(IND2(1)); 
figure(2); clf; NF = 10; NL = 2;  dxlim = 2e-3; dxlim_large = 5e-1; sgtitle(['z = ',num2str(r2(1),4),'\mum, ',num2str(r2(2),4),'\mum']);
for iz0 = IND2(1):IND2(end)
    clf; 
    subplot(2,2,1); hold on; xlabel(['z (\mu m)']);  plot(zbar(IND2)*1e6,abs(AL(IND2 )),'r','LineWidth',NL); plot(zbar(IND2)*1e6,abs(AL(IND2 )),'r','LineWidth',NL); scatter(zbar(iz0)*1e6,abs(AL(iz0)),'bo','filled');
    subplot(2,2,2); hold on; xlabel(['Re{A}']);  ylabel(['Im{A}']);  plot(real(AU(IND2 )),imag(AU(IND2 )),'k','LineWidth',NL);  plot(real(AL(IND2 )),imag(AL(IND2 )),'r','LineWidth',NL);   plot(real(AL2(IND2 )),imag(AL2(IND2 )),'m','LineWidth',NL); scatter(0,0,'ko','filled');
    scatter(real(AU(iz0 )),imag(AU(iz0)),'ko','filled');  scatter(real(AL(iz0 )),imag(AL(iz0)),'ro','filled');  scatter(real(AL2(iz0 )),imag(AL2(iz0)),'mo','filled');  xlim([-1,1]*dxlim); ylim([-2,2]*dxlim); axis equal; set(gca, 'FontSize',NF); legend('U','L_1','L_2','Location','Southeast');
    subplot(2,2,3); hold on; xlabel(['z (\mu m)']);  plot(zbar(IND2)*1e6,PhiL(IND2),'r','LineWidth',NL); scatter(zbar(iz0)*1e6,PhiL(iz0),'ro','filled'); plot(zbar(IND2)*1e6,zbar(IND2).*0 + pi*[-2:2],'k--');  ylim([-4,4]);
    subplot(2,2,4); hold on; xlabel(['z (\mu m)']);  plot(zbar(IND2)*1e6,PhiL2(IND2),'m','LineWidth',NL); scatter(zbar(iz0)*1e6,PhiL2(iz0),'mo','filled'); plot(zbar(IND2)*1e6,zbar(IND2).*0 + pi*[-2:2],'k--'); ylim([-4,4]);
    pause(0.5);
end



