# 1. Packages ----
pacman::p_load(
  BiocParallel, data.table, DESeq2, R6, broom, callr, dplyr,
  edgeR, ggplot2, here, limma, lobstr, magrittr, purrr, see, sloop,
  stringr, sva, testthat, tibble, tidyfast, tidyr, usethis, withr,
  wrapr, zeallot, argparse
);

# 2. General ----
#' Function: Make directory.
#' @param path A string representing the path to the directory to create.
#' @param p    A logical indicating whether or not to create all parent directories that do not exist in the path.
#' @return     A logical as to whether it succeeded.
#' @export
mkdir <- function(path, p = TRUE) { return(suppressWarnings(invisible(dir.create(path, recursive = p))))}

#' Function: Is orthogonal.
#' @param x Numeric vector to test for orthogonality.
#' @param y Numeric vector to test for orthogonality.
#' @return Logical if x and y are orthogonal.
#' @export
is.orth <- function(x, y) { return(sum(x * t(y)) == 0)}

#' Function: Paste but from ... args unquoted.
#' @param ... A list of expressions to convert to strings and concatenate without a seperator.
#' @return A string of the concatenated unquoted symbols.
#' @export
qc <- function(...) { paste0(rlang::enexprs(...))};

#' Function: Delimited separated values.
#' @param ...    A list of expressions to convert to strings delimited by: sep.wn within expressions and sep.bt between expressions.
#' @param sep.bt A string to delimit expressions.
#' @param sep.wn A string to delimit elements within each expression. The default (NULL), indicates no separation within each expression.
#' @return A string of delimited separated values of the input expression.
#' @export
dsv <- function(..., sep.bt = '.', sep.wn = NULL) { stringr::str_c(..., sep = sep.bt, collapse = sep.wn)}

#' Function: Prefixed delimited seprated values
#' @param ...    A list of expressions to convert to strings delimited by: sep.wn within expressions and sep.bt between expressions.
#' @param prefix A prefix string that will be prepended to the string resulting from the evaluation of ... as a string.
#' @param sep.bt A string to delimit expressions.
#' @param sep.wn A string to delimit elements within each expression. The default (NULL), indicates no separation within each expression.
#' @return A string of prefixed delimited separated values.
#' @export
pdsv <- function(..., prefix = ':', sep.bt = '.', sep.wn = NULL) { paste0(prefix, stringr::str_c(..., sep = sep.bt, collapse = sep.wn))}

#' Function: Check if code running in RStudio.
#' @return Logical that is TRUE if code is running in RStudio.
#' @export
in_rstudio <- function() {return(as.numeric(Sys.getenv('RSTUDIO')) %>% map(.,~ !is.na(.) && . == 1) %>% unlist())}


# 3. Rdatatable/data.table ----
#' Function: Read a data.table from a file.
#' @param file A string representing the path to the file to load into a data.table
#' @param ...  Additional arguments to be passed to data.table::fread .
#' @return     A data.table representation of the file pointed to by 'file'. Strings are converted to factors and na.strings = 'NA'.
#' @export
dt.read <- function(file, ...) { return(data.table::fread(file = file, na.strings = 'NA', stringsAsFactors = TRUE,...))}

#' Function: Write a data.table to a file.
#' @param dt   A data.table that will be written to a file.
#' @param file The path to the written file.
#' @param ...  Additional arguments passed through to data.table::fwrite .
#' @return     NULL
dt.write <- function(dt, file, ...) { return(data.table::fwrite(dt, file = file, quote = FALSE, na = 'NA', sep = '\t', ...))}

#' Function: Get the names of factor columns in a data.table, as a vector of strings.
#' @param dt The data.table to query for factor columns.
#' @return   A vector of strings representing the column names that are factors in the data.table.
#' @export
dt.getfacs <- function(dt) { return(dt[, lapply(.SD, is.factor)] %>% unlist() %>% which() %>% names())}

#' Return a subset of a data.table by selecting columns to keep.
#'
#' @param dt        A data.table to subset based on column names.
#' @param col.names The column names to keep, as a vector of strings.
#' @return          A data.table with columns only in col.names.
#' @export
dt.subset  <- function(dt, col.names, negate = FALSE) { if (!negate) return(dt[,..col.names]); sel.names <- !colnames(dt) %in% col.names; return(dt[,..sel.names])};

#' Function: Refactor all factor columns of data.table and character columns too, if indicated.
#' @param dt       A data.table with columns to refactor.
#' @param convert  A logical flag indicating whether to convert character columns to factor.
#' @return The input data.table, with factor columns refactored and character columns too, if appropriate.
#' @export
dt.refac   <- function(dt, convert = TRUE) {
  cols <- dt[, lapply(.SD, is.factor)] %>% unlist() %>% which() %>% names()
  if (convert) cols <- c(cols, dt[, lapply(.SD, is.character)] %>% unlist() %>% which() %>% names());
  dt[, (cols) := lapply(.SD, factor), .SDcols = cols]
  return(invisible(dt));
}

dt.constfacs <- function(dt) {
  return(colnames(dt)[which(dt[,lapply(.SD, function(x) { return((is.factor(x) && nlevels(x) <= 1) || (is.numeric(x) && length(unique(na.omit(x))) == 1))})] == TRUE)]);
}
str_frm <- as.formula;
str_add <- partial(str_c, collapse = ' + ');

# Linear Models -----
f.pval <- function(f, num_df, num_denom_df) { pf(q = f,df1 = num_df, df2 = num_denom_df, lower.tail = FALSE)[[1]]}

fac.test <- function(x, y, emp_pval = TRUE) {chisq.test(x,y,simulate.p.value = emp_pval);}
lm.pval <- function(mod, as.str = FALSE) { summ_lm <- summary(mod); pval <- f.pval(summ_lm$fstatistic[1L], summ_lm$fstatistic[2L], summ_lm$fstatistic[3L]); if(!as.str) return(pval) else return(format.pval(pval))}
pw.fac <- function(x,y) {
  stopifnot(is.factor(x), is.factor(y));
  chi_tb <- tidy(chisq.test(x,y,simulate.p.value = TRUE));
  df <- (length(unique(x)) - 1) * (length(unique(y)) - 1);
  nobs <- num_na(x,y);
  tib <- tibble(r = NA, stat = chi_tb$statistic, test = 'x2', pval = chi_tb$p.value, df = df);
  tib <- bind_cols(tib, nobs);
  return(tib);
}

pw.aov <- function(x,y) {
  nfac = sum(as.numeric(is.factor(x)),as.numeric(is.factor(y)));
  nnum = sum(as.numeric(is.numeric(x)), as.numeric(is.numeric(y)));
  stopifnot(nfac + nnum == 2, nfac == 1, nnum == 1);
  u <- if (is.factor(x)) x else y;
  v <- if (is.factor(x)) y else x;
  mod_lm  <- lm(v ~ u);
  summ_lm <- summary(mod_lm);
  nobs <- num_na(x,y);
  tib <- tibble(r = summ_lm$r.squared, stat = summ_lm$fstatistic[['value']], test = 't', pval = lm.pval(mod_lm), df = summ_lm$fstatistic[['numdf']]);
  tib <- bind_cols(tib, nobs);
  return(tib);
}

num_na <- function(x, y = NULL) {
  if (is.null(y)) y <- x;
  return(map2(x, y, function(a, b) { d <- as.numeric(any(is.numeric(x),is.numeric(y))); tibble_row(n = d, n.na = 1 - d, n.obs = 1);}) %>% bind_rows() %>% map(sum) %>% as_tibble());
}

pw.cor <- function(x,y) {
  mod_cor <- cor.test(x,y);
  nobs <- num_na(x,y);
  tib <- tibble(r = mod_cor$estimate[['cor']], stat = mod_cor$statistic[['t']], test = 't', pval = mod_cor$p.value, df =  mod_cor$parameter[['df']]);
  tib <- bind_cols(tib, nobs);
  return(tib);
}

#' @title Function: Hetero correlation.
#'
#' @param x A factor vector with more than 1 category or a numeric vector with variance.
#' @param y A factor vector with more than 1 category or a numeric vector with variance.
#'
#' @return
#' @export
hcor <- function(x, y) {
  nfac = sum(as.numeric(is.factor(x)),  as.numeric(is.factor(y)));
  nnum = sum(as.numeric(is.numeric(x)), as.numeric(is.numeric(y)));
  stopifnot(nfac + nnum == 2)
  if (nfac == 0)    return(pw.cor(x,y));
  if (nfac == 2)    return(pw.fac(x,y));
  if (is.factor(x)) return(pw.aov(x,y));
  if (is.factor(y)) return(pw.aov(y,x));
  stop('error: hcor');
}

#' Function: Compute correlation matrix from data.table columns and return results in tibble nested, wide, and long format.
#' @param dt A data.table with numeric and factor columns.
#' @return A list containing a list of tibbles (nest, wide, long) and a list of column names (from subset of data.table meeting criteria and the tibble correlations)
#' @export
cor.dt <- function(dt) {
  # Create lists to store return values
  cols <- list();
  tb   <- list();

  # Select columns which are numeric or are factors with levels > 1 and aren't Seq.
  # sel.num = num_columns - 1 of forthcoming tibble (we will add a rownames column)
  cols$sel <- colnames(dt)[which(dt[,lapply(.SD, function(x){return((is.factor(x) && nlevels(x) > 1) || is.numeric(x));})] == TRUE)]
  cols$sel <- cols$sel[which(cols$sel != 'Seq')];
  cols$sel.num <- length(cols$sel);

  # Create tibble to store pairwise correlations between each column.
  tb$nest  <- suppressWarnings(as_tibble(matrix(list(), nrow = cols$sel.num, ncol = cols$sel.num)));
  for (i in seq(1,cols$sel.num)){
    name_i <- cols$sel[i];
    x <- dt[,..name_i][[1]];
    for (j in seq(i,cols$sel.num)){
      name_j <- cols$sel[j];
      if (i != j) {
        y  <- dt[,..name_j][[1]];
        tb$nest[i,j] <- list(list(hcor(x, y)))
        tb$nest[j,i] <- tb$nest[i,j];
      } else {
        tb$nest[i,j] <- list(list(tibble(r = NA, stat = NA, test = '', pval = NA, df = NA, n = NA, n.na = NA, n.obs = NA)));
      }
    }
  }
  # Create nested tibble with shape [num_cols - 1, num_cols]
  tb$nest      <- set_names(tb$nest, cols$sel);
  tb$nest$Col  <- factor(cols$sel);
  tb$nest      <- tb$nest %>% relocate(Col);

  # Create names of nested tibble columns
  cols$ret     <- c('r', 'stat', 'test', 'pval', 'df', 'n', 'n.na', 'n.obs');
  cols$ret.num <- length(cols$ret);

  # Unnest nested tibbles to form a wide table of shape [num_cols - 1, 1 + 8*(num_cols - 1)]
  tb$wide <- suppressWarnings(tb$nest %>%
    unnest() %>%
    select(.,!colnames(.)[colnames(.) %in% cols$sel]) %>%
    set_names(nm = c('Col', expand_grid(sel = cols$sel, ret = cols$ret) %>% unite(id, sel:ret, sep=':') %>% pluck(1)))) %>%
    mutate_all(list(~na_if(.,""))) %>%
    mutate_if(is.character, function(x) {factor(str_to_title(x))});

  # Elongate wide tibble of shape [num_cols - 1, 1 + 8*(num_cols - 1)] -> shape [*, c(Col,Metric,Value,Test)]
  tb$long <- tb$wide %>%
    pivot_longer(cols = colnames(tb$wide)[which(!colnames(tb$wide) %like% 'Col|:test' )], names_to = "Metric", values_to = "Value") %>%
    rename(Var1 = Col) %>%
    relocate(c(Var1, Metric, Value)) %>%
    pivot_longer(!Var1:Value, names_to = 'Var2', values_to = 'Test') %>%
    relocate(c(Var1, Var2, Metric, Value, Test)) %>%
    mutate(Var2   = factor(str_to_title(gsub(':[^:]+$','',Var2))),
           Metric = factor(str_to_title(gsub('^[^:]+:','',Metric))),
           Test = factor(str_to_title(Test)));
  return(list(tb = tb, cols = cols));
}


# Section: PCA ----
#' Run PCA
#'
#' @param expr Matrix of RNA-Seq expression: [gene, samples]
#' @param meta data.table of [samples, k] where k is the number of metadata columns
#' @param top.gene Number of top variance genes to use for PCA. NULL indicates ALL
#' @param num.pc Number of PCs to return. 2 or 3.
#'
#' @return list of transformed data, pct variance
#'
#' @examples
run_pca <- function(expr, top.gene = NULL, dt.rn = 'Spec', var.dt.rn = 'Name', pcs.dt.rn = 'Gene') {
  num.gene <- nrow(expr);
  if (!is.null(top.gene)) {
    var.gene.sorted <- order(rowVars(expr), decreasing = TRUE);
    num.gene <- min(top.gene, num.gene);
    res <- prcomp(t(expr)[var.gene.sorted[seq_along(num.gene),]], center = TRUE)
  } else {
    res <- prcomp(t(expr), center = TRUE)
  }
  pcs.var <- res$sdev**2;
  pcs.pvar <- 100 * pcs.var/sum(pcs.var)
  pcs.cumvar <- 100 * cumsum(pcs.var)/sum(pcs.var)
  pca <- list();
  pca$all.dt <- setDT(as.data.frame(res$x), keep.rownames = dt.rn)
  pca$dt <- pca$all.dt[,1:3];
  pca$var.dt <- setDT(as.data.frame(t(data.frame(Var = pcs.var, PctVar = pcs.pvar, PctCumVar = pcs.cumvar))), keep.rownames = var.dt.rn);
  setnames(pca$var.dt, c(colnames(pca$var.dt)[1],colnames(pca$all.dt)[-1]));
  pca$pcs.dt <- setDT(as.data.frame(res$rotation), keep.rownames = pcs.dt.rn);
  pca$params <- list(num.genes = nrow(pca$pcs), mean = res$center, scale = res$scale);
  pca$dt[[dt.rn]] <- factor(pca$dt[[dt.rn]]);
  pca$all.dt[[dt.rn]] <- factor(pca$all.dt[[dt.rn]]);
  pca$var.dt[[var.dt.rn]] <- factor(pca$var.dt[[var.dt.rn]]);
  pca$pcs.dt[[pcs.dt.rn]] <- factor(pca$pcs.dt[[pcs.dt.rn]]);
  return(pca);
}

# Section: Design Matrix ----
#' Function: Render model.matrix from string formulae.
#' @param fstr A formula in string form.
#' @param data A data.frame or a wrapper thereof.
#' @return A list with the formula string input, a model.matrix formed from the formula & evaluated in the context of data.
#' @export
mm.str  <- function(fstr, data) { return(list(str = fstr, mm = make.mm(as.formula(fstr), data = data)))}

eqn.str  <- function(str.facs, str.covars = NULL, intercept = 0L) {
  stopifnot(is.integer(intercept), intercept >= -1, intercept <= 1);
  return(as.formula(paste0(c(qc(~ !!!intercept), str.covars, str.facs),collapse = ' + ')));
}
#' Function: Create model.matrix
#' @param ...    A formula to evaluate in the context of data as a model.matrix.
#' @param data   A data context (data.frame or wrapper) to evaluate a formula.
#' @param rm.pre A vector of prefixes to remove from the names of the columns of the model.matrix
#' @return A model.matrix.
#' @export
make.mm <- function(..., data, rm.pre = c()){
  mm <- model.matrix(..., data = data);
  idx <- which(as.data.table(mm)[,lapply(.SD, function(x){x != 0})][,lapply(.SD,as.numeric)][,lapply(.SD,sum)] == 0)
  if (length(idx) != 0) mm <- mm[,-idx];
  if (length(rm.pre) != 0) colnames(mm) <- gsub(paste0('^(',paste0(unlist(rm.pre), collapse='|'),')'),'',colnames(mm));
  return(mm);
}

# Section: Limma ----
# Alias: Remove Batch Effect
rm_batch_eff <- limma::removeBatchEffect;

# Section: sva ----
#' Function: Run sva
#'
#' @param expr        An wide data.frame (cols are samples) or wrapper containing expression data.
#' @param meta        A long data.frame (rows are samples) or wrapper containing metadata for the experiment/study.
#' @param f1          A string formula representing an alternative model.
#' @param f0          A string forula representing a null model.
#' @param top_gene    An integer indicating the max num of top variance genes to use for parameter estimation. NULL indicates all genes.
#' @param prefix      A string representing the prefix to be prepended to each surrogate variable column name.
#' @param use_seq  A logical switch: when FALSE, use classical SVA and when TRUE, use svaseq.
#' @return A list with the input wrapped as a list and with formulas replaced with model.matrices, the return object of either sva or svaseq,
#'         a data.table of the surrogate variables with column names prefixed by prefix, a data.table of the posterior probabilities.
#' @export
run_sva <- function(expr, meta, f1, f0, top_gene = NULL, prefix = "SV_", use_seq = FALSE) {
  x <- list(expr = expr, meta = meta, h1 = mm.str(f1, meta), h0 = mm.str(f0, meta), top_gene = top_gene);
  run_alg <- if (use_seq) sva::svaseq else sva::sva;
  y <- run_alg(dat = x$expr, mod = x$h1$mm, mod0 = x$h0$mm, vfilter = x$top_gene)
  sv <- data.table(spec = meta$spec, { z = y$sv; colnames(z) = paste0(prefix, seq(1, y$n.sv)); z; })
  pp <- data.table(gam = y$pprob.gam, b = y$pprob.b)
  return(list(dt.sv = sv, dt.pp = pp, argv = x))
}

#' Function: Run ComBat or ComBat-Seq for batch correction (no covariates).
#' @param dat     An expression matrix with shape [genes, samples].
#' @param batch   A vector of batch indicator variables of shape [samples].
#' @param group   A vector for the biological condition of interest of shapes [samples].
#' @param use_seq A logical switch that when TRUE, uses ComBat-Seq instead of ComBat.
#' @param ...     Other parameters to pass to either ComBat or Combat-Seq.
#' @return        Batch corrected expression.
run_combat <- function(dat, batch, use_seq = FALSE, ...) {
  if (use_seq) return(sva::ComBat_seq(dat, batch, ...));
  return(sva::ComBat(dat, batch, BPPARAM = bpparam(), ...));
}

# Section: DESeq2 ----
# Alias: Variance Stabilizing Transformation.
vst <- DESeq2::vst;

#' Function: Run DESeq2 using a full model.matrix.
#'
#' @param cnts A wide ( genes x samples) data.frame or wrapper of RNA-Seq integer cnts.
#' @param meta A long ( samples x k) data.frame or wrapper of RNA-Seq experimental metadata.
#' @param mm   A full model.matrix representing the design of the experiment.
#' @param rmtr A logical indicating whether to remove technical replicates
#' @param para A logical indicating whether or not to use parallelization via BiocParallel.
#' @return A learned DESeqDataSet object.
#' @export
run_deseq2 <- function(cnts, meta, mm = NULL, design = NULL, rmtr = FALSE, betaPrior = FALSE, run_vst = TRUE, para = TRUE) {
  stopifnot(sum(as.numeric(is.null(mm)), as.numeric(is.null(design))) == 1);
  dds <- DESeq2::DESeqDataSetFromMatrix(cnts, meta, ~ 1);
  if (rmtr) dds <- DESeq2::collapseReplicates(dds, colData(dds)$Spec)
  if (!is.null(design)) {
    dds <- DESeq2::DESeqDataSetFromMatrix(counts(dds), meta, design);
    dds <- DESeq2::DESeq(dds, betaPrior = betaPrior, parallel = para);
  } else {
    dds <- DESeq2::DESeq(dds, full = mm, betaPrior = FALSE, parallel = para);
  }
  ret = list(dds = dds, design = list(frm = design, mm = mm), rows = which(mcols(dds)$betaConv), vst = NULL);
  if (run_vst) ret$vst <- vst(counts(dds));
  return(ret);
}

#' Function: Run contrasts on DESeq2 model.
#'
#' @param dds   A learned DESeq2 model.
#' @param name  A string naming the contrast to be performed.
#' @param gA    A list of model.matrix levels and their relative weights in group A of a contrast: gA - gB.
#' @param gB    A list of model.matrix levels and their relative weights in group B of a contrast: gA - gB.
#' @param idx   A logical vector of length = nrow(counts(dds)) indicating the genes with estimated parameters. NULL indicates retain all genes.
#' @param optac A logical vector indicating whether Optional AdCa were included.
#' @param para  A logical indicating whether or not to use parallelization via BiocParallel.
#' @return A tibble with fields describing the contrast at multiple levels, even including columns for the factor levels themselves, as well a column list for the output of the Contrasts themselves.
#' @export
contr <- function(dds, name, gA, gB, idx = NULL, optac = TRUE, para = TRUE) {
  # Check to make sure gA and gB are not both references.
  stopifnot(any(!gA$Ref, !gB$Ref));

  # If neither is a reference, check to make sure contrasts are orthogonal.
  if(all(!gA$Ref, !gB$Ref)) stopifnot(is.orth(gA$Contr$Pct, gB$Contr$Pct))

  # Compute difference between groups, which merges the 3 tibbles (Num = NumberObsv, Pct = RelPctObsv, Ind = BinaryIndicator), but gives gB a '-' sign.
  diff_tb <- tibble(Level = gA$Contr$Level, Num = gA$Contr$Num - gB$Contr$Num, Pct = gA$Contr$Pct - gB$Contr$Pct, Ind = gA$Contr$Ind - gB$Contr$Ind);

  # Helper Function: Flatten diff_tb from a tibble of shape [Lvls, 1 + length(Num, Pct, Ind)] -> [1, 1 + 3 * Lvls], which is a single row.
  tb_to_row <- function(x, id) {
    y <- t(x[-1])
    colnames(y) <- x[[id]];
    tib3 <- add_column(as_tibble(y), Name = rownames(y), .before = 1) %>% gather(key = 'New_ID', value = 'Val', colnames(.)[-1]) %>% unite(Key, Name:New_ID)
    tib4 <- as_tibble(t(tib3[2]))
    colnames(tib4) <- tib3$Key
    return(tib4);
  }

  # Flatten diff tibble
  row_tb <- tb_to_row(diff_tb, 'Level')

  # Run contrasts without and with shrinkage
  res.raw    <- DESeq2::results(dds, contrast = diff_tb$Pct, parallel = para)
  res.shrink <- DESeq2::lfcShrink(dds, res = res.raw, type = 'ashr', quiet = TRUE, parallel = para);

  # Format contrasts results as data.table
  dt.raw     <- contr_to_dt(res.raw, idx)
  dt.shrink  <- contr_to_dt(res.shrink, idx)

  # Wrap output into a tibble row. Column Contrasts is a 3 element list of lists: the raw & shrunk contrast results in data.table form and the returned object and the diff_tb.
  tib <- tibble_row(
    TestName        = name,
    OptAdCa     = optac,
    Stage.A     = gA$Stage,
    Loc.A       = gA$Loc,
    Dys.A       = gA$Dys,
    Stage.B     = gB$Stage,
    Loc.B       = gA$Loc,
    Dys.B       = gB$Dys,
    NumLvl.A    = sum(gA$Contr$Ind),
    NumLvl.B    = sum(gB$Contr$Ind),
    NumSample.A = gA$SumNum,
    NumSample.B = gB$SumNum,
    NameLvl.A   = gA$Name,
    NameLvl.B   = gB$Name,
    row_tb,
    Contrast = list(list(Raw = list(dt = dt.raw, Results = res.raw), Shrink = list(dt = dt.shrink, Results = res.shrink), diff_tb = diff_tb))
  )
  return(tib)
}

#' Function: Convert contrast results to data.table
#' @param mat  A matrix representing the contrast results from DESeq2 of shape [num_genes, 5 or 6], with ncol depending on if its raw or with shrinkage.
#' @param ridx A logical vector of length num_genes indicating which genes to keep. NULL indicates all.
#' @return A data.table of the matrix mat with an additional column, Gene, ordered first.
#' @export
contr_to_dt <- function(mat, ridx = NULL) {
  if (!is.null(ridx)) mat <- mat[ridx, ]
  dt.mat   <- as.data.table(mat)
  num.cols <-  length(colnames(mat))
  if (num.cols != 5 && num.cols != 6) stop(str_c( 'fmt.as.dt given res with ', num.cols, ' columns which is not 5 or 6.'));
  sel.cols <- if (num.cols == 5) c('Mean', 'Log2FC', 'Log2FC.SE', 'Pval', 'FDR') else c('Mean', 'Log2FC', 'Log2FC.SE', 'Stat', 'Pval', 'FDR');
  setnames(dt.mat, sel.cols);
  dt.mat[, Gene := rownames(mat)]
  setcolorder(dt.mat, 'Gene')
  return(dt.mat)
}

PkgPerf <- R6::R6Class(
  "PkgPerf",
  private = list(
    .opt_lvl     = 0L,
    .num_threads = 0L,
    .num_nodes   = 0L,
    .check_args = function(x, lb = NULL, ub = NULL) {
      stopifnot()
      y  = floor(x)
      z = all(is.numeric(x), length(x) == 1, y >= lb, x == y)
      if (is.null(ub)) return(z)
      return(all(z, y <= ub))
    },
    .check_opt_lvl = function(x) {return(private$.check_args(x, 0L, 3L))},
    .check_num_threads = function(x) {return(private$.check_args(x, 0L, NULL))},
    .check_num_nodes = function(x) { return(private$.check_args(x, 1L, NULL))},
    .set_opt_lvl = function(x) {
      private$.opt_lvl <- as.integer(x)
      options(datatable.optimize = private$.opt_lvl)
      return(private$.opt_lvl)
    },
    .set_num_threads = function(x) {
      private$.num_threads <- as.integer(x)
      data.table::setDTthreads(private$.num_threads)
      return(private$.num_threads)
    },
    .set_num_nodes = function(x) {
      private$.num_nodes <- as.integer(x)
      BiocParallel::register(BiocParallel::MulticoreParam(private$.num_nodes))
      return(private$.num_nodes)
    }
  ),
  active = list(
    opt_lvl = function(x) {
      if (missing(x)) return(private$.opt_lvl)
      if (!private$.check_opt_lvl(x)) stop("`$x` is invalid", call. = FALSE)
      return(private$.set_opt_lvl(x))
    },
    num_threads = function(x) {
      if (missing(x)) return(private$.num_threads)
      if (!private$.check_num_threads(x)) stop("`$x` is invalid", call. = FALSE)
      return(private$.set_num_threads(x))
    },
    num_nodes = function(x) {
      if (missing(x)) return(private$.num_nodes)
      if (private$.check_num_nodes(x)) stop("`$x` is invalid", call. = FALSE)
      return(private$.set_num_nodes(x))
    }
  ),
  public = list(
    paths = list(),
    initialize = function(opt_lvl = 3L, num_threads = 16L, num_nodes = 16L) {
      stopifnot(
        private$.check_opt_lvl(opt_lvl),
        private$.check_num_threads(num_threads),
        private$.check_num_nodes(num_nodes)
      )
      private$.opt_lvl     <- opt_lvl
      private$.num_threads <- num_threads
      private$.num_nodes   <- num_nodes
      options(datatable.optimize = private$.opt_lvl)
      data.table::setDTthreads(private$.num_threads)
      BiocParallel::register(BiocParallel::MulticoreParam(private$.num_nodes))
    },
    get_opt_lvl     = function() {return(getOption('datatable.optimize')) },
    get_num_threads = function() {return(data.table::getDTthreads())},
    get_num_nodes   = function() {return(c(BiocParallel::multicoreWorkers(), parallel::detectCores()))}
  )
);
