#include <cerrno>
#include <cfenv>
#include <cmath>
#include <cstring>
#include <fstream>
#include <iostream>
#include <string>

#include <json.hpp>

#include "SimulatorMultiPart.cuh"
#include "../Commons/Physics.h"
#include "structs.hpp"
#include "myTimer.h"
#include "ForcesDescription/forces.hpp"

void testAndExit(nlohmann::json js, std::string name, std::string message)
{
    if (!js.contains(name))
    {
        std::cout << "ERROR. " << message << "\n";
        exit(0);
    }
}

template <class T>
void readOrExit(nlohmann::json js, std::string name, std::string message, T &value)
{
    if (!js.contains(name))
    {
        std::cout << message << "\n";
        exit(0);
    }
    else
    {
        value = js[name];
    }
}

template <class T>
void readOrDefault(nlohmann::json js, std::string name, std::string message, T &def, T value)
{
    if (!js.contains(name))
    {
        std::cout << message << "\n";
        def = value;
    }
    else
    {
        def = js[name];
    }
}

void readVec3(nlohmann::json js, std::string name, std::string message, float3 &def, float3 value)
{
    if (!js.contains(name))
    {
        std::cout << message;
        def = value;
    }
    else
    {
        def.x = js[name]["x"];
        def.y = js[name]["y"];
        def.z = js[name]["z"];
    }
}

void readSimulationParameters(nlohmann::json data, simulation_t &mySim)
{
    testAndExit(data, "SIMULATION", "Missing \"SIMULATION\" section in config file.");
    readOrExit(data["SOLVANT"], "eta", "Missing \"eta\" in section \"SOLVANT\"", mySim.eta);
    readOrExit(data["SIMULATION"], "totalTime", "Missing \"totalTime\" in section \"SIMULATION\"", mySim.totalTime);
    readOrExit(data["SIMULATION"], "deltaTime", "Missing \"deltaTime\" in section \"SIMULATION\"", mySim.deltaTime);
    readOrDefault(data["SIMULATION"], "outputScale", "Missing \"outputScale\"in section \"SIMULATION\". Setting default value to 1.0.", mySim.outputScale, 1.0f);
    readOrDefault(data["SIMULATION"], "temperature", "Missing \"temperature\"in section \"SIMULATION\". Setting default value to 273.", mySim.temperature, 273.0f);
    readOrExit(data["SIMULATION"], "outputDir", "Missing \"outputDir\" in section \"SIMULATION\".", mySim.outputDir);
    readOrDefault(data["SIMULATION"], "savePeriod", "Missing \"savePeriod\" in section \"SIMULATION\".Setting value to 1000.", mySim.savePeriod, 1000);
    readOrDefault(data["SIMULATION"], "restartSaveInterval", "Missing \"restartSaveInterval\" in section \"SIMULATION\".Setting value to 10000.", mySim.restartSaveInterval, 10000);
    readOrDefault(data["SIMULATION"], "savePeriod", "Missing \"savePeriod\" in section \"SIMULATION\".Setting value to 1000.", mySim.savePeriod, 1000);
    readOrDefault(data["SIMULATION"], "savePrecision", "Missing \"savePeriod\" in section \"SIMULATION\".Setting value to 3.", mySim.savePrecision, 3);
    readOrDefault(data["SIMULATION"], "overlapDistance", "Missing \"overlapDistance\" in section \"SIMULATION\".Setting value to 1.1.", mySim.overlapDistance, 1.1f);
    readOrExit(data["SIMULATION"], "density", "Missing \"density\" in section \"SIMULATION\".", mySim.density);
    readOrExit(data["SIMULATION"], "randomSeedPosition", "Missing \"randomSeedPosition\" in section \"SIMULATION\".", mySim.randomSeedPosition);
    readOrExit(data["SIMULATION"], "restartFile", "Missing \"restartFile\" in section \"SIMULATION\".", mySim.restartFile);
    readOrExit(data["ACCELERATION"], "maxListSize", "Missing \"maxListSize\" in section \"ACCELERATION\"", mySim.maxListSize);

    readOrDefault(data["ACCELERATION"], "Rc_multiplicator", "Missing \"Rc_multiplicator\" in section \"ACCELERATION\"", mySim.Rc_multiplicator, 1.05f);
    if (mySim.Rc_multiplicator < 1.0 || mySim.Rc > 2.0)
    {
        std::cout << "Rc_multiplicator value must be between 1 and 2. Default value of 1.05 will be used for this run\n";
        mySim.Rc_multiplicator = 1.05f;
    }
}

void readParticules(nlohmann::json data, std::vector<particles_t> &parts)
{
    int count = 0;
    for (auto p : data)
    {
        particles_t tmp;
        readOrExit(p, "radius", "Missing \"radius\" in section " + std::to_string(count) + " of \"PARTICLES\"", tmp.radius);
        readOrExit(p, "type", "Missing \"type\" in section " + std::to_string(count) + " of \"PARTICLES\"", tmp.type);
        readOrExit(p, "number", "Missing \"number\" in section " + std::to_string(count) + " of \"PARTICLES\"", tmp.number);
        readOrDefault(p, "psi", "No \"psi\" value found in section " + std::to_string(count) + " of \"PARTICLES\".Setting value to 0.0", tmp.psi, 0.0f);
        readOrExit(p, "volumicMass", "Missing \"volumicMass\" in section " + std::to_string(count) + " of \"PARTICLES\"", tmp.volumicMass);
        readOrDefault(p, "distribution", "Missing \"distribution\" in section " + std::to_string(count) + " of \"PARTICLES\".Setting value to \"normal\"", tmp.distribution, std::string("normal"));
        readOrDefault(p, "minRadius", "Missing \"minRadius\" in section " + std::to_string(count) + " of \"PARTICLES\".Setting value to 0.0", tmp.minRadius, 0.0f);
        readOrDefault(p, "maxRadius", "Missing \"maxRadius\" in section " + std::to_string(count) + " of \"PARTICLES\".Setting value to 0.0", tmp.maxRadius, 0.0f);
        readOrDefault(p, "radius_std_dev", "Missing \"radius_std_dev\" in section " + std::to_string(count) + " of \"PARTICLES\".Setting value to 0.0", tmp.std_dev, 0.0f);
        parts.push_back(tmp);
        count++;
    }
}

int main(int argc, char *argv[])
{
    std::ifstream current;

    if (argc == 1)
    {
        std::cout << "Error : No config file given. Exit!" << std::endl;
        exit(0);
    }

    if (argc > 1)
    {
        char const *option = argv[1];
        current.open(option);
        simulation_t mySim;

        if (!current)
        {
            std::cout << "Error : Cant open : " << option << "\n";
            exit(0);
        }

        std::vector<particles_t> myParts;

        std::vector<Yukawa> Yukawas;
        std::vector<Lennard_Jones> Lennard_Joness;
        std::vector<DLVO> DLVOs;

        nlohmann::json currentCfg;
        current >> currentCfg;
        current.close();

        readSimulationParameters(currentCfg, mySim);

        testAndExit(currentCfg, "PARTICLES", "Missing \"PARTICLES\" section in config file.");
        readParticules(currentCfg["PARTICLES"], myParts);

        testAndExit(currentCfg, "SOLVANT", "Missing \"SOLVANT\" section in config file.");
        readOrExit(currentCfg["SOLVANT"], "dielectricConstant", "Missing \"dielectricConstant\" in section \"SOLVANT\"", mySim.dielectricConstant);

        testAndExit(currentCfg, "FORCE", "Missing \"FORCES\" section in config file.");
        auto f = currentCfg["FORCE"];

        for (auto force : f)
        {
            if (force["name"] == "Yukawa")
            {
                Yukawa tmp;
                tmp.init(force, mySim.temperature);
                Yukawas.push_back(tmp);
            }
            if (force["name"] == "Lennard_Jones")
            {
                Lennard_Jones tmp;
                tmp.init(force, mySim.temperature);
                Lennard_Joness.push_back(tmp);
            }
            if (force["name"] == "DLVO")
            {
                DLVO tmp;
                tmp.init(force, mySim.temperature);
                DLVOs.push_back(tmp);
            }
        }

        myTimer timer = myTimer("Total time", false, 1000);
        SimulatorMultiPart *mySimulator = new SimulatorMultiPart(mySim, myParts, Yukawas, Lennard_Joness, DLVOs);
        timer.start();
        do
        {
            mySimulator->computeNext();
        } while (mySimulator->time < mySim.totalTime);
        timer.stop();
        timer.print();
        mySimulator->saveBin("final.bin");

        std::cout << "number of iteration:" << mySimulator->simIt << " grid rebuild count:" << mySimulator->gridRebuildCount << "\n";
        cudaDeviceReset();
    }
    return 0;
}
