#!/usr/bin/env python3
import os
import sys
import json
import numpy as np
import matplotlib.pyplot as plt
import csv

kb  = 1.38065e-23
PI = 3.141592653589793
E0 = 8.85418782e-12


def printquit(msg):
    print(msg)
    quit()

def lj_pot(epsilon, sigma,n,Rc,kb,T,radius):
    r = np.linspace(0.8*radius,radius*Rc , 1000)
    return r/radius,(4*( epsilon * np.power(sigma / (2*r), 2*n) - epsilon * np.power(sigma /(2*r), n)))

def computeDLVO(A,a0,a1,KT,epsi,psii,psij,Rc):
    r=np.linspace(a0+a1+0.3e-9,Rc*0.5*(a0+a1),1000);
    d=r-a0-a1
    j=0.5*(a0+a1)*PI*epsi*psii*psij;
    r2=r/(a0+a1);
    u1=(2*a0*a1)/(r*r-4*a0*a1);
    u2=(2*a0*a1)/(r*r);
    u3=np.log((r*r-4*a0*a1)/(r*r));
    u4=(-A/6)*(u1+u2+u3);
    u5=np.log((1+np.exp(-d*k))/(1.-np.exp(-d*k)));
    u6=np.log(1-np.exp(-2*d*k));
    u7=j*(u5+u6);
    u8=u4+u7;
    u=u8/(KT);
    return r2,u

def yukawa(u,k,temperature,psii,psij,a0,a1,Rc):
    r=np.linspace(a0+a1+0.3e-9,Rc*0.5*(a0+a1),1000);
    r2=r/(a0+a1);
    return r2,kb*temperature*u*psii*psij*(a0+a1)/r*np.exp(-k/(0.5*(a0+a1))*(r-a0-a1));

if(len(sys.argv)==1):
    printquit("No config file given!")

fname = sys.argv[1]
if(not os.path.isfile(fname)):
    printquit("Cant read "+fname+"!")

with open(fname) as json_file:
    data = json.load(json_file)
    if(not "FORCE" in data or not "PARTICLES"):
        printquit("Missing section FORCES or PARTICULES in config file!")
    
    if(not "SOLVANT" in data):
        printquit("Missing section SOLVANT!")
    
    if(not "dielectricConstant" in data["SOLVANT"]):
        printquit("Missing parameter dielectricConstant in section SOLVANT in config file!")
    dielectricConstant = data["SOLVANT"]["dielectricConstant"]
    
    if(not "SIMULATION" in data):
        printquit("Missing section SIMULATION in config file!")
    if(not "temperature" in data["SIMULATION"]):
        printquit("Missing parameter temperature in section SIMULATION of config file!")
    T = data["SIMULATION"]["temperature"]
    if(not "outputDir" in data["SIMULATION"]):
        printquit("Missing parameter temperature in section SIMULATION of config file!")
    outputDir = data["SIMULATION"]["outputDir"]

    if(not "radius" in data["PARTICLES"][0]):
        printquit("Missing radius in section PARTICLES in config file!")
    radius = data["PARTICLES"][0]["radius"]



    count =0
    for f in data["FORCE"]:
        if(f["name"] == "Lennard_Jones"):
            if(not "n" in f or not "well" in f):
                printquit("Missing parameter n or well in Lennard_Jones in FORCES in config file!")
            Rc = f["Rc"]
            n = f["n"]
            ljWell = f["well"]
            x,y = lj_pot(ljWell, 2*radius,n,Rc,kb,T,radius)
            fig, ax = plt.subplots()
            ax.plot(x, y)
            ax.set(xlabel='r/d',ylabel ='U/KBT')
            plt.ylim(-1.1*ljWell,1.1*ljWell)
            ax.grid()
            ax.set_title("Lennard-Jones with n = "+str(n)+ " ljWell = "+str(ljWell))
            fig.savefig(outputDir+"/potential"+str(count)+".pdf")
            #plt.show()
            plt.close(fig)
        if(f["name"]=="DLVO"):
            if(not "k" in f):
                printquit("Missing parameter k in DLVO in FORCES in config file!")
            k = f["k"]
            if(not "Hamaker" in f or not "psi" in data["PARTICLES"][0]):
                printquit("Missing parameter hamaker or psi in section PARTICLES in config file!")
            Rc = f["Rc"]    
            Hamaker = f["Hamaker"]
            p0=f["first"]
            p1=f["second"]
            psii = data["PARTICLES"][p0]["psi"] 
            psij = data["PARTICLES"][p1]["psi"]    
            x,y = computeDLVO(Hamaker,radius,radius,kb*T,E0*dielectricConstant,psii,psij,Rc)
            fig, ax = plt.subplots()
            ax.plot(x, y)
            ax.set(xlabel='r/d',ylabel ='U/KBT')
            mini = np.min(y)
            plt.ylim(mini*1.1,-2*mini)
            ax.grid()
            plt.suptitle("DLVO with k = "+  "{:.2e}".format(k))
            plt.title("beteween particle "+str(p0) +" with psi = "+str(psii)+" V and particle "+str(p1)+" with psi = "+str(psij)+ " V")
            fig.savefig(outputDir+"/potential"+str(count)+".pdf")
            #plt.show()
            plt.close(fig)
        if(f["name"]=="Yukawa"):
            if(not 'U' in f or not 'k' in f):
                printquit("Missing parameter u or k in Yukawa in FORCES in config file!")
            Rc = f["Rc"]
            u = f["U"]
            k = f["k"]  
            p0=f["first"]
            p1=f["second"]
            psii = data["PARTICLES"][p0]["psi"] 
            psij = data["PARTICLES"][p1]["psi"]  
            a0 = data["PARTICLES"][p0]["radius"] 
            a1 = data["PARTICLES"][p1]["radius"]              
            x,y = yukawa(u,k,T,psii,psij,a0,a1,Rc)
            fig, ax = plt.subplots()
            ax.plot(x, y)
            ax.set(xlabel='r/d',ylabel ='U/KBT')
            mini = np.min(y)
            plt.ylim(mini*1.1,-2*mini)
            ax.grid()
            plt.suptitle("Yukawa with k = "+  "{:.2e}".format(k) + " and u = "+"{:.2e}".format(k))
            plt.title("beteween particle "+str(p0) +" with psi = "+str(psii)+" V and particle "+str(p1)+" with psi = "+str(psij)+ " V")
            fig.savefig(outputDir+"/potential"+str(count)+".pdf")
            plt.show()
            plt.close(fig)            

