# Brownian Dynamic Simulator


![multipart.png](multipart.png)

*Yukawa simulation with 2 kind of particles with distribution of radiuses. Capture Visualization and analysis of atomistic simulation data with [OVITO](https://www.ovito.org/) the Open Visualization Tool.*


This software allows to make Brownian dynamics simulations based on interactions between colloids in suspension with the Lennard-Jones, Yukawa and DLVO potential. This tools can handle simulation with fixed particles radiuses or with a distribution of radiuses. The configuration is done by a json file. A second tool  enable analysis of agregation and percolation.

## Philosophy
There are many molecular simulation software packages, such as [laamps](https://www.lammps.org/) or [hoomd](https://glotzerlab.engin.umich.edu/hoomd-blue/), which are designed for complex simulations potentially on several servers with multiple GPUs. These programs can cover a wide range of simulations, but they involve a long learning curve, complex compilation processes and slower performance than a dedicated tool. 

Here the aim is to provide software that is as easy to use as possible and highly optimised to be fast.

## Technical overview
The aim is to propose an efficient GPU implementation. The neighbourhood search is based on a grid constructed efficiently using bucket sort and verlet lists.

## Requirement
### software

1. Cuda version>=10 
2. C++17 compiler
3. CMake
3. [boost](https://www.boost.org/) 1.8 or better (optional)


### Hardware
Nvidia GPU 10X0, 20X0, 30X0 and their server equivalent ;
### Tested configuration
OS : linux & Windows

Cuda 11.0,11.6,11.8,12.2

GPUs : GTX1080, GTX2080, GTX3090, Nvidia A40, Nvidia H100

## Usage
### Compiling

```
cd soma-dns
cmake -S . -B build
cmake --build build
```

When the build is finished the executable `./sim` will be located in the `/build/` folder. In case you used windows, it will be in the `/build/Debug` folder.


### Run a simulation
First, create a folder simulation output with a sub-folder  named XYZ

To run a simulation, just launch sim with a json file : `./sim mySimulation.json`

### Ouput
The ouptut of the simulation is a collection of XYZ files with this layout :

  `[particleID] [particle type] [position.x] [position.y] [position.z] [empty value] [radius]`

There is also a binary backup, `restart.bin`, from time to time (see configuration). This allows the simulation to be restarted in the event of a shutdown, as can be the case with calculation clusters that limit execution time.

### Analyse results
To run analysis type : `python3 Analyse.py mySimulation. json`

The results gives :

* a subdirectory **Analysis** with :
	* XYZ files with the cluster index
	* a csv file for each XYZ file with the list and size of clusters of particles
* a csv file named percolation.csv with for each XYZ file and percolation in 3 directions
* A pdf graphics of agregation curve

## Configuration
A configuration file must contain several mandatory sections :

* the PARTICLES section contains a description of the different types of particles as json array ;
* the SIMULATION section contains the global parameters of the simulation ;
* the SOLVENT section contains information about the solvent used in the simulation ;
* the FORCES section describe the potential to use in the simulation.
* the ACCELERATION section
* the ANALYSIS section
### Particles

| **Name of field** | **Type of value** | **Meaning** |
|-------------------|-------------------|-------------|
| type | int | A number for a kind of particle |
| radius | float | Size of particles |
| radius_std_dev | float | Only effective with SimulatorWithRadius. If 0, all particles will have the same radius. Otherwise, the radii of the particles will follow a [log normal distribution](https://en.wikipedia.org/wiki/Log-normal_distribution) or [normal distribution](https://en.wikipedia.org/wiki/Normal_distribution) |
| distribution | string | kind of radius distribution : normal or logNomal. Be careful with the normal distribution : it can generate negative rays. Do not forget to filter with the minimum radius |
| radiusMax | float | maximum radius for a distribution. |
| radiusMin | float | minimum radius for a distribution. |
| volumicMass | float | Volumic mass of the particle |
| number | integer | Count number of particules |
| psi | float | This psi value for dlvo or Yukawa  forces |

### Simulation
| **Name of field** | **Type of value** | **Meaning** |
|-------------------|-------------------|-------------|
| temperature | float | Temperature of the simulation |
|overlapDistance|float|Minimal distance between 2 particles at the beginning of the simulation. Expressed as a function of radius|
|deltaTime|float| time step of the simulation|
|totalTime|float|Duration of simulation|
|outputDir|string|Path to simulation output|
|outputScale|float|Scale of output file. Visualization and analyse with ovito need normalized output|
|savePeriod|int|if set to -1, no output; if set to 0, save file with logarithmic pattern. Else, number of iteration between 2 save|
|savePrecision|int|Number of digits of XYZ file save|
|restartSaveInterval|int|Represents  number of iterations after which the current state of a simulation is saved in a backup file to enable subsequent recovery. A file named restart.bin allows to restart simulation with full precision |
|randomSeedPosition|int|Seed used for random generation of radius and particle position|
|density|float|Density of colloids between 0 and 1|
|restartFile|string|Allow to restart simulation from XYZ file or binary file|


### Solvent
| **Name of field** | **Type of value** | **Meaning** |
|-------------------|-------------------|-------------|
|dielectricConstant|float|The dielectric constant of the fluid|
|eta| float|The dynamic viscosity of fluid|

### Forces
| **Name of field** | **Type of value** | **Meaning** |
|-------------------|-------------------|-------------|
| name | string | "Lennard_Jones", "DLVO" or "Yukawa. |

Forces have elements in common:

| **Name of field** | **Type of value** | **Meaning** |
|-------------------|-------------------|-------------|
| Rc | float | distance where the force reach 0 |
|first| int| type of particle in interraction|
|second| int| type of particle in interraction|

#### Lennard_Jones
We use the generalised [Lennard Jones](https://en.wikipedia.org/wiki/Lennard-Jones_potential) potential with r as the distance between 2 particles:

$$V(r) = 4\epsilon \left( \left(\frac{\sigma}{r} \right)^{2n} - \left(\frac{\sigma}{r} \right)^n \right)$$

with :
$\sigma$ = $a_{i}$+$a_{j}$,  $a_{i}$ and $a_{j}$ as radiuses of particles

| **Parameter** | **Type of value** | **Meaning** |
|-----------------------------|----------------------------|---------------------|
|n | int | power parameter|
|well| float|the bonding energy.|

#### DLVO
The potential used is :
$$V(r) = -\frac{A}{6}  ( \frac{2a_{i}2a_{j}}{r_{2}-(a_{i}+a_{j})} + \frac{2a_{i}2a_{j}}{r_{2}-(a_{i}-a_{j})}+\ln(\frac{r^{2}-(a_{i}+a_{j})^{2}}{r^{2}-(a_{i}-a_{j})^{2}}) +$$
$$\pi \epsilon \frac{a_{i}a_{j}}{a_{i}\times a_{j}}(\psi_{i}^{2}+\psi_{j}^{2})\times(\frac{(2\psi_{j}\psi_{j})}{(\psi_{i}^{2}+\psi_{j}^{2}}\ln\frac{1+\exp(-k(a_{i}+a_{j}))}{1-\exp(-k(a_{i}+a_{j}))}+\ln(1-\exp(-2k(a_{i}+a_{j})))$$

with :


* $A$ is Hamaker constant
* $a_{i}$ and $a_{i}$ are radiuses of 2 particles
* $psi_{i}$ and $psi_{j}$ are the charge of each particles
* $k$ is the inverse of  Debye length
 
| **Parameter** | **Type of value** | **Meaning** |
|-----------------------------|----------------------------|---------------------|
|Hamaker | float | Hamaker constant.|
|k| float |inverse of Debye length|
|rcdlvo|float|Distance expressed in radius to cut the potential|
|rcrep|float | Distance from which a repulsive force is applied expressed in diameter of particules|
|coeffdir|float|If distance between 2 particles <rcrep, a linear repulsive force  is applied with coeffdir slope

#### Yuakawa
The potential used is :

$$V(r) = Uq_{i}q_{j}\frac{2a}{r}\exp(-k(r-2a))$$

with:



* $U$ is potential energy
* $q_{i}$ and $q_{j}$ are charges of 2 colloids
* $a$ is radius of particle
* $k$ inverse of Debye length
 
| **Parameter** | **Type of value** | **Meaning** |
|-----------------------------|----------------------------|---------------------|
|k|float|inverse of Debye length|
|U|float|potential energy|
|rcrep|float | distance expressed in radius to cut the potential.|
|coeffdir|float|if distance between 2 particles <rcrep, a linear repulsive force  is applied with coeffdir slope


### Acceleration
| **Parameter** | **Type of value** | **Meaning** |
|-----------------------------|----------------------------|---------------------|
|maxListSize|int|This parameter sets the size of the neighbour lists. Depending on the search radius and/or the presence of particles of different sizes, it may be necessary to increase this value. If, during a simulation, the lists prove to be too small, the simulation stops and a message of the type `assert(count<lsize) fails`|
|Rc_multiplicator|float|This software uses verlet lists, the principle of which is to increase the search radius to avoid having to update these lists at each iteration. This variable sets the size of the new search radius by multiplying the search radius by this value. This value must be between 1 and 2. If it is 1, the lists are updated at each iteration. Our tests have shown that the optimal value is 1.05 on our GPUs.|
# Adding other force

It's possible to add your own force.  Simply write a json file with the following rules, as in the example below :


* the file name must be the same as the force name  in field "name"
*  "name" :  the name of the structure or class being generated. In this example, the name is "Lennard_Jones".
*   "variables" : an array that contains the variables needed for generating the code. Each variable is represented by an object with two properties: "name" (variable name) and "type" (variable type). In this example, the variables are "Rc" of type float, "well" of type float, "n" of type int, "second" of type int, and "first" of type int.
*   properties "Rc", "first" and "second" are mandatory
*   "potential_init": this is a string representing the initialization of a potential variable. The main idea is to calculate the constant part of a force once. if it's not used, just fill this propertie with "0.f;"  
*   "formula": this is a string representing a formula used in the generated code. It uses the previously defined variables as well as other variables or constants not present in this JSON file. The formula describes the force computation. It must be written in C++ style and it must return value in double3

```
{
  "name": "Lennard_Jones",
  "variables": [
    {
      "name": "Rc",
      "type": "float"
    },
    {
      "name": "well",
      "type": "float"
    },
    {
      "name": "n",
      "type": "int"
    },
    {
      "name": "second",
      "type": "int"
    },
    {
      "name": "first",
      "type": "int"
    }
  ],
  "potential_init":"4 * data.n * data.well * Constant::kb * temperature;",
  "formula":"const double r = r0 + r1;const double p = power(r/dist, data.n);double3 res = diff_p_pn * data.pot * p / dist/dist * (2 * p - 1);return res;"
}
```
List of data usable in formula
| **Parameter** | **Type of value** | **Meaning** |
|-----------------------------|----------------------------|---------------------|
|Constant::kb|float |Boltzmann constant|
|M_PI|float | Pi|
|r0 and r1|float| radiuses of the pair particles|
|dist|float| distance between the pair of particles|
|data.something|float or int|properties of your force are stored in a structure named data. To read a propertie named "foo", just type data.foo|
|diff_p_pn|double3| vector between the pair of particles|

### Analysis
| **Parameter** | **Type of value** | **Meaning** |
|-----------------------------|----------------------------|---------------------|
|cut_off_distance|float| If distance between 2 particles is lower than cut_off_distance multiplied by the sum of radiuses, the 2 particles are agregated

# Troubleshooting
If an error occur about `findCellBounds`, that means that a particle or more is/are out of the simulation. Take care about potential configuration and timestep.

If an error occur about `nlohmann`, that means that the json file  is malformed or incomplete.
