"""
config.py

File that configures the analysis to be conducted

Created by Kate Lonergan, ETH Zurich
2023

Modify datasets_to_consider to load in different datasets.

"""

from datetime import datetime
import os
import matplotlib.pyplot as plt
import sys


def datasets_to_consider():
    # List string names of different datasets to consider in a loop in the analysis.
    # Names must correspond to options listed in Config file to generate individual results folders
    datasets = ["Pickering"]
    return datasets


class Config:
    def __init__(self,
                 flag_save=False,  # Set to true to save results
                 flag_show_plots=True,  # Set to true to show plots as they are generated
                 save_root_path='results',
                 pd_save_sep=';',  # Set the separation for saved .csv files (e.g., ',' or ';')
                 dataset=None,
                 error_metrics=[],
                 data_norming=[],
                 colour_order=[4, 14, 0, 8, 16, 12, 15, 1, 10, 20, 6, 16, 2, 5, 22, 9, 19, 3, 13, 7, 18],  # Set the colour order for custom colour scheme (see Graphics Settings below)
                 main_colour=4,  # Which colour of the original, unordered colour scheme is the main colour
                 diverge_low_colour=4,  # Which colour of the original, unordered colour scheme should be used as the "cold" colour for a heatmap
                 diverge_high_colour=14,  # Which colour of the original, unordered colour scheme should be used as the "hot" colour for a heatmap
                 font_size=7,
                 font_family='Arial',
                 colour_scheme='batlow',  # Name of the colour scheme from Scientific Colour maps from Fabio Crameri (also see Graphics Settings below)
                 vis_n_solutions=5,  # Number of top solutions to visualise
                 track_top_n_solutions=5,  # Number of top solutions to track
                 n_bootstrap=5000,  # Set the number of boostrap samples for the robust cost correlation
                 tol_unique_top_pct=[1] + list(range(5, 55, 5)),  # Set the list for where CDF of unique solutions should be calculated
                 ):

        #################################################
        # Save information
        #################################################
        self.flag_save = flag_save
        self.flag_show_plots = flag_show_plots
        self.save_root_path = os.path.join(save_root_path, datetime.today().strftime("%Y-%m-%d"))
        self.save_figures_path = os.path.join(save_root_path, 'plots')
        self.pd_save_sep = pd_save_sep

        #################################################
        # Analysis information
        #################################################
        self.dataset = dataset
        self.error_metrics = error_metrics
        self.data_norming = data_norming
        self.vis_n_solutions = vis_n_solutions
        self.track_top_n_solutions = track_top_n_solutions
        self.n_bootstrap = n_bootstrap
        self.tol_unique_top_pct = tol_unique_top_pct

        # Specify what definitions of justice should be considered (dictionary)
        self.justice_proxies = {
            # Principle of affordability
            'Minimum cost': True,

            # Principle of autonomy
            'Autonomy': True,

            # Principle of decent work
            'Maximum job creation': True,
            'Equitable job creation FF': True,

            # Principle of environmental protection
            'Minimum land burden':  False,

            # Principle of equality
            'Equal cost per person': True,

            'Equal cost share by GDP': True,

            'Equal cost share by historical emissions':  False,

            'Equal job share by population': True,

            'Equal land burden (pct)': False,

            # Principle of intergenerational equality
            'Minimum nuclear': True,
        }

        # Compare what attributes should be considered
        # Attribute name: [[list of key words], # in colour scheme]
        self.compare_plot_attributes = {
            'Solar PV': [['pv', 'MW'], 1],
            'Onshore wind': [['wind', 'onshore', 'MW'], 0],
            'Offshore wind': [['wind', 'offshore', 'MW'], 2],
            'Battery': [['battery', 'MW'], 3],
            'Heatpumps': [['_hp_', 'MW'], 4],
            'Electrolysers': [['electrolysis', 'MW'], 5],
        }

        # Specify the column names for PV and wind
        self.pv_wind_techs = ['open_field_pv', 'roof_mounted_pv', 'wind_offshore', 'wind_onshore']

        #################################################
        # Set data paths
        #################################################

        # Country area
        self.country_area_path = os.path.join(*("data", "country_area", "country_area.csv"))
        self.country_area_sep = ','

        # Country names, 2 letter abbreviations, 3 letter abbreviations
        self.country_key_path = os.path.join(*("data", "country_codes", "country_key.csv"))
        self.country_key_sep = ';'

        # Current jobs
        self.jobs_path = os.path.join(*("data", "current", "current_jobs", "curr_all_jobs.csv"))
        self.jobs_sep = ';'

        # Employment factors
        self.emp_factors_path = os.path.join(*("data", "jobs", "employment_factors.csv"))
        self.emp_factors_sep = ';'

        # Historical emissions
        self.cum_emissions_path = os.path.join(*("data", "co2_emissions", "cumulative_share_co2_2021.csv"))
        self.cum_emissions_sep = ','

        # GDP data
        self.gdp_path = os.path.join(*("data", "gdp_data", "gdp.csv"))
        self.gdp_sep = ';'

        # Population path (regional)
        self.popu_regional_path = os.path.join(*("data", "population_data", "euro_regional_pop_formatted.csv"))
        self.popu_regional_sep = ';'

        # Technology land-use data
        self.tech_area_path = os.path.join(*("data", "technology_area", "tech_area_compiled.csv"))
        self.tech_area_sep = ';'

        # Set map path
        self.map_path = os.path.join(*("data", "custom_maps", "europe_cropped.txt"))

        # Voter preferences
        self.voter_preferences_path = os.path.join(*("data", "eu_surveys", "voter_preferences.csv"))
        self.voter_preferences_sep = ';'
        self.voter_preferences_subfolder = 'voter_preferences'

        #################################################
        # Dataset-specific information
        #################################################
        if self.dataset == 'Pickering':

            # Analysis information
            self.authors = ['Pickering et al. 2023']
            self.scenario_name = 'spores'
            self.scenario_path = os.path.join(*("data", "spores_data"))
            self.subfolder_names = ['slack-10']
            self.scenario_factors = ['total_investment_cost', 'nameplate_capacity',
                                     'primary_energy_supply', 'import_ratio'],
            self.final_consumption = 'final_consumption'

            # Update path name to dataset-specific results (plot folder updated at the end of class defn.)
            self.save_root_path = os.path.join(*(self.save_root_path, "Pickering"))
        else:
            print("Dataset not recognized. Please revise.")
            sys.exit()

        #################################################
        # Graphics settings
        #################################################
        self.save_figures_path = os.path.join(self.save_root_path, 'plots')  # Update to reflect dataset selected
        self.save_format = "svg"
        self.dpi = 600

        self.main_colour = main_colour
        self.colour_order = colour_order
        self.colour_scheme = colour_scheme
        self.colour_path = os.path.join(
                     *("data", "ScientificColourMaps7", colour_scheme, "DiscretePalettes", "batlow25.txt"))
        self.diverge_low_colour = diverge_low_colour
        self.diverge_high_colour = diverge_high_colour
        self.font_size = font_size
        self.font_family = font_family

        plt.rcParams["font.family"] = font_family
