#' generate_speciesGen_summary
#'
#' @name generate_speciesGen_summary
#' @return Using the outlier-including data port overall species lists and
#'   cumulative counts for plotting later.
#'
#' @export
generate_speciesGen_summary <- function(lemisDataFiltered){

  write.csv(lemisDataFiltered %>%
              ungroup() %>%
              filter(rank == "Species") %>%
              select(species_code, lemisName, corrected, correctedGenus, orderCorrected, group_) %>%
              group_by(species_code, lemisName, corrected, correctedGenus, orderCorrected, group_) %>%
              slice_head(n = 1),
            file = here("tables", paste0("12_speciesList_",
                                         Sys.Date(), ".csv")),
            row.names = FALSE)

  # Species by group --------------------------------------------------------

  speciesCountByGroup <- lemisDataFiltered %>%
    filter(rank == "Species") %>%
    group_by(group_) %>%
    summarise(nSpp = n_distinct(corrected))

  (speciesCountByGroupCapWild <- lemisDataFiltered %>%
      filter(rank == "Species") %>%
      group_by(group_, originCapWild) %>%
      summarise(nSpp = n_distinct(corrected)) %>%
      ungroup() %>%
      tidyr::pivot_wider(names_from = originCapWild,
                         values_from = nSpp) %>%
      left_join(speciesCountByGroup) %>%
      ungroup() %>%
      mutate_if(is.numeric, ~replace(., is.na(.), 0)))

  write.csv(speciesCountByGroupCapWild,
            file = here("tables", paste0("12_speciesCountByGroupCapWild_",
                                         Sys.Date(), ".csv")),
            row.names = FALSE)

  # Species by year ---------------------------------------------------------

  speciesCountByYear <- lemisDataFiltered %>%
    filter(!is.na(group_)) %>%
    filter(!group_ == "Miscellaneous") %>%
    filter(rank == "Species") %>%
    group_by(group_, sYear) %>%
    summarise(nSpp = n_distinct(corrected)) %>%
    mutate(
      group_ = factor(group_, levels = c(
        "Echinoderms and Cnidaria",
        "Crustaceans and Molluscs",
        "Arachnids",
        "Insecta and Myriapoda",
        "Lepidoptera",
        "Other Invertebrates",
        "Terrestrial Mammals",
        "Birds",
        "Reptiles",
        "Amphibians",
        "Fish",
        "Marine Mammals",
        "Porifera Sponges, Bryozoa, and Squirts",
        "Plants",
        "Miscellaneous"
      )),
      vert = case_when(group_ %in% c("Terrestrial Mammals",
                                     "Reptiles",
                                     "Birds",
                                     "Amphibians",
                                     "Fish",
                                     "Marine Mammals") ~ "Vertebrate",
                       group_ %in% c("Crustaceans and Molluscs",
                                     "Arachnids",
                                     "Insecta and Myriapoda",
                                     "Other Invertebrates",
                                     "Lepidoptera",
                                     "Echinoderms and Cnidaria",
                                     "Porifera Sponges, Bryozoa, and Squirts",
                                     "Plants",
                                     "Miscellaneous") ~ "Non"),
      vert = factor(vert, levels = c("Vertebrate", "Non")))

  write.csv(speciesCountByYear, file = here("tables", paste0("12_speciesCountByYear_",
                                                             Sys.Date(), ".csv")),
            row.names = FALSE)

  # Genus per group ---------------------------------------------------------

  genusCountByGroup <- lemisDataFiltered %>%
    filter(!is.na(correctedGenus) & !correctedGenus == "Above genus") %>%
    group_by(group_) %>%
    summarise(nGen = n_distinct(correctedGenus))

  (genusCountByGroupCapWild <- lemisDataFiltered %>%
      filter(!is.na(correctedGenus) & !correctedGenus == "Above genus") %>%
      group_by(group_, originCapWild) %>%
      summarise(nGen = n_distinct(correctedGenus)) %>%
      ungroup() %>%
      tidyr::pivot_wider(names_from = originCapWild,
                         values_from = nGen) %>%
      left_join(genusCountByGroup))

  sum(genusCountByGroupCapWild$nGen)
  sum(genusCountByGroupCapWild$Wild)

  sum(genusCountByGroupCapWild$Wild) /
    sum(genusCountByGroupCapWild$nGen) *100

  genusCountByGroupCapWild <- genusCountByGroupCapWild %>%
    mutate(percentGen = Wild/nGen *100)

  write.csv(genusCountByGroupCapWild,
            file = here("tables", paste0("12_genusCountByGroupCapWild_",
                                         Sys.Date(), ".csv")),
            row.names = FALSE)

  # Genus by year ---------------------------------------------------------

  genusCountByYear <- lemisDataFiltered %>%
    filter(!is.na(group_)) %>%
    filter(!group_ == "Miscellaneous") %>%
    filter(!is.na(correctedGenus) & !correctedGenus == "Above genus") %>%
    group_by(group_, sYear) %>%
    summarise(nGen = n_distinct(correctedGenus)) %>%
    mutate(
      group_ = factor(group_, levels = c(
        "Echinoderms and Cnidaria",
        "Crustaceans and Molluscs",
        "Arachnids",
        "Insecta and Myriapoda",
        "Lepidoptera",
        "Other Invertebrates",
        "Terrestrial Mammals",
        "Birds",
        "Reptiles",
        "Amphibians",
        "Fish",
        "Marine Mammals",
        "Porifera Sponges, Bryozoa, and Squirts",
        "Plants",
        "Miscellaneous"
      )),
      vert = case_when(group_ %in% c("Terrestrial Mammals",
                                     "Reptiles",
                                     "Birds",
                                     "Amphibians",
                                     "Fish",
                                     "Marine Mammals") ~ "Vertebrate",
                       group_ %in% c("Crustaceans and Molluscs",
                                     "Arachnids",
                                     "Insecta and Myriapoda",
                                     "Other Invertebrates",
                                     "Lepidoptera",
                                     "Echinoderms and Cnidaria",
                                     "Porifera Sponges, Bryozoa, and Squirts",
                                     "Plants",
                                     "Miscellaneous") ~ "Non"),
      vert = factor(vert, levels = c("Vertebrate", "Non")))

  write.csv(genusCountByYear, file = here("tables", paste0("12_genusCountByYear_",
                                                           Sys.Date(), ".csv")),
            row.names = FALSE)

  # Cumulative species by year ----------------------------------------------

  lemisDataFiltered <- lemisDataFiltered %>%
    filter(!is.na(group_)) %>%
    filter(!group_ == "Miscellaneous")

  speciesByYear <- lemisDataFiltered %>%
    filter(rank == "Species") %>%
    group_by(group_, corrected, sYear) %>%
    select(group_, corrected, sYear) %>%
    slice_head(n = 1)

  write.csv(speciesByYear,
            file = here("tables", paste0("12_speciesByYear_",
                                         Sys.Date(), ".csv")),
            row.names = FALSE)

  # subGroup <- speciesByYear %>%
  #   filter(group_ == "Amphibians")

  i <- 0
  cumSumList <- list()
  for(sg in unique(speciesByYear$group_)){
    # sg <- "Fish"
    subGroup <- speciesByYear %>%
      filter(group_ == sg)

    spVec <- NA
    for(sy in 2000:2022){
      # sy <- 2016
      i <- i+1
      subSpVec <- subGroup[subGroup$sYear == sy,]$corrected
      spVec <- c(spVec, subSpVec)
      nSpp <- length(unique(spVec[!is.na(spVec)]))

      cumSumList[[i]] <- data.frame(
        group_ = sg,
        sYear = sy,
        cumsum = nSpp)
    }
  }

  cumSumSpecies <- do.call(rbind, cumSumList)

  cumSumSpecies <- cumSumSpecies %>%
    mutate(vert = case_when(group_ %in% c("Terrestrial Mammals",
                                          "Reptiles",
                                          "Birds",
                                          "Amphibians",
                                          "Fish",
                                          "Marine Mammals") ~ "Vertebrate",
                            group_ %in% c("Crustaceans and Molluscs",
                                          "Arachnids",
                                          "Insecta and Myriapoda",
                                          "Other Invertebrates",
                                          "Lepidoptera",
                                          "Echinoderms and Cnidaria",
                                          "Porifera Sponges, Bryozoa, and Squirts",
                                          "Plants",
                                          "Miscellaneous") ~ "Non"),
           vert = factor(vert, levels = c("Vertebrate", "Non")))

  write.csv(cumSumSpecies, file = here("tables", paste0("12_cumSumSpecies_",
                                                        Sys.Date(), ".csv")),
            row.names = FALSE)

  return(
    list(
      "speciesCountByGroup" = speciesCountByGroup,
      "speciesCountByGroupCapWild" = speciesCountByGroupCapWild,
      "speciesCountByYear" = speciesCountByYear,
      "genusCountByGroupCapWild" = genusCountByGroupCapWild,
      "genusCountByYear" = genusCountByYear,
      "speciesByYear" = speciesByYear,
      "cumSumSpecies" = cumSumSpecies
    )
  )

}

