#' generate_orderWild_plots
#'
#' @name generate_orderWild_plots
#' @return Creates the plots that display the wild / captive rates of orders.
#'
#' @export
generate_orderWild_plots <- function(generalSummaries, filterOpt = filter){

  # targets::tar_load("generalSummaries_mag2_tenK")
  # generalSummaries <- generalSummaries_mag2_tenK

  palettesList <- generate_palette()
  # list2env(palettesList)
  list2env(palettesList, environment())
  names(groupPalette) <- groupPaletteDF$group_
  # targets::tar_load("generalSummaries_tenK")
  # generalSummaries <- generalSummaries_tenK

  indisByGroupByPurposeByOrder <- generalSummaries$indisByGroupByPurposeByOrder

  meanWild <- indisByGroupByPurposeByOrder %>%
    mutate(percentWild = source_Wild / totalWhole *100) %>%
    group_by(group_) %>%
    summarise(meanWild = round(mean(percentWild), digits = 2))

  names(groupPalette) <- groupPaletteDF %>%
    left_join(meanWild) %>%
    mutate(group_ = glue::glue("<b style='color:{group_colour}'>{group_}</b> <i>{meanWild}%</i>")) %>%
    pull(group_)

  logBreaks <- c(c(1) %o% 10^(1:20))

  logLabs <- trimws(sub("000B$", "T",
                        sub("000M$", "B",
                            sub("000K$", "M",
                                sub("000$", "K", format(logBreaks, scientific = FALSE))))))


  indisByGroupByPurposeByOrder_plotdata <- indisByGroupByPurposeByOrder %>%
    mutate(totalWhole = as.numeric(totalWhole),
           percentWild = as.numeric(source_Wild) / as.numeric(totalWhole) *100,
           vert = case_when(group_ %in% c("Terrestrial Mammals",
                                          "Reptiles",
                                          "Birds",
                                          "Amphibians",
                                          "Fish",
                                          "Marine Mammals") ~ "Vertebrates",
                            group_ %in% c("Crustaceans and Molluscs",
                                          "Arachnids",
                                          "Insecta and Myriapoda",
                                          "Other Invertebrates",
                                          "Lepidoptera",
                                          "Echinoderms and Cnidaria",
                                          "Porifera Sponges, Bryozoa, and Squirts",
                                          "Plants",
                                          "Miscellaneous") ~ "Invertebrates"),
           vert = factor(vert, levels = c("Vertebrates", "Invertebrates"))) %>%
    left_join(groupPaletteDF) %>%
    left_join(meanWild) %>%
    mutate(group_ = glue::glue("<b style='color:{group_colour}'>{group_}</b> <i>{meanWild}%</i>"))

  bigSpeciesOrders <- indisByGroupByPurposeByOrder_plotdata %>%
    filter(str_detect(group_, "Reptiles|Birds|Lepidoptera|Echinoderms")) %>%
    group_by(group_) %>%
    filter(totalSpecies == max(totalSpecies)) %>%
    ungroup() %>%
    mutate(ynudge = c(-3, +8, -5, +10),
           xnudge = c(5000000, -2550000, 1200, 7000000),
           vjust = c(1.1, -0.2, 1.1, -0.2))

  captiveFishCluster <-
    data.frame(
      xloc = 150,
      yloc = 10,
      label = "Multiple orders of\ncaptive/ranched fish",
      xend = 6000,
      yend = 3,
      vjust = -0.2,
      vert = factor("Vertebrates", levels = c("Vertebrates", "Invertebrates"))
    )

  line90 <- data.frame(
    xloc = Inf,
    yloc = 89,
    label = paste0(round(generalSummaries$orderOverPercentage90$over90per[1], digits = 0),
                   "% of orders\nabove 90% wild"),
    vert = factor("Vertebrates", levels = c("Vertebrates", "Invertebrates"))
  )

  orderWildPlot <- indisByGroupByPurposeByOrder_plotdata %>%
    ggplot() +
    geom_vline(xintercept = 1, alpha = 0.25, linetype = 2) +
    geom_point(aes(x = totalWhole, y = percentWild, colour = reorder(group_, -percentWild),
                   size = totalSpecies, shape = vert)) +
    geom_text(data = captiveFishCluster,
              aes(x = xloc, y = yloc, label = label, vjust = vjust), hjust = 0.5,
              fontface = 4, colour = "#4d4d4d", lineheight = 0.95) +
    geom_curve(data = captiveFishCluster,
               aes(x = xloc, y = yloc, xend = xend, yend = yend),
               arrow = arrow(type = "closed", angle = 25, length = unit(2, "mm")),
               linewidth = 1.2, colour = "#4d4d4d") +
    geom_text(data = bigSpeciesOrders,
              aes(x = totalWhole + xnudge,
                  y = percentWild + ynudge,
                  label = orderCorrected, vjust = vjust), hjust = 0.5,
              fontface = 4, colour = "#4d4d4d") +
    geom_curve(data = bigSpeciesOrders,
               aes(x = totalWhole + xnudge,
                   y = percentWild + ynudge,
                   xend = totalWhole, yend = percentWild),
               arrow = arrow(type = "closed", angle = 25, length = unit(2, "mm")),
               linewidth = 1.2, colour = "#4d4d4d") +
    geom_hline(yintercept = 90,
               alpha = 0.25, linetype = 2) +
    geom_text(data = line90,
              aes(x = xloc, y = yloc, label = label), hjust = 1, vjust = 1,
              fontface = 4, colour = "#4d4d4d", lineheight = 0.95) +
    scale_x_log10(breaks = logBreaks,
                  labels = logLabs) +
    scale_colour_manual(values = groupPalette) +
    scale_y_continuous(expand = c(0.01,0), breaks = seq(0,100,25)) +
    scale_size_continuous(breaks = seq(0, 2000, 1000), range = c(1, 8)) +
    coord_cartesian(clip = "off") +
    facet_wrap(vert~., scales = "free") +
    labs(x = "Number of individuals traded", y = "Percentage\nwild sourced",
         colour = "<b>Group</b> & <i>mean wild source %</i>",
         size = "<b>Traded species</b>") +
    theme_bw() +
    theme(
      text = element_text(colour = "#4d4d4d"),
      line = element_line(colour = "#4d4d4d"),
      axis.ticks = element_line(colour = "#4d4d4d"),
      axis.title = element_text(face = 2),
      axis.title.y = element_text(face = 2, angle = 0, hjust = 1),
      axis.text.y = element_markdown(),
      legend.text = element_markdown(),
      legend.title = element_markdown(),
      strip.background = element_blank(),
      strip.text = element_markdown(hjust = 0, face = 4, size = 14,
                                    lineheight = 1.2,colour = "#4D4D4D"),
      panel.grid = element_blank(),
      panel.border = element_blank(),
      axis.line.y.left = element_line(),
      axis.line.x.bottom = element_line(),
      axis.line = element_line(colour = "#4D4D4D")
    ) +
    guides(shape = "none", size = guide_legend(override.aes = list(colour = "#4d4d4d")))

  orderWildPlot

  ggsave(here("figures", paste0("wildRatesOrder_", Sys.Date(), "_", filterOpt, ".png")),
         width = 340, height = 160, units = "mm")
  ggsave(here("figures", paste0("wildRatesOrder_", Sys.Date(), "_", filterOpt, ".pdf")),
         width = 340, height = 160, units = "mm")

  return(orderWildPlot)

}
