#' generate_general_summaries
#'
#' @name generate_general_summaries
#' @return a
#'
#' @export
generate_general_summaries <- function(lemisDataProcessed, filterOpt = filter){

  # lemisDataProcessed <- lemisDataProcessed_genPlus_tenK
  # Species and genus counts table ------------------------------------------

  lemisDataProcessed <- lemisDataProcessed %>%
    filter(!is.na(group_))

  speciesCountByGroup <- lemisDataProcessed %>%
    filter(rank == "Species") %>%
    group_by(group_) %>%
    summarise(total = n_distinct(corrected))

  (speciesCountByGroupCapWild <- lemisDataProcessed %>%
      filter(rank == "Species") %>%
      group_by(group_, originCapWild) %>%
      summarise(total = n_distinct(corrected)) %>%
      ungroup() %>%
      tidyr::pivot_wider(names_from = originCapWild,
                         values_from = total) %>%
      left_join(speciesCountByGroup))

  genusCountByGroup <- lemisDataProcessed %>%
    filter(!is.na(correctedGenus) & !correctedGenus == "Above genus") %>%
    group_by(group_) %>%
    summarise(total = n_distinct(correctedGenus))

  (genusCountByGroupCapWild <- lemisDataProcessed %>%
      filter(!is.na(correctedGenus) & !correctedGenus == "Above genus") %>%
      group_by(group_, originCapWild) %>%
      summarise(total = n_distinct(correctedGenus)) %>%
      ungroup() %>%
      tidyr::pivot_wider(names_from = originCapWild,
                         values_from = total) %>%
      left_join(genusCountByGroup))

  names(speciesCountByGroupCapWild) <- c("group_", paste0("Species_", names(speciesCountByGroupCapWild)[-1]))
  names(genusCountByGroupCapWild) <- c("group_", paste0("Genus_", names(genusCountByGroupCapWild)[-1]))

  overallSpeciesGenusCounts <- speciesCountByGroupCapWild %>%
    left_join(genusCountByGroupCapWild)

  sum(overallSpeciesGenusCounts$Species_total)
  sum(overallSpeciesGenusCounts$Genus_total)

  overallSpeciesGenusCounts %>%
    mutate(broaderGroups = case_when(group_ %in% c("Arachnids",
                                                   "Insecta and Myriapoda",
                                                   "Crustaceans and Molluscs",
                                                   "Echinoderms and Cnidaria",
                                                   "Lepidoptera",
                                                   "Other Invertebrates",
                                                   "Porifera Sponges, Bryozoa, and Squirts") ~ "invertebrate",
                                     group_ %in% c("Terrestrial Mammals",
                                                   "Reptiles",
                                                   "Birds",
                                                   "Amphibians") ~ "terrestrial vertebrate",
                                     group_ %in% c("Fish",
                                                   "Marine Mammals") ~ "marine vertebrate")) %>%
    group_by(broaderGroups) %>%
    summarise(totalSpp = sum(Species_total),
              totalGen = sum(Genus_total))

  overallSpeciesGenusCounts %>%
    select(group_, Species_total)

  overallSpeciesGenusCounts <- overallSpeciesGenusCounts %>%
    mutate(percentWild_species = (Species_Wild / Species_total)*100,
           percentRanched_species = (Species_Ranched / Species_total)*100,
           percentCaptive_species = (Species_Captive / Species_total)*100)

  overallSpeciesGenusCounts %>%
    select(group_, percentWild_species, percentRanched_species)

  overallSpeciesGenusCounts <- overallSpeciesGenusCounts %>%
    mutate(percentWild_genus = (Genus_Wild / Genus_total)*100,
           percentRanched_genus = (Genus_Ranched / Genus_total)*100,
           percentCaptive_genus = (Genus_Captive / Genus_total)*100)

  write.csv(overallSpeciesGenusCounts, file = here("tables", paste0("16_speciesAndGenusByGroup_",
                                                                    Sys.Date(), "_", filterOpt, ".csv")),
            row.names = FALSE)

  # Individuals by group by purpose -----------------------------------------

  wholeIndisByGroup <- lemisDataProcessed %>%
    group_by(group_) %>%
    summarise(totalWhole = format(sum(quantity, na.rm = TRUE), scientific = FALSE))

  liveIndisByGroup <- lemisDataProcessed %>%
    filter(description == "LIV") %>%
    group_by(group_) %>%
    summarise(totalLive = format(sum(quantity, na.rm = TRUE), scientific = FALSE))

  wholeIndisByGroupByPurpose <- lemisDataProcessed %>%
    mutate(purpose = case_when(
      purpose == "B" ~  "Breeding in captivity or artificial propagation",
      purpose == "E" ~                                      "Educational",
      purpose == "G" ~                                  "Botanic gardens",
      purpose == "H" ~                                 "Hunting Trophies",
      purpose == "L" ~   "Law Enforcement / Judicial / Forensic use only",
      # purpose == "M" ~                              "Biomedical research",
      purpose == "P" ~                                         "Personal",
      # purpose == "Q" ~                   "Circuses/traveling exhibitions",
      purpose %in% c("S", "M") ~                  "Scientific / Research",
      purpose == "T" ~                                       "Commercial",
      purpose == "Y" ~        "Reintroduction/introduction into the wild",
      purpose %in% c("Z", "Q") ~                        "Zoos / Circuses",
      TRUE ~ "Other")) %>%
    mutate(purpose = paste0("purpose_", gsub("/", "_or_", gsub(" ", "_", purpose)))) %>%
    group_by(group_, purpose) %>%
    summarise(total = format(sum(quantity, na.rm = TRUE), scientific = FALSE)) %>%
    pivot_wider(names_from = c("purpose"),
                values_from = "total")

  wholeIndisByGroupBySource <- lemisDataProcessed %>%
    mutate(originCapWild = paste0("source_", originCapWild)) %>%
    group_by(group_, originCapWild) %>%
    summarise(totalWhole = format(sum(quantity, na.rm = TRUE), scientific = FALSE)) %>%
    pivot_wider(names_from = c("originCapWild"),
                values_from = "totalWhole")

  indisByGroupByPurpose <- wholeIndisByGroup %>%
    left_join(wholeIndisByGroupBySource) %>%
    left_join(liveIndisByGroup) %>%
    left_join(wholeIndisByGroupByPurpose)

  # indisByGroupByPurpose %>%
  #   mutate(across(!c(group_), as.numeric)) %>%
  #   replace(is.na(.), 0) %>%
  #   mutate(percentWild = source_Wild / totalWhole*100) %>%
  #   select(group_, totalWhole, source_Wild, percentWild)

  write.csv(indisByGroupByPurpose, file = here("tables", paste0("16_indisByGroupByPurpose_",
                                                                Sys.Date(), "_", filterOpt, ".csv")),
            row.names = FALSE)

  # Order summary -----------------------------------------------------------

  wholeIndisByGroupByOrder <- lemisDataProcessed %>%
    filter(!is.na(orderCorrected) & !orderCorrected == "Above order") %>%
    group_by(orderCorrected, group_) %>%
    summarise(totalWhole = format(sum(quantity, na.rm = TRUE), scientific = FALSE))

  liveIndisByGroupByOrder <- lemisDataProcessed %>%
    filter(description == "LIV") %>%
    filter(!is.na(orderCorrected) & !orderCorrected == "Above order") %>%
    group_by(orderCorrected, group_) %>%
    summarise(totalLive = format(sum(quantity, na.rm = TRUE), scientific = FALSE))

  wholeIndisByGroupByPurposeByOrder <- lemisDataProcessed %>%
    filter(!is.na(orderCorrected) & !orderCorrected == "Above order") %>%
    mutate(purpose = case_when(
      purpose == "B" ~  "Breeding in captivity or artificial propagation",
      purpose == "E" ~                                      "Educational",
      purpose == "G" ~                                  "Botanic gardens",
      purpose == "H" ~                                 "Hunting Trophies",
      purpose == "L" ~   "Law Enforcement / Judicial / Forensic use only",
      # purpose == "M" ~                              "Biomedical research",
      purpose == "P" ~                                         "Personal",
      # purpose == "Q" ~                   "Circuses/traveling exhibitions",
      purpose %in% c("S", "M") ~                  "Scientific / Research",
      purpose == "T" ~                                       "Commercial",
      purpose == "Y" ~        "Reintroduction/introduction into the wild",
      purpose %in% c("Z", "Q") ~                        "Zoos / Circuses",
      TRUE ~ "Other")) %>%
    mutate(purpose = paste0("purpose_", gsub("/", "_or_", gsub(" ", "_", purpose)))) %>%
    group_by(orderCorrected, group_, purpose) %>%
    summarise(total = format(sum(quantity, na.rm = TRUE), scientific = FALSE)) %>%
    pivot_wider(names_from = c("purpose"),
                values_from = "total")

  wholeIndisByGroupBySourceByOrder <- lemisDataProcessed %>%
    filter(!is.na(orderCorrected) & !orderCorrected == "Above order") %>%
    mutate(originCapWild = paste0("source_", originCapWild)) %>%
    group_by(orderCorrected, group_, originCapWild) %>%
    summarise(totalWhole = format(sum(quantity, na.rm = TRUE), scientific = FALSE)) %>%
    pivot_wider(names_from = c("originCapWild"),
                values_from = "totalWhole")

  indisByGroupByPurposeByOrder <- wholeIndisByGroupByOrder %>%
    left_join(wholeIndisByGroupBySourceByOrder) %>%
    left_join(liveIndisByGroupByOrder) %>%
    left_join(wholeIndisByGroupByPurposeByOrder)

  indisByGroupByPurposeByOrder <- indisByGroupByPurposeByOrder %>%
    ungroup() %>%
    mutate(across(!c(group_, orderCorrected), as.numeric)) %>%
    replace(is.na(.), 0)

  speciesByOrder <- lemisDataProcessed %>%
    filter(rank == "Species") %>%
    group_by(orderCorrected, group_) %>%
    summarise(totalSpecies = n_distinct(corrected))

  indisByGroupByPurposeByOrder <- indisByGroupByPurposeByOrder %>%
    left_join(speciesByOrder) %>%
    replace(is.na(.), 0)

  write.csv(indisByGroupByPurposeByOrder, file = here("tables", paste0("16_indisByGroupByPurposeByOrder_",
                                                                       Sys.Date(), "_", filterOpt, ".csv")),
            row.names = FALSE)


  # Wild impact rates by order, indis, weight, entries ----------------------

  wildCapByOrderPercentages_whole <- lemisDataProcessed %>%
    filter(!is.na(orderCorrected) & !orderCorrected == "Above order") %>%
    group_by(orderCorrected, group_, originCapWild) %>%
    summarise(totalWhole = sum(quantity, na.rm = TRUE)) %>%
    pivot_wider(id_cols = c("orderCorrected", "group_"), names_from = "originCapWild",
                names_prefix = "whole_",
                values_from = "totalWhole", values_fill = 0) %>%
    ungroup() %>%
    mutate(total_whole = whole_Wild + whole_Captive + whole_Other + whole_Ranched,
           percent_whole_Wild = whole_Wild / total_whole *100,
           percent_whole_Captive = whole_Captive / total_whole *100,
           percent_whole_Ranched = whole_Ranched / total_whole *100)

  wildCapByOrderPercentages_entries <- lemisDataProcessed %>%
    filter(!is.na(orderCorrected) & !orderCorrected == "Above order") %>%
    group_by(orderCorrected, group_, originCapWild) %>%
    summarise(totalEntries = n()) %>%
    pivot_wider(id_cols = c("orderCorrected", "group_"), names_from = "originCapWild",
                names_prefix = "entries_",
                values_from = "totalEntries", values_fill = 0) %>%
    ungroup() %>%
    mutate(total_entries = entries_Wild + entries_Captive + entries_Other + entries_Ranched,
           percent_entries_Wild = entries_Wild / total_entries *100,
           percent_entries_Captive = entries_Captive / total_entries *100,
           percent_entries_Ranched = entries_Ranched / total_entries *100)

  wildCapByOrderPercentages <- wildCapByOrderPercentages_entries %>%
    left_join(wildCapByOrderPercentages_whole)

  write.csv(wildCapByOrderPercentages, file = here("tables", paste0("16_wildCapByOrderPercentages_",
                                                                    Sys.Date(), "_", filterOpt, ".csv")),
            row.names = FALSE)

  # Wild cap rates by species -----------------------------------------------

  speciesLevelWildCap <- lemisDataProcessed %>%
    filter(rank == "Species") %>%
    group_by(corrected) %>%
    mutate(totalQty = sum(quantity, na.rm = TRUE)) %>%
    group_by(originCapWild, corrected, group_) %>%
    mutate(sumQty = sum(quantity, na.rm = TRUE)) %>%
    select(totalQty, sumQty, group_) %>%
    slice_head(n = 1) %>%
    mutate(percentQty = sumQty/totalQty *100,
           vert = case_when(group_ %in% c("Terrestrial Mammals",
                                          "Reptiles",
                                          "Birds",
                                          "Amphibians",
                                          "Fish",
                                          "Marine Mammals") ~ "Vertebrates",
                            group_ %in% c("Crustaceans and Molluscs",
                                          "Arachnids",
                                          "Insecta and Myriapoda",
                                          "Other Invertebrates",
                                          "Lepidoptera",
                                          "Echinoderms and Cnidaria",
                                          "Porifera Sponges, Bryozoa, and Squirts",
                                          "Plants",
                                          "Miscellaneous") ~ "Invertebrates"),
           vert = factor(vert, levels = c("Vertebrates", "Invertebrates")))


  write.csv(speciesLevelWildCap, file = here("tables", paste0("16_wildCapBySpeciesPercentages_",
                                                              Sys.Date(), "_", filterOpt, ".csv")),
            row.names = FALSE)

  # Wild impact rates order -------------------------------------------------------

  orderOverPercentage90 <- indisByGroupByPurposeByOrder %>%
    mutate(percentWild = source_Wild / totalWhole) %>%
    mutate(percentOver90_TF = if_else(percentWild >= 0.90, "over90", "under90")) %>%
    group_by(percentOver90_TF) %>%
    summarise(over90Count = n_distinct(orderCorrected),
              over90per = n_distinct(orderCorrected) / nrow(indisByGroupByPurposeByOrder)*100)

  orderOverPercentage75 <- indisByGroupByPurposeByOrder %>%
    mutate(percentWild = source_Wild / totalWhole)%>%
    mutate(percentWild = if_else(is.na(percentWild), 0, percentWild),
           percentOver75_TF = if_else(percentWild >= 0.75, "over75", "under75")) %>%
    group_by(percentOver75_TF) %>%
    summarise(over75Count = n_distinct(orderCorrected),
              over75per = n_distinct(orderCorrected) / nrow(indisByGroupByPurposeByOrder)*100)

  orderOverPercentage50 <- indisByGroupByPurposeByOrder %>%
    mutate(percentWild = source_Wild / totalWhole)%>%
    mutate(percentWild = if_else(is.na(percentWild), 0, percentWild),
           percentOver50_TF = if_else(percentWild >= 0.50, "over50", "under50")) %>%
    group_by(percentOver50_TF) %>%
    summarise(over50Count = n_distinct(orderCorrected),
              over50per = n_distinct(orderCorrected) / nrow(indisByGroupByPurposeByOrder)*100)

  # Bonus Vertebrate summary ------------------------------------------------------

  vertSummary <- liveIndisByGroup %>%
    mutate(vert = if_else(group_ %in% c("Terrestrial Mammals",
                                        "Reptiles",
                                        "Birds",
                                        "Amphibians",
                                        "Fish",
                                        "Marine Mammals"), "Vertebrate", "Non")) %>%
    group_by(vert) %>%
    summarise(total = sum(as.numeric(totalLive), na.rm = TRUE))

  generalSummaries <- list(
    "overallSpeciesGenusCounts" = overallSpeciesGenusCounts,
    "indisByGroupByPurpose" = indisByGroupByPurpose,
    "indisByGroupByPurposeByOrder" = indisByGroupByPurposeByOrder,
    "wildCapByOrderPercentages" = wildCapByOrderPercentages,
    "speciesLevelWildCap" = speciesLevelWildCap,
    "orderOverPercentage90" = orderOverPercentage90,
    "orderOverPercentage75" = orderOverPercentage75,
    "orderOverPercentage50" = orderOverPercentage50,
    "vertSummary" = vertSummary
  )

  return(generalSummaries)

  # Extra in text summaries -------------------------------------------------

  # totalMax <- lemisDataProcessed %>%
  #   filter(description == "LIV") %>%
  #   pull(quantity) %>%
  #   sum(na.rm = TRUE)
  #
  # format(totalMax, scientific = FALSE)
  #
  # lemisDataProcessed %>%
  #   filter(correctedGenus == "Above genus") %>%
  #   pull(quantity) %>% sum(na.rm = TRUE) %>% format(scientific = FALSE, big.mark = ",")

}
