
library(optparse)

Main <- function(){
  option_list <- list( 
      make_option(c("--samplelevel-mode"), action="store_true", default=FALSE, 
          help="make plot with each row representing a sample with a CNV on chrom"),
      make_option(c("--density-mode"), action="store_true", default=FALSE, 
          help="make density plot with chrom starts"),
      make_option("--ref-genome", default="hg19", action="store", type="character",
                  help="name of reference genome build (hg19/hg18/hg38) [default %default]"),
      make_option("--fam", default=NA, action="store", type="character",
          help="name of input fam file [REQUIRED, default %default]"),
      make_option("--cnv-bed", default=NA, action="store", type="character",
          help="name of input cnv bed file [REQUIRED, default %default]"),
      make_option("--out-pdf", default=NA, action="store", type="character",
          help="name of output pdf file [REQUIRED, default %default]")
      )

  # get command line options, if help option encountered print help and exit,
  # otherwise if options not found on command line then set defaults, 
  opt_parser <- OptionParser(option_list=option_list)
  opt <- parse_args(opt_parser)
  if (is.na(opt[["fam"]]) | is.na(opt[["cnv-bed"]]) | is.na(opt[["out-pdf"]])) {
    print_help(opt_parser)
    stop("missing at least one REQUIRED opt")
  }

  # get chrom sizes based on defined reference genome build
  chrom_sizes_list <- GetChromSizes(opt[["ref-genome"]])
  chrom_sizes <- data.frame(chrom=character(), size=numeric())
  for (x in names(chrom_sizes_list)) { 
    chrom_sizes <- rbind(chrom_sizes,
                         data.frame(chrom=x, size=chrom_sizes_list[[x]]))
  }
  chrom_sizes$chrom <- gsub("chr","",chrom_sizes$chrom)
  
  # read fam file, get all unique iids
  fam <- read.table(opt[["fam"]], stringsAsFactors=F)
  iids <- fam[,2]
  n_samples <- length(iids)

  # read bed file 
  cnv <- read.table(opt[["cnv-bed"]], stringsAsFactors=F, sep="\t")
  colnames(cnv) <- c("chrom","start0","end","interval","cnvtype","iid")
  cnv$start <- cnv$start0 + 1
  cnv$chrom <- gsub("chr","",cnv$chrom)

  # subset bed file on samples in fam
  cnv <- subset(cnv, iid %in% iids)

  # make the CNV plot, per-chromosome
  chrom_start <- 1
  pdf(opt[["out-pdf"]])
  for (chrnum in 1:22) {
    chrom_str <- as.character(chrnum)
    chrom_end <- chrom_sizes[chrom_sizes$chrom == chrom_str, "size"]
    cnv.x <- subset(cnv, chrom == chrom_str)

    # get number of iids carrying at least one cnv in this chromosome
    iids_1 <- iids[iids %in% cnv.x$iid]
    n_samples_1 <- length(iids_1)

    if (opt[["samplelevel-mode"]] == TRUE) {
      # init plot 
      plot(1, type="n", xlab="", ylab="", main=chrom_str,
           xlim=c(1, chrom_end), ylim=c(1, n_samples_1))
      for (i in 1:length(iids_1)) {
        iid_i <- iids_1[i]
        cnv.x.y <- subset(cnv.x, iid == iid_i)
        if (nrow(cnv.x.y) > 0) {
          for (j in rownames(cnv.x.y)) {
            if (cnv.x.y[j, "cnvtype"] == "DEL") {
              color.x.y <- "red"
            } else if (cnv.x.y[j, "cnvtype"] == "DUP") {
              color.x.y <- "blue"
            } else {
              color.x.y <- "black"
            }
            segments(cnv.x.y[j,"start"], i,
                     cnv.x.y[j,"end"], i,
                     lwd=3,
                     col=color.x.y)
          }
        }
      }
    } else if (opt[["density-mode"]] == TRUE) {
      cnv.x$midpt <- (cnv.x$start + cnv.x$end) / 2
      plot(density(cnv.x$midpt), main=chrom_str, xlab="", ylab="density")
    } else {
      # init plot 
      plot(1, type="n", xlab="", ylab="", main=chrom_str,
           xlim=c(1, chrom_end), ylim=c(1, nrow(cnv.x)))
      cnv.x <- cnv.x[order(cnv.x$start), ]
      rownames(cnv.x) <- 1:nrow(cnv.x)
      for (i in rownames(cnv.x)) {
        if (cnv.x[i, "cnvtype"] == "DEL") {
          color.x <- "red"
        } else if (cnv.x[i, "cnvtype"] == "DUP") {
          color.x <- "blue"
        } else {
          color.x <- "black"
        }
        segments(as.numeric(cnv.x[i,"start"]), as.numeric(i),
                 as.numeric(cnv.x[i,"end"]), as.numeric(i),
                 lwd=3,
                 col=color.x)
      
      }

    }

    cnv.x$midpt <- (cnv.x$start + cnv.x$end) / 2
    step.x <- (max(cnv.x$midpt) - min(cnv.x$midpt)) / (nrow(cnv.x)-1)
    cnv.x$idx <- seq(min(cnv.x$midpt), max(cnv.x$midpt), by=step.x)
    # print(summary(lm(idx~midpt, data=cnv.x)))

  }
  dev.off()
}


GetChromSizes <- function(ref_genome) {
    # from :
    # https://hgdownload.cse.ucsc.edu/goldenpath/hg19/bigZips/hg19.chrom.sizes
    if (ref_genome == "hg19") {
        chrom_sizes = list(
                                    "chrX"=155270560,
                                    "chrY"=59373566,
                                    "chr1"=249250621,
                                    "chr2"=243199373,
                                    "chr3"=198022430,
                                    "chr4"=191154276,
                                    "chr5"=180915260,
                                    "chr6"=171115067,
                                    "chr7"=159138663,
                                    "chr8"=146364022,
                                    "chr9"=141213431,
                                    "chr10"=135534747,
                                    "chr11"=135006516,
                                    "chr12"=133851895,
                                    "chr13"=115169878,
                                    "chr14"=107349540,
                                    "chr15"=102531392,
                                    "chr16"=90354753,
                                    "chr17"=81195210,
                                    "chr18"=78077248,
                                    "chr19"=59128983,
                                    "chr20"=63025520,
                                    "chr21"=48129895,
                                    "chr22"=51304566
                          )
      } else if (ref_genome == "hg18") {
        # from :
        # http://hgdownload.cse.ucsc.edu/goldenpath/hg18/bigZips/hg18.chrom.sizes
        chrom_sizes = list(
                                    "chrX"=154913754,
                                    "chrY"=57772954,
                                    "chr1"=247249719,
                                    "chr2"=242951149,
                                    "chr3"=199501827,
                                    "chr4"=191273063,
                                    "chr5"=180857866,
                                    "chr6"=170899992,
                                    "chr7"=158821424,
                                    "chr8"=146274826,
                                    "chr9"=140273252,
                                    "chr10"=135374737,
                                    "chr11"=134452384,
                                    "chr12"=132349534,
                                    "chr13"=114142980,
                                    "chr14"=106368585,
                                    "chr15"=100338915,
                                    "chr16"=88827254,
                                    "chr17"=78774742,
                                    "chr18"=76117153,
                                    "chr19"=63811651,
                                    "chr20"=62435964,
                                    "chr21"=46944323,
                                    "chr22"=49691432
                        )
      } else if  (ref_genome == "hg38") {
        # from :
        # http://hgdownload.cse.ucsc.edu/goldenpath/hg38/bigZips/hg38.chrom.sizes
        chrom_sizes = list(
                                    "chrX"=156040895,
                                    "chrY"=57227415,
                                    "chr1"=248956422,
                                    "chr2"=242193529,
                                    "chr3"=198295559,
                                    "chr4"=190214555,
                                    "chr5"=181538259,
                                    "chr6"=170805979,
                                    "chr7"=159345973,
                                    "chr8"=145138636,
                                    "chr9"=138394717,
                                    "chr10"=133797422,
                                    "chr11"=135086622,
                                    "chr12"=133275309,
                                    "chr13"=114364328,
                                    "chr14"=107043718,
                                    "chr15"=101991189,
                                    "chr16"=90338345,
                                    "chr17"=83257441,
                                    "chr18"=80373285,
                                    "chr19"=58617616,
                                    "chr20"=64444167,
                                    "chr21"=46709983,
                                    "chr22"=50818468
                            )
  }
  return(chrom_sizes)
}

if (interactive() == F) {
  Main()
}
