%% Fit the three-pathway model to the 25TPO experiment
% This script will perform a fitting of a the three-pathway model developed
% in this work to the 25TPO experimental data. To perform the fitting you
% should simply run the script and select the following files:
% 25TPO-32.txt
% 25TPO-40.txt
% 25TPO-44.txt
% The analytical fundtion for the thin-zone three-pathway model is
% generated using Denis Constales' Multi-Zone TAP Reactor Theory:
% https://doi.org/10.1016/j.ces.2004.05.023
%
% The script generates a figure showing the model fits. It probably helps
% to make it fullscreen as the text is quite large.

%% Import the data
% Reads in and parses data from multiple Reece Lab TAP experiments
clearvars -except fileName filePath

%%%%%% MARCH 11th 2021 UPDATE %%%%%%
% ADDED: Now adds a "zero" time read so analytical solutions are easier to
% work with. The labjack does not pull a "zero" time read, so we duplicate
% the first point. It's not perfect but it'll do...
%%%%%% MARCH 11th 2021 UPDATE %%%%%%
%
%%%%% DECEMBER 9th 2022 UPDATE %%%%%
% ADDED: Now you can read in multiple files at once to make this a bit more
% efficient moving forward. It will parse the masses from the filenames.
% This version does not output pulsecat as I can't be bothered writing
% something that accounts for different pulse numbers in files... you can
% easy recreate it with reshape(peak(:,:,k),1,[]).
%%%%% DECEMBER 9th 2022 UPDATE %%%%%
%
% Reads XXXXX-Y.txt where XXXX is the filename and Y is the mass
% File is formatted as XY time/signal data
% Creates five items in the workspace
% peak(Y,N,K)     : All of the pulses in a N by Y format
%                 N = Number of Pulses Y = Length of one pulse
%                 K = Mass string index
% peak_bline(Y,N,K) : All of the pulses (baseline corrected) in a N by Y format
%                 N = Number of Pulses Y = Length of one pulse
%                 K = Mass string index
% peak_norm(Y,N,K)  : All of the pulses (area normalised) in a N by Y format
%                 N = Number of Pulses Y = Length of one pulse
%                 K = Mass string index
% pulseno       : Number of pulses
% timesingle(Y) : Time array of length Y
%                 Y = Length of one pulse
% timecat(Y*N)  : Time data to go with pulsecat in 1D array
%                 N = Number of Pulses Y = Length of one pulse
% mass(K)       : List of masses read in as a number
% massstr(K)    : List of masses read in as a string

% Clears up the variablespace to make sure you are starting from scratch.
clear peak pulsecat timecat timesingle pulseno peak_bline peak_norm M0

% Opens file explorer window - try/catch is used to save the filepath.
try
    [fileName,filePath] = uigetfile({'*.*'},'Please select TAP data','MultiSelect','on',filePath);
catch
    [fileName,filePath] = uigetfile({'*.*'},'Please select TAP data','MultiSelect','on');
end

if isa(fileName,'char')
    fprintf('It seems you have only loaded one file... exiting script\n')
    return    
end

% Reads Nx2 Matrix of raw data this is currently a very slow part of the
% script and can be made more efficient using a textscan loop similar to
% the ReadTAPData script, but it works so no complaining...
for k = 1:length(fileName)
    data = readmatrix([filePath,fileName{k}]);
    fprintf('Finished reading file %s\n',fileName{k})
    fprintf('Finished reading file with %d lines\n',length(data))
    
    fname = fileName{k};
    searchloc = length(fname)-4:-1:length(fname)-7;
    for j = searchloc
        if fname(j) == '-'
            break
        end
    end
    mass(k) = str2num(fname(j:length(fname)-4));
    massstr{k} = fname(j:length(fname)-4);
    
    % Splits into X (Time) and Y (Signal) data
    xdata = data(:,1);
    ydata = data(:,2);
    
    % Finds end of pulses by finding sampling time, this assumes a constant
    % sampling time but this should always be the case (so far)
    pulseend = find(xdata==max(xdata));
    
    % Counts number of pulses
    pulseno = length(pulseend);
    
    if k > 1
        if pulseno < p
            fprintf('Files do not contain the same number of pulses, removing extra points\n')
            peak(:,p,:) = [];
            peak_bline(:,p,:) = [];
            peak_norm(:,p,:) = [];
            M0(p,:) = [];
        elseif pulseno > p
            pulseno = p;
        end
    end
    
    % Creates X array for single pulse, appends a zero value as it is missing
    % in the raw data
    timesingle = [0;xdata(1:pulseend)];
    
    % Creates a new sequential time array
    timecat = linspace(min(xdata),max(xdata)*pulseno,length(xdata));
    
    % Grabs each pulse and saves into peak, duplicates the first data point to
    % account for the no t=0 read in the labjack.
    for i = 1:pulseno
        if i == 1
            peak(:,i,k) = [ydata(1);ydata(1:pulseend)];
        else
            peak(:,i,k) = [ydata(pulseend(i-1)+1);ydata(pulseend(i-1)+1:pulseend(i))];
        end
    end
    
    % Locations for baseline correction. Uses last 0.1s of pulse to calculate
    % baseline rather than a fixed number of points.
    tend = max(timesingle);
    bloc = find(timesingle>(tend-0.1));
    
    for i = 1:pulseno    % Loop over each peak
        peak_bline(:,i,k) = peak(:,i,k)-mean(peak(bloc,i,k)); % Baseline corrects data.
        M0(i,k) = trapz(timesingle,peak_bline(:,i,k)); % Calculates M0
        peak_norm(:,i,k) = peak_bline(:,i,k)/M0(i,k); % Area normalises the peak (M0 = 1).
    end
    
    % For uneven pulse size debugging
    [~,p,~] = size(peak); 
    
    % Transposes M0 for easier Excel copying
    fprintf('Finished processing %d pulses\n',pulseno)
    
    clear ans colltime data datapoints datastart endfile fid filestart i textdata
    
    % Clears the workspace
    clear xdata ydata pulseend i data
end
fprintf('NOTE: As the LabJack T7 does not pull a t=0 read \nwe duplicate the first data point as t=0 \nto help fitting with the analytical solutions.\n')
fprintf('Masses read in: ')
for i = 1:length(mass)
    fprintf('%.1f ',mass(i))
end
fprintf('\n')

%% Process the data
% Looks for Argon on mass 40
inert = find(mass == 40);

% Looks for reactant signal, assumes 28 to start with, but if it cannot
% find 28 then it searches for 32. 
reactant = find(mass == 28);
if isempty(reactant)
    reactant = find(mass == 32);
end

% Looks for product signal on mass 44
product = find(mass == 44);

% If we cannot find inert (40), reactant (28 or 32), or product (44) the
% script will exit as it cannot continue.
if isempty(inert)
    fprintf('No mass 40 found...\n')
    fprintf('Exiting script\n')
    return
end
if isempty(reactant)
    fprintf('Neither mass 32 or mass 28 were found...\n')
    fprintf('Exiting script\n')
    return
end
if isempty(product)
    fprintf('No mass 44 found...\n')
    fprintf('Exiting script\n')
    return
end

% Here we will process the data. We will use our pre-calculated calibration
% coefficients for each mass in order to esimate the number of molecules
% being produced.
calib_mass = [28,32,44];

calib_vals = [1.060,0.932,1.077]; % Taken from 2023-02-10

% Normalised M0 to the inert, smooth the inert value to remove noise.
    M0n = M0./smooth(M0(:,inert),4);

% Gas Mixture of reactant
% Reactant/Argon
gasmix = 0.20/0.80;

% Correct for CO2 crack at 28
if mass(reactant) == 28
    crack28 = 0.1245;
    M0n(:,reactant) = M0n(:,reactant)-M0n(:,product)*crack28;
end

% Estimate 0 conversion M0 value
zero_conv = gasmix;

M0react = M0n(:,reactant)/calib_vals(calib_mass == mass(reactant))/gasmix;
M0prod  = M0n(:,product)/calib_vals(calib_mass == mass(product))/gasmix;

peak_react = peak_norm(:,:,reactant).*M0react';
peak_prod = peak_norm(:,:,product).*M0prod';
peak_inert  = peak_norm(:,:,inert);

M0s = [M0react,M0prod];

%% Perform fitting of the data
% Initiate ka values for curve fitting where
% x0 = [ka1,kr1,ka2,kr2,ka3]
% these are selected to speed up the regression
x0 = [19,5,200,500,100];
rate_constants = [];
pulseno = 150;

for i = 1:pulseno
    % First we must calculate the diffusivity for the Argon using a curve
    % fitting to the inert Argon curve using the one zone TAP reactor curve
    % function.
    
    % Set up the x variable which is the time variable
    xdata = timesingle;
    % x0 is fitted variable (residence time) and we provide an intial
    % guess.
    x0_1Z    = 0.01;
    
    % Using the lsqcurvefit solver to regress a value for the residence
    % time of the Argon curve
    
    % This makes the solver shut up.
    options = optimoptions('lsqcurvefit',...
        'Display','off');
    
    % Run the solver
    x = ...
        lsqcurvefit(@calc_1Z,x0_1Z,xdata,peak_inert(:,i)',...
        -inf,+inf,options);    
    
    % Re-calculate the fitted curve using the regressed value
    G0 = calc_1Z(x,xdata);
    
    % Use the calculated residence time to back-calculate diffusivity, we
    % do this by using pre-calculated values for length + voidage. We then
    % scale the diffusivity to the reactant and product masses.
    L = 5.881;
    eb = 0.464;
    Dinert = eb*L^2/x/2;
    Dreact = Dinert*sqrt(40)/sqrt(mass(reactant));
    Dprod = Dinert*sqrt(40)/sqrt(mass(product));
    Dreact_T(i,1) = Dreact;
    
    % Calculate diffusional response for product and reactant
    x_prod = eb*L^2/Dprod/2;
    x_react = eb*L^2/Dreact/2;    
    G_prod = calc_1Z(x_prod,xdata);
    G_react = calc_1Z(x_react,xdata);
    
    % Identify L1 and L3 based on the location of the thin zone within the
    % catalyst zone: cat_loc = 0 - 1.
    cat_loc = 0.5;
    L1 = L*cat_loc;
    L3 = L*(1-cat_loc);
    
    % This makes the solver shut up.
    options = optimoptions('lsqcurvefit',...
        'Display','off');
    
    % Run the solver for the three-pathway model
    % Infinite Upper Bounds
    UB = [inf,inf,inf,inf,inf];
    % Lower bounds are set to zero to stop getting negative rate constants.
    LB = [0  ,0  ,0  ,0  ,0  ];
    
    % Curve fitting algorithm from MATLAB
    [x,resnorm,residual,~,~,~,jacobian] = ...
        lsqcurvefit(@(x,xdata)SimulateCurves(x,xdata,Dreact,Dprod,L1,L3,eb,mass(reactant)),...
        x0,xdata,[peak_react(:,i),peak_prod(:,i)],...
        LB,UB,... % LB, UB
        options);
    
    % Calculate confidence intervals for each parameter
    conf = nlparci(x,residual,'jacobian',jacobian);
    
    % Generate curves using fitted values and extrapolated to time + 20s
    % just incase the curve does not return to 0 before.
    xtrapolated = 0:xdata(2)-xdata(1):xdata(end)+20;
    curves = SimulateCurves(x,xtrapolated,Dreact,Dprod,L1,L3,eb,mass(reactant));
    
    % Store intergrated and simualted curves
    M0curves(i,:) = trapz(xtrapolated(1:length(xdata)),curves(1:length(xdata),:));
    allcurves(:,:,i) = [G0',curves(1:length(xdata),:)];
    
    % Export rate constants + confidence intervals
    dx = diff(conf');
    
    % Save the rate constants in the format I like:
    % rate_constants = [ka1, e_ka1, kr1, e_kr1...]
    for ii = numel(x):-1:1
        rate_constants(i,2*ii-[0 1]) = [dx(ii),x(ii)];
    end
    
    % Use previously calculated rate constants as initial guess to speed
    % things up.
    x0 = x;
    
    subplot(2,3,1)
    hold off
    plot(timesingle(1:3:end),peak_inert(1:3:end,i),'o','Color',[0.5,0.5,0.5])
    hold on
    plot(timesingle,G0,'k','LineWidth',2)
    title(['Pulse Set Number : ',num2str(i),' of : ',num2str(pulseno)]);
    ylabel('Normalised Exit Flux (1/s)')
    xlabel('Time (s)')
    legend('Argon','Model')
    xlim([0,1])
    set(gca,'linewidth',2,'FontWeight','bold','FontSize',12)
    
    subplot(2,3,2)
    hold off
    plot(timesingle(1:3:end),peak_react(1:3:end,i),'bo')
    hold on
    plot(xtrapolated(1:length(xdata)),curves(1:length(xdata),1),'k','LineWidth',2)
%     plot(timesingle,G_react/max(G_react)*max(curves(:,1)),'-','Color',[0.7,0.7,0.7],'LineWidth',2)
    title(['Pulse Set Number : ',num2str(i),' of : ',num2str(pulseno)]);
    ylabel('Normalised Exit Flux (1/s)')
    xlabel('Time (s)','FontWeight', 'bold')
    legend('Oxygen','Model')
%     legend(massstr{reactant},'Model','Diffusion Only')
    xlim([0,1])
    set(gca,'linewidth',2,'FontWeight','bold','FontSize',12)
    
    subplot(2,3,3)
    hold off
    plot(timesingle(1:3:end),peak_prod(1:3:end,i),'ro')
    hold on
    plot(xtrapolated(1:length(xdata)),curves(1:length(xdata),2),'k','LineWidth',2)
%     plot(timesingle,G_prod/max(G_prod)*max(curves(:,2)),'-','Color',[0.7,0.7,0.7],'LineWidth',2)
    title(['Pulse Set Number : ',num2str(i),' of : ',num2str(pulseno)]);
    ylabel('Normalised Exit Flux (1/s)')
    xlabel('Time (s)','FontWeight', 'bold')
    legend('CO2','Model')
%     legend(massstr{product},'Model','Diffusion Only')
    xlim([0,3])
    set(gca,'linewidth',2,'FontWeight','bold','FontSize',12)
    
    subplot(2,3,4:6)
    hold off
    plot(1:i,M0react(1:i),'ob')
    hold on
    plot(1:i,M0curves(1:i,1),'-b')
    plot(1:i,M0prod(1:i),'or')
    plot(1:i,M0curves(1:i,2),'-r')
    legend('Experiment','Simulated')
    xlabel('Pulse Set Number')
    ylabel('Integrated Exit Flux')
    set(gca,'linewidth',2,'FontWeight','bold','FontSize',12)
    set(gcf,'color','w')
    
    
    fprintf('Fitting curve %d of %d\n',i,pulseno)
    drawnow
    
    pause(0.1)
end

%M0_all = [M0react,M0prod,M0curves];
fprintf('Done!\n')

%% Function that generates the three curves

function curves = SimulateCurves(x,xdata,Dreact,Dprod,L1,L3,eb,reactmass)
% Uses the RevAdsReact functions to calculate the reactant and product
% curves, this is the objective function that the lsqcurvefit will see.
timesingle = xdata;

% Unpack the fitted variables [ka,kd,kr]
ka1 = x(1); kr1 = x(2); 
ka2 = x(3); kr2 = x(4);
ka3 = x(5);

% Calculate the laplace variable (s)
tlen    = length(timesingle);
tmax    = max(timesingle);
dt = timesingle(2)-timesingle(1);
k       = -tlen/2:tlen/2-1;
k(k==0) = 1E-10;
s       = (2*k*pi/(tmax))*1i;

G1 = abs(ifft(RAR_33site_R_TZ(Dreact,Dreact,L1,L3,eb,eb,ka1,ka2,ka3,s))*tlen/tmax);
G2 = abs(ifft(RAR_33site_P_TZ(Dprod,Dprod,Dreact,Dreact,L1,L3,eb,eb,ka1,ka2,ka3,kr1,kr2,s))*tlen/tmax);

% If you pulse O2 then you can make 2 CO2 molecules per molecule of O2.
if reactmass == 32
    G2 = G2*2;
end

curves = [G1',G2'];
end

%% Model Functions
function G1 = RAR_33site_R_TZ(Dr1,Dr3,L1,L3,e1,e3,ka1,ka2,ka3,s)
% 3 Pathway model - Reactant

%RAR_33SITE_R_TZ
%    G1 = RAR_33SITE_R_TZ(DR1,DR3,L1,L3,E1,E3,KA1,KA2,KA3,S)

%    This function was generated by the Symbolic Math Toolbox version 8.4.
%    12-Apr-2023 16:37:37

t2 = L1.^2;
t3 = L3.^2;
t4 = 1.0./Dr1;
t5 = 1.0./Dr3;
t6 = e1.*s.*t2.*t4;
t7 = e3.*s.*t3.*t5;
t8 = sqrt(t6);
t9 = sqrt(t7);
t10 = cosh(t8);
t11 = sinh(t9);
G1 = (Dr3.*t8.*t9)./(L3.*ka1.*t8.*t10.*t11+L3.*ka2.*t8.*t10.*t11+L3.*ka3.*t8.*t10.*t11+Dr3.*t8.*t9.*t10.*cosh(t9)+L1.*L3.*e1.*s.*t11.*sinh(t8));
end

function G2 = RAR_33site_P_TZ(Dp1,Dp3,Dr1,Dr3,L1,L3,e1,e3,ka1,ka2,ka3,kr1,kr2,s)
% 3 Pathway Model - Product

%RAR_33SITE_P_TZ
%    G2 = RAR_33SITE_P_TZ(DP1,DP3,DR1,DR3,L1,L3,E1,E3,KA1,KA2,KA3,KR1,KR2,S)

%    This function was generated by the Symbolic Math Toolbox version 8.4.
%    12-Apr-2023 16:37:38

t2 = L1.^2;
t3 = L3.^2;
t4 = 1.0./Dp1;
t5 = 1.0./Dp3;
t6 = 1.0./Dr1;
t7 = 1.0./Dr3;
t8 = e1.*s.*t2.*t4;
t9 = e3.*s.*t3.*t5;
t10 = e1.*s.*t2.*t6;
t11 = e3.*s.*t3.*t7;
t12 = sqrt(t8);
t13 = sqrt(t9);
t14 = sqrt(t10);
t15 = sqrt(t11);
t16 = cosh(t12);
t17 = cosh(t14);
t18 = sinh(t15);
G2 = (Dp3.*L3.*t12.*t13.*t14.*t16.*t18.*(ka1.*kr1.*kr2+ka2.*kr1.*kr2+ka1.*kr1.*s+ka2.*kr2.*s))./((kr1+s).*(kr2+s).*(Dp3.*t12.*t13.*t16.*cosh(t13)+L1.*L3.*e1.*s.*sinh(t12).*sinh(t13)).*(L3.*ka1.*t14.*t17.*t18+L3.*ka2.*t14.*t17.*t18+L3.*ka3.*t14.*t17.*t18+Dr3.*t14.*t15.*t17.*cosh(t15)+L1.*L3.*e1.*s.*t18.*sinh(t14)));
end

function G0 = calc_1Z(x,xdata)
% One zone TAP reactor with no adsorption function.
timesingle = xdata;

% Calculate the laplace variable (s)
tlen    = length(timesingle);
tmax    = max(timesingle);
dt = timesingle(2)-timesingle(1);
k       = -tlen/2:tlen/2-1;
k(k==0) = 1E-10;
s       = (2*k*pi/(tmax))*1i;

% Setup Physical Parameters
ff = 1./cosh(sqrt(2*s*x));
G0 = abs(ifft(ff))*(1/dt);
end