########
#loading of annotated sequences (.gb) and comparisons file obtained from Easyfig software (.out)
#definition of functions to read annotations from GenBank files and comparisons files
#plot graphical schemes of sequences and comparisons between them
########


install.packages("genoPlotR")
install.packages("ggbio")

## try http:// if https:// URLs are not supported
source("https://bioconductor.org/biocLite.R")
biocLite("ggbio")

if (!requireNamespace("BiocManager", quietly = TRUE))
  install.packages("BiocManager")
BiocManager::install("ggbio")


library(genoPlotR)
library(ggbio)
library(ade4)
library(grid)
library(Hmisc, dependencies=TRUE)
library(BSgenome)
"dependencies ‘Hmisc’, ‘biovizBase’, ‘BSgenome’, ‘VariantAnnotation’, ‘rtracklayer’, ‘GenomicFeatures’, ‘OrganismDbi’ are not available for package ‘ggbio’"

##################################data##############################

#sgs1 genomic sequences
#genomes sans CG3036
melanosgs1noCG3036<-try(read_dna_seg_from_file("D.melanogaster_NT_033779_2L_noCG3036.gb"))
sechelliasgs1noCG3036<-try(read_dna_seg_from_file("D.sechellia_CM015602.1_2L_noCG3036.gb"))
simulanssgs1noCG3036<-try(read_dna_seg_from_file("D.simulans_Scf_2L_noCG3036.gb"))
maurisgs1noCG3036<-try(read_dna_seg_from_file("D.mauritiana_CM015614.1_2L_noCG3036.gb"))
santosgs1noCG3036<-try(read_dna_seg_from_file("D.santomea_2L_noCG3036.gb"))
yakubasgs1noCG3036<-try(read_dna_seg_from_file("D.yakuba_2L_extraction_noCG3036.gb"))
#teissieri n'a pas CG3036, on peut garder teissierisgs1
teissierisgs1<-try(read_dna_seg_from_file("D.teissieri_2L_sgs1.gb"))
erectasgs1noCG3036<-try(read_dna_seg_from_file("D.erecta_NW_020825200.1_PacBio_noCG3036.gb"))
eugracilissgs1noCG3036<-try(read_dna_seg_from_file("D.eugracilis_NW_016077548_noCG3036.gb"))
suzukiisgs1noCG3036<-try(read_dna_seg_from_file("D.suzukii_Andromeda_2L_noCG3036.gb"))
biarmipessgs1noCG3036<-try(read_dna_seg_from_file("D.biarmipes_SZUW01002387.1_2L_noCG3036.gb"))
takahashiisgs1noCG3036<-try(read_dna_seg_from_file("D.takahashii_NW_016039488.1_noCG3036.gb"))
subobscurasgs1noCG3036<-try(read_dna_seg_from_file("D.subobscura_contig_170_noCG3036.gb"))
obscurasgs1noCG3036<-try(read_dna_seg_from_file("D.obscura_contig_36_noCG3036.gb"))
pseudoobscurasgs1noCG3036<-try(read_dna_seg_from_file("D.pseudoobscura_NC_046681.1_noCG3036.gb"))
ananassaesgs1noCG3036<-try(read_dna_seg_from_file("D.ananassae_NW_022473445.1_noCG3036.gb"))
bipectinatasgs1noCG3036<-try(read_dna_seg_from_file("D.bipectinata_contig_54_noCG3036.gb"))
ficusphilasgs1noCG3036<-try(read_dna_seg_from_file("D.ficusphila_contig_412_noCG3036.gb"))
eleganssgs1noCG3036<-try(read_dna_seg_from_file("D.elegans_contig_814_noCG3036.gb"))
rhopaloasgs1noCG3036<-try(read_dna_seg_from_file("D.rhopaloa_contig_2_noCG3036.gb"))
triaurariasgs1noCG3036<-try(read_dna_seg_from_file("D.triauraria_contig_599_noCG3036.gb"))
rufasgs1noCG3036<-try(read_dna_seg_from_file("D.rufa_contig_97_noCG3036.gb"))
kikkawaisgs1noCG3036<-try(read_dna_seg_from_file("D.kikkawai_contig_310_noCG3036.gb"))
jambulinasgs1noCG3036noSnsl<-try(read_dna_seg_from_file("D.jambulina_contig_114_noCG3036noSnsl.gb"))

#comparison sgs1 genomic sequences
#melanogaster group
sgs1_mel_sech_noCG3036<-try(read_comparison_from_blast("sgs1_mel_sech_noCG3036.out"))
sgs1_sech_sim_noCG3036<-try(read_comparison_from_blast("sgs1_sech_sim_noCG3036.out"))
sgs1_sim_mau_noCG3036<-try(read_comparison_from_blast("sgs1_sim_mau_noCG3036.out"))
sgs1_mau_santo_noCG3036<-try(read_comparison_from_blast("sgs1_mau_santo_noCG3036.out"))
sgs1_santo_yak_noCG3036<-try(read_comparison_from_blast("sgs1_santo_yak_noCG3036.out"))
sgs1_yak_teissieri_noCG3036<-try(read_comparison_from_blast("sgs1_yak_teissieri_noCG3036.out"))
sgs1_teissieri_erecta_noCG3036<-try(read_comparison_from_blast("sgs1_teissieri_erecta_noCG3036.out"))
sgs1_erecta_eugracilis_noCG3036<-try(read_comparison_from_blast("sgs1_erecta_eugracilis_noCG3036.out"))
sgs1_eugracilis_suzukii_noCG3036<-try(read_comparison_from_blast("sgs1_eugracilis_suzukii_noCG3036.out"))
sgs1_suzukii_biarmipes_noCG3036<-try(read_comparison_from_blast("sgs1_suzukii_biarmipes_noCG3036.out"))
sgs1_biarmipes_takahashii_noCG3036<-try(read_comparison_from_blast("sgs1_biarmipes_takahashii_noCG3036.out"))

#other species
sgs1_subobscura_obscura_noCG3036<-try(read_comparison_from_blast("sgs1_subobscura_obscura_noCG3036.out"))
sgs1_obscura_pseudoobscura_noCG3036<-try(read_comparison_from_blast("sgs1_obscura_pseudoobscura_noCG3036.out"))
sgs1_pseudoobscura_ananassae_noCG3036<-try(read_comparison_from_blast("sgs1_pseudoobscura_ananassae_noCG3036.out"))
sgs1_ananassae_bipectinata_noCG3036<-try(read_comparison_from_blast("sgs1_ananassae_bipectinata_noCG3036.out"))
sgs1_bipectinata_ficusphila_noCG3036<-try(read_comparison_from_blast("sgs1_bipectinata_ficusphila_noCG3036.out"))
sgs1_ficusphila_elegans_noCG3036<-try(read_comparison_from_blast("sgs1_ficusphila_elegans_noCG3036.out"))
sgs1_elegans_rhopaloa_noCG3036<-try(read_comparison_from_blast("sgs1_elegans_rhopaloa_noCG3036.out"))
sgs1_rhopaloa_melanogaster_noCG3036<-try(read_comparison_from_blast("sgs1_rhopaloa_melanogaster_noCG3036.out"))
sgs1_melanogaster_triauraria_noCG3036<-try(read_comparison_from_blast("sgs1_melanogaster_triauraria_noCG3036.out"))
sgs1_triauraria_rufa_noCG3036<-try(read_comparison_from_blast("sgs1_triauraria_rufa_noCG3036.out"))
sgs1_rufa_jambulina_noCG3036noSnsl<-try(read_comparison_from_blast("sgs1_rufa_jambulina_noCG3036noSnsl.out"))
sgs1_jambulina_kikkawai_noCG3036noSnsl<-try(read_comparison_from_blast("sgs1_jambulina_kikkawai_noCG3036noSnsl.out"))

#sgs3X genomic sequences
melanogaster_sgs3bis<-try(read_dna_seg_from_file("D.melanogaster_sgs3X.gb"))
sechellia_sgs3bis<-try(read_dna_seg_from_file("D.sechellia_ChrX_NC_045954_sgs3X.gb"))
simulans_sgs3bis<-try(read_dna_seg_from_file("D.simulans_ChrX_NC_052525_sgs3X.gb"))
mauritiana_sgs3bis<-try(read_dna_seg_from_file("D.mauritiana_ChrX _NC_046672_sgs3X.gb"))
santomea_sgs3bis<-try(read_dna_seg_from_file("D.santomea_CAGO_X_sgs3X.gb"))
yakuba_sgs3bis<-try(read_dna_seg_from_file("D.yakuba_NY73PB_X_sgs3X.gb"))
teissieri_sgs3bis<-try(read_dna_seg_from_file("D.teissieri_GT53_ChrX_sgs3X.gb"))
erecta_sgs3bis<-try(read_dna_seg_from_file("D.erecta_NW_020825209_sgs3X.gb"))
eugracilis_sgs3bis<-try(read_dna_seg_from_file("D.eugracilis_NW_016077443_sgs3X.gb"))
suzukii_sgs3bis<-try(read_dna_seg_from_file("D.suzukii_WWNF01000011.1_sgs3X.gb"))
biarmipes_sgs3bis<-try(read_dna_seg_from_file("D.biarmipes_SZUW01002661.1_sgs3X.gb"))
takahashii_sgs3bis<-try(read_dna_seg_from_file("D.takahashii_GCF_000224235.1_sgs3X.gb"))
rhopaloa_sgs3bis<-try(read_dna_seg_from_file("D.rhopaloa_contig_3_sgs3X.gb"))
elegans_sgs3bis<-try(read_dna_seg_from_file("D.elegans_contig_149_sgs3X.gb"))
ficusphila_sgs3bis<-try(read_dna_seg_from_file("D.ficusphila_contig_198_sgs3X.gb"))
bipectinata_sgs3bis<-try(read_dna_seg_from_file("D.bipectinata_contig_50_sgs3X.gb"))
ananassae_sgs3bis<-try(read_dna_seg_from_file("D.ananassae_NW_022473271_sgs3X.gb"))
pseudoobscura_sgs3bis<-try(read_dna_seg_from_file("D.pseudoobscura_NC_046683_sgs3X.gb"))
obscura_sgs3bis<-try(read_dna_seg_from_file("D.obscura_contig_73_sgs3X.gb"))
subobscura_sgs3bis<-try(read_dna_seg_from_file("D.subobscura_contig_167_sgs3X.gb"))
triauraria_sgs3bis<-try(read_dna_seg_from_file("D.triauraria_contig_650_sgs3X.gb"))
rufa_sgs3bis<-try(read_dna_seg_from_file("D.rufa_contig_112_sgs3X.gb"))
jambulina_sgs3bis<-try(read_dna_seg_from_file("D.jambulina_contig_6_sgs3X.gb"))
kikkawai_sgs3bis<-try(read_dna_seg_from_file("D.kikkawai_contig_284_sgs3X.gb"))

#comparison sgs3X genomic sequences
#melanogaster group
sgs3bis_short_melanogaster_sechellia<-try(read_comparison_from_blast("sgs3bis_short_melanogaster_sechellia.out"))
sgs3bis_short_sechellia_simulans<-try(read_comparison_from_blast("sgs3bis_short_sechellia_simulans.out"))
sgs3bis_short_simulans_mauritiana<-try(read_comparison_from_blast("sgs3bis_short_simulans_mauritiana.out"))
sgs3bis_short_mauritiana_santomea<-try(read_comparison_from_blast("sgs3bis_short_mauritiana_santomea.out"))
sgs3bis_short_santomea_yakuba<-try(read_comparison_from_blast("sgs3bis_short_santomea_yakuba.out"))
sgs3bis_short_yakuba_teissieri<-try(read_comparison_from_blast("sgs3bis_short_yakuba_teissieri.out"))
sgs3bis_short_teissieri_erecta<-try(read_comparison_from_blast("sgs3bis_short_teissieri_erecta.out"))
sgs3bis_short_erecta_eugracilis<-try(read_comparison_from_blast("sgs3bis_short_erecta_eugracilis.out"))
sgs3bis_short_eugracilis_suzukii<-try(read_comparison_from_blast("sgs3bis_short_eugracilis_suzukii.out"))
sgs3bis_short_suzukii_biarmipes<-try(read_comparison_from_blast("sgs3bis_short_suzukii_biarmipes.out"))
sgs3bis_short_biarmipes_takahashii<-try(read_comparison_from_blast("sgs3bis_short_biarmipes_takahashii.out"))

#other species
sgs3bis_obscura_subobscura<-try(read_comparison_from_blast("sgs3bis_obscura_subobscura.out"))
sgs3bis_subobscura_pseudoobscura<-try(read_comparison_from_blast("sgs3bis_subobscura_pseudoobscura.out"))
sgs3bis_pseudoobscura_ananassae<-try(read_comparison_from_blast("sgs3bis_pseudoobscura_ananassae.out"))
sgs3bis_ananassae_bipectinata<-try(read_comparison_from_blast("sgs3bis_ananassae_bipectinata.out"))
#sgs3bis_ananassae_bipectinata_test<-try(read_comparison_from_blast("sgs3bis_ananassae_bipectinata_test.out"))
sgs3bis_bipectinata_ficusphila<-try(read_comparison_from_blast("sgs3bis_bipectinata_ficusphila.out"))
sgs3bis_ficusphila_elegans<-try(read_comparison_from_blast("sgs3bis_ficusphila_elegans.out"))
sgs3bis_elegans_rhopaloa<-try(read_comparison_from_blast("sgs3bis_elegans_rhopaloa.out"))
sgs3bis_rhopaloa_melanogaster<-try(read_comparison_from_blast("sgs3bis_rhopaloa_melanogaster.out"))
sgs3bis_melanogaster_triauraria<-try(read_comparison_from_blast("sgs3bis_melanogaster_triauraria.out"))
sgs3bis_triauraria_rufa<-try(read_comparison_from_blast("sgs3bis_triauraria_rufa.out"))
sgs3bis_rufa_jambulina<-try(read_comparison_from_blast("sgs3bis_rufa_jambulina.out"))
sgs3bis_jambulina_kikkawai<-try(read_comparison_from_blast("sgs3bis_jambulina_kikkawai.out"))

#triauraria vs bipectinata
sgs3bis_bipectinata_triauraria<-try(read_comparison_from_blast("sgs3bis_bipectinata_triauraria.out"))


#sgs3 sgs7 sgs8 genomic sequences
melanogaster_sgs3<-try(read_dna_seg_from_file("D.melanogaster _3L_sgs3.gb"))
simulans_sgs3<-try(read_dna_seg_from_file("D.simulans_Scf_3L_sgs3.gb"))
sechellia_sgs3<-try(read_dna_seg_from_file("D.sechellia_CM015604.1_3L_sgs3.gb"))
mauritiana_sgs3<-try(read_dna_seg_from_file("D.mauritiana_CM015616.1_3L_sgs3.gb"))
santomea_sgs3<-try(read_dna_seg_from_file("D.santomea_CAGO_3L_sgs3.gb"))
yakuba_sgs3<-try(read_dna_seg_from_file("D.yakuba_3L_sgs3.gb"))
teissieri_sgs3<-try(read_dna_seg_from_file("D.teissieri_3L_sgs3.gb"))
erecta_sgs3<-try(read_dna_seg_from_file("D.erecta_NW_020825198.1_sgs3.gb"))
eugracilis_sgs3<-try(read_dna_seg_from_file("D.eugracilis_NW_016078254.1_sgs3.gb"))
suzukii_sgs3<-try(read_dna_seg_from_file("D.suzukii_Cepheus_sgs3.gb"))
biarmipes_sgs3<-try(read_dna_seg_from_file("D.biarmipes_SZUW01002707.1_sgs3.gb"))
takahashii_sgs3<-try(read_dna_seg_from_file("D.takahashii_NW_016039132.1_3L_sgs3.gb"))
elegans_sgs3<-try(read_dna_seg_from_file("D.elegans_contig_208_sgs3.gb"))
rhopaloa_sgs3<-try(read_dna_seg_from_file("D.rhopaloa_contig_163_sgs3.gb"))
ficusphila_sgs3<-try(read_dna_seg_from_file("D.ficusphila_contig_286_sgs3.gb"))
kikkawai_sgs3<-try(read_dna_seg_from_file("D.kikkawai_contig_80_sgs3.gb"))
ananassae_sgs3<-try(read_dna_seg_from_file("D.ananassae_NW_022473439.1_3L_sgs3.gb"))
bipectinata_sgs3<-try(read_dna_seg_from_file("D.bipectinata_contig_1_sgs3.gb"))
pseudoobscura_sgs3<-try(read_dna_seg_from_file("D.pseudoobscura_NC_046683.1_3L_sgs3.gb"))
subobscura_sgs3<-try(read_dna_seg_from_file("D.subobscura_contig_115_sgs3.gb"))
obscura_sgs3<-try(read_dna_seg_from_file("D.obscura_contig_1_sgs3.gb"))
triauraria_sgs3<-try(read_dna_seg_from_file("D.triauraria_contig_937_sgs3.gb"))
rufa_sgs3<-try(read_dna_seg_from_file("D.rufa_contig_86_sgs3.gb"))
jambulina_sgs3<-try(read_dna_seg_from_file("D.jambulina_contig_27_sgs3.gb"))

#comparison sgs3 sgs7 sgs8 genomic sequences
#mlanogaster group
sgs3_melanogaster_sechellia<-try(read_comparison_from_blast("sgs3_melanogaster_sechellia.out"))
sgs3_sechellia_simulans<-try(read_comparison_from_blast("sgs3_sechellia_simulans.out"))
sgs3_simulans_mauritiana<-try(read_comparison_from_blast("sgs3_simulans_mauritiana.out"))
sgs3_mauritiana_santomea<-try(read_comparison_from_blast("sgs3_mauritiana_santomea.out"))
sgs3_santomea_yakuba<-try(read_comparison_from_blast("sgs3_santomea_yakuba.out"))
sgs3_yakuba_teissieri<-try(read_comparison_from_blast("sgs3_yakuba_teissieri.out"))
sgs3_teissieri_erecta<-try(read_comparison_from_blast("sgs3_teissieri_erecta.out"))
sgs3_erecta_eugracilis<-try(read_comparison_from_blast("sgs3_erecta_eugracilis.out"))
sgs3_eugracilis_suzukii<-try(read_comparison_from_blast("sgs3_eugracilis_suzukii.out"))
sgs3_suzukii_biarmipes<-try(read_comparison_from_blast("sgs3_suzukii_biarmipes.out"))
sgs3_biarmipes_takahashii<-try(read_comparison_from_blast("sgs3_biarmipes_takahashii.out"))
sgs3_takahashii_ficusphila<-try(read_comparison_from_blast("sgs3_takahashii_ficusphila.out"))

#other species
sgs3_obscura_subobscura<-try(read_comparison_from_blast("sgs3_obscura_subobscura.out"))
sgs3_subobscura_pseudoobscura<-try(read_comparison_from_blast("sgs3_subobscura_pseudoobscura.out"))
sgs3_pseudoobscura_ananassae<-try(read_comparison_from_blast("sgs3_pseudoobscura_ananassae.out"))
sgs3_anana_bipec<-try(read_comparison_from_blast("sgs3_ananassae_bipectinata.out"))
sgs3_bipectinata_ficusphila<-try(read_comparison_from_blast("sgs3_bipectinata_ficusphila.out"))
sgs3_ficusphila_elegans<-try(read_comparison_from_blast("sgs3_ficusphila_elegans.out"))
sgs3_elegans_rhopaloa<-try(read_comparison_from_blast("sgs3_elegans_rhopaloa.out"))
sgs3_rhopaloa_melanogaster<-try(read_comparison_from_blast("sgs3_rhopaloa_melanogaster.out"))
sgs3_melanogaster_triauraria<-try(read_comparison_from_blast("sgs3_melanogaster_triauraria.out"))
sgs3_triauraria_rufa<-try(read_comparison_from_blast("sgs3_triauraria_rufa.out"))
sgs3_rufa_jambulina<-try(read_comparison_from_blast("sgs3_rufa_jambulina.out"))
sgs3_jambulina_kikkawai<-try(read_comparison_from_blast("sgs3_jambulina_kikkawai.out"))


#sgs3 sgs7 sgs8 zoom genomic sequences
melanogastersgsregion<-try(read_dna_seg_from_file("D.melanogaster_3L_sgsregion.gb"))
sechelliasgsregion<-try(read_dna_seg_from_file("D.sechellia_CM015604.1_3L_sgsregion.gb"))
simulanssgsregion<-try(read_dna_seg_from_file("D.simulans_Scf_3L_sgsregion.gb"))
mauritianasgsregion<-try(read_dna_seg_from_file("D.mauritiana_CM015616.1_3L_sgsregion.gb"))
santomeasgsregion<-try(read_dna_seg_from_file("D.santomea_CAGO_3L_sgsregion.gb"))
yakubasgsregion<-try(read_dna_seg_from_file("D.yakuba_3L_sgsregion.gb"))
teissierisgsregion<-try(read_dna_seg_from_file("D.teissieri_3L_sgsregion.gb"))
erectasgsregion<-try(read_dna_seg_from_file("D.erecta_NW_020825198.1_sgsregion.gb"))
eugracilissgsregion<-try(read_dna_seg_from_file("D.eugracilis_NW_016078254.1_sgsregion.gb"))
suzukiisgsregion<-try(read_dna_seg_from_file("D.suzukii_Cepheus_sgsregion.gb"))
biarmipessgsregion<-try(read_dna_seg_from_file("D.biarmipes_SZUW01002707.1_sgsregion.gb"))
takahashiisgsregion<-try(read_dna_seg_from_file("D.takahashii_NW_016039132.1_3L_sgsregion.gb"))
obscurasgsregion<-try(read_dna_seg_from_file("D.obscura_contig_1_sgs3_sgsregion.gb"))
subobscurasgsregion<-try(read_dna_seg_from_file("D.subobscura_contig_115_sgs3_sgsregion.gb"))
pseudoobscurasgsregion<-try(read_dna_seg_from_file("D.pseudoobscura_NC_046683.1_3L_sgs3_sgsregion.gb"))
ananassaesgsregion<-try(read_dna_seg_from_file("D.ananassae_NW_022473439.1_3L_sgs3_sgsregion.gb"))

#comparison sgs3 sgs7 sgs8 zoom genomic sequences
sgs3_melanogaster_sechellia_sgsregion<-try(read_comparison_from_blast("sgs3_melanogaster_sechellia_sgsregion.out"))
sgs3_sechellia_simulans_sgsregion<-try(read_comparison_from_blast("sgs3_sechellia_simulans_sgsregion.out"))
sgs3_simulans_mauritiana_sgsregion<-try(read_comparison_from_blast("sgs3_simulans_mauritiana_sgsregion.out"))
sgs3_mauritiana_erecta_sgsregion_2<-try(read_comparison_from_blast("sgs3_mauritiana_erecta_sgsregion_2.out"))
sgs3_erecta_teissieri_sgsregion_2<-try(read_comparison_from_blast("sgs3_erecta_teissieri_sgsregion_2.out"))
sgs3_teissieri_yakuba_sgsregion_2<-try(read_comparison_from_blast("sgs3_teissieri_yakuba_sgsregion_2.out"))
sgs3_yakuba_santomea_sgsregion_2<-try(read_comparison_from_blast("sgs3_yakuba_santomea_sgsregion_2.out"))
sgs3_santomea_eugracilis_sgsregion_2<-try(read_comparison_from_blast("sgs3_santomea_eugracilis_sgsregion_2.out"))
sgs3_eugracilis_suzukii_sgsregion<-try(read_comparison_from_blast("sgs3_eugracilis_suzukii_sgsregion.out"))
sgs3_suzukii_biarmipes_sgsregion<-try(read_comparison_from_blast("sgs3_suzukii_biarmipes_sgsregion.out"))
sgs3_biarmipes_takahashii_sgsregion<-try(read_comparison_from_blast("sgs3_biarmipes_takahashii_sgsregion.out"))
sgs3_obscura_subobscura_sgsregion<-try(read_comparison_from_blast("sgs3_obscura_subobscura_sgsregion.out"))
sgs3_subobscura_pseudoobscura_sgsregion<-try(read_comparison_from_blast("sgs3_subobscura_pseudoobscura_sgsregion.out"))
sgs3_pseudoobscura_ananassae_sgsregion<-try(read_comparison_from_blast("sgs3_pseudoobscura_ananassae_sgsregion.out"))


#2 à 2
sgs3_melanogaster_melanogaster<-try(read_comparison_from_blast("sgs3_melanogaster_melanogaster.out"))
sgs3_erecta_erecta<-try(read_comparison_from_blast("sgs3_erecta_erecta.out"))
sgs3_santomea_santomea<-try(read_comparison_from_blast("sgs3_santomea_santomea.out"))
sgs3_yakuba_yakuba<-try(read_comparison_from_blast("sgs3_yakuba_yakuba.out"))
sgs3_teissieri_teissieri<-try(read_comparison_from_blast("sgs3_teissieri_teissieri.out"))
sgs3_suzukii_suzukii<-try(read_comparison_from_blast("sgs3_suzukii_suzukii.out"))
sgs3_biarmipes_biarmipes<-try(read_comparison_from_blast("sgs3_biarmipes_biarmipes.out"))
sgs3_takahashii_takahashii<-try(read_comparison_from_blast("sgs3_takahashii_takahashii.out"))

#sgs region teissieri vs mauritiana
sgs3_teissieri_mauritiana_sgsregion<-try(read_comparison_from_blast("sgs3_teissieri_mauritiana_sgsregion.out"))

#sgs region melanogaster vs mauritiana
sgs3_melanogaster_mauritiana_sgsregion<-try(read_comparison_from_blast("sgs3_melanogaster_mauritiana_sgsregion.out"))


################################################################################
# File reading functions: read dna_seg
################################################################################
# SUPPORT FUNCTIONS FOR READ_DNA_SEG_FROM_GENBANK
# Get start and end position for a currentFeature Object
# Please note, the function makes sure that only start and end of the
# following two types are ok:
# 'XXX (numeric)..(numeric)' or 'XXX complement((numeric)..(numeric))',
# where XXX may contain a-Z, 0-9 and
# also ! " # $ % & ' ( ) * + , - . / : ; < = > ? @ [ \ ] ^ _ ` { | } ~.
get_start <- function(line){
  ifelse (length(grep("complement", line)) > 0,
          start <- as.numeric(gsub("_|[[:blank:]]|[[:alpha:]]|\\(|\\)|\\.\\..*", "",
                                   grep("^[[:graph:]]+ complement\\([[:digit:]]+\\.\\.[[:digit:]]+\\)$",
                                        line, value=TRUE))),
          start <- as.numeric(gsub("_|[[:blank:]]|[[:alpha:]]|\\(|\\)|\\.\\..*", "",
                                   grep("^[[:graph:]]+ [[:digit:]]+\\.\\.[[:digit:]]+$", line, value=TRUE))))
  start
}
get_end <- function(line){
  ifelse(length(grep("complement", line)) > 0,
         end <-
           as.numeric(gsub("_|[[:blank:]]|[[:alpha:]]|\\(|\\)|.*\\.\\.", "",
                           grep("^[[:graph:]]+ complement\\([[:digit:]]+\\.\\.[[:digit:]]+\\)$",
                                line, value=TRUE))),
         
         end <- as.numeric(gsub("_|[[:blank:]]|[[:alpha:]]|\\(|\\)|.*\\.\\.", "",
                                grep("^[[:graph:]]+ [[:digit:]]+\\.\\.[[:digit:]]+$",
                                     line, value=TRUE))))
  end
}
# Extracts data from feature lines.
extract_data <- function(extract, cF){
  extract <- gsub(extract, "", grep(paste("^",extract,sep=""), cF, value=TRUE))
  if (length(extract) == 0) { extract <- "NA" }
  extract[1]
}
# Added for support, to run an embl file directly
read_dna_seg_from_embl <- function(file, tagsToParse=c("CDS"), ...){
  read_dna_seg_from_file(file, tagsToParse, fileType="embl", ...)
}
# Added for support, to run an genbank file directly
read_dna_seg_from_genbank <- function(file, tagsToParse=c("CDS"), ...){
  read_dna_seg_from_file(file, tagsToParse, fileType="genbank", ...)
}

# MAIN FUNCTION
# Read genes from a GenBank file
read_dna_seg_from_file <- function(file, tagsToParse=c("CDS"),
                                   fileType="detect", meta_lines=2,
                                   gene_type="auto", header=TRUE,
                                   extra_fields=NULL, ...){
  
  # Import data from file into variable
  importedData <- readLines(file)
  
  # Find file type
  TYPE <- "Unknown"
  if (fileType == "detect" || fileType == "Detect" || fileType == "DETECT") {
    if (length(grep(">", importedData[1]))) {TYPE <- "Fasta"}
    if (length(grep("^ID", importedData))) {TYPE <- "EMBL"}
    if (length(grep("^LOCUS", importedData))) {TYPE <- "Genbank"}
  }
  else if (fileType == "EMBL" || fileType == "embl" || fileType == "Embl") {
    TYPE <- "EMBL"
  }
  else if (fileType == "Genbank" || fileType == "GENBANK" || fileType == "genbank") {
    TYPE <- "Genbank"
  }
  else if (fileType == "Ptt" || fileType == "PTT" || fileType == "ptt") {
    TYPE <- "PTT"
  }
  else if (fileType == "Fasta" || fileType == "FASTA" || fileType == "fasta") {
    TYPE <- "Fasta"
  }
  if (TYPE == "Unknown") {
    stop("fileType has to be either 'detect', 'embl', 'genbank' or 'ptt'. Note if file type is .ptt, please specify this rather than using 'detect'.")
  }
  
  # If type is PTT, call already made function
  if(TYPE == "PTT") {
    dna_seg <- read_dna_seg_from_ptt(file, meta_lines, header, ...)
    return(dna_seg)
  }
  else if (TYPE == "Fasta"){
    dna_seg <- read_dna_seg_from_fasta(file)
    return(dna_seg)
  }
  
  # If type isn't PTT do everything else...
  else {
    
    # Extract and name main segments
    if(TYPE == "Genbank") {
      mainSegments <- grep("^[[:alnum:]]", importedData)
      names(mainSegments) <- gsub("*| .*", "", grep("^[[:alnum:]]",
                                                    importedData, value=TRUE))
    }
    
    # SIMPLE ERROR HANDLING
    if(TYPE == "Genbank") {
      if(length(grep("FEATURES|DEFINITION", names(mainSegments))) < 2) {
        stop("FEATURES or DEFINITION segment missing in GBK File.")
      }
      if(length(grep("LOCUS", names(mainSegments))) != 1) {
        stop("Number of LOCUS should be 1.")
      }
    }
    if(TYPE == "EMBL") {
      if(length(grep("^ID|^FH", importedData)) < 2){
        stop("ID or FH segment missing in EMBL File.")
      }
      if(length(grep("^ID", importedData)) != 1) {
        stop("Number of ID lines should be 1.")
      }
    }
    
    # Extract META data
    if(TYPE == "EMBL") {
      seg_name <- gsub("^DE[[:blank:]]+", "",
                       grep("^DE", importedData, value=T))
    }
    if(TYPE == "Genbank") {
      seg_name <- gsub("DEFINITION {1,}", "",
                       importedData[mainSegments["DEFINITION"]])
    }
    
    # Extract features only, handles whether FEATURES is the last (or not)
    # entry in GBK file
    if(TYPE == "Genbank") {
      ifelse(which(names(mainSegments) == "FEATURES") == length(mainSegments),
             dataFeatures <-
               importedData[mainSegments["FEATURES"]:
                              (length(importedData) - 1)],
             dataFeatures <-
               importedData[mainSegments["FEATURES"]:
                              (mainSegments[which(names(mainSegments) ==
                                                    "FEATURES")+1] - 1)])
    }
    if(TYPE == "EMBL") {
      dataFeatures <- grep("^FT", importedData, value=T)
    }
    
    # SIMPLE ERROR HANDLING
    if(TYPE == "Genbank") {
      if(length(dataFeatures) < 2){ stop("No FEATURES in GBK file.") }
    }
    if(TYPE == "EMBL") {
      if(length(dataFeatures) < 1){ stop("No FEATURES in GBK file.") }
    }
    
    
    # Extract each start line for each feature
    if(TYPE == "Genbank") {
      startLineOfFeature <- c(1:length(dataFeatures))[- grep("^ {6,}",
                                                             dataFeatures)]
    }  
    if(TYPE == "EMBL") {
      startLineOfFeature <- grep("FT   [[:alnum:]]", dataFeatures)
    }
    startLineOfFeature <- c(startLineOfFeature, length(dataFeatures)+1)
    
    
    # Define variables for storage
    nF <- length(startLineOfFeature)-1
    name <- character()
    start <- numeric()
    end <- numeric()
    strand <- numeric()
    length <- numeric()
    pid <- character()
    gene <- character()
    synonym <- character()
    product <- character()
    color <- character()
    proteinid <- character()
    feature <- character()
    geneType <- character()
    extra <- list()
    for (tag in extra_fields){
      extra[[tag]] <- character()
    }
    
    # Loop over all features                     
    for(counter in 1:nF){
      
      # Get feature, normally 20ish lines.
      currentFeature <- (dataFeatures[ startLineOfFeature[counter] :
                                         (startLineOfFeature[counter+1]-1) ])
      
      # Clean up feature, decreases number of lines etc.
      if(TYPE == "Genbank") {
        currentFeature <-
          gsub("^ |:|\"| $", "",
               gsub("[[:blank:]]+|[[:space:]]+",
                    " ", strsplit(paste(currentFeature,
                                        collapse=""), "   /")[[1]]))
      }
      if(TYPE == "EMBL") {
        currentFeature <-
          gsub("^ |:|\"| $", "",
               gsub("[[:blank:]]+|[[:space:]]+", " ",
                    strsplit(paste(gsub("FT", "", currentFeature),
                                   collapse=""), "   /")[[1]]))
      }
      
      # If feature is of a type to parse. Default is only CDS tags.
      if(length(grep(gsub(" [[:print:]]+", "", currentFeature[1]),
                     tagsToParse)) > 0){
        # Create list with exons to parse
        tag <- gsub(" [[:graph:]]+", "", currentFeature[1])
        qualif <- gsub("[[:graph:]]+ ", "", currentFeature[1])
        exonVector <- strsplit(gsub("[[:alpha:]]|_| |\\(|\\)|", "",
                                    currentFeature[1]), ",")
        if (length(grep("complement", currentFeature[1])) > 0){
          exonVector <- paste(tag, " complement(", exonVector[[1]], ")",
                              sep="")
        }
        if (length(grep("complement", currentFeature[1])) == 0){
          exonVector <- paste(tag, " ", exonVector[[1]], sep="")
        }
        
        # If there are more than one exon, insert introns
        if (length(exonVector) > 1) {
          exonVector2 <- 0
          for (i in 1:(length(exonVector)-1)) {
            if (length(grep("complement", currentFeature[1])) > 0) {
              exonVector2 <-
                c(exonVector2,exonVector[i],
                  paste(tag, "_intron complement(", get_end(exonVector[i])+1,
                        "..", get_start(exonVector[i+1])-1, ")", sep=""))
            }
            if (length(grep("complement", currentFeature[1])) == 0) {
              exonVector2 <- c(exonVector2,exonVector[i],
                               paste(tag, "_intron ", get_end(exonVector[i])+1,
                                     "..", get_start(exonVector[i+1])-1,
                                     sep=""))
            }
          }
          exonVector2[length(exonVector)*2] <- exonVector[length(exonVector)]
          exonVector2 <- exonVector2[2:length(exonVector2)]
          exonVector <- exonVector2
        }
        
        # For each exon in currentFeature 
        for (currentExon in exonVector) {
          
          # Set currentExon to currentFeature line 1
          currentFeature[1] <- currentExon
          
          # SIMPLE ERROR HANDLING AND Only continue parsing if start and stop
          # is valid...
          # Extract gene name or ID AND start and end, THEN, check if it's ok.
          ifelse(length(grep("gene=", currentFeature)) > 0,
                 nameTEMP <- extract_data("gene=", currentFeature),
                 nameTEMP <- extract_data("locus_tag=", currentFeature))
          startTEMP <-get_start(currentFeature[1])
          endTEMP <- get_end(currentFeature[1])
          if(length(startTEMP) == 0 || length(endTEMP) == 0) {
            warning(paste("Start and stop position invalid for "),
                    nameTEMP, ". This entry has been excluded.", sep="") }
          
          # Continue if start and end is ok... Otherwise, skip this feature
          if (length(startTEMP) > 0 && length(endTEMP) > 0) {
            
            # Save name, start, end, length
            name <- c(name, nameTEMP)
            start <- c(start, startTEMP)
            end <- c(end, endTEMP)
            length <- c(length, (get_end(currentFeature[1]) -
                                   get_start(currentFeature[1]) + 1)/3 - 1)
            
            # Set strand to 1 or -1
            ifelse (length(grep("complement", currentFeature[1])) > 0,
                    strand <- c(strand, -1), strand <- c(strand, 1))
            
            # Extract PID
            if (TYPE == "Genbank") {
              pid <- c(pid, extract_data("db_xref=GI", currentFeature))
            }
            if (TYPE == "EMBL") {
              pidTEMP <- extract_data("db_xref=UniProtKB/Swiss-Prot",
                                      currentFeature)
              if (pidTEMP == "NA"){
                pidTEMP <- extract_data("db_xref=UniProtKB/TrEMBL",
                                        currentFeature)
              }
              pid <- c(pid, pidTEMP)
            }
            
            # Extract gene
            ifelse(length(grep("gene=", currentFeature)) > 0,
                   gene <- c(gene, extract_data("gene=", currentFeature)),
                   gene <- c(gene, "-"))
            
            # Extract synonym
            synonym <- c(synonym, extract_data("locus_tag=", currentFeature))
            
            # Extract protein ID
            proteinid <- c(proteinid, extract_data("protein_id=",
                                                   currentFeature))
            
            # Extract product
            product <- c(product, extract_data("product=", currentFeature))
            
            # Extract color
            color <- c(color, extract_data("(color|colour)=", currentFeature))
            
            # Extract extra
            for (tag in names(extra)){
              extra[[tag]] <- c(extra[[tag]],
                                extract_data(paste(tag, "=", sep=""),
                                             currentFeature))
            }
            
            # Set geneType
            if (length(grep("intron", currentFeature[1])) > 0){
              geneType <- c(geneType, "introns")
            }
            if (length(grep("intron", currentFeature[1])) == 0){
              geneType <- c(geneType, gene_type)
            }
            
            # Return tag feature info, with or without added _pseudo tag
            if (length(grep("^pseudo", currentFeature)) > 0) {
              feature <- c(feature,
                           paste(gsub(" [[:print:]]+", "",
                                      currentFeature[1]), "_pseudo", sep=""))
            }
            if (length(grep("^pseudo", currentFeature)) == 0) {
              feature <- c(feature, gsub(" [[:print:]]+", "",
                                         currentFeature[1]))
            }
            
            
            # end of parse if start and end ok
          }
          
          # End of exon loop
        }
        
        # End of CDS loop
      }
      
      # End of loop over all features
    }
    
    # SIMPLE ERROR HANDLING
    if(is.numeric(start) == FALSE) stop("Start is not numeric.")
    if(is.numeric(end) == FALSE) stop("End is not numeric.")
    if(is.numeric(length) == FALSE) stop("Length is not numeric.")
    if(is.numeric(strand) == FALSE) stop("Strand is not numeric.")
    if(is.character(pid) == FALSE) stop("PID is not character.")
    if(is.character(name) == FALSE) stop("Name is not character.")
    if(is.character(gene) == FALSE) stop("Gene is not character.")
    if(is.character(synonym) == FALSE) {
      stop("Synonym is not character.")
    }
    if(is.character(product) == FALSE) {
      stop("Product is not character.")
    }
    if(is.character(proteinid) == FALSE) {
      stop("Protein ID is not character.")
    }
    if(is.character(feature) == FALSE) {
      stop("Feature is not character.")
    }
    # Check color
    # Eventually, change Artemis colors to their RGB equivalent
    artCol <- artemisColors()
    if (length(color) > 0 && all(color %in% c("NA", artCol$n))) {
      for (i in 1:length(color)){
        if (color[i] != "NA") color[i] <- artCol$colors[artCol$n == color[i]]
      }
    }
    
    # If gene_type is auto, change form arrows to blocks, otherwise arrows
    if (length(grep("intron", geneType))>=1 && gene_type == "auto") {
      geneType[geneType== "auto"] <- "exons"
    }
    if (length(grep("intron", geneType))==0 && gene_type == "auto") {
      geneType[geneType== "auto"] <- "bars"
    }
    
    # Cut table to include only added features    
    table <- data.frame(name=name, start=start, end=end, strand=strand,
                        length=length, pid=pid, gene=gene, synonym=synonym,
                        product=product, proteinid=proteinid, feature=feature,
                        gene_type=geneType, stringsAsFactors=FALSE)
    if (!all(color == "NA")){
      color[color == "NA"] <- "grey"
      table$col <- color
    }
    ## Adding extra fields
    for (tag in extra_fields){
      table[[tag]] <- extra[[tag]]
    }
    # SIMPLE ERROR HANDLING
    if (dim(table)[1] == 0)
      return (NULL)
    #stop("Nothing to return in table. I.e. no features extracted.")
    
    # Go to next function
    read_dna_seg(table, seg_name, ...)
    
    # End of not PTT
  }
  
  # End of genbank to dna_seg
}
read_dna_seg_from_fasta <- function(file, ...){
  # read data
  data <- readLines(file)
  title <- data[1]
  if (length(grep("^>\\w+", title, perl=TRUE)) < 1){
    stop(paste(file, "does not seem like a valid fasta file"))
  }
  name <- unlist(strsplit(title, " ", fixed=TRUE))[1]
  name <- substr(name, 2, nchar(name))
  seq <- data[-1]
  len <- sum(nchar(seq))
  dna_seg <- as.dna_seg(data.frame(name=name, start=1, end=len, strand=1,
                                   stringsAsFactors=FALSE), ...)
  return(dna_seg)
}
# reading genes from a file. Use source=tab or ptt to specify type
read_dna_seg_from_ptt <- function(file, meta_lines=2, header=TRUE, ...){
  # reads meta info
  seg_name <- readLines(file, n=1)
  seg_name <- strsplit(seg_name, "/,|-/", fixed=TRUE)[[1]][1]
  # reads ptt table
  ptt <- read.table(file, skip=meta_lines, as.is=TRUE, header=header,
                    sep="\t", quote="")
  if (header){
    names(ptt) <- tolower(names(ptt))
  }
  else {
    names(ptt) <- c("location", "strand", "length", "pid", "gene",
                    "synonym", "code", "cog", "product")
  }
  # parse location
  location <- strsplit(ptt$location, "..", fixed=TRUE)
  start <- as.numeric(sapply(location, function(x) x[[1]]))
  end <- as.numeric(sapply(location, function(x) x[[2]]))
  # parse strand
  strand <- ptt$strand
  strand[strand=="-"] <- -1
  strand[strand=="+"] <- 1
  strand <- as.numeric(strand)
  # parse gene name from name or synonym if not present
  name <- ifelse(ptt$gene == "-", ptt$synonym, ptt$gene)
  table <- data.frame(name=name, start=start, end=end, strand=strand,
                      length=ptt$length, pid=ptt$pid, gene=ptt$gene,
                      synonym=ptt$synonym, code=ptt$code, cog=ptt$cog,
                      product=ptt$product,
                      stringsAsFactors=FALSE)
  read_dna_seg(table, seg_name, ...)
}
read_dna_seg_from_tab <- function(file, header=TRUE, ...) {
  table <- read.table(file, as.is=TRUE, header=header, sep="\t", quote="")
  if (ncol(table) < 4) stop("Insufficent number of columns in table")
  col_names <-  c("name", "start", "end", "strand")
  names(table)[1:length(col_names)] <- col_names
  # parse name from file name by default
  seg_name <- basename(file)
  read_dna_seg(table, seg_name, ...)
}

# trim dna_seg given x limits
trim.dna_seg <- function(x, xlim=NULL, ...){
  dna_seg <- x
  if (!is.null(xlim)){
    if (!is.numeric(xlim)) stop("xlim must be numeric")
    if (length(xlim) != 2) stop("xlim must be length 2")
    dna_seg <- dna_seg[dna_seg$start >= xlim[1] & dna_seg$end <= xlim[2],]
  }
  dna_seg
}
# reverse dna_seg
reverse.dna_seg <- function(x, ...){
  dna_seg <- x
  start <- -dna_seg$end
  dna_seg$end <- -dna_seg$start
  dna_seg$start <- start
  dna_seg$strand <- -dna_seg$strand
  dna_seg
}

read_dna_seg <- function(table, seg_name, reverse=FALSE, xlim=NULL, ...){
  # check args
  if (ncol(table) < 4) stop("Insufficent number of columns in table")
  if (nrow(table) < 1) stop("No lines in table")
  col_names <-  c("name", "start", "end", "strand")
  if (!all(col_names %in% names(table)))
    stop("Table should contain at least columns name, start, end and strand")
  # make dna_seg object, set seg_name attribute
  dna_seg <- as.dna_seg(table, ...)
  dna_seg <- trim.dna_seg(dna_seg, xlim)
  if (reverse) dna_seg <- reverse.dna_seg(dna_seg)
  attr(dna_seg, "seg_name") <- seg_name
  dna_seg
}

#####################command plot########################################

###############sgs1####################

#fig melanogaster group no CG3036
plot_gene_map(dna_segs=list( melanosgs1noCG3036, sechelliasgs1noCG3036, simulanssgs1noCG3036, maurisgs1noCG3036, santosgs1noCG3036, yakubasgs1noCG3036, teissierisgs1, erectasgs1noCG3036, eugracilissgs1noCG3036, suzukiisgs1noCG3036, biarmipessgs1noCG3036, takahashiisgs1noCG3036),comparisons= list(sgs1_mel_sech_noCG3036, sgs1_sech_sim_noCG3036, sgs1_sim_mau_noCG3036, sgs1_mau_santo_noCG3036, sgs1_santo_yak_noCG3036, sgs1_yak_teissieri_noCG3036, sgs1_teissieri_erecta_noCG3036, sgs1_erecta_eugracilis_noCG3036, sgs1_eugracilis_suzukii_noCG3036, sgs1_suzukii_biarmipes_noCG3036, sgs1_biarmipes_takahashii_noCG3036),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D.melanogaster", "D.sechellia", "D.simulans", "D.mauritiana", "D.santomea", "D.yakuba", "D.teissieri", "D.erecta", "D.eugracilis", "D.suzukii", "D.biarmipes", "D. takahashii"))

#fig espèces éloignées no CG3036 no willistoni
plot_gene_map(dna_segs=list( subobscurasgs1noCG3036, obscurasgs1noCG3036, pseudoobscurasgs1noCG3036, ananassaesgs1noCG3036, bipectinatasgs1noCG3036, ficusphilasgs1noCG3036, eleganssgs1noCG3036, rhopaloasgs1noCG3036, melanosgs1noCG3036, triaurariasgs1noCG3036, rufasgs1noCG3036, jambulinasgs1noCG3036noSnsl, kikkawaisgs1noCG3036),comparisons= list(sgs1_subobscura_obscura_noCG3036, sgs1_obscura_pseudoobscura_noCG3036, sgs1_pseudoobscura_ananassae_noCG3036, sgs1_ananassae_bipectinata_noCG3036, sgs1_bipectinata_ficusphila_noCG3036, sgs1_ficusphila_elegans_noCG3036, sgs1_elegans_rhopaloa_noCG3036, sgs1_rhopaloa_melanogaster_noCG3036, sgs1_melanogaster_triauraria_noCG3036, sgs1_triauraria_rufa_noCG3036, sgs1_rufa_jambulina_noCG3036noSnsl, sgs1_jambulina_kikkawai_noCG3036noSnsl),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D.subobscura", "D.obscura", "D.pseudoobscura", "D.ananassae", "D.bipectinata", "D.ficusphila", "D.elegans", "D.rhopaloa", "D. melanogaster", "D.triauraria", "D.rufa", "D.jambulina", "D.kikkawai"))


###############sgs3bis###############
#melanogaster group
plot_gene_map(dna_segs=list(melanogaster_sgs3bis, sechellia_sgs3bis, simulans_sgs3bis, mauritiana_sgs3bis, santomea_sgs3bis, yakuba_sgs3bis, teissieri_sgs3bis, erecta_sgs3bis, eugracilis_sgs3bis, suzukii_sgs3bis, biarmipes_sgs3bis, takahashii_sgs3bis),comparisons= list(sgs3bis_melanogaster_sechellia, sgs3bis_sechellia_simulans, sgs3bis_simulans_mauritiana, sgs3bis_mauritiana_santomea, sgs3bis_santomea_yakuba, sgs3bis_yakuba_teissieri, sgs3bis_teissieri_erecta, sgs3bis_erecta_eugracilis, sgs3bis_eugracilis_suzukii, sgs3bis_suzukii_biarmipes, sgs3bis_biarmipes_takahashii),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D.melanogaster", "D.sechellia","D.simulans", "D.mauritiana", "D.santomea", "D.yakuba", "D.teissieri", "D.erecta", "D.eugracilis", "D.suzukii", "D.biarmipes", "D.takahashii"))

#short window
plot_gene_map(dna_segs=list(melanogaster_sgs3bis, sechellia_sgs3bis, simulans_sgs3bis, mauritiana_sgs3bis, santomea_sgs3bis, yakuba_sgs3bis, teissieri_sgs3bis, erecta_sgs3bis, eugracilis_sgs3bis, suzukii_sgs3bis, biarmipes_sgs3bis, takahashii_sgs3bis),comparisons= list(sgs3bis_short_melanogaster_sechellia, sgs3bis_short_sechellia_simulans, sgs3bis_short_simulans_mauritiana, sgs3bis_short_mauritiana_santomea, sgs3bis_short_santomea_yakuba, sgs3bis_short_yakuba_teissieri, sgs3bis_short_teissieri_erecta, sgs3bis_short_erecta_eugracilis, sgs3bis_short_eugracilis_suzukii, sgs3bis_short_suzukii_biarmipes, sgs3bis_short_biarmipes_takahashii),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D.melanogaster", "D.sechellia","D.simulans", "D.mauritiana", "D.santomea", "D.yakuba", "D.teissieri", "D.erecta", "D.eugracilis", "D.suzukii", "D.biarmipes", "D.takahashii"))

#other species
plot_gene_map(dna_segs=list(obscura_sgs3bis, subobscura_sgs3bis, pseudoobscura_sgs3bis, ananassae_sgs3bis, bipectinata_sgs3bis, ficusphila_sgs3bis, elegans_sgs3bis, rhopaloa_sgs3bis, melanogaster_sgs3bis, triauraria_sgs3bis, rufa_sgs3bis, jambulina_sgs3bis, kikkawai_sgs3bis),comparisons= list(sgs3bis_obscura_subobscura, sgs3bis_subobscura_pseudoobscura, sgs3bis_pseudoobscura_ananassae, sgs3bis_ananassae_bipectinata, sgs3bis_bipectinata_ficusphila, sgs3bis_ficusphila_elegans, sgs3bis_elegans_rhopaloa, sgs3bis_rhopaloa_melanogaster, sgs3bis_melanogaster_triauraria, sgs3bis_triauraria_rufa, sgs3bis_rufa_jambulina, sgs3bis_jambulina_kikkawai),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. obscura", "D. subobscura", "D. pseudoobscura", "D. ananassae", "D. bipectinata", "D. ficusphila", "D. elegans", "D. rhopaloa","D.melanogaster", "D. triauraria", "D. rufa", "D. jambulina", "D. kikkawai"), offsets = c(12000,24000,22000,8000,0,40000,25000,20000,37000,-10000,9000,3000,50500))

#triauraria vs bipectinata
plot_gene_map(dna_segs=list(bipectinata_sgs3bis, triauraria_sgs3bis),comparisons= list(sgs3bis_bipectinata_triauraria),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. bipectinata", "D. triauraria"))

##############sgs3################
#melanogaster group
plot_gene_map(dna_segs=list(melanogaster_sgs3, sechellia_sgs3, simulans_sgs3, mauritiana_sgs3, santomea_sgs3, yakuba_sgs3, teissieri_sgs3, erecta_sgs3, eugracilis_sgs3, suzukii_sgs3, biarmipes_sgs3, takahashii_sgs3, ficusphila_sgs3),comparisons= list(sgs3_melanogaster_sechellia, sgs3_sechellia_simulans, sgs3_simulans_mauritiana, sgs3_mauritiana_santomea, sgs3_santomea_yakuba, sgs3_yakuba_teissieri, sgs3_teissieri_erecta, sgs3_erecta_eugracilis, sgs3_eugracilis_suzukii, sgs3_suzukii_biarmipes, sgs3_biarmipes_takahashii, sgs3_takahashii_ficusphila),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D.melanogaster", "D.sechellia","D.simulans", "D.mauritiana", "D.santomea", "D.yakuba", "D.teissieri", "D.erecta", "D.eugracilis", "D.suzukii", "D.biarmipes", "D.takahashii", "D. ficusphila"),offsets = c(1500,12000,11500,7000,7000,7500,6000,11500,7000,0,4000,-2000,7000))

#sgs region melanogaster group
plot_gene_map(dna_segs=list(melanogastersgsregion, sechelliasgsregion, simulanssgsregion, mauritianasgsregion, erectasgsregion, teissierisgsregion, yakubasgsregion, santomeasgsregion, eugracilissgsregion, suzukiisgsregion, biarmipessgsregion, takahashiisgsregion),comparisons= list(sgs3_melanogaster_sechellia_sgsregion, sgs3_sechellia_simulans_sgsregion, sgs3_simulans_mauritiana_sgsregion, sgs3_mauritiana_erecta_sgsregion_2, sgs3_erecta_teissieri_sgsregion_2, sgs3_teissieri_yakuba_sgsregion_2, sgs3_yakuba_santomea_sgsregion_2, sgs3_santomea_eugracilis_sgsregion_2, sgs3_eugracilis_suzukii_sgsregion, sgs3_suzukii_biarmipes_sgsregion, sgs3_biarmipes_takahashii_sgsregion),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. melanogaster", "D. sechellia","D. simulans", "D. mauritiana","D. santomea", "D. yakuba", "D. teissieri","D. erecta", "D. eugracilis", "D. suzukii", "D. biarmipes", "D. takahashii"), offsets = c(-8000,2000,1500,1500,1000,-2200,-1000,-2000,-2700,0,100,-1000))

#other species
plot_gene_map(dna_segs=list( obscura_sgs3, subobscura_sgs3, pseudoobscura_sgs3, ananassae_sgs3, bipectinata_sgs3, ficusphila_sgs3, elegans_sgs3, rhopaloa_sgs3, melanogaster_sgs3, triauraria_sgs3, rufa_sgs3, jambulina_sgs3, kikkawai_sgs3),comparisons= list(sgs3_obscura_subobscura, sgs3_subobscura_pseudoobscura, sgs3_pseudoobscura_ananassae, sgs3_anana_bipec, sgs3_bipectinata_ficusphila, sgs3_ficusphila_elegans, sgs3_elegans_rhopaloa, sgs3_rhopaloa_melanogaster, sgs3_melanogaster_triauraria, sgs3_triauraria_rufa, sgs3_rufa_jambulina, sgs3_jambulina_kikkawai),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D.obscura", "D.subobscura", "D.pseudoobscura", "D. ananassae", "D. bipectinata", "D.ficusphila", "D. elegans", "D. rhopaloa", "D. melanogaster", "D.triauraria", "D. rufa", "D. jambulina", "D. kikkawai"),offsets = c(-7500,22000,11000,11500,-5000,6000,6000,0,3000,43000,41500,-7000,18500))

#sgs region other species
plot_gene_map(dna_segs=list(obscurasgsregion, subobscurasgsregion, pseudoobscurasgsregion, ananassaesgsregion),comparisons= list(sgs3_obscura_subobscura_sgsregion, sgs3_subobscura_pseudoobscura_sgsregion, sgs3_pseudoobscura_ananassae_sgsregion),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. obscura", "D. subobscura", "D. pseudoobscura", "D. ananassae"), offsets = c(0, 0, -200, 0))

#sgs region melanogaster vs mauritiana
plot_gene_map(dna_segs=list(melanogastersgsregion, mauritianasgsregion),comparisons= list(sgs3_melanogaster_mauritiana_sgsregion),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. melanogaster", "D. mauritiana"))

#sgs region teissieri vs mauritiana
plot_gene_map(dna_segs=list(teissierisgsregion, mauritianasgsregion),comparisons= list(sgs3_teissieri_mauritiana_sgsregion),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. teissieri", "D. mauritiana"))

#comparisons 2 by2
plot_gene_map(dna_segs=list(melanogaster_sgs3, melanogaster_sgs3),comparisons= list(sgs3_melanogaster_melanogaster),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. melanogaster", "D. melanogaster"))
plot_gene_map(dna_segs=list(erecta_sgs3, erecta_sgs3),comparisons= list(sgs3_erecta_erecta),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. erecta", "D. erecta"))
plot_gene_map(dna_segs=list(santomea_sgs3, santomea_sgs3),comparisons= list(sgs3_santomea_santomea),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. santomea", "D. santomea"))
plot_gene_map(dna_segs=list(yakuba_sgs3, yakuba_sgs3),comparisons= list(sgs3_yakuba_yakuba),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. yakuba", "D. yakuba"))
plot_gene_map(dna_segs=list(teissieri_sgs3, teissieri_sgs3),comparisons= list(sgs3_teissieri_teissieri),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. teissieri", "D.teissieri"))
plot_gene_map(dna_segs=list(suzukii_sgs3, suzukii_sgs3),comparisons= list(sgs3_suzukii_suzukii),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. suzukii", "D. suzukii"))
plot_gene_map(dna_segs=list(biarmipes_sgs3, biarmipes_sgs3),comparisons= list(sgs3_biarmipes_biarmipes),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. biarmipes", "D. biarmipes"))
plot_gene_map(dna_segs=list(takahashii_sgs3, takahashii_sgs3),comparisons= list(sgs3_takahashii_takahashii),annotation_height=1, annotation_cex = 0.1, seg_plot_height=10, gene_type="exons",dna_seg_scale=FALSE,scale=TRUE, fixed_gap_length=10, dna_seg_labels=c("D. takahashii", "D. takahashii"))
