# Functions to analyse mortality on the ZIMS (or similar) data  
# Name:        bastaZIMS.R
# Date:        25 Jan 2018
# Version:     1.0.1
# Modified by: Fernando

# General Comments: 
# -----------------
# - based on FunctionsPrim006.R. 
# - Added minimum age.
# 26 Apr. 2018:
#    a) Corrected the .CalcParLike.minAge() and .CalcAgePost.minAge() for 
#       minAge > 0: the likelihood was misspecified (missing etas for the 
#       survival function, 26/Apr/2018)
#    b) The .CalcParLike.noMinAge() had the indicator for uncensoring as
#       exponent instead of multyplying it.
#    c) The posterior for the parameters did not include the priors for 
#       gamma and eta.
# ============================================================================#

# ----------------------------------- 
# A) FUNCTIONS AVAILABLE TO THE USER: 
# ----------------------------------- 

# A.1) Data check function:
# -------------------------
DataCheck <- function(object, ...) UseMethod("DataCheck")

DataCheck <- function(object, silent = FALSE) {
  if (!silent) {
    cat("\nData check in progress:\n=======================\n")
  }
  prbList <- list()
  prbList$n <- nrow(object)
  prbn <- 0
  # Find relevant column:
  colsnames <- c("Birth.Date", "Min.Birth.Date", "Max.Birth.Date", 
                 "Entry.Date", "Depart.Date", "Depart.Type")
  idcols <- which(!colsnames %in% colnames(object))
  prbList$stopExec <- FALSE
  if (length(idcols) > 0) {
    prbn <- prbn + 1
    if (!silent) {
      cat("\nCritical column names missing:\n")
      for (i in 1:length(idcols)) {
        cat(sprintf("Column '%s' missing.\n", colsnames[idcols]))
      }
      cat("Column names required are:\n")
      cat(sprintf("%s.\n", paste(colsnames, collapse = ", ")))
    }
    pbrList$missCols <- idcols
    prbList$stopExec <- TRUE
  } else {
    # Convert to dates:
    object$Birth.Date <- as.Date(object$Birth.Date, format = "%Y-%m-%d")
    object$Min.Birth.Date <- as.Date(object$Min.Birth.Date, format = "%Y-%m-%d")
    object$Max.Birth.Date <- as.Date(object$Max.Birth.Date, format = "%Y-%m-%d")
    object$Entry.Date <- as.Date(object$Entry.Date, format = "%Y-%m-%d")
    object$Depart.Date <- as.Date(object$Depart.Date, format = "%Y-%m-%d")
    
    # Find NA records in dates:
    prbList$nas <- list()
    nnas <- 0
    for (i in 1:length(colsnames)) {
      idna <- which(is.na(object[[colsnames[i]]]))
      if (length(idna) > 0) {
        prbList$nas[[colsnames[i]]] <- idna
        nnas <- nnas + 1
      } else {
        prbList$nas[[colsnames[i]]] <- "None"
      }
    }
    if (nnas > 0) {
      if (!silent) {
        cat(sprintf("\nThere are %s columns with NA records\n", nnas))
        cat("Use print(object) to review NA records.\n")
      }
      prbList$stopExec <- TRUE
    }
    
    # Find date ranges:
    prbList$DateRan <- apply(object[, colsnames[-6]], 2, range, na.rm = TRUE)
    
    # Find inconsistencies between dates:
    datepr <- c("Min Birth > Birth", "Birth > Max Birth", 
                "Min Birth > Max Birth", "Birth > Entry", "Min Birth > Entry",
                "Max Birth > Entry", "Entry > Depart")
    nchpr <- nchar(datepr)
    maxch <- max(nchpr)
    prbList$MinBBirth <- which(object$Min.Birth.Date > object$Birth.Date)
    prbList$BirthMaxB <- which(object$Birth.Date > object$Max.Birth.Date)
    prbList$MinBMaxB <- which(object$Min.Birth.Date > object$Max.Birth.Date)
    prbList$BirthEntr <- which(object$Birth.Date > object$Entry.Date)
    prbList$MinBEntr <- which(object$Min.Birth.Date > object$Entry.Date)
    prbList$MaxBEntr <- which(object$Max.Birth.Date > object$Entry.Date)
    prbList$EntrDep <- which(object$Entry.Date > object$Depart.Date)
    idlens <- grep("MinBBirth", names(prbList)):grep("EntrDep", names(prbList))
    dateslen <- sapply(idlens, function(pr) length(prbList[[pr]]))
    names(dateslen) <- names(prbList)[idlens]
    if (any(dateslen > 0)) {
      if (!silent) {
        cat("\nWe found the following number of\nrecords with inconsistencies between dates:\n")
        for (i in which(dateslen > 0)) {
          cat(sprintf("%s%s: %s records\n", 
                      datepr[i], paste(rep(" ", maxch + 2 - nchpr[i]),
                                       collapse = ""), dateslen[i]))
        }
        cat("\nUse print(object) to find inconsistent records.\n")
      }
      prbList$stopExec <- TRUE
    } else {
      if (!silent) {
        cat("\nNo inconsistencies between dates.\n")
      }
    }
    # Find types of departue types
    prbList$DepartType <- table(object$Depart.Type)
    departType <- as.character(object$Depart.Type)
    if (!all(departType %in% c("C", "D"))) {
      prbList$stopExec <- TRUE
    }
    
    # Uncensored individuals:
    prbList$idUnCens <- which(departType == "D")
    prbList$nUnCens <- length(prbList$idUnCens)
    
    # Censored individuals:
    prbList$idCens <- which(departType == "C")
    prbList$nCens <- length(prbList$idCens)
    
    # Birth times to be estimated:
    prbList$idNoBirth <- which(object$Birth.Date != object$Min.Birth.Date)
    prbList$nNoBirth <- length(prbList$idNoBirth)
  }
  
  # Output:
  class(prbList) <- "bastaCheck"
  return(prbList)
}

print.bastaCheck <- function(bastacheck) {
  cat(sprintf("DATA CHECK: %s\n", Sys.time()))
  cat("================================\n\n")
  if ("missCols" %in% names(bastacheck)) {
    cat("Data object is missing the following columns:\n")
    cat(sprintf("%s.\n", paste(bastacheck$missCols, collapse = ", ")))
    cat("The analysis cannot be performed without these columns.")
  } else {
    cat("NUMBER OF RECORDS:\n")
    cat("----------------\n")
    # Total number of records:
    cat(sprintf("Total number of records  : %s\n", bastacheck$n))
    
    # Number of censored records:
    cat(sprintf("Number censored (C)      : %s\n", bastacheck$nCens))
    # Number of censored records:
    cat(sprintf("Number uncensored (D)    : %s\n", bastacheck$nUnCens))
    # Number of records with unknown birth:
    cat(sprintf("Number with unknown birth: %s\n", bastacheck$nNoBirth))
    
    
    # Dates columns with NAs:
    cat("\nNAs IN DATES COLUMNS:\n")
    cat("---------------------\n")
    nalen <- sapply(1:length(bastacheck$nas), 
                    function(ii) {
                      if (bastacheck$nas[[ii]][1] != "None") {
                        nna <- length(bastacheck$nas[[ii]])
                      } else {
                        nna <- 0
                      }
                      return(nna)
                    })
    names(nalen) <- names(bastacheck$nas)
    idna <- which(nalen > 0)
    nachar <- nchar(names(bastacheck$nas))
    mnach <- max(nachar)
    if (any(nalen > 0)) {
      cat("NAs found in the following dates columns:\n")
      for (ii in idna) {
        cat(sprintf("%s:\n", names(bastacheck$nas)[ii]))
        print(bastacheck$nas[[ii]])
        cat("\n")
      }
    } else {
      cat("No NAs found in dates columns.\n")
    }
    
    
    # Dates ranges:
    cat("\nDATES RANGES:\n")
    cat("-------------\n")
    dtab <- bastacheck$DateRan
    idna <- which(is.na(dtab))
    if (length(idna) > 0) dtab[idna] <- "NANA-NA-NA"
    cat(sprintf("%s\n", paste(colnames(dtab), collapse = "\t")))
    for(ii in 1:2) {
      cat(sprintf("%s\n", paste(dtab[ii, ], collapse = "\t")))
    }
    
    
    # Inconsistencies in the dates columns:
    cat("\nINCONSISTENCIES BETWEEN DATES COLUMNS:\n")
    cat("--------------------------------------\n")
    
    datepr <- c("Min Birth > Birth", "Birth > Max Birth", 
                "Min Birth > Max Birth", "Birth > Entry", "Min Birth > Entry",
                "Max Birth > Entry", "Entry > Depart")
    idlens <- c("MinBBirth", "BirthMaxB", "MinBMaxB", "BirthEntr", "MinBEntr",
                "MaxBEntr", "EntrDep")
    nchpr <- nchar(datepr)
    maxch <- max(nchpr)
    dateslen <- sapply(idlens, function(pr) length(bastacheck[[pr]]))
    names(dateslen) <- names(bastacheck)[idlens]
    if (any(dateslen > 0)) {
      cat("Records with inconsistencies between dates:\n")
      for (i in which(dateslen > 0)) {
        cat(sprintf("\n%s:\n", datepr[i]))
        print(bastacheck[[idlens[i]]])
      }
    } else {
      cat("\nNone.\n")
    }
    
    # Inconsistencies in Depart Type:
    cat("\nINCONSISTENCIES IN DEPARTURE TYPES:\n")
    cat("-----------------------------------\n")
    if (length(which(!names(bastacheck$DepartType) %in% c("C", "D")))) {
      cat(sprintf("Departure type codes found: %s\n", 
                  paste(names(bastacheck$DepartType), collapse = ", ")))
      cat("\nWarning: only departure types required\nare C (censored) and D (dead).")
    } else {
      cat("None\n")
    }
  }
}  

# A.2) main basta function:
# -------------------------
bastaZIMS <- function(object, ...) UseMethod("bastaZIMS")

bastaZIMS.default <- function(object, covform = NULL, model = "GO", 
                              shape = "simple", covarsStruct = "fused", 
                              minAge = 0, niter = 11000, burnin = 1001, 
                              thinning = 20, nsim = 1, parallel = FALSE, 
                              ncpus = 2, updateJumps = TRUE, lifeTable = TRUE, 
                              negSenescence = FALSE, ...) {
  argList <- list(...)
  # Check dataset:
  dcheck <- DataCheck(object, silent = TRUE)
  if (dcheck$stopExec) {
    stop("\nProblems detected with the data.\nUse DataCheck() to find problems.", call. = FALSE)
  }
  intVars <- c("algObj", "datObj", "ageObj", "covObj", "defTheta", "fullParObj",
               "parObj", "parCovObj", ".Random.seed", "updateJumps", "niter",
               "nsim", "burnin", "thinning", "minAge", ".CalcMort", 
               '.CalcSurv', ".CalcMort.numeric", ".CalcMort.matrix", 
               ".CalcSurv.numeric", ".CalcSurv.matrix", ".CalcCumHaz",
               ".CalcCumHaz.matrix", ".CalcCumHaz.numeric")
  keep <- seq(burnin, niter, thinning)
  algObj <- .CreateAlgObj(model, shape, covform, covarsStruct, niter, burnin,
                          thinning, updateJumps, nsim, minAge, negSenescence)
  datObj <- .PrepDataObj(object)
  ageObj <- .CreateAgeObj(datObj, algObj)
  covObj <- .CreateCovObj(object, datObj, algObj)
  defTheta <- .SetDefaultTheta(algObj)
  fullParObj <- .BuildFullParObj(defTheta, algObj, covObj)
  parObj <- .DefineParObj(fullParObj)
  parCovObj <- .CalcCovPars(parObj = parObj, parCov = NULL, 
                            covObj = covObj, datObj = datObj, 
                            type = "both")
  .CalcMort <- function(theta, ...) UseMethod(".CalcMort")
  .CalcMort.matrix <- .DefineMortMatrix(algObj)
  .CalcMort.numeric <- .DefineMortNumeric(algObj)
  
  # b) Cummulative hazard:
  .CalcCumHaz <- function(theta, ...) UseMethod(".CalcCumHaz")
  .CalcCumHaz.matrix <- .DefineCumHazMatrix(algObj)
  .CalcCumHaz.numeric <- .DefineCumHazNumeric(algObj)
  
  # c) Survival:
  .CalcSurv <- function(theta, ...) UseMethod(".CalcSurv")
  .CalcSurv.matrix <- .DefineSurvMatrix(algObj)
  .CalcSurv.numeric <- .DefineSurvNumeric(algObj)
  
  # Create covariate names object:
  covsNames <- list(cat = NA, con = NA, class = NA)
  if (inherits(covObj, c("cateCov", "bothCov"))) {
    covsNames$cat <- covObj$cat
  }
  if (inherits(covObj, c("contCov", "bothCov"))) {
    covsNames$con <- covObj$cont
  }
  covsNames$class <- class(covObj)[1]
  
  # Assign variables to globalenvironment()
  for (name in intVars) assign(name, get(name), envir = globalenv())
  
  # Find jump sd:
  cat("Running sequence to find jump SDs...\n\n")
  outJumps <- .RunMCMC(1, saveAll = TRUE, printProgr = FALSE, UpdJumps = TRUE,
                       parJumps = NA)
  Start <- Sys.time()
  if (nsim > 1) {
    cat("Multiple simulations started...\n\n") 
    if (parallel) {
      opp <- options()
      options(warn = -1)
      sfInit(parallel = TRUE, cpus = ncpus)
      sfExport(list = c(intVars, ".Random.seed"))
      # sfSource(file = "/Users/fernando/FERNANDO/PROJECTS/4.PACKAGES/BaSTA.ZIMS/bastaZIMS/pkg/R/bastazims.r")
      sfLibrary(BaSTA.ZIMS)
      bastaOut <- sfClusterApplyLB(1:nsim, .RunMCMC, UpdJumps = FALSE, 
                                   parJumps = outJumps$jumps)
      sfRemoveAll(hidden = TRUE)
      sfStop()
      options(opp)
    } else {
      bastaOut <- lapply(1:nsim, .RunMCMC)
    }
  } else {
    cat("Simulation started...\n\n")
    bastaOut <- lapply(1:nsim, .RunMCMC)
  }
  End <- Sys.time()
  compTime <- round(as.numeric(End-Start, units = units(End - Start)), 2)
  cat(sprintf("Total MCMC computing time: %.2f %s.\n\n", compTime, 
              units(End - Start)))
  names(bastaOut) <- paste("sim.", 1:nsim, sep = "")
  bastaSumars <- .ExtractParalOut(bastaOut, keep, fullParObj, covsNames, nsim,
                                  datObj, algObj, defTheta, .CalcMort, 
                                  .CalcMort.numeric, .CalcMort.matrix, 
                                  .CalcSurv, .CalcSurv.matrix, 
                                  .CalcSurv.numeric, covObj)
  bastaFinal <- bastaSumars
  bastaFinal$runs <- bastaOut
  bastaFinal$fullpar <- fullParObj
  bastaFinal$simthe <- defTheta
  bastaFinal$covs <- covsNames
  bastaFinal$settings <- c(niter, burnin, thinning, nsim)
  names(bastaFinal$settings) <- c("niter", "burnin", "thinning", "nsim")
  bastaFinal$modelSpecs <- 
    c(model, shape, minAge, covarsStruct,
      paste(names(covObj$cat), collapse = ", "), 
      paste(names(covObj$cont), collapse = ", "))
  names(bastaFinal$modelSpecs) <- c("model", "shape", "min. age", 
                                    "Covar. structure", "Categorical", 
                                    "Continuous")
  bastaFinal$lifeTable <- .CalcLifeTable(bastaOut, lifeTable, covObj, algObj,
                                         datObj)
  class(bastaFinal) <- "bastaZIMS"
  return(bastaFinal)
}

# A.3) plotting bastaZIMS outputs:
# --------------------------------
plot.bastaZIMS <- function(x, plot.type = "traces", trace.name = "theta",
                           densities = FALSE, noCIs = FALSE, 
                           addKM = TRUE, ...) {
  args <- list(...)
  nv <- ifelse(plot.type == "traces", x$settings['nsim'], length(x$survQuant))
  
  if ("col" %in% names(args)) {
    Palette <- args$col
    if (length(Palette) < nv) {
      ncwarn <- ifelse(plot.trace, "simulation", "covariates")
      warning(sprintf("Insufficient number of colors. Not all %s will be displayed.",
                      ncwarn), call. = FALSE)
    }
  } else {
    if (nv <= 9) {
      Palette <- c('#E41A1C', '#377EB8', '#4DAF4A', '#984EA3', '#FF7F00', 
                   '#FFFF33', '#A65628', '#F781BF', '#999999')
    } else {
      Palette <- rainbow(nv)
    }
  }
  if ("lwd" %in% names(args)) {
    lwd <- args$lwd
  } else {
    lwd <- 1
  }
  if ("lty" %in% names(args)) {
    lty <- args$lty
  } else {
    lty <- 1
  }
  op <- par(no.readonly = TRUE)
  # ======= #
  # TRACES:
  # ======= #
  if (plot.type == "traces") {
    nsim <- length(x$runs)
    
    if (trace.name == "theta") {
      pcol <- x$simthe$length
      prow <- x$fullpar$theta$len / pcol
      npar <- x$simthe$length
      allnpar <- x$fullpar$theta$len
    } else if (trace.name == "gamma") {
      npar <- ncol(x$runs$sim.1$params$gamma)
      if (is.null(npar)) {
        stop("'gamma' parameters not calculated (not propHaz)", call. = FALSE)
      }
      pcol <- ceiling(npar / 2)
      prow <- ceiling(npar / pcol)
      allnpar <- x$fullpar$gamma$len
    } else {
      #etaPar <- ("eta" %in% colnames(x$params)) 
      npar <- ifelse("eta" %in% colnames(x$params), 1, NA)
      if (is.na(npar)) {
        stop("'eta' parameters not calculated (no minAge)", call. = FALSE)
      }
      pcol <- 1
      prow <- 1
      allnpar <- 1
    }
    keep <- seq(1, x$setting["niter"], x$settings["thinning"])
    par (mfrow = c(prow, pcol), mar = c(4, 4, 1, 1))
    for (pp in 1:allnpar) {
      ylim <- 
        range(sapply(1:nsim, function(ii) 
          range(x$runs[[ii]]$params[[trace.name]][, pp]))) 
      xlim <- c(0, nrow(x$runs[[1]]$params[[trace.name]]))
      plot(xlim, ylim, col = NA, xlab = "", ylab = "", 
           main = colnames(x$runs[[1]]$params[[trace.name]])[pp])
      for (tt in 1:nsim) {
        lines(keep, x$runs[[tt]]$params[[trace.name]][keep, pp], 
              col = Palette[tt])
      }
    }
    # =========== #
    # DEMO RATES:
    # =========== #
  } else if (plot.type == "demorates") {
    par(mfrow = c(2, 1), mar = c(4, 4, 1, 1)) 
    demvname <- c("Mortality", "Survival")
    names(demvname) <- c("mort", "surv")
    for (demv in c("mort", "surv")) {
      ylim <- c(0, 0)
      if ("xlim" %in% names(args)) {
        xlim <- args$xlim
      } else {
        xlim <- c(0, 0)
      }
      vars <- names(x$mort)
      minAge <- as.numeric(x$modelSpecs["min. age"])
      for (nta in 1:length(x$mort)) {
        cuts <- x$cuts[[nta]]
        ylim <- range(c(ylim, x[[demv]][[nta]][, cuts]), 
                      na.rm = TRUE)
        if (! "xlim" %in% names(args)) {
          xlim <- range(c(xlim, x$x[cuts] + minAge), na.rm = TRUE)
        }
      }
      plot(xlim, ylim, col = NA, xlab = "", ylab = demvname[demv])
      if (minAge > 0) abline(v = minAge, lty = 2)
      nn <- 0
      for (nta in 1:length(x$mort)) {
        nn <- nn + 1
        cuts <- x$cuts[[nta]]
        yy <- x[[demv]][[nta]][, cuts]
        xx <- x$x[cuts]
        if (!noCIs) {
          polygon(c(xx, rev(xx)) + minAge, c(yy[2, ], rev(yy[3, ])), 
                  col = adjustcolor(Palette[nn], alpha.f = 0.25),
                  border = NA)
        }
        lwdd <- ifelse(length(lwd) > 1, lwd[nn], lwd)
        ltyy <- ifelse(length(lty) > 1, lty[nn], lty)
        lines(xx + minAge, yy[1, ], lwd = lwdd, col = Palette[nn], 
              lty = ltyy)
      }
      if (nn > 1 & demv == 'surv') {
        legend('topright', vars, col = Palette[1:nn], pch = 15, bty = 'n')
      }
    }
    # ================ #
    # GOODNESS OF FIT: 
    # ================ #
  } else {
    ncat <- length(x$surv)
    catname <- names(x$surv)
    pcol <- ceiling(ncat / 2)
    prow <- ceiling(ncat / pcol)
    
    par(mfrow = c(prow, pcol), mar = c(4, 4, 1, 1)) 
    for (nta in 1:ncat) {
      ylim <- c(0, 1)
      xlim <- c(0, 0)
      minAge <- as.numeric(x$modelSpecs["min. age"])
      cuts <- x$cuts[[nta]]
      if (! "xlim" %in% names(args)) {
        xlim <- range(c(xlim, x$x[cuts] + minAge), na.rm = TRUE)
      } else {
        xlim <- args$xlim
      }
      if (is.data.frame(x$lifeTable[[nta]])) {
        lifeTab <- x$lifeTable[[nta]]
      } else {
        lifeTab <- x$lifeTable[[nta]]$Mean
      }
      
      plot(xlim, ylim, col = NA, xlab = "", ylab = "Survival", 
           main = catname[nta])
      if (minAge > 0) abline(v = minAge, lty = 2)
      nn <- 0
      cuts <- x$cuts[[nta]]
      yy <- x$surv[[nta]][, cuts]
      xx <- x$x[cuts]
      if (!noCIs) {
        polygon(c(xx, rev(xx)) + minAge, c(yy[2, ], rev(yy[3, ])), 
                col = adjustcolor(Palette[1], alpha.f = 0.25),
                border = NA)
      }
      lwdd <- ifelse(length(lwd) > 1, lwd[1], lwd)
      ltyy <- ifelse(length(lty) > 1, lty[1], lty)
      lines(lifeTab$Ages, lifeTab$lx, type = "s")
      lines(xx + minAge, yy[1, ], lwd = lwdd, col = Palette[1], 
            lty = ltyy)
      if (nta == ncat) {
        legend('topright', c("Kapplan-Meier", "Estimated survival"), 
               col = c(1, Palette[1]), lwd = 2, bty = 'n')
      }
    }
  }
  par(op)
}

# A.4) Printing bastaZIMS outputs:
# ---------------------------------
print.bastaZIMS <- function(x, ...) {
  extraArgs <- list(...)
  if (length(extraArgs) > 0) {
    if (!is.element('digits', names(extraArgs))){
      digits <- 4
    } else {
      digits <- extraArgs$digits
    }
  } else {
    digits <- 4
  }
  if ("ModelSpecs" %in% names(x)) {
    x$modelSpecs <- x$ModelSpecs
  }
  cat("\nCall:\n")
  cat(paste("Model             \t\t: ", x$modelSpecs[1], "\n", sep = ""))
  cat(paste("Shape             \t\t: ", x$modelSpecs[2], "\n", sep = ""))
  cat(paste("Minimum age       \t\t: ", x$modelSpecs[3], "\n", sep = ""))
  cat(paste("Covars. structure \t\t: ", x$modelSpecs[4], "\n", sep = ""))
  cat(paste("Cat. covars.      \t\t: ", x$modelSpecs[5], "\n", sep = ""))
  cat(paste("Cont. covars.     \t\t: ", x$modelSpecs[6], "\n", 
            collapse = ""))
  
  cat("\nCoefficients:\n")
  print.default(x$coefficients, digits, ...)
  cat("\nConvergence:\n")
  cat(x$convmessage)
  if (is.na(x$DIC[1])){
    cat("\nDIC not calculated.")
  } else {
    cat(sprintf("\nDIC = %s", round(x$DIC["DIC"], 2)))
  }
}

# A.5) Summary for bastaZIMS outputs:
# -----------------------------------
summary.bastaZIMS <-
  function(object,...){
    extraArgs       <- list(...)
    if (length(extraArgs) > 0) {
      if (!is.element('digits', names(extraArgs))){
        digits <- 4
      } else {
        digits <- extraArgs$digits
      }
    } else {
      digits <- 4
    }
    if ("ModelSpecs" %in% names(object)) {
      object$modelSpecs <- object$ModelSpecs
    }
    if ("version" %in% names(object)) {
      cat(sprintf("\nOutput from BaSTA version %s\n", object$version))
    }
    cat("\nCall:\n")
    cat(paste("Model             \t\t: ", object$modelSpecs[1], "\n", sep = ""))
    cat(paste("Shape             \t\t: ", object$modelSpecs[2], "\n", sep = ""))
    cat(paste("Covars. structure \t\t: ", object$modelSpecs[3], "\n", sep = ""))
    cat(paste("Minimum age       \t\t: ", object$modelSpecs[4], "\n", sep = ""))
    cat(paste("Cat. covars.      \t\t: ", object$modelSpecs[5], "\n", sep = ""))
    cat(paste("Cont. covars.     \t\t: ", object$modelSpecs[6], "\n", 
              collapse = ""))
    
    cat("\nModel settings:\n")
    print(object$set)
    
    
    cat("\nMean Kullback-Leibler\ndiscrepancy calibration (KLDC):\n")
    if (object$K[1] != "Not calculated") {
      if ("qkl1" %in% names(object$K)) {
        meanKLcalib  <- t((object$K$qkl1 + object$K$qkl2) / 2)
      } else {
        meanKLcalib  <- (object$K$q12 + object$K$q21) / 2
      }
      print.default(meanKLcalib, digits = digits)
    } else {
      if (object$set['nsim'] == 1) {
        cat("KLDC was not calculated due to insufficient number\n",
            " of simulations to estimate convergence.\n")
      } else {
        cat("KLDC was not calculated due to lack of convergence,\n",
            "or because covariates were not included in the model.\n")
      }
    }
    
    
    cat("\nCoefficients:\n")
    print.default(object$coefficients, digits = digits)
    
    cat("\nConvergence:\n")
    if ("Convergence" %in% names(object)) {
      object$convergence <- object$Convergence
    }
    if (object$convergence[1] == "Not calculated") {
      if (object$set['nsim'] == 1) {
        cat("\nConvergence calculations require more than one run.",
            "\nTo estimate potential scale reduction run at least",
            "two simulations.\n")
      } else {
        cat("\nWarning: Convergence not reached for some parameters",
            " (i.e. 'PotScaleReduc' values larger than 1.1).",
            "\nThese estimates should not be used for inference.\n")
      }
    } else {
      if (all(object$convergence[, "Rhat"] < 1.1)) {
        cat("Appropriate convergence reached for all parameters.\n")
      } else {
        cat("\nWarning: Convergence not reached for some parameters",
            " (i.e. 'PotScaleReduc' values larger than 1.1).",
            "\nThese estimates should not be used for inference.\n")
      }
    } 
    cat("\nDIC:\n")
    if (!is.na(object$DIC[1])){
      cat(object$DIC["DIC"],"\n")
      if ("Convergence" %in% names(object)) {
        warning("Model fit in versions older than BaSTA 1.5 had a mistake in the",
                "\ncalculation of DIC values. In case you are interested in\n",
                "comparing the fit of different models, please try to run them\n",
                "with BaSTA versions 1.5 or above.",
                " We apologise for the inconvenience.", call. = FALSE)
      }
    } else {
      if (object$set['nsim'] == 1) {
        cat("DIC was not calculated due to insufficient number",
            "of simulations to estimate convergence.\n")
      } else {
        cat("DIC was not calculated due to lack of convergence.\n")
      }
    }
    ans <- c(list(coefficients = object$coef, DIC = object$DIC,
                  KullbackLeibler = object$KullbackLeibler, 
                  convergence = object$convergence,
                  modelSpecs = object$modelSpecs, settings = object$set))
    return(invisible(ans))
  }


# A.6) Function to calculate pace shape measures:
# -----------------------------------------------
CalcPaceShape <- function(object) {
  if (!inherits(object, "bastaZIMS")) {
    cat("/n Error: this function can only be used with BaSTA.ZIMS outputs (i.e. class = 'bastaZIMS'.")
  } else {
    x <- object$x
    Dx <- x[2] - x[1]
    nca <- length(object$surv)
    nga <- ifelse(is.matrix(object$surv[[1]]), 0, length(object$surv[[1]]))
    canames <- names(object$surv)
    pstat <- list()
    for (ca in 1:nca) {
      pstat[[canames[ca]]] <- list()
      if (nga > 0) {
        ganames <- names(object$surv[[ca]])
        for (ga in 1:nga) {
          ps <- c(ex = .CalcEx(Sx = object$surv[[ca]][[ga]][1, ], dx = Dx),
                  Hx = .CalcHx(Sx = object$surv[[ca]][[ga]][1, ], dx = Dx),
                  Gx = .CalcGx(Sx = object$surv[[ca]][[ga]][1, ], dx = Dx),
                  Cv = .CalcCVx(x = x, Sx = object$surv[[ca]][[ga]][1, ], 
                                dx = Dx))
          pstat[[canames[ca]]][[ganames[ga]]] <- ps
        }
      } else {
        ps <- c(ex = .CalcEx(Sx = object$surv[[ca]][1, ], dx = Dx),
                Hx = .CalcHx(Sx = object$surv[[ca]][1, ], dx = Dx),
                Gx = .CalcGx(Sx = object$surv[[ca]][1, ], dx = Dx),
                Cv = .CalcCVx(x = x, Sx = object$surv[[ca]][1, ], dx = Dx))
        pstat[[canames[ca]]] <- ps
      }
    }
    return(pstat)    
  }
}

# ----------------------------------- 
# B)     INTERNAL FUNCTIONS:          
# ----------------------------------- 

# B.1) Functions to manage user inputs:
# -------------------------------------
# Algorithm object function:
.CreateAlgObj <- function(model, shape, covform, covarsStruct, niter, burnin,
                          thinning, updateJumps, nsim, minAge, negSenescence) {
  return(list(model = model, shape = shape, covform = covform, 
              covStruc = covarsStruct, niter = niter, burnin = burnin, 
              thinning = thinning, updJump = updateJumps, nsim = nsim, 
              minAge = minAge, negSenescence = negSenescence))
}

# Prepare data object:
.PrepDataObj <- function(object) {
  n <- nrow(object)
  # Calculate Julian times:
  bi <- round(as.numeric(as.Date(object$Birth.Date, format = "%Y-%m-%d")) / 
                365.25, 2) + 1970
  bil <- round(as.numeric(as.Date(object$Min.Birth.Date, format = "%Y-%m-%d")) /
                 365.25, 2) + 1970
  biu <- round(as.numeric(as.Date(object$Max.Birth.Date, format = "%Y-%m-%d")) /
                 365.25, 2) + 1970
  entry <- round(as.numeric(as.Date(object$Entry.Date, format = "%Y-%m-%d")) /
                   365.25, 2) + 1970
  depart <- round(as.numeric(as.Date(object$Depart.Date, format = "%Y-%m-%d")) /
                    365.25, 2) + 1970

  # Extract ages:
  # if (length(which(biu > entry)) > 0) {
  #   biu[biu > entry] <- entry[biu > entry]
  # }
  xl <- depart - bi
  xt <- rep(0, n)
  xt[entry != bi] <- (entry - bi)[entry != bi]
  
  
  # Entry and departure types:
  entryType <- as.character(object$Entry.Type)
  departType <- as.character(object$Depart.Type)
  
  # Censored individuals:
  idCens <- which(departType %in% c("O", "C"))
  nCens <- length(idCens)
  
  # Birth times to be estimated:
  idNoBirth <- which(bil != biu)
  nNoBirth <- length(idNoBirth)
  
  # Output:
  return(list(xl = xl, xt = xt, bi = bi, bil = bil, biu = biu, 
              entry = entry, depart = depart, idCens = idCens,
              idNoBirth = idNoBirth, n = n, nCens = nCens, nNoBi = nNoBirth))
}

# Create and managing covariages functions:
# -----------------------------------------
.CreateCovObj <- function(object, dataObj, algObj) {
  covObj <- list()
  covClass <- c("noCov", "noCovType")
  if (is.null(algObj$covform)) {
    covObj$covs <- NULL
  } else {
    covMat <- .MakeCovMat(algObj$covform, data = object)
    covType <- .FindCovType(covMat)
    if (algObj$covStruc == "fused") {
      covClass[1] <- "fused"
      if (!is.null(covType$cat)) {
        covObj$inMort <- covMat[, covType$cat]
        covObj$imLen <- ncol(covObj$inMort)
      } else {
        covClass[1] <- "propHaz"
      }
      if (!is.null(covType$cont)) {
        covObj$propHaz <- matrix(covMat[, c(covType$int, covType$cont)], 
                                 ncol = length(c(covType$int, covType$cont)),
                                 dimnames = list(NULL, c(names(covType$int), 
                                                         names(covType$cont))))
        covObj$phLen <- ncol(covObj$propHaz)
      } else {
        covClass[1] <- "inMort"
      }
    } else if (algObj$covStruc == "all.in.mort") {
      if (is.null(covType$int) & is.null(covType$cat)) {
        covObj$inMort <- cbind(1, covMat)
        colnames(covObj$inMort) <- c("Intercept", colnames(covMat))
      } else {
        covObj$inMort <- covMat
      }
      covObj$imLen <- ncol(covObj$inMort)
      covClass[1] <- "inMort"
    } else {
      if (!is.null(covType$int)) {
        covObj$propHaz <- 
          matrix(covMat[, -covType$int], dataObj$n, ncol(covMat) -1, 
                 dimnames = list(NULL, colnames(covMat)[-covType$int]))
      } else if (!is.null(covType$cat)) {
        covObj$propHaz <- 
          matrix(covMat[, -covType$cat[1]], dataObj$n, ncol(covMat) -1, 
                 dimnames = list(NULL, colnames(covMat)[-covType$cat[1]]))
      } else {
        covObj$propHaz <- covMat
      }
      covObj$phLen <- ncol(covObj$propHaz)
      covClass[1] <- "propHaz"
    }
    if (!is.null(covType$cat) & !is.null(covType$cont)) {
      covClass[2] <- "bothCov"
      covObj$cat <- covType$cat
      covObj$cont <- covType$cont
    } else if (!is.null(covType$cat)) {
      covClass[2] <- "cateCov"
      covObj$cat <- covType$cat
    } else if (!is.null(covType$cont)) {
      covClass[2] <- "contCov"
      covObj$cont <- covType$cont
    }
  }
  class(covObj) <- covClass
  return(covObj)
}

.FindCovType <- function(covMat) {
  # This functions finds and returns if an intercecpt was included 
  # and which covariates are categorical or continuous.
  if (!is.null(covMat)) {
    lu <- apply(covMat, 2, function(x) length(unique(x)))
    ru <- apply(covMat, 2, range)
    idcat <- which(lu == 2 & apply(ru, 2, sum) == 1)
    if (length(idcat) == 0) {
      idcat <- NULL
    }
    idint <- which(lu == 1)
    if (length(idint) == 0) {
      idint <- NULL
    }
    idcon <- which(lu > 2)
    if (length(idcon) == 0) {
      idcon <- NULL
    }
  }
  else {
    idcat <- NULL
    idint <- NULL
    idcon <- NULL
  }
  return(list(int = idint, cat = idcat, cont = idcon))
}

.MakeCovMat <- function(covform, data) {
  covs <- model.matrix(covform, data = data)
  return(covs)
}

# Functions to create and manage age object:
# ------------------------------------------
.CreateAgeObj <- function(datObj, algObj) {
  bi <- datObj$bi
  xl <- datObj$xl
  xt <- datObj$xt
  if (datObj$nNoBi > 0) {
    biint <- cbind(datObj$bil, datObj$biu)[datObj$idNoBirth, ]
    bi[datObj$idNoBirth] <- apply(biint, 1, 
                                  function(bb) runif(1, bb[1], bb[2]))
    xl[datObj$idNoBirth] <- datObj$depart[datObj$idNoBirth] - 
      bi[datObj$idNoBirth]
    xt[datObj$idNoBirth] <- datObj$entry[datObj$idNoBirth] - 
      bi[datObj$idNoBirth]
  }
  ageObj <- list(xl = xl, xt = xt, bi = bi)
  if (algObj$minAge > 0) {
    ageObj$ma$xlu <- which(ageObj$xl >= algObj$minAge)
    ageObj$ma$xtu <- which(ageObj$xt >= algObj$minAge)
    ageObj$ma$xll <- which(ageObj$xl < algObj$minAge)
    ageObj$ma$xtl <- which(ageObj$xt < algObj$minAge)
  }
  ageObj$ucens <- rep(1, datObj$n)
  ageObj$ucens[datObj$idCens] <- 0
  class(ageObj) <- ifelse(algObj$minAge == 0, "noMinAge", "minAge")
  return(ageObj)
}

.SampleAgeObj <- function(ageObj, datObj, algObj) {
  ageObjn <- ageObj
  if (datObj$nNoBi > 0) {
    ageObjn$bi[datObj$idNoBirth] <- 
      round(.rtnorm(datObj$nNoBi, ageObj$bi[datObj$idNoBirth], 
                   0.2, lower = datObj$bil[datObj$idNoBirth], 
                   upper = datObj$biu[datObj$idNoBirth]), 2)
    ageObjn$xl[datObj$idNoBirth] <- datObj$depart[datObj$idNoBirth] -
      ageObjn$bi[datObj$idNoBirth]
    ageObjn$xt[datObj$idNoBirth] <- datObj$entry[datObj$idNoBirth] -
      ageObjn$bi[datObj$idNoBirth]
  }
  if (algObj$minAge > 0) {
    ageObjn$ma$xlu <- which(ageObjn$xl >= algObj$minAge)
    ageObjn$ma$xtu <- which(ageObjn$xt >= algObj$minAge)
    ageObjn$ma$xll <- which(ageObjn$xl < algObj$minAge)
    ageObjn$ma$xtl <- which(ageObjn$xt < algObj$minAge)
  }
  return(ageObjn)
}

# Functions to manage parameters:
# -------------------------------
.SetDefaultTheta  <- function(algObj) {
  if (algObj$model == "EX") {
    nTh <- 1
    startTh <- 0.2 
    jumpTh <- 0.1
    priorMean <- 0.06
    priorSd <- 1
    nameTh <- "b0"
    lowTh <- 0
    jitter <- 0.5
  } else if (algObj$model == "GO") {
    nTh <- 2 
    startTh <- c(-2, 0.01) 
    jumpTh <- c(0.1, 0.1)
    priorMean <- c(-3, 0.01)
    priorSd <- c(5, 1) # change 2020-12-10
    nameTh <- c("b0", "b1")
    lowTh <- c(-Inf, 0)
    if (algObj$negSenescence) lowTh[2] <- -Inf
    jitter <- c(0.5, 0.2) 
    if (algObj$shape == "bathtub") {
      lowTh <- c(-Inf, 0)
    }
  } else if (algObj$model == "WE") {
    nTh <- 2
    startTh <- c(1.5, 0.2) 
    jumpTh <- c(.01, 0.1)
    priorMean <- c(1.5, .05)
    priorSd <- c(1, 1)
    nameTh <- c("b0", "b1")
    lowTh <- c(0, 0)
    jitter <- c(0.5, 0.2) 
  } else if (algObj$model == "LO") {
    nTh <- 3 
    startTh <- c(-2, 0.01, 1e-04) 
    jumpTh <- c(0.1, 0.1, 0.1) 
    priorMean <- c(-3, 0.01, 1e-10)
    priorSd <- c(1, 1, 1)
    nameTh <- c("b0", "b1", "b2")
    lowTh <- c(-Inf, 0, 0)
    jitter <- c(0.5, 0.2, 0.5) 
  }
  if (algObj$shape == "Makeham") {
    nTh <- nTh + 1 
    startTh <- c(0, startTh) 
    jumpTh <- c(0.1, jumpTh) 
    priorMean <- c(0, priorMean)
    priorSd <- c(1, priorSd)
    nameTh <- c("c", nameTh)
    lowTh <- c(0, lowTh)
    jitter <- c(0.25, jitter) 
  } else if (algObj$shape == "bathtub") {
    nTh <- nTh + 3 
    startTh <- c(-0.1, 0.6, 0, startTh)
    jumpTh <- c(0.1, 0.1, 0.1, jumpTh) 
    priorMean <- c(-2, 0.01, 0, priorMean)
    priorSd <- c(1, 5, 1, priorSd) # change 2020-12-10
    nameTh <- c("a0", "a1", "c", nameTh)
    lowTh <- c(-Inf, 0, 0, lowTh)
    jitter <- c(0.5, 0.2, 0.2, jitter) 
  }
  defaultTheta  <- list(length = nTh, start = startTh, jump = jumpTh, 
                        priorMean = priorMean, priorSd = priorSd, name = nameTh, 
                        low = lowTh, jitter = jitter)
  attr(defaultTheta, "model") = algObj$model
  attr(defaultTheta, "shape") = algObj$shape
  return(defaultTheta)
}

.BuildFullParObj <- function(defTheta, algObj, covObj) {
  parObj <- list()
  
  # Mortality function parameters:
  parObj$theta <- list()
  if (inherits(covObj, c("inMort", "fused"))) {
    covnames <- names(covObj$cat)
    ncovs <- covObj$imLen
    fullNames <- paste(rep(defTheta$name, each = ncovs), 
                       rep(covnames, defTheta$length), sep = ".")
  } else {
    covnames <- ""
    ncovs <- 1
    fullNames <- defTheta$name
  }
  parObj$theta$start <- matrix(defTheta$start, ncovs, defTheta$length,
                               dimnames = list(covnames, defTheta$name),
                               byrow = TRUE)
  parObj$theta$priorMu <- matrix(defTheta$priorMean, ncovs, defTheta$length,
                                 dimnames = list(covnames, defTheta$name),
                                 byrow = TRUE)
  parObj$theta$priorSd <- matrix(defTheta$priorSd, ncovs, defTheta$length,
                                 dimnames = list(covnames, defTheta$name),
                                 byrow = TRUE)
  parObj$theta$low <- matrix(defTheta$low, ncovs, defTheta$length,
                             dimnames = list(covnames, defTheta$name),
                             byrow = TRUE)
  parObj$theta$jitter <- matrix(defTheta$jitter, ncovs, defTheta$length,
                                dimnames = list(covnames, defTheta$name),
                                byrow = TRUE)
  parObj$theta$dim <- dim(parObj$theta$start)
  parObj$theta$len <- length(parObj$theta$start)
  parObj$theta$names <- fullNames
  if (inherits(covObj, c("bothCov", "contCov"))) {
    phcovs <- names(covObj$cont)
    nphcovs <- length(phcovs)
    parObj$gamma <- list()
    parObj$gamma$start <- rep(0, nphcovs)
    names(parObj$gamma$start) <- phcovs
    parObj$gamma$priorMu <- rep(0, nphcovs)
    names(parObj$gamma$priorMu) <- phcovs
    parObj$gamma$priorSd <- rep(5, nphcovs)
    names(parObj$gamma$priorSd) <- phcovs
    parObj$gamma$low <- rep(-Inf, nphcovs)
    names(parObj$gamma$low) <- phcovs
    parObj$gamma$jitter <- rep(0.2, nphcovs)
    names(parObj$gamma$jitter) <- phcovs
    parObj$gamma$len <- length(parObj$gamma$start)
    parObj$gamma$names <- phcovs
  }
  
  # Proportional hazards parameters:
  if (inherits(covObj, "propHaz")) {
    phcovs <- c()
    if (inherits(covObj, c("cateCov", "bothCov"))) {
      phcovs <- names(covObj$cat)
    }
    if (inherits(covObj, c("bothCov", "contCov"))) {
      phcovs <- c(phcovs, names(covObj$cont))
    }
    nphcovs <- length(phcovs)
    parObj$gamma <- list()
    parObj$gamma$start <- rep(0, nphcovs - 1)
    names(parObj$gamma$start) <- phcovs[-1]
    parObj$gamma$priorMu <- rep(0, nphcovs - 1)
    names(parObj$gamma$priorMu) <- phcovs[-1]
    parObj$gamma$priorSd <- rep(5, nphcovs - 1)
    names(parObj$gamma$priorSd) <- phcovs[-1]
    parObj$gamma$low <- rep(-Inf, nphcovs - 1)
    names(parObj$gamma$low) <- phcovs[-1]
    parObj$gamma$jitter <- rep(0.2, nphcovs - 1)
    names(parObj$gamma$jitter) <- phcovs[-1]
    parObj$gamma$len <- length(parObj$gamma$start)
    parObj$gamma$names <- phcovs[-1]
  }
  if (algObj$minAge > 0) {
    parObj$eta$start <- 0.01
    parObj$eta$priorMu <- 0
    parObj$eta$priorSd <- 1
    parObj$eta$low <- 0
    parObj$eta$jitter <- 0.01
    parObj$eta$len <- 1
    parObj$eta$names <- "minAge"
  }
  class(parObj) <- c(class(covObj)[1], ifelse(algObj$minAge > 0, "minAge",
                                              "noMinAge"))
  return(parObj)
}

.DefineParObj <- function(fullParObj) {
  parObj <- list()
  parObj$theta <- fullParObj$theta$start
  if (inherits(fullParObj,  c("fused", "propHaz"))) {
    parObj$gamma <- fullParObj$gamma$start
  } else {
    parObj$gamma <- 0
  }
  if (inherits(fullParObj, "minAge")) {
    parObj$eta <- fullParObj$eta$start
  }
  class(parObj) <- class(fullParObj)
  return(parObj)
}

.CalcCovPars <- function(parObj, parCov, covObj, datObj, type = "theta") {
  if (is.null(parCov)) {
    parCovObjn <- list()
  } else {
    parCovObjn <- parCov
  }
  if (type %in% c("theta", "both")) {
    if (inherits(covObj, c("fused", "inMort"))) {
      parCovObjn$theta <- covObj$inMort %*% parObj$theta
    } else {
      parCovObjn$theta <- matrix(1, nrow = datObj$n) %*% parObj$theta
    }
  }
  if (type %in% c("gamma", "both")) {
    if (inherits(parObj, c("fused", "propHaz"))) {
      parCovObjn$gamma <- covObj$propHaz %*% parObj$gamma
    } else {
      parCovObjn$gamma <- rep(0, datObj$n)
    }
  }
  return(parCovObjn)
}

.JitterPars <- function(parObj, fullParObj) {
  parObjn <- parObj
  nthe <- fullParObj$theta$len
  parObjn$theta[1:nthe] <- .rtnorm(nthe, c(parObj$theta), 
                                   c(fullParObj$theta$jitter), 
                                   low = c(fullParObj$theta$low))
  if (inherits(parObj, c("fused", "propHaz"))) {
    parObjn$gamma <- .rtnorm(fullParObj$gamma$len,
                             fullParObj$gamma$start, 
                             fullParObj$gamma$jitter)
  }
  if (inherits(parObj, "minAge")) {
    parObjn$eta <- .rtnorm(fullParObj$eta$len, fullParObj$eta$start, 
                           fullParObj$eta$jitter, lower = fullParObj$eta$low)
  }
  return(parObjn)
}

.SamplePars <- function(parObj, parJumps, fullParObj, type = "theta", pp) {
  parObjn <- parObj
  if (type == 'theta') {
    nthe <- fullParObj$theta$len
    parObjn$theta[pp] <- .rtnorm(1, parObj$theta[pp], parJumps$theta[pp], 
                                 low = fullParObj$theta$low[pp])
  } else if (type == 'gamma') {
    if (inherits(parObj, c("fused", "propHaz"))) {
      parObjn$gamma[pp] <- .rtnorm(1, parObj$gamma[pp], parJumps$gamma[pp],
                                   low = fullParObj$gamma$low[pp])
    }
  } else {
    if (inherits(parObj, "minAge")) {
      parObjn$eta <- .rtnorm(1, parObj$eta, parJumps$eta, 
                             low = fullParObj$eta$low)
    }
  }
  return(parObjn)
}

.CalcHastRatio <- function(pNow, pNew, parJumps, type = "theta", pp) {
  if (type == 'theta') {
    hRatio <- .dtnorm(pNow$theta[pp], pNew$theta[pp], parJumps$theta[pp], 
                                 low = fullParObj$theta$low[pp], log = TRUE) -
      .dtnorm(pNew$theta[pp], pNow$theta[pp], parJumps$theta[pp], 
              low = fullParObj$theta$low[pp], log = TRUE)
  } else if (type == 'gamma') {
    if (inherits(pNow, c("fused", "propHaz"))) {
      hRatio <- .dtnorm(pNow$gamma[pp], pNew$gamma[pp], 
                                   parJumps$gamma[pp],
                                   low = fullParObj$gamma$low[pp], log = TRUE) -
        .dtnorm(pNew$gamma[pp], pNow$gamma[pp], 
                parJumps$gamma[pp],
                low = fullParObj$gamma$low[pp], log = TRUE)
    }
  } else {
    if (inherits(pNow, "minAge")) {
      hRatio <- .dtnorm(pNow$eta, pNew$eta, parJumps$eta, 
                             low = fullParObj$eta$low, log = TRUE) -
        .dtnorm(pNew$eta, pNow$eta, parJumps$eta, 
                low = fullParObj$eta$low, log = TRUE)
    }
  }
  return(hRatio)
}

.SetParJumps <- function(parObj, fullParObj) {
  parJump <- parObj
  nthe <- fullParObj$theta$len
  parJump$theta[1:nthe] <- rep(0.1, nthe)
  if (inherits(parObj, c("fused", "propHaz"))) {
    parJump$gamma <- rep(0.1, fullParObj$gamma$len)
  }
  if (inherits(parObj, "minAge")) {
    parJump$eta <- 0.001
  }
  return(parJump)
}

# Function to create output and jumps matrices:
.CreateOutputMat <- function(fullParObj, datObj, niter, burnin, 
                             type = 'params', parObj = NULL) {
  if (type == 'params') {
    outMat <- list()
    outMat$theta <- matrix(0, niter, fullParObj$theta$len,
                           dimnames = list(NULL, fullParObj$theta$names))
    if (!is.null(parObj)) {
      outMat$theta[1, ] <- c(parObj$theta)
    }
    if (inherits(fullParObj, c("fused", "propHaz"))) {
      outMat$gamma <- matrix(0, niter, fullParObj$gamma$len, 
                             dimnames = list(NULL, fullParObj$gamma$names))
      if (!is.null(parObj)) {
        outMat$gamma[1, ] <- parObj$gamma
      }
    }
    if (inherits(fullParObj, "minAge")) {
      outMat$eta <- matrix(0, niter, 1)
      if (!is.null(parObj)) {
        outMat$eta[1, ] <- parObj$eta
      }
    }
  } else if (type == 'ages') {
    if (datObj$nNoBi > 0) {
      outMat <- matrix(0, 0, datObj$nNoBi, 
                       dimnames = list(NULL, 
                                       sprintf("ID.%s", datObj$idNoBirth)))
    }
  } else {
    outMat <- list()
    outMat$theta <- matrix(0, burnin, fullParObj$theta$len,
                           dimnames = list(NULL, fullParObj$theta$names))
    if (inherits(fullParObj, c("fused", "propHaz"))) {
      outMat$gamma <- matrix(0, burnin, fullParObj$gamma$len, 
                             dimnames = list(NULL, fullParObj$gamma$names))
    }
    if (inherits(fullParObj, "minAge")) {
      outMat$eta <- matrix(0, burnin, 1)
    }
  }
  return(outMat)
}


# B.2) Basic survival analysis functions:
# ---------------------------------------
# a) General mortality functions:
.DefineMortMatrix <- function(algObj) {
  if (algObj$model == "EX") {
    mortfun <- function(theta, x) c(theta) * rep(1, length(x))
  } else if (algObj$model == "GO") {
    if (algObj$shape == "simple") {
      mortfun <- function(theta, x) {
        exp(theta[ ,"b0"] + theta[, "b1"] * x)
      }
    } else if (algObj$shape == "Makeham") {
      mortfun <- function(theta, x) {
        theta[, "c"] + exp(theta[, "b0"] + theta[, "b1"] * x)
      }
    } else {
      mortfun <- function(theta, x) {
        exp(theta[, "a0"] - theta[, "a1"] * x) + theta[, "c"] + 
          exp(theta[, "b0"] + theta[, "b1"] * x)
      }
    }
  } else if (algObj$model == "WE") {
    if (algObj$shape == "simple") {
      mortfun <- function(theta, x) {
        theta[, "b0"] * theta[, "b1"]^theta[, "b0"] * 
          x^(theta[, "b0"] - 1)
      }
    } else if (algObj$shape == "Makeham") {
      mortfun <- function(theta, x) {
        theta[, "c"] + theta[, "b0"] * theta[, "b1"]^theta[, "b0"] * 
          x^(theta[, "b0"] - 1)
      }
    } else {
      mortfun <- function(theta, x) {
        exp(theta[, "a0"] - theta[, "a1"] * x) + theta[, "c"] + 
          theta[, "b0"] * theta[, "b1"]^theta[, "b0"] * 
          x^(theta[, "b0"] - 1)
      }
    }
  } else if (algObj$model == "LO") {
    if (algObj$shape == "simple") {
      mortfun <- function(theta, x) {
        exp(theta[, "b0"] + theta[, "b1"] * x) / 
          (1 + theta[, "b2"] * exp(theta[, "b0"]) / 
             theta[, "b1"] * (exp(theta[, "b1"] * x) - 1))
      }
    } else if (algObj$shape == "Makeham") {
      mortfun <- function(theta, x) {
        theta[, "c"] + exp(theta[, "b0"] + theta[, "b1"] * x) / 
          (1 + theta[, "b2"] * exp(theta[, "b0"]) / 
             theta[, "b1"] * (exp(theta[, "b1"] * x) - 1))
      }
    } else {
      mortfun <- function(theta, x) {
        exp(theta[, "a0"] - theta[, "a1"] * x) + theta[, "c"] + 
          exp(theta[, "b0"] + theta[, "b1"] * x) / 
          (1 + theta[, "b2"] * exp(theta[, "b0"]) / 
             theta[, "b1"] * (exp(theta[, "b1"] * x) - 1))
      }
    }
  }
  return(mortfun)
}

.DefineMortNumeric <- function(algObj) {
  if (algObj$model == "EX") {
    mortfun <- function(theta, x) c(theta) * rep(1, length(x))
  } else if (algObj$model == "GO") {
    if (algObj$shape == "simple") {
      mortfun <- function(theta, x) {
        exp(theta["b0"] + theta["b1"] * x)
      }
    } else if (algObj$shape == "Makeham") {
      mortfun <- function(theta, x) {
        theta["c"] + exp(theta["b0"] + theta["b1"] * x)
      }
    } else {
      mortfun <- function(theta, x) {
        exp(theta["a0"] - theta["a1"] * x) + theta["c"] + 
          exp(theta["b0"] + theta["b1"] * x)
      }
    }
  } else if (algObj$model == "WE") {
    if (algObj$shape == "simple") {
      mortfun <- function(theta, x) {
        theta["b0"] * theta["b1"]^theta["b0"] * 
          x^(theta["b0"] - 1)
      }
    } else if (algObj$shape == "Makeham") {
      mortfun <- function(theta, x) {
        theta["c"] + theta["b0"] * theta["b1"]^theta["b0"] * 
          x^(theta["b0"] - 1)
      }
    } else {
      mortfun <- function(theta, x) {
        exp(theta["a0"] - theta["a1"] * x) + theta["c"] + 
          theta["b0"] * theta["b1"]^theta["b0"] * 
          x^(theta["b0"] - 1)
      }
    }
  } else if (algObj$model == "LO") {
    if (algObj$shape == "simple") {
      mortfun <- function(theta, x) {
        exp(theta["b0"] + theta["b1"] * x) / 
          (1 + theta["b2"] * exp(theta["b0"]) / 
             theta["b1"] * (exp(theta["b1"] * x) - 1))
      }
    } else if (algObj$shape == "Makeham") {
      mortfun <- function(theta, x) {
        theta["c"] + exp(theta["b0"] + theta["b1"] * x) / 
          (1 + theta["b2"] * exp(theta["b0"]) / 
             theta["b1"] * (exp(theta["b1"] * x) - 1))
      }
    } else {
      mortfun <- function(theta, x) {
        exp(theta["a0"] - theta["a1"] * x) + theta["c"] + 
          exp(theta["b0"] + theta["b1"] * x) / 
          (1 + theta["b2"] * exp(theta["b0"]) / 
             theta["b1"] * (exp(theta["b1"] * x) - 1))
      }
    }
  }
  return(mortfun)
}

# b) General cumulative hazards functions:
.DefineCumHazMatrix <- function(algObj) {
  if (algObj$model == "EX") {
    cumhazfun <- function(theta, x) c(theta) * x
  } else if (algObj$model == "GO") {
    if (algObj$shape == "simple") {
      cumhazfun <- function(theta, x) {
        exp(theta[, "b0"]) / theta[, "b1"] * 
          (exp(theta[, "b1"] * x) - 1)
      }
    } else if (algObj$shape == "Makeham") {
      cumhazfun <- function(theta, x) {
        theta[, "c"] * x + exp(theta[, "b0"]) / theta[, "b1"] * 
          (exp(theta[, "b1"] * x) - 1)
      }
    } else {
      cumhazfun <- function(theta, x) {
        exp(theta[, "a0"]) / theta[, "a1"] * (1 - exp(-theta[, "a1"] * x)) + 
          theta[, "c"] * x + exp(theta[, "b0"]) / theta[, "b1"] * 
          (exp(theta[, "b1"] * x) - 1)
      }
    }
  } else if (algObj$model == "WE") {
    if (algObj$shape == "simple") {
      cumhazfun <- function(theta, x) {
        (theta[, "b1"] * x)^theta[, "b0"]
      }      
    } else if (algObj$shape == "Makeham") {
      cumhazfun <- function(theta, x) {
        theta[, "c"] * x + (theta[, "b1"] * x)^theta[, "b0"]
      }
    } else {
      cumhazfun <- function(theta, x) {
        exp(theta[, "a0"]) / theta[, "a1"] * (1 - exp(-theta[, "a1"] * x)) +
          theta[, "c"] * x + (theta[, "b1"] * x)^theta[, "b0"]
      }
    }
  } else if (algObj$model == "LO") {
    if (algObj$shape == "simple") {
      cumhazfun <- function(theta, x) {
        log(1 + theta[, "b2"] * exp(theta[, "b0"]) / theta[, "b1"] * 
              (exp(theta[, "b1"] * x) - 1)) * (1 / theta[, "b2"])
      }
    } else if (algObj$shape == "Makeham") {
      cumhazfun <- function(theta, x) {
        theta[, "c"] * x + log(1 + theta[, "b2"] * exp(theta[, "b0"]) / 
                                 theta[, "b1"] * 
                                 (exp(theta[, "b1"] * x) - 1)) * 
          (1 / theta[, "b2"])
      }
    } else {
      cumhazfun <- function(theta, x) {
        exp(theta[, "a0"]) / theta[, "a1"] * (1 - exp(-theta[, "a1"] * x)) +
          theta[, "c"] * x + log(1 + theta[, "b2"] * 
                                   exp(theta[, "b0"]) / theta[, "b1"] * 
                                   (exp(theta[, "b1"] * x) - 1)) *
          (1 / theta[, "b2"])
      }
    }
  }
  return(cumhazfun)
}

.DefineCumHazNumeric <- function(algObj) {
  if (algObj$model == "EX") {
    cumhazfun <- function(theta, x) c(theta) * x
  } else if (algObj$model == "GO") {
    if (algObj$shape == "simple") {
      cumhazfun <- function(theta, x) {
        exp(theta["b0"]) / theta["b1"] * 
          (exp(theta["b1"] * x) - 1)
      }
    } else if (algObj$shape == "Makeham") {
      cumhazfun <- function(theta, x) {
        theta["c"] * x + exp(theta["b0"]) / theta["b1"] * 
          (exp(theta["b1"] * x) - 1)
      }
    } else {
      cumhazfun <- function(theta, x) {
        exp(theta["a0"]) / theta["a1"] * (1 - exp(-theta["a1"] * x)) + 
          theta["c"] * x + exp(theta["b0"]) / theta["b1"] * 
          (exp(theta["b1"] * x) - 1)
      }
    }
  } else if (algObj$model == "WE") {
    if (algObj$shape == "simple") {
      cumhazfun <- function(theta, x) {
        (theta["b1"] * x)^theta["b0"]
      }      
    } else if (algObj$shape == "Makeham") {
      cumhazfun <- function(theta, x) {
        theta["c"] * x + (theta["b1"] * x)^theta["b0"]
      }
    } else {
      cumhazfun <- function(theta, x) {
        exp(theta["a0"]) / theta["a1"] * (1 - exp(-theta["a1"] * x)) +
          theta["c"] * x + (theta["b1"] * x)^theta["b0"]
      }
    }
  } else if (algObj$model == "LO") {
    if (algObj$shape == "simple") {
      cumhazfun <- function(theta, x) {
        log(1 + theta["b2"] * exp(theta["b0"]) / theta["b1"] * 
              (exp(theta["b1"] * x) - 1)) * (1 / theta["b2"])
      }
    } else if (algObj$shape == "Makeham") {
      cumhazfun <- function(theta, x) {
        theta["c"] * x + log(1 + theta["b2"] * exp(theta["b0"]) / 
                               theta["b1"] * 
                               (exp(theta["b1"] * x) - 1)) * 
          (1 / theta["b2"])
      }
    } else {
      cumhazfun <- function(theta, x) {
        exp(theta["a0"]) / theta["a1"] * (1 - exp(-theta["a1"] * x)) +
          theta["c"] * x + log(1 + theta["b2"] * 
                                 exp(theta["b0"]) / theta["b1"] * 
                                 (exp(theta["b1"] * x) - 1)) *
          (1 / theta["b2"])
      }
    }
  }
  return(cumhazfun)
}

# c) General survival functions:
.DefineSurvMatrix <- function(algObj) {
  if (algObj$model == "EX") {
    survfun <- function(theta, x) exp(- c(theta) * x)
  } else if (algObj$model == "GO") {
    if (algObj$shape == "simple") {
      survfun <- function(theta, x) {
        exp(exp(theta[, "b0"]) / theta[, "b1"] * 
              (1 - exp(theta[, "b1"] * x)))
      }
    } else if (algObj$shape == "Makeham") {
      survfun <- function(theta, x) {
        exp(-theta[, "c"] * x + exp(theta[, "b0"]) / theta[, "b1"] * 
              (1 - exp(theta[, "b1"] * x)))
      }
    } else {
      survfun <- function(theta, x) {
        exp(exp(theta[, "a0"]) / theta[, "a1"] * (exp(-theta[, "a1"] * x) - 1) - 
              theta[, "c"] * x + exp(theta[, "b0"]) / theta[, "b1"] * 
              (1 - exp(theta[, "b1"] * x)))
      }
    }
  } else if (algObj$model == "WE") {
    if (algObj$shape == "simple") {
      survfun <- function(theta, x) {
        exp(-(theta[, "b1"] * x)^theta[, "b0"])
      }      
    } else if (algObj$shape == "Makeham") {
      survfun <- function(theta, x) {
        exp(-theta[, "c"] * x - (theta[, "b1"] * x)^theta[, "b0"])
      }
    } else {
      survfun <- function(theta, x) {
        exp(exp(theta[, "a0"]) / theta[, "a1"] * (exp(-theta[, "a1"] * x) - 1) -
              theta[, "c"] * x - (theta[, "b1"] * x)^theta[, "b0"])
      }
    }
  } else if (algObj$model == "LO") {
    if (algObj$shape == "simple") {
      survfun <- function(theta, x) {
        (1 + theta[, "b2"] * exp(theta[, "b0"]) / theta[, "b1"] * 
           (exp(theta[, "b1"] * x) - 1))^(-1 / theta[, "b2"])
      }
    } else if (algObj$shape == "Makeham") {
      survfun <- function(theta, x) {
        exp(-theta[, "c"] * x) * (1 + theta[, "b2"] * exp(theta[, "b0"]) / 
                                    theta[, "b1"] * (exp(theta[, "b1"] * x) - 1))^(-1 / theta[, "b2"])
      }
    } else {
      survfun <- function(theta, x) {
        exp(exp(theta[, "a0"]) / theta[, "a1"] * (exp(-theta[, "a1"] * x) - 1) -
              theta[, "c"] * x) * (1 + theta[, "b2"] * 
                                     exp(theta[, "b0"]) / theta[, "b1"] * 
                                     (exp(theta[, "b1"] * x) - 1))^(-1 / theta[, "b2"])
      }
    }
  }
  return(survfun)
}

.DefineSurvNumeric <- function(algObj) {
  if (algObj$model == "EX") {
    survfun <- function(theta, x) exp(- c(theta) * x)
  } else if (algObj$model == "GO") {
    if (algObj$shape == "simple") {
      survfun <- function(theta, x) {
        exp(exp(theta["b0"]) / theta["b1"] * 
              (1 - exp(theta["b1"] * x)))
      }
    } else if (algObj$shape == "Makeham") {
      survfun <- function(theta, x) {
        exp(-theta["c"] * x + exp(theta["b0"]) / theta["b1"] * 
              (1 - exp(theta["b1"] * x)))
      }
    } else {
      survfun <- function(theta, x) {
        exp(exp(theta["a0"]) / theta["a1"] * (exp(-theta["a1"] * x) - 1) - 
              theta["c"] * x + exp(theta["b0"]) / theta["b1"] * 
              (1 - exp(theta["b1"] * x)))
      }
    }
  } else if (algObj$model == "WE") {
    if (algObj$shape == "simple") {
      survfun <- function(theta, x) {
        exp(-(theta["b1"] * x)^theta["b0"])
      }      
    } else if (algObj$shape == "Makeham") {
      survfun <- function(theta, x) {
        exp(-theta["c"] * x - (theta["b1"] * x)^theta["b0"])
      }
    } else {
      survfun <- function(theta, x) {
        exp(exp(theta["a0"]) / theta["a1"] * (exp(-theta["a1"] * x) - 1) -
              theta["c"] * x - (theta["b1"] * x)^theta["b0"])
      }
    }
  } else if (algObj$model == "LO") {
    if (algObj$shape == "simple") {
      survfun <- function(theta, x) {
        (1 + theta["b2"] * exp(theta["b0"]) / theta["b1"] * 
           (exp(theta["b1"] * x) - 1))^(-1 / theta["b2"])
      }
    } else if (algObj$shape == "Makeham") {
      survfun <- function(theta, x) {
        exp(-theta["c"] * x) * (1 + theta["b2"] * exp(theta["b0"]) / 
                                  theta["b1"] * (exp(theta["b1"] * x) - 1))^(-1 / theta["b2"])
      }
    } else {
      survfun <- function(theta, x) {
        exp(exp(theta["a0"]) / theta["a1"] * (exp(-theta["a1"] * x) - 1) -
              theta["c"] * x) * (1 + theta["b2"] * 
                                   exp(theta["b0"]) / theta["b1"] * 
                                   (exp(theta["b1"] * x) - 1))^(-1 / theta["b2"])
      }
    }
  }
  return(survfun)
}


# B.3) Functions to calculate pace-shape measures:
# ------------------------------------------------
# life expectancy:
.CalcEx <- function(Sx, dx) sum(Sx * dx) / Sx[1]

# Keyfitz's entropy:
.CalcHx <- function(Sx, dx) {
  Sx1 <- Sx[Sx > 0]; Sx1 <- Sx1 / Sx1[1]
  -sum(Sx1 * log(Sx1) * dx) / sum(Sx1 * dx)
}

# Gini coefficient:
.CalcGx <- function(Sx, dx) {
  Sx <- Sx / Sx[1]
  Sx <- Sx[Sx > 0]
  return(1 - 1 / sum(Sx * dx) * sum(Sx^2 * dx))
}

# Coefficient of variation:
.CalcCVx <- function(x, Sx, dx) {
  Sx <- Sx / Sx[1]
  idd <- which(Sx > 0)
  Sx <- Sx[idd]
  x <- (x - x[1])[idd]
  dS <- -diff(Sx)
  dS <- dS / sum(dS)
  ex <- sum(Sx * dx)
  return(sqrt(sum((x[-length(x)] + dx/2 - ex)^2 * dS)) / ex)
}

# Function to calculate final pace-shape measures:
.CalcPS <- function(object) {
  xv <- object$x
  dx <- xv[2] - xv[1]
  PSqMat <- lapply(names(object$covs$cat), function(ss) {
    idcov <- grep(ss, colnames(object$theta))
    thetaMat <- object$theta[, idcov]
    colnames(thetaMat) <- defTheta$name
    surv <- apply(thetaMat, 1, function(th) .CalcSurv(th, xv))
    Ex <- apply(surv, 2, .CalcEx, dx = dx)
    Hx <- apply(surv, 2, .CalcHx, dx = dx)
    Epx <- - log(Hx)
    Gx <- apply(surv, 2, .CalcGx, dx = dx)
    PSq <- rbind(c(mean(Ex), quantile(Ex, c(0.025, 0.975))),
                 c(mean(Hx), quantile(Hx, c(0.025, 0.975))),
                 c(mean(Epx), quantile(Epx, c(0.025, 0.975), na.rm = T)),
                 c(mean(Gx), quantile(Gx, c(0.025, 0.975))))
    dimnames(PSq) <- list(c("LifeExp", "LifeTableEntropy", "LifespanEqual",
                            "Gini"), 
                          c("Mean", "2.5%", "97.5%"))
    return(PSq)
  })
  names(PSqMat) <- names(object$cov$cat)
  return(PSqMat)
}

# d) Random sampling function for ages at death:
# SampleAges <- function(n, th, dx = 0.1, lower = 0, upper = Inf) {
#   xv <- seq(lower, ifelse(upper == Inf, 1000, upper), dx)
#   Fx <- 1 - .CalcSurv(th, xv)
#   if (max(Fx) == 1) {
#     Fx <- Fx[1:which(Fx == 1)[1]]
#   }
#   xr <- xv[findInterval(runif(n, min = min(Fx), max = max(Fx)), Fx)]
#   return(xr)
# }

# B.4) Truncated distribution functions:
# --------------------------------------
# Truncated normal:
.rtnorm <- function(n, mean, sd, lower = -Inf, upper = Inf) {
  Flow <- pnorm(lower, mean, sd)
  Fup <- pnorm(upper, mean, sd)
  ru <- runif(n, Flow, Fup)
  rx <- qnorm(ru, mean, sd)
  return(rx)
}

.dtnorm <- function(x, mean, sd, lower = -Inf, upper = Inf, log = FALSE) {
  Flow <- pnorm(lower, mean, sd)
  Fup <- pnorm(upper, mean, sd)
  densx <- dnorm(x, mean, sd) / (Fup - Flow)
  if (log) densx <- log(densx)
  return(densx)
}

.ptnorm <- function(q, mean, sd, lower = -Inf, upper = Inf, log = FALSE) {
  p <- (pnorm(q, mean, sd) - pnorm(lower, mean, sd)) / 
    (pnorm(upper, mean, sd) - pnorm(lower, mean, sd))
  if (log) {
    p <- log(p)
  }
  return(p)
}

.qtnorm <- function (p, mean = 0, sd = 1, lower = -Inf, upper = Inf) {
  p2 <- (p) * (pnorm(upper, mean, sd) - pnorm(lower, mean, sd)) + 
    pnorm(lower, mean, sd)
  q <- qnorm(p2, mean, sd)
  return(q)
}

# Truncated Gamma:
.rtgamma <- function(n, shape, rate = 1, scale = 1/rate, lower = -Inf, 
                    upper = Inf) {
  Flow <- pgamma(lower, shape, scale = scale)
  Fup <- pgamma(upper, shape, scale = scale)
  ru <- runif(n, Flow, Fup)
  rx <- qgamma(ru, shape, scale = scale)
  return(rx)
}

.dtgamma <- function(x, shape, rate = 1, scale = 1/rate, lower = -Inf, 
                    upper = Inf, log = FALSE) {
  Flow <- pgamma(lower, shape, scale = scale)
  Fup <- pgamma(upper, shape, scale = scale)
  densx <- dgamma(x, shape, scale = scale) / (Fup - Flow)
  if (log) densx <- log(densx)
  return(densx)
}

.ptgamma <- function(q, shape, rate = 1, scale = 1/rate, lower = -Inf, 
                    upper = Inf, log = FALSE) {
  p <- (pgamma(q, shape, scale = scale) - 
          pgamma(lower, shape, scale = scale)) / 
    (pgamma(upper, shape, scale = scale) - 
       pgamma(lower, shape, scale = scale))
  if (log) {
    p <- log(p)
  }
  return(p)
}

.qtgamma <- function (p, shape, rate = 1, scale = 1/rate, lower = -Inf, 
                     upper = Inf) {
  p2 <- (p) * (pgamma(upper, shape, scale = scale) - 
                 pgamma(lower, shape, scale = scale)) + 
    pgamma(lower, shape, scale = scale)
  q <- qgamma(p2, shape, scale = scale)
  return(q)
}

# B.5) Likelihoods and posteriors:
# --------------------------------
# a) Likelihood for mortality parameters:
.CalcParLike <- function(parObj, ...) UseMethod(".CalcParLike")

.CalcParLike.noMinAge <- function(parObj, ageObj, parCovObj, datObj, algObj) {
  # All individuals
  parLike <- exp(parCovObj$gamma) * 
    (.CalcCumHaz(parCovObj$theta, ageObj$xt) -
    .CalcCumHaz(parCovObj$theta, ageObj$xl)) +
    (log(.CalcMort(parCovObj$theta, ageObj$xl)) + parCovObj$gamma) * 
    ageObj$ucens
  return(parLike)
}

.CalcParLike.minAge <- function(parObj, ageObj, parCovObj, datObj, algObj) {
  # All individuals
  parLike <- rep(0, datObj$n)
  # Survival xl above minAge:
  parLike[ageObj$ma$xlu] <- -parObj$eta * algObj$minAge - 
    exp(parCovObj$gamma[ageObj$ma$xlu]) * 
    .CalcCumHaz(parCovObj$theta[ageObj$ma$xlu, ], 
                   ageObj$xl[ageObj$ma$xlu] - algObj$minAge) +
    (log(.CalcMort(parCovObj$theta[ageObj$ma$xlu, ], 
                   ageObj$xl[ageObj$ma$xlu] - algObj$minAge)) + 
       parCovObj$gamma[ageObj$ma$xlu]) * ageObj$ucens[ageObj$ma$xlu]
  
  # Survival xl below minAge:
  parLike[ageObj$ma$xll] <- (-parObj$eta * ageObj$xl[ageObj$ma$xll]) +
    log(parObj$eta) * ageObj$ucens[ageObj$ma$xll]
  
  # Truncation below minAge:
  parLike[ageObj$ma$xtl] <- parLike[ageObj$ma$xtl] +
    parObj$eta * ageObj$xt[ageObj$ma$xtl]
  
  # Truncation above minAge:
  parLike[ageObj$ma$xtu] <- parLike[ageObj$ma$xtu] + 
    exp(parCovObj$gamma[ageObj$ma$xtu]) *
    (.CalcCumHaz(parCovObj$theta[ageObj$ma$xtu, ], ageObj$xt[ageObj$ma$xtu] -
                  algObj$minAge)) + parObj$eta * algObj$minAge
    
  return(parLike)
}

# b) parameter posterior:
.CalcParPost <- function(parObj, fullParObj, parLike) {
  # Sum to calculate posterior:  
  mortPostSum <- sum(parLike) + 
    sum(.dtnorm(c(parObj$theta), c(fullParObj$theta$priorMu), 
                c(fullParObj$theta$priorSd), 
                low = c(fullParObj$theta$low), log = TRUE))
  if ("gamma" %in% names(fullParObj)) {
    mortPostSum <- mortPostSum + 
      sum(.dtnorm(parObj$gamma, fullParObj$gamma$priorMu,
                  fullParObj$gamma$priorSd, log = TRUE))
  }
  if ("eta" %in% names(fullParObj)) {
    mortPostSum <- mortPostSum + 
      .dtnorm(parObj$eta, fullParObj$eta$priorMu,
              fullParObj$eta$priorSd, 
              low = c(fullParObj$eta$low), log = TRUE)
  }
  return(mortPostSum)
}

# c) Posterior for age at death or censoring:
.CalcAgePost <- function(ageObj, ...) UseMethod(".CalcAgePost")

.CalcAgePost.noMinAge <- function(ageObj, parCovObj, datObj) {
  agePost <- exp(parCovObj$gamma) * 
    (-.CalcCumHaz(parCovObj$theta, ageObj$xl)) +
    (log(.CalcMort(parCovObj$theta, ageObj$xl)) + 
       parCovObj$gamma) * ageObj$ucens
  return(agePost)
}

.CalcAgePost.minAge <- function(ageObj, parCovObj, datObj) {
  agePost <- rep(0, datObj$n)
  # Survival xl above minAge:
  agePost[ageObj$ma$xlu] <- -parObj$eta * algObj$minAge - 
    exp(parCovObj$gamma[ageObj$ma$xlu]) * 
    .CalcCumHaz(parCovObj$theta[ageObj$ma$xlu, ], 
                ageObj$xl[ageObj$ma$xlu] - algObj$minAge) +
    (log(.CalcMort(parCovObj$theta[ageObj$ma$xlu, ], 
                   ageObj$xl[ageObj$ma$xlu] - algObj$minAge)) + 
       parCovObj$gamma[ageObj$ma$xlu]) * ageObj$ucens[ageObj$ma$xlu]
  
  # Survival xl below minAge:
  agePost[ageObj$ma$xll] <- (-parObj$eta * ageObj$xl[ageObj$ma$xll]) +
    log(parObj$eta) * ageObj$ucens[ageObj$ma$xll]
  
  return(agePost)
}

# d) Priors for times of birth:
.CalcBirthPrior <- function(ageObj, datObj) {
  bPrior <- rep(0, datObj$n)
  bPrior[datObj$idNoBirth] <- dunif(ageObj$bi[datObj$idNoBirth],
                                    datObj$bil[datObj$idNoBirth],
                             datObj$biu[datObj$idNoBirth], log = TRUE)
  return(bPrior)
}

# e) Posterior for times of birth:
.CalcBirthPost <- function(ageObj, agePost, datObj) {
  if (length(datObj$idNoBirth) > 0) {
    post <- agePost + .CalcBirthPrior(ageObj, datObj)
  } else {
    post <- 0
  }
  return(post)
}

# --------------------
# B.6) MCMC functions:
# --------------------
# a) Function to update jumps:
.UpdateJumps <- function(parJumps, jumpsMat, iter, iterUpd, updTarg,
                         type = "theta") {
  if (type == 'theta') {
    jumps <- parJumps$theta
    if (ncol(jumpsMat$theta) > 1) {
      updRate <- apply(jumpsMat$theta[iter - ((iterUpd - 1):0), ], 2, sum) / 
        iterUpd  
    } else {
      updRate <- sum(jumpsMat$theta[iter - ((iterUpd - 1):0), ]) / 
        iterUpd  
    }
    updRate[updRate == 0] <- 1e-2
    if (is.matrix(jumps)) {
      jumps <- jumps * 
        matrix(updRate, nrow(jumps), ncol(jumps)) / updTarg
    } else {
      jumps <- jumps * updRate / updTarg
    }
    parJumps$theta <- jumps
  } else if (type == "gamma") {
    jumps <- parJumps$gamma
    if (fullParObj$gamma$len == 1) {
      updRate <- sum(jumpsMat$gamma[iter - ((iterUpd - 1):0), ]) / iterUpd
    } else {
      updRate <- apply(jumpsMat$gamma[iter - ((iterUpd - 1):0), ], 2, sum) / 
        iterUpd
    }
    updRate[updRate == 0] <- 1e-2
    jumps <- jumps * updRate / updTarg
    parJumps$gamma <- jumps
  } else {
    updRate <- sum(jumpsMat$eta[iter - ((iterUpd - 1):0), ]) / iterUpd
    parJumps$eta <- parJumps$eta * updRate / updTarg
  }
  return(parJumps)
}

# b) MCMC function:
.RunMCMC <- function(sim, saveAll = TRUE, printProgr = FALSE, UpdJumps = TRUE,
                     parJumps = NA) {
  # fname <- sprintf("~/FERNANDO/PROJECTS/4.PACKAGES/BaSTA.ZIMS/logs/logSim%s.txt",
  #                  sim)
  # file.create(fname, overwrite = TRUE)
  
  #rm(".Random.seed", envir = .GlobalEnv); runif(1)
  # Index for thinned sequence:
  keep <- seq(algObj$burnin, algObj$niter, algObj$thinning)
  
  # -------------------
  # INITIAL PARAMETERS:
  # -------------------
  #cat("Start INITIAL PARAMETERS\n", file = fname, append = TRUE)
  # Birth times:
  agesNow <- .SampleAgeObj(ageObj, datObj, algObj)
  
  # Parameters:
  parsNow <- parObj
  parCovNow <- parCovObj
  
  if (sim > 1) {
    rm(".Random.seed", envir = .GlobalEnv); runif(1)
    thJit <- 0
    mortPostNow <- NA
    while((is.na(mortPostNow) | mortPostNow == -Inf) & thJit <= 50) {
      thJit <- thJit + 1
      parsNow <- .JitterPars(parObj, fullParObj)
      parCovNow <- .CalcCovPars(parsNow, parCovNow, covObj, datObj, 
                                type = "both")
      parLikeNow <- .CalcParLike(parsNow, agesNow, parCovNow, datObj, algObj)
      mortPostNow <- .CalcParPost(parsNow, fullParObj, parLikeNow)
    }
  } else {
    parLikeNow <- .CalcParLike(parsNow, agesNow, parCovNow, datObj, algObj)
    mortPostNow <- .CalcParPost(parsNow, fullParObj, parLikeNow)
  }
  
  #cat("End INITIAL PARAMETERS\n", file = fname, append = TRUE)
  # ----------------
  # START THE MODEL:
  # ----------------
  #cat("Start START THE MODEL\n", file = fname, append = TRUE)
  # Age estimation:
  agePostNow <- .CalcAgePost(agesNow, parCovNow, datObj)
  
  # Birth estimation:
  bPostNow <- .CalcBirthPost(agesNow, agePostNow, datObj)
  
  #cat("End START THE MODEL\n", file = fname, append = TRUE)
  # ---------------------
  # JUMPS FOR METROPOLIS:
  # ---------------------
  #cat("Start JUMPS FOR METROPOLIS\n", file = fname, append = TRUE)
  if (UpdJumps) {
    # Start jumps for Metropolis algorithm:
    niter <- ifelse(algObj$niter / 4 < 5000, 5000, ceiling(algObj$niter / 4))
    niter <- 5000
    burnin <- niter
    parJumps <- .SetParJumps(parObj, fullParObj)
    jumpsMat <- .CreateOutputMat(fullParObj, datObj, niter, burnin, 
                                 type = 'jumps')
    iterUpd <- 50
    updTarg <- 0.25
    updSeq <- seq(1, burnin, iterUpd)
    nUpdSeq <- length(updSeq)
    jumpsParMat <- jumpsMat
    jumpsParMat$theta <- jumpsParMat$theta[updSeq, ]
    if (inherits(parObj, c("fused", "propHaz"))) {
      jumpsParMat$gamma <- jumpsParMat$gamma[updSeq, ]
    }
    if (algObj$minAge > 0) {
      jumpsParMat$eta <- jumpsParMat$eta[updSeq]
    }
  }
  #cat("End JUMPS FOR METROPOLIS\n", file = fname, append = TRUE)
  
  # --------------------------
  # CONSTRUCT OUTPUT MATRICES:
  # --------------------------
  #cat("Start CONSTRUCT OUTPUT MATRICES\n", file = fname, append = TRUE)
  # Mortality parameters:
  parsOut <- .CreateOutputMat(fullParObj, datObj, niter, burnin, 
                              type = 'params', parObj = parsNow)
  
  # Birth:
  if (length(datObj$idNoBirth) > 0) {
    birthMat <- .CreateOutputMat(fullParObj, datObj, niter, burnin, 
                                 type = 'ages')
  } else {
    birthMat <- NA
  }
  
  # Likelihood and posterior:
  likePostMat <- matrix(0, niter, 2, dimnames = list(NULL, c("Like", "Post")))
  likePostMat[1, ] <- c(sum(parLikeNow), mortPostNow)

  #cat("End CONSTRUCT OUTPUT MATRICES\n", file = fname, append = TRUE)
  
  # ---------
  # RUN MCMC:
  # ---------
  if(printProgr) progrBar <- txtProgressBar(min = 2, max = niter, style = 3)
  
  Start <- Sys.time()
  for (iter in 2:niter) {
    # -------------------------------
    # 1. Update mortality parameters:
    # -------------------------------
    for (tg in c("theta", 'gamma', "eta")) {
      if (tg %in% names(fullParObj)) {
        ntg <- fullParObj[[tg]]$len
        for (pp in 1:ntg) {
          napost <- TRUE
          nat <- 0
          while (napost & nat <= 2) {
            nat <- nat + 1
            parsNew <- .SamplePars(parsNow, parJumps, fullParObj, pp = pp, 
                                   type = tg)
            parCovNew <- .CalcCovPars(parsNew, parCovNow, covObj, datObj, 
                                      type = tg)
            parLikeNew <- .CalcParLike(parsNew, agesNow, parCovNew, datObj,
                                       algObj)
            mortPostNew <- .CalcParPost(parsNew, fullParObj, parLikeNew)
            hRatio <- .CalcHastRatio(parsNow, parsNew, parJumps, type = tg,
                                     pp)
            napost <- ifelse(is.na(mortPostNew), TRUE, FALSE)
          }
          postRatio <- exp(mortPostNew - mortPostNow + hRatio)
          if (!is.na(postRatio) & postRatio > runif(1)) {
            parsNow <- parsNew
            parCovNow <- parCovNew
            parLikeNow <- parLikeNew
            mortPostNow <- mortPostNew
            if (UpdJumps & iter <= burnin) {
              jumpsMat[[tg]][iter, pp] <- 1
            }
          }
        }
        agePostNow <- .CalcAgePost(agesNow, parCovNow, datObj)
        bPostNow <- .CalcBirthPost(agesNow, agePostNow, datObj)
      }
    }
    
    # --------------------------
    # 2. Propose times of birth:
    # --------------------------
    if (datObj$nNoBi > 0) {
      agesNew <- .SampleAgeObj(agesNow, datObj, algObj)
      agePostNew <- .CalcAgePost(agesNew, parCovNow, datObj)
      bPostNew <- .CalcBirthPost(agesNew, agePostNew, datObj)
      
      r <- exp(bPostNew - bPostNow)[datObj$idNoBirth]
      
      idUpd3 <- datObj$idNoBirth[r > runif(datObj$nNoBi)]
      idUpd3 <-idUpd3[!(is.na(idUpd3))]
      if (length(idUpd3) > 0) {
        agesNow$bi[idUpd3] <- agesNew$bi[idUpd3]
        agesNow$xl[idUpd3] <- agesNew$xl[idUpd3]
        agesNow$xt[idUpd3] <- agesNew$xt[idUpd3]
        agePostNow[idUpd3] <- agePostNew[idUpd3]
        bPostNow[idUpd3] <- bPostNew[idUpd3]
        if (algObj$minAge > 0) {
          agesNow$ma$xlu <- which(agesNow$xl >= algObj$minAge)
          agesNow$ma$xtu <- which(agesNow$xt >= algObj$minAge)
          agesNow$ma$xll <- which(agesNow$xl < algObj$minAge)
          agesNow$ma$xtl <- which(agesNow$xt < algObj$minAge)
        }
      }
      parLikeNow <- .CalcParLike(parsNow, agesNow, parCovNow, datObj, algObj)
      mortPostNow <- .CalcParPost(parsNow, fullParObj, parLikeNow)
    }
    
    # -------------------------------------
    # 3. Dynamic Metropolis to update jumps:
    # --------------------------------------
    if (UpdJumps) {
      if (iter %in% updSeq) {
        parJumps <- .UpdateJumps(parJumps, jumpsMat, iter, iterUpd, updTarg)
        idpar <- which(updSeq == iter)
        if (is.matrix(jumpsParMat$theta)) {
          jumpsParMat$theta[idpar, ] <- c(parJumps$theta)
        } else {
          jumpsParMat$theta[idpar] <- c(parJumps$theta)
        }
        if (iter == max(updSeq)) {
          if (is.matrix(jumpsParMat$theta)) {
            parForMeanJump <- jumpsParMat$theta[floor(nUpdSeq / 2):nUpdSeq, ]
            meanParJump <- apply(parForMeanJump, 2, mean)
          } else {
            parForMeanJump <- jumpsParMat$theta[floor(nUpdSeq / 2):nUpdSeq]
            meanParJump <- mean(parForMeanJump, 2)
          }
          
          parJumps$theta <- matrix(meanParJump, 
                                   fullParObj$theta$dim[1], 
                                   fullParObj$theta$dim[2], 
                                   dimnames = 
                                     dimnames(fullParObj$theta$start))
        }
        if (inherits(parObj, c("fused", "propHaz"))) {
          parJumps <- .UpdateJumps(parJumps, jumpsMat, iter, iterUpd, updTarg,
                                   type = "gamma")
          if (is.matrix(jumpsParMat$gamma)) {
            jumpsParMat$gamma[idpar, ] <- c(parJumps$gamma)
          } else {
            jumpsParMat$gamma[idpar] <- c(parJumps$gamma)
          }
          if (iter == max(updSeq)) {
            if (is.matrix(jumpsParMat$gamma)) {
              parForMeanJump <- jumpsParMat$gamma[floor(nUpdSeq / 2):nUpdSeq, ]
              parJumps$gamma <- apply(parForMeanJump, 2, mean)
              names(parJumps$gamma) <- names(parsNow$gamma)
            } else {
              parForMeanJump <- jumpsParMat$gamma[floor(nUpdSeq / 2):nUpdSeq]
              parJumps$gamma <- mean(parForMeanJump)
            }
          }
        }
        if (inherits(parObj, "minAge")) {
          parJumps <- .UpdateJumps(parJumps, jumpsMat, iter, iterUpd, updTarg,
                                   type = "eta")
          jumpsParMat$eta[idpar] <- c(parJumps$eta)
          if (iter == max(updSeq)) {
            parForMeanJump <- jumpsParMat$eta[floor(nUpdSeq / 2):nUpdSeq]
            parJumps$eta <- mean(parForMeanJump)
          }
        }
      }
    }
    
    # -------------------------------
    # 4. Fill in the output matrices:
    # -------------------------------
    parsOut$theta[iter, ] <- c(parsNow$theta)
    if (inherits(parObj, c("fused", "propHaz"))) {
      parsOut$gamma[iter, ] <- parsNow$gamma
    }
    if (inherits(parObj, "minAge")) {
      parsOut$eta[iter] <- parsNow$eta
    }
    likePostMat[iter, ] <- c(sum(parLikeNow), mortPostNow)
    
    if (iter %in% keep & saveAll) {
      if (datObj$nNoBi > 0) {
        birthMat <- rbind(birthMat, agesNow$bi[datObj$idNoBirth])
      } else {
        birthMat <- NA
      }
    } 
    if(printProgr) setTxtProgressBar(progrBar, iter)
    
    # -------------------------------
    # 5. Write log:
    # -------------------------------
    # if (iter %in% logsec) {
    #   cat(sprintf("Simulation: %s\nIteration: %s\nTime:%s\n", sim, iter, 
    #               Sys.time()), file = fname, append = FALSE)
    # }
  }
  End <- Sys.time()
  if (!saveAll) {
    birthMat <- NA
  }
  # EXTRACT RESULTS:
  return(list(params = parsOut, birth = birthMat, likePost = likePostMat,
              keep = keep, jumps = parJumps))
}

# B.7) Functions to manage MCMC outputs:
# --------------------------------------
# a) Extract thinned sequences from multiple runs, calculate coefficients,
#    DIC and quantiles for mortality and survival:
.ExtractParalOut <- function(bastaOut, keep, fullParObj, covsNames, nsim, 
                             datObj, algObj, defTheta, .CalcMort, 
                             .CalcMort.numeric, .CalcMort.matrix, 
                             .CalcSurv, .CalcSurv.matrix, 
                             .CalcSurv.numeric, covObj) {
  parMat <- bastaOut[[1]]$params$theta[keep, ]
  parnames <- fullParObj$theta$names
  theMat <- parMat
  likePost <- bastaOut[[1]]$likePost[keep, ]
  birthMat <- bastaOut[[1]]$birth
  if (covsNames$class %in% c("propHaz", "fused")) {
    parMat <- cbind(parMat, bastaOut[[1]]$params$gamma[keep, ])
    parnames <- c(parnames, fullParObj$gamma$names)
  }
  if (inherits(fullParObj, "minAge")) {
    parMat <- cbind(parMat, bastaOut[[1]]$params$eta[keep])
    parnames <- c(parnames, "eta")
  }
  for (i in 2:nsim) {
    if (is.matrix(theMat)) {
      theMat <- rbind(theMat, bastaOut[[i]]$params$theta[keep, ])
    } else {
      theMat <- c(theMat, bastaOut[[i]]$params$theta[keep, ])
    }
    if (covsNames$class == "inMort") {
      pmat <- bastaOut[[i]]$params$theta[keep, ]
      if (inherits(fullParObj, "minAge")) {
        pmat <- cbind(pmat, bastaOut[[i]]$params$eta[keep])
      }
    } else {
      pmat <- cbind(bastaOut[[i]]$params$theta[keep, ],
                    bastaOut[[i]]$params$gamma[keep, ])
      if (inherits(fullParObj, "minAge")) {
        pmat <- cbind(pmat, bastaOut[[i]]$params$eta[keep])
      }
    }
    if (is.matrix(parMat)) {
      parMat <- rbind(parMat, pmat)
    } else {
      parMat <- c(parMat, pmat)
      parMat <- matrix(parMat, ncol = 1)
    }
    likePost <- rbind(likePost, bastaOut[[i]]$likePost[keep, ])
    birthMat <- rbind(birthMat, bastaOut[[i]]$birth)
  }
  colnames(parMat) <- parnames
  coeffs <- cbind(apply(parMat, 2, mean), apply(parMat, 2, sd), 
                  t(apply(parMat, 2, quantile, c(0.025, 0.975))))
  colnames(coeffs) <- c("Mean", "SE", "2.5%", "97.5%")
  if (nsim > 1) {
    nthin <- length(keep)
    idSims <- rep(1:algObj$nsim, each = nthin)
    Means <- apply(parMat, 2, function(x) 
      tapply(x, idSims, mean))
    Vars <- apply(parMat, 2, function(x) 
      tapply(x, idSims, var))
    meanall <- apply(Means, 2, mean)
    B <- nthin / (algObj$nsim - 1) * apply(t((t(Means) - meanall)^2), 2, sum)
    W <- 1 / algObj$nsim * apply(Vars, 2, sum)
    Varpl <- (nthin - 1) / nthin * W + 1 / nthin * B
    Rhat <- sqrt(Varpl / W)
    Rhat[Varpl==0] <- 1
    conv <- cbind(B, W, Varpl, Rhat)
    rownames(conv) <- colnames(parMat)
    coeffs <- cbind(coeffs, conv[, 'Rhat'])
    colnames(coeffs) <- c(colnames(coeffs)[-ncol(coeffs)], "PotScaleReduc")
    idnconv <- which(conv[, 'Rhat'] > 1.1)
    if (length(idnconv) == 0) {
      # DIC:
      Dave <- mean(- 2 * likePost[, 'Like'])
      pD <- 1/2 * var(-2 * likePost[, 'Like'])
      DIC <- pD + Dave
      Dmode <- Dave - 2 * pD
      k <- fullParObj$theta$len + 
        ifelse("gamma" %in% names(fullParObj), fullParObj$gamma$len, 0) +
        ifelse("eta" %in% names(fullParObj), fullParObj$eta$len, 0)
      modSel <- c(Dave, Dmode, pD, k, DIC)
      names(modSel) <- c("D.ave", "D.mode", "pD", "k", "DIC")
      convmessage <- "All parameters converged properly.\n"
    } else {
      modSel <- NA
      convmessage <- "Convergence not reached for some parameters.\n"
    }
  } else {
    modSel <- NA
    convmessage <- "Convergence not calculated due to\ninsuficcient number of simulations.\n"
  }
  cat(convmessage)
  # Kullback-Leibler:
  kulLeib <- .CalcKulbackLeibler(coeffs, covObj, defTheta, fullParObj, 
                                 algObj)
  # Mortality, survival and density quantiles:
  xv <- seq(0, max(datObj$xl) * 10, 0.1)
  tempOut <- list(params = parMat, theta = theMat)
  mortQuan <- .CalcDemoFunQuan(tempOut, xv, covsNames, defTheta, 
                               funtype = "mort", .CalcMort, 
                               .CalcMort.numeric, .CalcMort.matrix, 
                               .CalcSurv, .CalcSurv.matrix, 
                               .CalcSurv.numeric)
  survQuan <- .CalcDemoFunQuan(tempOut, xv, covsNames, defTheta, 
                               funtype = "surv", .CalcMort, 
                               .CalcMort.numeric, .CalcMort.matrix, 
                               .CalcSurv, .CalcSurv.matrix, 
                               .CalcSurv.numeric)
  densQuan <- .CalcDemoFunQuan(tempOut, xv, covsNames, defTheta, 
                               funtype = "dens", .CalcMort, 
                               .CalcMort.numeric, .CalcMort.matrix, 
                               .CalcSurv, .CalcSurv.matrix, 
                               .CalcSurv.numeric)
  PSQuan <- .CalcDemoFunQuan(tempOut, xv, covsNames, defTheta, 
                               funtype = "PS", .CalcMort, 
                               .CalcMort.numeric, .CalcMort.matrix, 
                               .CalcSurv, .CalcSurv.matrix, 
                               .CalcSurv.numeric)
  cuts <- list()
  for (nta in names(survQuan)) {
    cuts[[nta]] <- which(survQuan[[nta]][1, ] > 0.05)
  }
  out <- list(params = parMat, theta = theMat, coefficients = coeffs, 
              names = parnames, DIC = modSel, KullbackLeibler = kulLeib, 
              PS = PSQuan, mort = mortQuan, surv = survQuan, dens = densQuan, 
              x = xv, cuts = cuts, convergence = conv, 
              convmessage = convmessage)
  
  return(out)
}

# b) Calculcate Kulback-Leibler discrepancies between parameters:
.CalcKulbackLeibler <- function(coef, covObj, defTheta, fullParObj, algObj,
                                datObj) {
  if (!is.null(covObj$cat) & 
      !(length(covObj$cat) == 2 & class(covObj)[1] == "propHaz")) {
    if (length(covObj$cat) > 1) {
      if (inherits(covObj, c("fused", "inMort"))) {
        parNames <- defTheta$name
        nPar <- defTheta$length
        low <- defTheta$low
        nCat <- length(covObj$cat)
        namesCat <- names(covObj$cat)
      } else {
        parNames <- "gamma"
        nCat <- length(covObj$cat) - 1
        namesCat <- names(covObj$cat)[-1]
        nPar <- 1
        low <- -Inf
      }
      nComb <- (nCat - 1)^2 - ((nCat - 1)^2 - (nCat - 1)) / 2
      covComb1 <- c()
      covComb2 <- c()
      klMat1 <- matrix(0, nPar, nComb, dimnames = list(parNames, NULL))
      klMat2 <- klMat1
      comb <- 0
      for (i in 1:nCat) {
        for (j in 1:nCat) {
          if (i > j) {
            comb <- comb + 1
            covComb1 <- c(covComb1, 
                          sprintf("%s - %s", namesCat[i], namesCat[j]))
            covComb2 <- c(covComb2, 
                          sprintf("%s - %s", namesCat[j], namesCat[i]))
            for (p in 1:nPar) {
              if (inherits(covObj, c("fused", "inMort"))) {
                idP <- sapply(c(i, j), function(ij) 
                  which(rownames(coef) == 
                          sprintf("%s.%s", parNames[p], namesCat[ij])))
              } else {
                idP <- sapply(c(i, j), function(ij) 
                  which(rownames(coef) == namesCat[ij]))
              }
              parRan <- range(sapply(1:2, 
                                     function(pp) .qtnorm(c(0.001, 0.999), 
                                                          coef[idP[pp], 1],
                                                          coef[idP[pp], 2], 
                                                          lower = low[p])))
              parVec <- seq(parRan[1], parRan[2], length = 100)
              dp <- parVec[2] - parVec[1]
              parDens <- sapply(1:2, function(pp) 
                .dtnorm(seq(parRan[1], parRan[2], length = 100), 
                        coef[idP[pp], 1], coef[idP[pp], 2], lower = low[p]))
              klMat1[p, comb] <- sum(parDens[, 1] * 
                                       log(parDens[, 1] / parDens[, 2]) * dp)
              klMat2[p, comb] <- sum(parDens[, 2] * 
                                       log(parDens[, 2] / parDens[, 1]) * dp)
            }
          }
        }
      }
      colnames(klMat1) <- covComb1
      colnames(klMat2) <- covComb2
      qKlMat1 <- (1 + (1 - exp(-2 * klMat1)^(1 / 2))) / 2
      qKlMat2 <- (1 + (1 - exp(-2 * klMat2)^(1 / 2))) / 2
      mqKl <- (qKlMat1 + qKlMat2) / 2
      outList <- list(kl1 = klMat1, kl2 = klMat2, qkl1 = qKlMat1, 
                      qkl2 = qKlMat2, mqKl = mqKl)
    } else {
      outList <- "Not calculated"
    }
  } else {
    outList <- "Not calculated"
  }
  return(outList)
}

# c) Functions to calculate survival and mortality quantiles:
.CalcDemoFunQuan <- function(out, x, covsNames, defTheta, 
                             funtype = "mort", .CalcMort, 
                             .CalcMort.numeric, .CalcMort.matrix, 
                             .CalcSurv, .CalcSurv.matrix, 
                             .CalcSurv.numeric) {
  covinf <- list(th = list(), ga = list())
  fullm <- list(th = list(), ga = list())
  if (covsNames$class == "inMort") {
    fullm$th <- out$theta
    covinf$th$num <- ifelse(length(covsNames$cat) == 0, 0, 
                            length(covsNames$cat))
    if (is.na(covsNames$cat)[1]) {
      covinf$th$name <- ""
    } else {
      covinf$th$name <- names(covsNames$cat)
    }
    fullm$ga$cat <- 0
    fullm$ga$con <- 0
    covinf$ga$caname <- ""
    covinf$ga$coname <- ""
    covinf$ga$canum <- 0
    covinf$ga$conum <- 0
  } else if (covsNames$class == "fused") {
    fullm$th <- out$theta
    covinf$th$num <- ifelse(length(covsNames$cat) == 0, 0, 
                            length(covsNames$cat))
    if (is.na(covsNames$cat)[1]) {
      covinf$th$name <- ""
    } else {
      covinf$th$name <- names(covsNames$cat)
    }
    covinf$ga$caname <- ""
    covinf$ga$canum <- 0
    covinf$ga$coname <- names(covsNames$con)
    covinf$ga$conum <- length(covsNames$con)
    fullm$ga$cat <- 0
    concol <- colnames(out$params)[(ncol(out$theta)+1):ncol(out$params)]
    ideta <- grep("eta", concol)
    if (length(ideta) > 0) {
      concol <- concol[-ideta]
    }
    fullm$ga$con <- out$params[, concol]
  } else {
    fullm$th <- out$theta
    covinf$th$num <- 0
    covinf$th$name <- ""
    if (is.na(covsNames$cat)[1]) {
      covinf$ga$caname <- ""
      covinf$ga$canum <- 0
      fullm$ga$cat <- 0
    } else {
      covinf$ga$caname <- names(covsNames$cat)
      covinf$ga$canum <- length(covsNames$cat)
      concol <- colnames(out$params)[(ncol(out$theta)+1):ncol(out$params)]
      ideta <- grep("eta", concol)
      if (length(ideta) > 0) {
        concol <- concol[-ideta]
      }
      fullm$ga$cat <- cbind(0, out$params[, covinf$ga$caname[-1]])
      colnames(fullm$ga$cat) <- covinf$ga$caname
    }
    if (is.na(covsNames$con)[1]) {
      covinf$ga$coname <- ""
      covinf$ga$conum <- 0
      fullm$ga$con <- 0
    } else {
      covinf$ga$coname <- names(covsNames$con)
      covinf$ga$conum <- length(covsNames$con)
      fullm$ga$con <- out$params[, covinf$ga$coname]
    }
  }
  ideta <- grep("eta", colnames(out$params))
  # if (length(ideta) == 1) {
  #   etav <- out$params[, ideta]
  # } else {
  #   etav <- rep(0, nrow(out$params))
  # }
  demoQuan <- list()
  for (nta in covinf$th$name) {
    if (nta == "") {
      ntal <- "nocov"
      catname <- "nocov"
    } else {
      ntal <- nta
      catname <- nta
    }
    #demoQuan[[ntal]] <- list()
    for (nga in covinf$ga$caname) {
      if (nga == "") {
        ngal <- "nocov"
      } else {
        ngal <- nga
        catname <- nga
      }
      if (is.matrix(fullm$th)) {
        thm <- fullm$th[, grep(nta, colnames(fullm$th))]
      } else {
        thm <- matrix(fullm$th, ncol = 1)
      }
      if (!is.matrix(thm)) thm <- matrix(thm, ncol = 1)
      colnames(thm) <- defTheta$name
      if (covinf$ga$canum <= 1) {
        gaca <- fullm$ga$cat
      } else {
        gaca <- fullm$ga$cat[, grep(nga, colnames(fullm$ga$cat))]
      }
      if (covinf$ga$conum == 0) {
        gaco <- fullm$ga$con
      } else {
        if (length(covObj$con) > 1) {
          gaco <- sum(fullm$ga$con * apply(covObj$propHaz[, names(covObj$cont)],
                                           2, mean))
        } else {
          gaco <- fullm$ga$con * mean(covObj$propHaz[, names(covObj$cont)])
        }
      }
      gam <- gaca + gaco
      if (length(gam) == 1) {
        gam <- rep(gam, nrow(thm))
      }
      if (funtype %in% c("mort", "surv", "dens")) {
        demof <- sapply(1:nrow(thm), function(ii) {
          if (funtype == "mort") {
            demf <- .CalcMort(thm[ii, ], x) * exp(gam[ii])
          } else if (funtype == "surv") {
            demf <- .CalcSurv(thm[ii, ], x)^{exp(gam[ii])}
          } else if (funtype == "dens") {
            demf <- .CalcMort(thm[ii, ], x) * exp(gam[ii]) * 
              .CalcSurv(thm[ii, ], x)^{exp(gam[ii])}
          }
          return(demf)
        })
        demofave <- apply(demof, 1, mean)
        demofci <- apply(demof, 1, quantile, c(0.025, 0.975), na.rm = TRUE)
        demoffin <- rbind(demofave, demofci)
        rownames(demoffin) <- c("Mean", "2.5%", "97.5%")
        demoQuan[[catname]] <- demoffin
      } else {
        Deltax <- x[2] - x[1]
        surv <- sapply(1:nrow(thm), function(ii) {
          sx <- .CalcSurv(thm[ii, ], x)^{exp(gam[ii])}
          return(sx)
        })
        Ex <- apply(surv, 2, .CalcEx, dx = Deltax)
        Hx <- apply(surv, 2, .CalcHx, dx = Deltax)
        Epx <- - log(Hx)
        Gx <- apply(surv, 2, .CalcGx, dx = Deltax)
        PSq <- rbind(c(mean(Ex), quantile(Ex, c(0.025, 0.975))),
                     c(mean(Hx), quantile(Hx, c(0.025, 0.975))),
                     c(mean(Epx), quantile(Epx, c(0.025, 0.975), na.rm = T)),
                     c(mean(Gx), quantile(Gx, c(0.025, 0.975))))
        dimnames(PSq) <- list(c("LifeExp", "LifeTableEntropy", "LifespanEqual",
                                "Gini"), 
                              c("Mean", "2.5%", "97.5%"))
        demoQuan[[catname]] <- list(PS = PSq, Ex = Ex, Hx = Hx, Epx = Epx,
                                    Gx = Gx)
      }
    }
  }
  return(demoQuan)
}


# d) Function to calculate lifetable output:
.CalcLifeTable <- function(bastaOut, lifeTable, covObj, algObj, datObj) {
  cat("Constructing life table... ")
  if (datObj$nNoBi > 0) {
    nthin <- ceiling((algObj$niter - algObj$burnin + 1) / algObj$thinning)
    bMat <- matrix(datObj$bi, datObj$n, nthin * algObj$nsim)
    for (sim in 1:algObj$nsim) {
      bMat[datObj$idNoBirth, 1:nthin + (sim - 1) * nthin] <- 
        t(bastaOut[[sim]]$birth)
    }
  } else {
    bMat <- matrix(datObj$bi, ncol = 1)
  }
  
  # Mean, lower and upper Births:
  Bmat <- cbind(Mean = apply(bMat, 1, mean),
                Median = apply(bMat, 1, quantile, 0.5),
                Lower = apply(bMat, 1, quantile, 0.025),
                Upper = apply(bMat, 1, quantile, 0.975))
  
  # Last age:
  ageLast <- datObj$depart - Bmat
  
  # First age:
  ageFirst <- datObj$entry - Bmat
  ageFirst[ageFirst < 0] <- 0
  
  # Index:
  departType <- rep("D", datObj$n)
  departType[datObj$idCens] <- "C"
  
  # Find maximum age:
  maxAge <- max(ageLast)
  ageFact <- 1
  
  # Verify that maxAge is larger than 3, otherwise change the scale to months:
  # if (maxAge < 3) {
  #   ageFact <- 12
  #   maxAge <- maxAge * ageFact
  #   ageFirst <- ageFirst * ageFact
  #   ageLast <- ageLast * ageFact
  # } else {
  #   ageFact <- 1
  # }
  
  # Age vector:
  agev <- (algObj$minAge * ageFact):maxAge
  nage <- length(agev)
  
  # Create Life tables:
  LT  <- list()
  if (is.null(covObj$cat)) {
    covNames <- c("noCov")
  } else {
    covNames <- names(covObj$cat)
  }
  for (covar in covNames) {
    if (covar == "noCov") {
      idx <- 1:datObj$n
    } else {
      if (inherits(covObj, c("fused", "inMort"))) {
        covcat <- "inMort"
      } else {
        covcat <- "propHaz"
      }
      if (covcat == "propHaz" & !covar %in% colnames(covObj[[covcat]])) {
        if (length(covNames) > 2) {
          idx <- which(apply(covObj[[covcat]][, covNames[-1]], 1, sum) == 0)
        } else {
          idx <- which(covObj[[covcat]][, covNames[-1]] == 0)
        }
      } else {
        idx <- which(covObj[[covcat]][, covar] == 1)
      }
    }
    LT[[covar]] <- list()
    for (ltt in 1:ncol(Bmat)) {
      x <- ageLast[idx, ltt]
      xt <- ageFirst[idx, ltt]
      depType <- departType[idx]
      
      # Outputs:
      Nx <- Dx <- ax <- rep(0, nage)
      for (xx in 1:nage) {
        # A) EXPOSURES:
        # Find how many entered the interval (including truncated):
        idNx <- which(xt < agev[xx] + 1 & x >= agev[xx])
        
        # Extract ages and departType:
        xf <- xt[idNx]
        xl <- x[idNx]
        dt <- depType[idNx]
        
        # proportion of truncation in interval:
        trp <- xf - agev[xx]
        trp[trp < 0] <- 0
        
        # proportion of censoring:
        cep <- agev[xx] + 1 - xl
        cep[cep < 0] <- 0
        cep[dt == "D"] <- 0
        
        # Calculate exposures:
        nexp <- 1 - trp - cep
        Nx[xx] <- sum(nexp)
        
        # B) DEATHS:
        # Calculate total deaths in the interval:
        idDx <- which(dt == "D" & xl < agev[xx] + 1)
        Dx[xx] <- sum(nexp[idDx])
        
        # C) PROPORTION LIVED BY THOSE THAT DIED IN INTERVAL:
        if (Dx[xx] > 1) {
          ylived <- xl[idDx] - agev[xx]
          ax[xx] <- sum(ylived) / length(idDx)
        } else {
          ax[xx] <- 0
        }
      }
      # Age-specific mortality probability:
      qx <- Dx / Nx
      
      # Age-specific survival probability:
      px <- 1 - qx
      
      # Survivorship (or cumulative survival):
      lx <- c(1, cumprod(px))[1:nage]
      
      # Number of individual years lived within the interval:
      Lx <- lx * (1 + ax * qx)
      Lx[is.na(Lx)] <- 0
      
      # Total number of individual years lived after age x:
      Tx <- rev(cumsum(rev(Lx)))
      
      # Remaining life expectancy after age x:
      ex <- Tx / lx
      ex[is.na(ex)] <- 0
      
      # Life-table:
      LT[[covar]][[colnames(Bmat)[ltt]]] <- 
        data.frame(Ages = agev, Nx = Nx, Dx = Dx, lx = lx, px = px,
                   qx = qx, Lx = Lx, Tx = Tx, ex = ex)      
    }
  }
  cat("done.\n")
  return(LT)
}
