import numpy
from copy import deepcopy

from .interface import ComputerInterface

from xtb.interface import Calculator, Param
from xtb.libxtb import VERBOSITY_MUTED
from xtb.utils import get_solvent


class Options(object):
    def __init__(self, method=None):
        if method is not None:
            self.method = method
        else:
            self.method = Param.GFN2xTB
        self.verbose = VERBOSITY_MUTED
        self.solvent = None


class XTBInterface(ComputerInterface):
    def __init__(self, options):
        if options is not None:
            self.options = options
        else:
            self.options = Options()
        self.maxderiv = 1

    def max_deriv(self):
        return self.maxderiv

    def energy(self, mol):
        charges = numpy.asarray(deepcopy(mol.charges))
        coords = numpy.asarray(deepcopy(mol.coords))
        natm = len(charges)
        coords.reshape((natm, 3))
        calc = Calculator(self.options.method, charges, coords)
        calc.set_verbosity(self.options.verbose)
        calc.set_accuracy(0.01)
        if self.options.solvent is not None:
            calc.set_solvent(get_solvent(self.options.solvent))
        res = calc.singlepoint()
        return res.get_energy()

    def gradient(self, mol):
        charges = numpy.asarray(deepcopy(mol.charges))
        coords = numpy.asarray(deepcopy(mol.coords))
        natm = len(charges)
        coords.reshape((natm, 3))
        calc = Calculator(self.options.method, charges, coords)
        calc.set_verbosity(self.options.verbose)
        calc.set_accuracy(0.001)
        if self.options.solvent is not None:
            calc.set_solvent(get_solvent(self.options.solvent))
        res = calc.singlepoint()
        E = res.get_energy()
        g = res.get_gradient().reshape(-1)
        return E, g
