import unittest
import os
import numpy
from omega.system import Molecule, MolSystem

has_pyscf = True
try:
    from omega.pyscf_interface import PyscfInterface, Options
except ImportError:
    has_pyscf = False


class SystemTest(unittest.TestCase):
    def setUp(self):
        self.thresh = 1e-12

    @unittest.skipUnless(has_pyscf, "Requires PySCF")
    def test_sys_h5py(self):
        import h5py

        # create H2 molecule
        op = Options()
        op.method = "b3lyp"
        op.charge = 0
        op.spin = 0
        op.basis = "ccpvtz"
        computer = PyscfInterface(op)
        mol = Molecule()
        mol.add((0.0, 0.0, 0.76), charge=1)
        mol.add((0.0, 0.0, 0.0), charge=1)
        sys = MolSystem(mol=mol)
        sys.compute_forces(computer, order=4)

        # save
        f = h5py.File("_test.h5", 'w')
        sys.save_to_hdf5(f)
        f.close()

        nsys = MolSystem(filename="_test.h5")
        os.remove("_test.h5")
        dE = abs(sys.E - nsys.E)
        dF1 = numpy.linalg.norm(sys.F1 - nsys.F1)/numpy.sqrt(sys.F1.size)
        dF2 = numpy.linalg.norm(sys.F2 - nsys.F2)/numpy.sqrt(sys.F2.size)
        dF3 = numpy.linalg.norm(sys.F3 - nsys.F3)/numpy.sqrt(sys.F3.size)
        dF4 = numpy.linalg.norm(sys.F4 - nsys.F4)/numpy.sqrt(sys.F4.size)
        self.assertTrue(dE < self.thresh)
        self.assertTrue(dF1 < self.thresh)
        self.assertTrue(dF2 < self.thresh)
        self.assertTrue(dF3 < self.thresh)
        self.assertTrue(dF4 < self.thresh)


if __name__ == '__main__':
    unittest.main()
