import unittest
from omega.system import Molecule, UnitCell

has_pyscf = True
try:
    import pyscf
except ImportError:
    has_pyscf = False

has_ase = True
try:
    import ase
except ImportError:
    has_ase = False


class PBCInterfaceTest(unittest.TestCase):
    @unittest.skipUnless(has_pyscf, "Requires PySCF")
    def test_pyscf_lda(self):
        from omega.pyscf_interface import PBCPyscfInterface, PBCOptions
        ab = 3.370137329
        l1b = [0.0, ab, ab]
        l2b = [ab, 0.0, ab]
        l3b = [ab, ab, 0.0]
        mol = Molecule()
        mol.add((0.0, 0.0, 0.0), name="C", unit="Bohr")
        mol.add((ab/2, ab/2, ab/2), name="C", unit="Bohr")
        uc = UnitCell(mol=mol, lattice=[l1b, l2b, l3b], unit="Bohr")
        options = PBCOptions()
        options.method = 'lda'
        options.basis = 'gth-dzvp'
        options.pseudo = 'gth-pade'
        options.kmesh = [2, 2, 2]
        computer = PBCPyscfInterface(options)
        Eout = computer.energy(uc)
        Eref = -10.8572451224117
        self.assertTrue(abs(Eref - Eout) < 1e-10)

    @unittest.skipUnless(has_pyscf, "Requires PySCF")
    def test_pyscf_hf(self):
        from omega.pyscf_interface import PBCPyscfInterface, PBCOptions
        ab = 3.370137329
        l1b = [0.0, ab, ab]
        l2b = [ab, 0.0, ab]
        l3b = [ab, ab, 0.0]
        mol = Molecule()
        mol.add((0.0, 0.0, 0.0), name="C", unit="Bohr")
        mol.add((ab/2, ab/2, ab/2), name="C", unit="Bohr")
        uc = UnitCell(mol=mol, lattice=[l1b, l2b, l3b], unit="Bohr")
        options = PBCOptions()
        options.method = 'hf'
        options.basis = 'gth-dzvp'
        options.pseudo = 'gth-pade'
        options.kmesh = [2, 2, 2]
        computer = PBCPyscfInterface(options)
        Eout = computer.energy(uc)
        Eref = -11.0283567336213
        self.assertTrue(abs(Eref - Eout) < 1e-10)


if __name__ == '__main__':
    unittest.main()
