import unittest
import numpy
import subprocess
import os
import sys

from omega.system import Molecule, MolSystem

has_pyscf = True
try:
    import pyscf
except ImportError:
    has_pyscf = False

has_xtb = True
try:
    import xtb
except ImportError:
    has_xtb = False

has_ase = True
try:
    import ase
except ImportError:
    has_ase = False

has_qchem = True
if sys.version_info[0] < 3:
    DEVNULL = open(os.devnull, 'w')
else:
    DEVNULL = subprocess.DEVNULL
x = subprocess.call(["which", "qchem"], stdout=DEVNULL)
if x == 1:
    has_qchem = False

has_ase_qchem = has_ase and has_qchem
if has_ase_qchem:
    try:
        from ase.calculators import qchem
    except ImportError:
        has_ase_qchem = False


class InterfaceTest(unittest.TestCase):
    @unittest.skipUnless(has_pyscf, "Requires PySCF")
    def test_pyscf(self):
        from omega.pyscf_interface import PyscfInterface, Options
        msys = MolSystem(filename="xyz/h2o_b3lyp_sto3g.xyz")

        # compute analytic Hessian at reference geometry
        op = Options()
        op.method = "b3lyp"
        op.charge = 0
        op.spin = 0
        op.basis = "sto-3g"
        computer = PyscfInterface(op)
        msys.compute_forces(computer, order=2)
        msys.get_normal_modes()
        ref = 0.0006605202743806
        diff = abs(ref - numpy.trace(msys.P2))
        self.assertTrue(
            diff < 1e-14, "Difference PySCF check: {}".format(diff))

    @unittest.skipUnless(has_xtb, "Requires XTB")
    def test_xtb(self):
        from omega.xtb_interface import XTBInterface
        eref = -85.429555765504
        gref = 0.000364794299
        computer = XTBInterface(None)
        msys = MolSystem(filename="xyz/cholesterol.xyz")
        msys.compute_forces(computer, order=1)
        de = abs(eref - msys.E)
        dg = abs(gref - numpy.linalg.norm(msys.F1))
        self.assertTrue(
            de < 1e-9, "Difference XTB energy check: {}".format(de))
        self.assertTrue(
            dg < 1e-7, "Difference XTB gradient check: {}".format(dg))

    @unittest.skipUnless(has_ase, "Requires ASE")
    def test_ase(self):
        from omega.ase_interface import ASEInterface
        from omega.finite_difference import fd_d1_0
        from ase.calculators.lj import LennardJones

        tol = 1e-7
        He2 = Molecule()
        He2.add((0.0, 0.0, 0.0), name='He')
        He2.add((0.0, 0.0, 1.5), name='He')
        calc = LennardJones()
        computer = ASEInterface(calc)
        E, go = computer.gradient(He2)
        gr, _ = fd_d1_0(
            He2, He2.coords, computer, E, diag2=False, delta=0.0004)

        diff = numpy.linalg.norm(go - gr)/numpy.sqrt(2)
        self.assertTrue(diff < tol)

    @unittest.skipUnless(has_ase_qchem, "Requires ASE and QChem")
    def test_ase_qchem(self):
        from omega.ase_interface import ASEInterface
        from omega.finite_difference import fd_d1_0
        from ase.calculators.qchem import QChem

        tol = 1e-4
        default_parameters = {
            'method': 'hf',
            'basis': '6-31G',
            'jobtype': None,
            'charge': 0}
        calc = QChem()
        calc.parameters = default_parameters
        computer = ASEInterface(calc)

        msys = MolSystem(filename="xyz/h2o_b3lyp_sto3g.xyz")
        msys.compute_forces(computer, order=1)
        gr, _ = fd_d1_0(
            msys.mol, msys.mol.coords, computer, 0.0, diag2=False, delta=0.005)
        subprocess.call(["rm", "qchem.inp"])
        subprocess.call(["rm", "qchem.out"])
        diff = numpy.linalg.norm(msys.F1 - gr)/numpy.sqrt(gr.shape[0])
        self.assertTrue(diff < tol)


if __name__ == '__main__':
    unittest.main()
