import unittest
import numpy
from omega.system import Molecule, MolSystem

has_pyscf = True
try:
    from omega.pyscf_interface import PyscfInterface, Options
except ImportError:
    has_pyscf = False


class FDSymTest(unittest.TestCase):
    def setUp(self):
        self.thresh = 5e-6

    @unittest.skipUnless(has_pyscf, "Requires PySCF")
    def test_H2(self):
        H2 = Molecule()

        # reference geometry
        H2.add((0.0, 0.0, 0.0), name='H')
        H2.add((0.0, 0.0, 0.74), name='H')
        N = H2.natom*3

        op = Options()
        op.method = "b3lyp"
        op.basis = "ccpvdz"
        computer = PyscfInterface(op)

        sys = MolSystem(mol=H2)
        sys.compute_forces(computer, order=4)

        L = numpy.sqrt(float(N*N*N))
        DF3a = numpy.linalg.norm(sys.F3 - sys.F3.transpose((0, 2, 1)))/L
        DF3b = numpy.linalg.norm(sys.F3 - sys.F3.transpose((1, 0, 2)))/L
        DF3c = numpy.linalg.norm(sys.F3 - sys.F3.transpose((1, 2, 0)))/L
        DF3d = numpy.linalg.norm(sys.F3 - sys.F3.transpose((2, 0, 1)))/L
        DF3e = numpy.linalg.norm(sys.F3 - sys.F3.transpose((2, 1, 0)))/L

        s3 = "Error in 3rd order force constants: {}"
        self.assertTrue(DF3a < self.thresh, s3.format(DF3a))
        self.assertTrue(DF3b < self.thresh, s3.format(DF3b))
        self.assertTrue(DF3c < self.thresh, s3.format(DF3c))
        self.assertTrue(DF3d < self.thresh, s3.format(DF3d))
        self.assertTrue(DF3e < self.thresh, s3.format(DF3e))

        L = float(N*N)
        DF4a = numpy.linalg.norm(sys.F4 - sys.F4.transpose((0, 1, 3, 2)))/L
        DF4b = numpy.linalg.norm(sys.F4 - sys.F4.transpose((0, 1, 2, 3)))/L
        DF4c = numpy.linalg.norm(sys.F4 - sys.F4.transpose((2, 3, 0, 1)))/L
        DF4d = numpy.linalg.norm(sys.F4 - sys.F4.transpose((2, 3, 1, 0)))/L

        s4 = "Error in 4th  order force constants: {}"
        self.assertTrue(DF4a < self.thresh, s4.format(DF4a))
        self.assertTrue(DF4b < self.thresh, s4.format(DF4b))
        self.assertTrue(DF4c < self.thresh, s4.format(DF4c))
        self.assertTrue(DF4d < self.thresh, s4.format(DF4d))


if __name__ == '__main__':
    unittest.main()
