import unittest
import numpy
from omega.system import Molecule, MolSystem, UnitCell
from omega.finite_difference import fd_d1_0

has_pyscf = True
try:
    from omega.pyscf_interface import PyscfInterface, Options
except ImportError:
    has_pyscf = False


class FDGradTest(unittest.TestCase):
    @unittest.skipUnless(has_pyscf, "Requires PySCF")
    def test_H2(self):
        H2 = Molecule()
        # reference geometry
        H2.add((0.0, 0.0, 0.0), name='H')
        H2.add((0.0, 0.0, 0.74), name='H')

        op = Options()
        op.method = "b3lyp"
        op.basis = "ccpvdz"
        computer = PyscfInterface(op)

        sys = MolSystem(mol=H2)
        sys.compute_forces(computer, order=1)

        ref = sys.F1
        out, _ = fd_d1_0(H2, H2.coords, computer, sys.E, diag2=False)

        diff = numpy.linalg.norm(ref - out)
        self.assertTrue(diff < 1e-6, "Error: {}".format(diff))

    #@unittest.skipUnless(has_pyscf, "Requires PySCF")
    #def test_diamond_HF(self):
    #    from omega.pyscf_interface import PBCPyscfInterface, PBCOptions
    #    ab = 3.370137329
    #    l1b = [0.0, ab, ab]
    #    l2b = [ab, 0.0, ab]
    #    l3b = [ab, ab, 0.0]
    #    mol = Molecule()
    #    mol.add((0.0, 0.05, 0.0), name="C", unit="Bohr")
    #    mol.add((ab/2, ab/2, ab/2), name="C", unit="Bohr")
    #    uc = UnitCell(mol=mol, lattice=[l1b, l2b, l3b], unit="Bohr")
    #    options = PBCOptions()
    #    options.method = 'hf'
    #    options.basis = 'gth-dzvp'
    #    options.pseudo = 'gth-pade'
    #    options.kmesh = [2,2,2]
    #    computer = PBCPyscfInterface(options)
    #    E = computer.energy(uc)
    #    out,_ = fd_d1_0(uc, uc.mol.coords, computer, E, diag2=False)
    #    _,ref = computer.gradient(uc)
    #    print(out)
    #    print(ref)
    #    print("Madelung contribution is omitted!!!")
    #    self.assertTrue(numpy.linalg.norm(out - ref) < 1e-6)

    @unittest.skipUnless(has_pyscf, "Requires PySCF")
    def test_diamond_pbe(self):
        from omega.pyscf_interface import PBCPyscfInterface, PBCOptions
        ab = 3.370137329
        l1b = [0.0, ab, ab]
        l2b = [ab, 0.0, ab]
        l3b = [ab, ab, 0.0]
        mol = Molecule()
        mol.add((0.0, 0.05, 0.0), name="C", unit="Bohr")
        mol.add((ab/2, ab/2, ab/2), name="C", unit="Bohr")
        uc = UnitCell(mol=mol, lattice=[l1b, l2b, l3b], unit="Bohr")
        options = PBCOptions()
        options.method = 'pbe'
        options.basis = 'gth-dzvp'
        options.pseudo = 'gth-pade'
        options.kmesh = [2, 2, 2]
        computer = PBCPyscfInterface(options)
        E = computer.energy(uc)
        out, _ = fd_d1_0(uc, uc.mol.coords, computer, E, diag2=False)
        _, ref = computer.gradient(uc)
        self.assertTrue(numpy.linalg.norm(out - ref) < 1e-6)

    @unittest.skipUnless(has_pyscf, "Requires PySCF")
    def test_Si_pbe(self):
        from omega.pyscf_interface import PBCPyscfInterface, PBCOptions
        ab = 5.1672
        l1b = [0.0, ab, ab]
        l2b = [ab, 0.0, ab]
        l3b = [ab, ab, 0.0]
        mol = Molecule()
        mol.add((0.0, 0.05, 0.0), name="Si", unit="Bohr")
        mol.add((ab/2, ab/2, ab/2), name="Si", unit="Bohr")
        uc = UnitCell(mol=mol, lattice=[l1b, l2b, l3b], unit="Bohr")
        options = PBCOptions()
        options.method = 'pbe'
        options.basis = 'gth-dzvp'
        options.pseudo = 'gth-pade'
        options.kmesh = [2, 2, 2]
        computer = PBCPyscfInterface(options)
        E = computer.energy(uc)
        out, _ = fd_d1_0(uc, uc.mol.coords, computer, E, diag2=False)
        _, ref = computer.gradient(uc)
        self.assertTrue(numpy.linalg.norm(out - ref) < 1e-6)


if __name__ == '__main__':
    unittest.main()
