import unittest
import numpy

from omega.system import Molecule
from omega.coord.bond import get_bond_list
from omega.coord.fragment import get_fragments
from omega.coord.angle import get_bond_angles
from omega.coord.dihedral import get_dihedrals


class AngleTest(unittest.TestCase):
    def test_angle_list1(self):
        tol = 1e-14
        mymol = Molecule()
        mymol.add((0.0, 0.0, 0.0), name="H")
        mymol.add((0.0, 0.0, 0.8), name="H")
        mymol.add((0.0, 0.8, 0.0), name="H")
        mymol.add((0.0, 0.8, 0.8), name="H")

        blist = get_bond_list(mymol)
        fragments = get_fragments(mymol, blist)
        alist = get_bond_angles(blist, fragments, mymol)
        self.assertTrue(len(alist) == 4)
        for a in alist:
            self.assertTrue(abs(a.theta - numpy.pi/2) < tol)

    def test_angle_list2(self):
        mymol = Molecule()
        mymol.add((0.0, 0.0, 0.0), name="H")
        mymol.add((0.0, 0.0, 0.8), name="H")
        mymol.add((0.0, 1.8, 0.0), name="H")
        mymol.add((0.0, 1.8, 0.8), name="H")

        blist = get_bond_list(mymol)
        fragments = get_fragments(mymol, blist)
        alist = get_bond_angles(blist, fragments, mymol)
        self.assertTrue(len(alist) == 0)

    def test_angle_value(self):
        tol = 1e-14
        mymol = Molecule()
        mymol.add((0.0, 0.0, 0.0), name="H")
        mymol.add((0.0, 0.0, 0.8), name="H")
        mymol.add((0.0, 0.8, 0.0), name="H")
        mymol.add((0.0, 0.8, 0.8), name="H")

        blist = get_bond_list(mymol)
        fragments = get_fragments(mymol, blist)
        alist = get_bond_angles(blist, fragments, mymol)
        self.assertTrue(len(alist) == 4)
        for a in alist:
            self.assertTrue(abs(a.evaluate(mymol.coords) - numpy.pi/2) < tol)

    def test_dihedral_list1(self):
        mymol = Molecule()
        mymol.add((0.0, 0.0, 0.0), name="H")
        mymol.add((0.0, 0.0, 0.8), name="H")
        mymol.add((0.0, 0.8, 0.0), name="H")
        mymol.add((0.0, 0.8, 0.8), name="H")

        blist = get_bond_list(mymol)
        fragments = get_fragments(mymol, blist)
        dlist = get_dihedrals(blist, fragments, mymol)
        self.assertTrue(len(dlist) == 4)

    def test_dihedral_list2(self):
        mymol = Molecule()
        mymol.add((+0.000000000,  0.000000000,  0.000000000), name='C', unit="Bohr")
        mymol.add((+0.000000000,  0.000000000,  3.200000000), name='C', unit="Bohr")
        mymol.add((+0.000000000,  2.050000000,  0.000000000), name='H', unit="Bohr")
        mymol.add((-1.775352078, -1.025000000,  0.000000000), name='H', unit="Bohr")
        mymol.add((+1.775352078, -1.025000000,  0.000000000), name='H', unit="Bohr")
        mymol.add((+1.775352078,  1.025000000,  3.200000000), name='H', unit="Bohr")
        mymol.add((-1.775352078,  1.025000000,  3.200000000), name='H', unit="Bohr")
        mymol.add((+0.000000000, -2.050000000,  3.200000000), name='H', unit="Bohr")

        blist = get_bond_list(mymol)
        fragments = get_fragments(mymol, blist)
        dlist = get_dihedrals(blist, fragments, mymol)
        self.assertTrue(len(dlist) == 9)

    def test_dihedral_value(self):
        tol = 1e-14
        mymol = Molecule()
        mymol.add((+0.000000000,  0.000000000,  0.000000000), name='C', unit="Bohr")
        mymol.add((+0.000000000,  0.000000000,  3.200000000), name='C', unit="Bohr")
        mymol.add((+0.000000000,  2.050000000,  0.000000000), name='H', unit="Bohr")
        mymol.add((-1.775352078, -1.025000000,  0.000000000), name='H', unit="Bohr")
        mymol.add((+1.775352078, -1.025000000,  0.000000000), name='H', unit="Bohr")
        mymol.add((+1.775352078,  1.025000000,  3.200000000), name='H', unit="Bohr")
        mymol.add((-1.775352078,  1.025000000,  3.200000000), name='H', unit="Bohr")
        mymol.add((+0.000000000, -2.050000000,  3.200000000), name='H', unit="Bohr")

        blist = get_bond_list(mymol)
        fragments = get_fragments(mymol, blist)
        dlist = get_dihedrals(blist, fragments, mymol)
        self.assertTrue(len(dlist) == 9)
        for d in dlist:
            self.assertTrue(abs(d.evaluate(mymol.coords) - d.phi) < tol)


if __name__ == '__main__':
    unittest.main()
