import logging
import numpy
from .bond import get_bond_list
from .angle import get_bond_angles
from .dihedral import get_dihedrals
from .fragment import get_fragments, get_fragment_bonds
from ..finite_difference import get_fd1
from ..system import is_linear


class RIC(object):
    """A class to store redundant internal coordinates.

    This class stores the redundant internal coodinate system
    of a given molecule. The RICs themselves store their initial
    value for the purposes of identifying singular/ill-defined
    displacements.

    Attributes:
        blist (list): list of bonds (includes artificial
            bonds between fragments)
        flist (list): list of fragments
        alist (list): list of bond angles
        dlist (list): list of dihedrals
        nb (int): number of bonds
        na (int): number of angles
        nd (int): number of dihedrals
        nric (int): number of internals
        nc (int): number of cartesians
        ntr (int): number of trans/rot DOFs
        ncu (int): number of significant coordinates
    """
    def __init__(self, mol, dihedrals=True):
        self.blist = get_bond_list(mol)
        self.flist = get_fragments(mol, self.blist)
        bflist = get_fragment_bonds(mol, self.flist)
        self.blist += bflist
        self.alist = get_bond_angles(self.blist, self.flist, mol)
        if dihedrals:
            self.dlist = get_dihedrals(self.blist, self.flist, mol)
        else:
            self.dlist = []

        self.nb = len(self.blist)
        self.na = len(self.alist)
        self.nd = len(self.dlist)
        self.nric = self.nb + self.na + self.nd
        self._fdwarned = False

        self.nc = 3*mol.natom
        self.ntr = (5 if is_linear(mol.coords) else 6)
        self.ncu = self.nc - self.ntr
        if self.nric < self.ncu:
            logging.error("Failed to form sufficient internal coordinates")
            raise Exception("Failed to form sufficient internal coordinates")

        logging.debug("Redunant internal coordinates:")
        logging.debug(" {:5d} bonds".format(self.nb))
        logging.debug(" {:5d} angles".format(self.na))
        logging.debug(" {:5d} dihedrals".format(self.nd))

    def bmatrix_fd(self, coords, delta=5e-4):
        """Return the Wilsonian B-matrix using finite differences."""
        fc, bc = get_fd1(coords, delta=delta)
        nf = len(fc)
        nb = len(bc)
        assert(nf == nb and nf == self.nc)
        B = numpy.zeros((self.nric, self.nc))
        for i in range(nf):
            vf = self.evaluate(fc[i])
            vb = self.evaluate(bc[i])
            B[:, i] = (vf - vb)/(2*delta)
        return B

    def bmatrix(self, coords):
        """Return the Wilsonian B-matrix using analytic differentiation."""
        B = numpy.zeros((self.nric, self.nc))
        for i in range(self.nb):
            gv = self.blist[i].grad(coords)
            B[i, :] = gv
        for i in range(self.na):
            gv = self.alist[i].grad(coords)
            B[self.nb + i, :] = gv
        for i in range(self.nd):
            gv = self.dlist[i].grad(coords)
            B[self.nb + self.na + i, :] = gv
        return B

    def bmatrix_gen(self, coords):
        """Return the Wilsonian B-matrix, prefer analytic derivatives."""
        try:
            B = self.bmatrix(coords)
        except AttributeError:
            www = "Analytic B-matrix is not implemented," 
            www += "using finite-difference..."
            if not self._fdwarned:
                logging.warning(www)
                self._fdwarned = True
            B = self.bmatrix_fd(coords)
        return B

    def evaluate(self, coords):
        """Return the values of the RICs given the cartesian coordinates."""
        vals = numpy.zeros(self.nric)
        for i, b in enumerate(self.blist):
            vals[i] = b.evaluate(coords)
        for j, a in enumerate(self.alist):
            vals[j + self.nb] = a.evaluate(coords)
        for k, d in enumerate(self.dlist):
            vals[k + self.nb + self.na] = d.evaluate(coords)
        return vals
