import numpy
from .bond import Bond, get_adjacency


def _get_fragment(fragment, adj):
    new = set()
    for x in fragment:
        for a in adj[x]:
            if a not in fragment:
                new.add(a)
    if new:
        fragment = fragment.union(new)
        return _get_fragment(fragment, adj)
    else:
        return fragment


def get_fragments(mol, blist):
    """Return a list of fragments given a molecule and bond list."""
    remain = [i for i in range(mol.natom)]
    flist = []
    count = 0
    adj = get_adjacency(blist, mol.natom)
    while remain and count < 1000000:
        fragment = set()
        fragment.add(remain[0])
        fragment = _get_fragment(fragment, adj)
        flist.append(fragment)
        remain = [x for x in remain if x not in fragment]
        count += 1
    return flist


def get_fragment_bonds(mol, flist, method="first"):
    """Return a list of artificial bonds between fragments."""
    blist = []
    # get matrix of fragment distances
    lf = len(flist)
    dmat = numpy.zeros((lf, lf))
    for i, f1 in enumerate(flist):
        for j, f2 in enumerate(flist):
            if i == j:
                dmat[i, j] = 0.0
            elif method.lower() == "first":
                ii = list(f1)[0]
                jj = list(f2)[0]
                c1 = numpy.asarray(mol.coords[ii])
                c2 = numpy.asarray(mol.coords[jj])
                dist = numpy.linalg.norm(c1 - c2)
                dmat[i, j] = dist

    for i, f1 in enumerate(flist[:lf - 1]):
        # find the next closest fragment
        amin = numpy.amin(dmat[(i + 1):, i])
        idx = numpy.where(dmat[:, i] == amin)[0][0]
        f2 = flist[idx]
        if method.lower() == "first":
            i = list(f1)[0]
            j = list(f2)[0]
            c1 = numpy.asarray(mol.coords[i])
            c2 = numpy.asarray(mol.coords[j])
            e1str = mol.names[i].decode("utf-8")
            e2str = mol.names[j].decode("utf-8")
            dist = numpy.linalg.norm(c1 - c2)
            blist.append(Bond(i, j, e1str, e2str, dist))
        else:
            raise Exception("unrecognized method: {}".format(method))
    return blist
