import logging
import sys
from omega.constants import hartree_to_cm_1
from omega.system import Molecule, UnitCell, PBCSystem
from omega.pyscf_interface import PBCPyscfInterface, PBCOptions
from ase.dft.kpoints import special_points, get_bandpath, kpoint_convert

logging.basicConfig(
    format='%(levelname)s:%(message)s', level=logging.INFO, stream=sys.stdout)

ab = 3.370137329
l1b = [0.0, ab, ab]
l2b = [ab, 0.0, ab]
l3b = [ab, ab, 0.0]
mol = Molecule()
mol.add((0.0, 0.0, 0.0), name="C", unit="Bohr")
mol.add((ab/2, ab/2, ab/2), name="C", unit="Bohr")
uc = UnitCell(mol=mol, lattice=[l1b, l2b, l3b], unit="Bohr")

nq = 30
points = special_points['fcc']
G = points['G']
X = points['X']
W = points['W']
K = points['K']
L = points['L']
band_qpts, qpath, sp_points = get_bandpath(
    [L, G, X, W, K, G], uc.lattice, npoints=nq)
qpts = kpoint_convert(uc.lattice, skpts_kc=band_qpts)

options = PBCOptions()
options.method = 'pbe'
options.basis = 'gth-szv'
options.pseudo = 'gth-pade'
options.kmesh = [1, 1, 1]
options.scf_conv = 1e-9
options.precision = 1e-11
options.verbose = 1
computer = PBCPyscfInterface(options)

w = []
system = PBCSystem(cell=uc)
system.compute_hessian(computer, supercell=[2, 2, 2])
for q in qpts:
    wq = system.get_phonons(q)
    w.append(hartree_to_cm_1*wq)

outfile = "diamond_pbe.dat"
f = open(outfile, 'w')
f.write("# Special points:\n")
for point, label in zip(sp_points, ['L', 'G', 'X', 'W', 'K', 'G']):
    f.write("# %0.6f %s\n" % (point, label))
for i, wq in zip(qpath, w):
    f.write("{:.7f} ".format(i))
    for wi in wq:
        f.write("{:.5f}  ".format(wi))
    f.write("\n")
f.close()
