import unittest
import numpy
from omega.system import MolSystem
from omega.pyscf_interface import PyscfInterface, Options
from omega import transrot

from harmony.matvec import get_matvec, get_matvec_proj


class MatvecTest(unittest.TestCase):
    def test_benzene_full(self):
        sys = MolSystem(filename="h5/Benzene_b3lyp_sto3g.h5")
        Mi2 = 1.0/numpy.sqrt(numpy.asarray(sys.M))
        n = len(Mi2)
        F2m = numpy.einsum('ij,i,j->ij', sys.F2, Mi2, Mi2)
        op = Options()
        op.method = "b3lyp"
        op.basis = "sto3g"
        computer = PyscfInterface(op)

        matvec = get_matvec(sys.mol, computer, delta=0.005)

        vec = numpy.random.rand(n)
        ref = numpy.einsum('ij,j->i', F2m, vec)
        out = matvec(vec)
        diff = numpy.linalg.norm(out - ref)/numpy.linalg.norm(ref)
        self.assertTrue(diff < 1e-4, "Difference: {}".format(diff))

    def test_benzene_full_big(self):
        sys = MolSystem(filename="h5/Benzene_b3lyp_sto3g.h5")
        Mi2 = 1.0/numpy.sqrt(numpy.asarray(sys.M))
        n = len(Mi2)
        F2m = numpy.einsum('ij,i,j->ij', sys.F2, Mi2, Mi2)
        op = Options()
        op.method = "b3lyp"
        op.basis = "sto3g"
        computer = PyscfInterface(op)

        matvec = get_matvec(sys.mol, computer, delta=0.005)

        vec = 1.0e8*numpy.random.rand(n)
        ref = numpy.einsum('ij,j->i', F2m, vec)
        out = matvec(vec)
        diff = numpy.linalg.norm(out - ref)/numpy.linalg.norm(ref)
        self.assertTrue(diff < 1e-4, "Difference: {}".format(diff))

    def test_benzene_proj(self):
        sys = MolSystem(filename="h5/Benzene_b3lyp_sto3g.h5")
        Mi2 = 1.0/numpy.sqrt(numpy.asarray(sys.M))
        n = len(Mi2)
        F2m = numpy.einsum('ij,i,j->ij', sys.F2, Mi2, Mi2)
        op = Options()
        op.method = "b3lyp"
        op.basis = "sto3g"
        computer = PyscfInterface(op)
        dvecs = transrot.transrot(
                sys.mol.natom, sys.mol.coords, sys.I, sys.R,
                linear=sys.linear, masses=sys.M, mweight=True)
        P = transrot.half_projector_against(dvecs, full=True)
        F2mP = numpy.einsum('ij,jk,kl->il', P, F2m, P)

        matvec = get_matvec_proj(P, sys.mol, computer, delta=0.005)

        vec = numpy.random.rand(n)
        ref = numpy.einsum('ij,j->i', F2mP, vec)
        out = matvec(vec)
        diff = numpy.linalg.norm(out - ref)/numpy.linalg.norm(ref)
        self.assertTrue(diff < 1e-4, "Difference: {}".format(diff))

    def test_h2o(self):
        pass


if __name__ == '__main__':
    unittest.main()
