import logging
import numpy
try:
    from numpy.random import default_rng
    legacy = False
except ImportError:
    logging.warning("Legacy numpy.random detected")
    legacy = True


def _random_uniform(low, high):
    if legacy:
        u = (high - low)*numpy.random.rand() - 0.5*(high + low)
    else:
        rng = default_rng()
        u = rng.uniform(low=low, high=high)
    return u


def _random_int(low, high):
    if legacy:
        i = numpy.random.randint(low, high)
    else:
        rng = default_rng()
        i = rng.integers(low, high)
    return i


def _rademacher(n):
    if legacy:
        v = 2.0*numpy.random.rand(n) - numpy.ones(n)
    else:
        rng = default_rng()
        v = rng.uniform(low=-1.0, high=1.0, size=(n))
    return numpy.asarray([-1.0 if vv < 0.0 else 1.0 for vv in v])


def _gaussian(n):
    if legacy:
        return numpy.random.normal(0.0, 1.0, n)
    else:
        rng = default_rng()
        return rng.normal(0.0, 1.0, n)


def _rayleigh(n):
    if legacy:
        v = 2.0*numpy.random.rand(n) - numpy.ones(n)
    else:
        rng = default_rng()
        v = rng.uniform(low=-1.0, high=1.0, size=(n))
    return numpy.sqrt(n)*v/numpy.linalg.norm(v)


def get_random(n, method):
    if method.lower() == "rademacher":
        return _rademacher(n)
    elif method.lower() == "gaussian":
        return _gaussian(n)
    elif method.lower() == "rayleigh":
        return _rayleigh(n)
    else:
        raise Exception("Unrecognized method: {}".format(method))


def get_importance(prob):
    np = prob.shape[0]
    maxp = prob.max()
    sump = prob.sum()
    max_it = 10000
    it = 0
    while it < max_it:
        u = _random_uniform(0, 1)
        i = _random_int(0, np)
        if prob[i]/maxp > u:
            return (i, prob[i]/sump)
        it += 1
    assert(it == max_it)
    raise Exception("Rejection sampling failed")
