import numpy
import logging

from omega import constants
from omega import transrot

from .fit import zpe, Hfunc, Afunc, Sfunc, Sfunc_rot, Sfunc_cut
from .fit import chebyshev_fit
from .evaluation import eval_chebyshev_matvec_batch, eval_chebyshev_batch
from . import trace
from .matvec import get_matvec, get_matvec_half_proj


# TODO: try this again with preconditioner
def get_lowest_modes(sys, computer, nmodes, ref=None):
    n = len(sys.M)
    from .davidson import davidson
    if ref is not None:
        Mi2 = 1.0/numpy.sqrt(numpy.asarray(ref.M))
        F2m = numpy.einsum('ij,i,j->ij', ref.F2, Mi2, Mi2)
        eref, vref = numpy.linalg.eigh(F2m)

    c = 1000.0

    def matvec(vec):
        if ref is not None:
            matvec_temp = get_matvec(sys.mol, computer, delta=0.0048)
            vtemp = numpy.matmul(vref, vec)
            return c*numpy.matmul(vref.transpose(), matvec_temp(vtemp))
        else:
            matvec_temp = get_matvec(sys.mol, computer, delta=0.0048)
            return c*matvec_temp(vec)

    e, v = davidson(nmodes, n, matvec)
    return e/c, v


def get_function(string, beta, I, rotor, cutoff):
    if string.lower()[0] == 'z':
        func = zpe
    elif string.lower()[0] == 'h' or string.lower()[0] == 'u':
        def func(x): return Hfunc(x, beta)
    elif string.lower()[0] == 'a' or string.lower()[0] == 'g':
        def func(x): return Afunc(x, beta)
    elif string.lower()[0] == 's':
        if rotor is not None:
            sthr = rotor
            def func(x): return Sfunc_rot(x, beta, I, sthr)
        elif cutoff is not None:
            alpha = 1.0*constants.hartree_to_cm_1
            cut = cutoff
            def func(x): return Sfunc_cut(x, beta, cut, alpha)
        else:
            def func(x): return Sfunc(x, beta)
    else:
        raise Exception("unrecognized quantity: {}".format(string))
    return func


def get_fits(quantities, beta, I, fit_params):
    o = fit_params["order"] if "order" in fit_params else 24
    wmin = fit_params["wmin"] if "wmin" in fit_params else 50.0
    wmax = fit_params["wmax"] if "wmax" in fit_params else 8000.0
    rotor = fit_params["rotor"] if "rotor" in fit_params else None
    cutoff = fit_params["cutoff"] if "cutoff" in fit_params else None
    nx = fit_params["nx"] if "nx" in fit_params else 1000
    points = fit_params["points"] if "points" in fit_params else "linear"
    if rotor is not None:
        rotor /= constants.hartree_to_cm_1
    if cutoff is not None:
        cutoff /= constants.hartree_to_cm_1
    wmax /= constants.hartree_to_cm_1
    wmin /= constants.hartree_to_cm_1

    funcs = [get_function(s, beta, I, rotor, cutoff) for s in quantities]
    polys = [chebyshev_fit(f, wmin, wmax, order=o, nx=nx, points=points) for f in funcs]
    return polys


class Harmonic(object):
    """Harmonic thermodynamic analysis driver.

    Attributes:
        sys (MolSystem): System representing the molecule of interest
        computer (ComputerInterface): Object for computing energies/gradients
    """
    def __init__(self, sys, computer):
        self.sys = sys
        self.computer = computer
        self.dvecs = transrot.transrot(
                sys.mol.natom, sys.mol.coords, sys.I, sys.R,
                linear=sys.linear, masses=sys.M, mweight=True)
        self.P = transrot.projector_against(self.dvecs)

    def run(self, fit_params, fit="chebyshev",
            quantities=["ZPE", "Hvib", "Avib", "Svib"], T=298.15,
            nsample=100, check=True, method="Rademacher", diff="sym",
            delta=0.0012, full=False):

        if fit.lower()[:1] == "c":
            data = self._run_cheby(
                fit_params, quantities=quantities, T=T, nsample=nsample,
                check=check, method=method, diff=diff, delta=delta)
        elif fit.lower()[:1] == "l":
            data = self._run_lanczos(
                fit_params, quantities=quantities, T=T, nsample=nsample,
                check=check, method=method, diff=diff, delta=delta)
        else:
            raise Exception("Unrecognized fit method: {}".format(fit))
        if full:
            return data
        else:
            out = [numpy.average(x) for x in data]
            return out

    def _run_cheby(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                   T=298.15, nsample=100, check=True, method="Rademacher", diff="sym", delta=0.0012):
        """Run the calculation."""
        if check:
            E, F1 = self.computer.gradient(self.sys.mol)
            gn = numpy.linalg.norm(F1)/numpy.sqrt(F1.shape[0])
            if gn > 1e-5:
                logging.warning("geometry is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E))
            logging.info("Gradient norm: {} Ha".format(gn))

        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)
        n = len(self.sys.M)

        nout = len(quantities)
        fits = get_fits(quantities, beta, self.sys.I, fit_params)
        matvec = get_matvec(self.sys.mol, self.computer, diff=diff)

        coeffs = [X.coef for X in fits]

        def Xv(v): return eval_chebyshev_matvec_batch(
            v, matvec, coeffs, fits[0].domain, proj=self.P)

        if method.lower()[:7] == "hutch++":
            if len(method.lower()) > 7:
               kappa = int(method.lower()[7:])
            else:
               kappa = 1
            out = trace.stochastic_trace_pp(nout, n, Xv, nsample, kappa=kappa, names=quantities)
        else:
            out = trace.stochastic_trace(nout, n, Xv, nsample, method, names=quantities)
        return out

    def _run_lanczos(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                     T=298.15, nsample=100, check=True, method="Rademacher", diff="sym", delta=0.0012):

        if check:
            E, F1 = self.computer.gradient(self.sys.mol)
            gn = numpy.linalg.norm(F1)/numpy.sqrt(F1.shape[0])
            if gn > 1e-5:
                logging.warning("geometry is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E))
            logging.info("Gradient norm: {} Ha".format(gn))

        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)

        Ph = transrot.half_projector_against(self.dvecs)
        n = Ph.shape[1]
        matvec = get_matvec_half_proj(Ph, self.sys.mol, self.computer, diff=diff, delta=delta)

        rotor = fit_params["rotor"] if "rotor" in fit_params else None
        cutoff = fit_params["cutoff"] if "cutoff" in fit_params else None
        if rotor is not None:
            rotor /= constants.hartree_to_cm_1
        if cutoff is not None:
            cutoff /= constants.hartree_to_cm_1
        mmax = fit_params["order"] if "order" in fit_params else 24

        funcs = [get_function(s, beta, self.sys.I, rotor, cutoff) for s in quantities]
        return trace.stochastic_lanczos(n, matvec, nsample, funcs, quantities, mmax, Ph=Ph)

    def run_imp(self, fit_params, Vec, prob, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                T=298.15, nsample=100, check=True):
        """Run the calculation with importance sampling."""
        if check:
            E, F1 = self.computer.gradient(self.sys.mol)
            gn = numpy.linalg.norm(F1)/numpy.sqrt(F1.shape[0])
            if gn > 1e-5:
                logging.warning("geometry is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E))
            logging.info("Gradient norm: {} Ha".format(gn))

        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)
        n = len(self.sys.M)

        nout = len(quantities)
        fits = get_fits(quantities, beta, self.sys.I, fit_params)
        matvec = get_matvec(self.sys.mol, self.computer)

        coeffs = [X.coef for X in fits]

        def Xv(v): return eval_chebyshev_matvec_batch(
            v, matvec, coeffs, fits[0].domain)

        out = trace.stochastic_trace_imp(nout, n, Xv, nsample, Vec, prob, P=self.P, names=quantities)
        return out

    def run_exact(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                  T=298.15, check=True, diff="sym"):
        """Run the calculation, computing the trace exactly."""
        if check:
            E, F1 = self.computer.gradient(self.sys.mol)
            gn = numpy.linalg.norm(F1)/numpy.sqrt(F1.shape[0])
            if gn > 1e-5:
                logging.warning("geometry is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E))
            logging.info("Gradient norm: {} Ha".format(gn))

        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)
        n = len(self.sys.M)

        matvec = get_matvec(self.sys.mol, self.computer, diff=diff)
        nout = len(quantities)
        fits = get_fits(quantities, beta, self.sys.I, fit_params)

        coeffs = [X.coef for X in fits]

        def Xv(v): return eval_chebyshev_matvec_batch(
            v, matvec, coeffs, fits[0].domain, proj=self.P)

        out = trace.exact_trace(nout, n, Xv)
        return out

    def run_diff(self, ref, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                 T=298.15, nsample=100, check=True, method="Rayleigh", diff="sym"):
        raise Exception("This function is depracated, use HarmonicDiff object instead")

    def run_diff_imp(self, ref, fit_params, Vec, prob, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                     T=298.15, nsample=100, check=True, method="Rayleigh", diff="sym"):
        raise Exception("This function is depracated, use HarmonicDiff object instead")

    def run_diff_exact(self, ref, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                       T=298.15, nsample=100, check=True, diff="sym"):
        raise Exception("This function is depracated, use HarmonicDiff object instead")


class HarmonicDiff(object):
    """Harmonic thermodynamic analysis for the difference of two systems

    Attributes:
        sys (MolSystem): System representing the molecule of interest
        ref (MolSystem): System representing the reference hessian
        computer (ComputerInterface): Object for computing energies/gradients
    """
    def __init__(self, sys, ref, computer, computer2=None):
        self.sys = sys
        self.ref = ref
        self.computer = computer
        self.computer2 = computer2
        self.dvecs = transrot.transrot(
                sys.mol.natom, sys.mol.coords, sys.I, sys.R,
                linear=sys.linear, masses=sys.M, mweight=True)
        self.P = transrot.projector_against(self.dvecs)
        self.dvecs_ref = transrot.transrot(
                ref.mol.natom, ref.mol.coords, ref.I, ref.R,
                linear=ref.linear, masses=ref.M, mweight=True)
        self.Pref = transrot.projector_against(self.dvecs_ref)

    def run(self, fit_params, fit="chebyshev", quantities=["ZPE", "Hvib", "Avib", "Svib"],
            T=298.15, nsample=100, check=True, method="Rayleigh", diff="sym", full=False):
        """Run the calculation."""
        if check:
            E,F1 = self.computer.gradient(self.sys.mol)
            gn = numpy.linalg.norm(F1)/numpy.sqrt(F1.shape[0])
            if gn > 1e-5:
                logging.warning("geometry is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E))
            logging.info("Gradient norm: {} Ha".format(gn))

        if fit.lower()[:1] == "c":
            data = self._run_cheby(
                fit_params, quantities=quantities, T=T,
                nsample=nsample, method=method, diff=diff)
        elif fit.lower()[:1] == "l":
            data = self._run_lanczos(
                fit_params, quantities=quantities, T=T,
                nsample=nsample, method=method, diff=diff)
        else:
            raise Exception("Unrecognized fit method: {}".format(fit))
        if full:
            return data
        else:
            out = [numpy.average(x) for x in data]
            return out

    def _run_cheby(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                   T=298.15, nsample=100, method="Rayleigh", diff="sym"):
        """Run the calculation."""
        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)
        n = len(self.sys.M)

        matvec = get_matvec(self.sys.mol, self.computer, diff=diff)
        nout = len(quantities)
        fits = get_fits(quantities, beta, self.sys.I, fit_params)

        coeffs = [X.coef for X in fits]

        def Xv1(v): return eval_chebyshev_matvec_batch(
            v, matvec, coeffs, fits[0].domain, proj=self.P)

        if self.ref.F2 is not None:
            Mi2 = 1.0/numpy.sqrt(numpy.asarray(self.ref.M))
            F2m = numpy.einsum('ij,i,j->ij', self.ref.F2, Mi2, Mi2)

            def Xv2(v):
                Pv = numpy.matmul(self.Pref, v)
                temp = eval_chebyshev_batch(F2m, coeffs, fits[0].domain)
                return [numpy.matmul(t, Pv) for t in temp]

        else:
            if self.computer2 is None:
                raise Exception("No Hessian in reference and no computer provided!")
            matvec2 = get_matvec(self.ref.mol, self.computer2, diff=diff)

            def Xv2(v): return eval_chebyshev_matvec_batch(
                v, matvec2, coeffs, fits[0].domain, proj=self.Pref)

        def Xv(v):
            T1 = Xv1(v)
            T2 = Xv2(v)
            return [t1 - t2 for t1, t2 in zip(T1, T2)]

        if method.lower() == "hutch++":
            out = trace.stochastic_trace_pp(nout, n, Xv, nsample, names=quantities)
        else:
            out = trace.stochastic_trace(nout, n, Xv, nsample, method, P=None, names=quantities)
        return out

    def _run_lanczos(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                     T=298.15, nsample=100, method="Rayleigh", diff="sym"):

        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)

        Ph = transrot.half_projector_against(self.dvecs)
        Phref = transrot.half_projector_against(self.dvecs_ref)
        n = Ph.shape[1]
        matvec = get_matvec_half_proj(Ph, self.sys.mol, self.computer, diff=diff)

        if self.ref.F2 is not None:
            Mi2 = 1.0/numpy.sqrt(numpy.asarray(self.ref.M))
            F2m = numpy.einsum('ij,i,j->ij', self.ref.F2, Mi2, Mi2)

            def matvec2(v):
                Pv = numpy.matmul(Phref, v)
                x = numpy.matmul(F2m, Pv)
                return numpy.matmul(Phref.transpose(), x)

        else:
            if self.computer2 is None:
                raise Exception("No Hessian in reference and no computer provided!")
            matvec2 = get_matvec_half_proj(Phref, self.ref.mol, self.computer2, diff=diff)

        rotor = fit_params["rotor"] if "rotor" in fit_params else None
        cutoff = fit_params["cutoff"] if "cutoff" in fit_params else None
        if rotor is not None:
            rotor /= constants.hartree_to_cm_1
        if cutoff is not None:
            cutoff /= constants.hartree_to_cm_1
        mmax = fit_params["order"] if "order" in fit_params else 24

        funcs = [get_function(s, beta, self.sys.I, rotor, cutoff) for s in quantities]
        return trace.stochastic_lanczos_diff(n, matvec, matvec2, nsample, funcs, quantities, mmax, Ph1=Ph, Ph2=Phref)

    def run_imp(self, fit_params, Vec, prob, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                T=298.15, nsample=100, check=True, method="Rayleigh", diff="sym"):
        """Run the calculation with importance sampling."""
        raise Exception("This method of importance sampling should not be used")
        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)
        n = len(self.sys.M)

        matvec = get_matvec(self.sys.mol, self.computer, diff=diff)
        nout = len(quantities)
        fits = get_fits(quantities, beta, self.sys.I, fit_params)

        coeffs = [X.coef for X in fits]

        def Xv1(v): return eval_chebyshev_matvec_batch(
            v, matvec, coeffs, fits[0].domain)

        Mi2 = 1.0/numpy.sqrt(numpy.asarray(self.ref.M))
        F2m = numpy.einsum('ij,i,j->ij', self.ref.F2, Mi2, Mi2)

        def Xv2(v):
            temp = eval_chebyshev_batch(F2m, coeffs, fits[0].domain)
            return [numpy.matmul(t, v) for t in temp]

        out = trace.stochastic_trace_diff_imp(nout, n, Xv1, Xv2, nsample, Vec, prob, P1=self.P, P2=self.Pref, names=quantities)
        return out

    def run_exact(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                  T=298.15, nsample=100, check=True, diff="sym"):
        """Run the calculation with the exact trace."""
        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)
        n = len(self.sys.M)

        matvec = get_matvec(self.sys.mol, self.computer, diff=diff)
        fits = get_fits(quantities, beta, self.sys.I, fit_params)

        coeffs = [X.coef for X in fits]

        def Xv1(v): return eval_chebyshev_matvec_batch(
            v, matvec, coeffs, fits[0].domain, proj=self.P)

        Mi2 = 1.0/numpy.sqrt(numpy.asarray(self.ref.M))
        F2m = numpy.einsum('ij,i,j->ij', self.ref.F2, Mi2, Mi2)

        def Xv2(v):
            v1 = numpy.matmul(self.Pref, v)
            temp = eval_chebyshev_batch(
                F2m, coeffs, fits[0].domain)
            return [numpy.matmul(t, v1) for t in temp]

        def Xv(v):
            T1 = Xv1(v)
            T2 = Xv2(v)
            return [t1 - t2 for t1, t2 in zip(T1, T2)]

        out = trace.exact_trace(len(quantities), n, Xv)
        return out


class HarmonicBinding(object):
    """Harmonic thermodynamic analysis for binding

    Attributes:
        sys (MolSystem): System representing the bound system
        sys1 (MolSystem): System representing the first subsystem
        sys2 (MolSystem): System representing the second subsystem
        computer (ComputerInterface): Object for computing energies/gradients
    """
    def __init__(self, sys, sys1, sys2, computer):
        self.sys = sys
        self.sys1 = sys1
        self.sys2 = sys2
        self.computer = computer
        self.dvecs = transrot.transrot(
                sys.mol.natom, sys.mol.coords, sys.I, sys.R,
                linear=sys.linear, masses=sys.M, mweight=True)
        self.P = transrot.projector_against(self.dvecs)

        self.dvecs1 = transrot.transrot(
                sys1.mol.natom, sys1.mol.coords, sys1.I, sys1.R,
                linear=sys1.linear, masses=sys1.M, mweight=True)
        self.P1 = transrot.projector_against(self.dvecs1)

        self.dvecs2 = transrot.transrot(
                sys2.mol.natom, sys2.mol.coords, sys2.I, sys2.R,
                linear=sys2.linear, masses=sys2.M, mweight=True)
        self.P2 = transrot.projector_against(self.dvecs2)

    def run(self, fit_params, fit="Chebyshev", quantities=["ZPE", "Hvib", "Avib", "Svib"],
            T=298.15, nsample=100, check=True, method="Rayleigh", diff="sym", full=False):
        if fit.lower()[:1] == "c":
            data = self._run_cheby(
                fit_params, quantities=quantities, T=T,
                nsample=nsample, check=check, method=method, diff=diff)
        elif fit.lower()[:1] == "l":
            data = self._run_lanczos(
                fit_params, quantities=quantities, T=T,
                nsample=nsample, check=check, method=method, diff=diff)
        else:
            raise Exception("Unrecognized fit method: {}".format(fit))

        if full:
            return data
        else:
            out = [numpy.average(x) for x in data]
            return out

    def _run_cheby(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                   T=298.15, nsample=100, check=True, method="Rayleigh", diff="sym"):
        if check:
            E, F1 = self.computer.gradient(self.sys.mol)
            gn = numpy.linalg.norm(F1)/numpy.sqrt(F1.shape[0])
            if gn > 1e-5:
                logging.warning("geometry is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E))
            logging.info("Gradient norm: {} Ha".format(gn))

            E1, F11 = self.computer.gradient(self.sys1.mol)
            gn = numpy.linalg.norm(F11)/numpy.sqrt(F11.shape[0])
            if gn > 1e-5:
                logging.warning("geometry of subsystem 1 is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E1))
            logging.info("Gradient norm: {} Ha".format(gn))

            E2, F12 = self.computer.gradient(self.sys2.mol)
            gn = numpy.linalg.norm(F12)/numpy.sqrt(F12.shape[0])
            if gn > 1e-5:
                logging.warning("geometry of subsystem 2 is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E2))
            logging.info("Gradient norm: {} Ha".format(gn))

        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)
        n = len(self.sys.M)
        n1 = len(self.sys1.M)
        n2 = len(self.sys2.M)

        nout = len(quantities)
        fits = get_fits(quantities, beta, self.sys.I, fit_params)
        matvec = get_matvec(self.sys.mol, self.computer, diff=diff)
        matvec1 = get_matvec(self.sys1.mol, self.computer, diff=diff)
        matvec2 = get_matvec(self.sys2.mol, self.computer, diff=diff)

        coeffs = [X.coef for X in fits]

        def Xv(v): return eval_chebyshev_matvec_batch(
            v, matvec, coeffs, fits[0].domain, proj=self.P)

        def Xv1(v):
            xs = [numpy.zeros(n) for i in range(nout)]
            out = eval_chebyshev_matvec_batch(
                v, matvec1, coeffs, fits[0].domain, proj=self.P1)
            for x, o in zip(xs, out):
                x[:n1] = o
            return xs

        def Xv2(v):
            xs = [numpy.zeros(n) for i in range(nout)]
            out = eval_chebyshev_matvec_batch(
                v, matvec2, coeffs, fits[0].domain, proj=self.P2)
            for x, o in zip(xs, out):
                x[n1:] = o
            return x

        def Xvfull(v):
            Tv = Xv(v)
            T1 = Xv1(v[:n1])
            T2 = Xv2(v[n1:])
            return [t - t1 - t2 for t, t1, t2 in zip(Tv, T1, T2)]

        if method.lower() == "hutch++":
            out = trace.stochastic_trace_pp(nout, n, Xvfull, nsample, names=quantities)
        else:
            out = trace.stochastic_trace(
                nout, n, Xvfull, nsample, method, names=quantities)
        return out

    def _run_lanczos(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                     T=298.15, nsample=100, check=True, method="Rayleigh", diff="sym"):
        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)

        Ph = transrot.half_projector_against(self.dvecs)
        Ph1 = transrot.half_projector_against(self.dvecs1)
        Ph2 = transrot.half_projector_against(self.dvecs2)
        n = Ph.shape[0]
        n1 = Ph1.shape[0]
        n2 = Ph2.shape[0]
        matvec = get_matvec_half_proj(Ph, self.sys.mol, self.computer, diff=diff)
        if self.sys1.F2 is not None:
            Mi2 = 1.0/numpy.sqrt(numpy.asarray(self.sys1.M))
            F2m = numpy.einsum('ij,i,j->ij', self.sys1.F2, Mi2, Mi2)

            def matvec1(v):
                Pv = numpy.matmul(Ph1, v)
                x = numpy.matmul(F2m, Pv)
                return numpy.matmul(Ph1.transpose(), x)

        else:
            matvec1 = get_matvec_half_proj(
                Ph1, self.sys1.mol, self.computer, diff=diff)

        if self.sys2.F2 is not None:
            Mi2 = 1.0/numpy.sqrt(numpy.asarray(self.sys2.M))
            F2m = numpy.einsum('ij,i,j->ij', self.sys2.F2, Mi2, Mi2)

            def matvec2(v):
                Pv = numpy.matmul(Ph2, v)
                x = numpy.matmul(F2m, Pv)
                return numpy.matmul(Ph2.transpose(), x)

        else:
            matvec2 = get_matvec_half_proj(
                Ph2, self.sys2.mol, self.computer, diff=diff)

        rotor = fit_params["rotor"] if "rotor" in fit_params else None
        cutoff = fit_params["cutoff"] if "cutoff" in fit_params else None
        if rotor is not None:
            rotor /= constants.hartree_to_cm_1
        if cutoff is not None:
            cutoff /= constants.hartree_to_cm_1
        mmax = fit_params["order"] if "order" in fit_params else 24

        funcs = [get_function(
            s, beta, self.sys.I, rotor, cutoff) for s in quantities]
        return trace.stochastic_lanczos_bind(
            n, n1, n2, matvec, matvec1, matvec2, nsample,
            funcs, quantities, mmax, Ph=Ph, Ph1=Ph1, Ph2=Ph2)

    def run_exact(self, fit_params, quantities=["ZPE", "Hvib", "Avib", "Svib"],
                  T=298.15, check=True, diff="sym"):
        """Run the calculation with the exact trace."""
        if check:
            E, F1 = self.computer.gradient(self.sys.mol)
            gn = numpy.linalg.norm(F1)/numpy.sqrt(F1.shape[0])
            if gn > 1e-5:
                logging.warning("geometry is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E))
            logging.info("Gradient norm: {} Ha".format(gn))

            E1, F11 = self.computer.gradient(self.sys1.mol)
            gn = numpy.linalg.norm(F11)/numpy.sqrt(F11.shape[0])
            if gn > 1e-5:
                logging.warning("geometry of subsystem 1 is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E1))
            logging.info("Gradient norm: {} Ha".format(gn))

            E2, F12 = self.computer.gradient(self.sys2.mol)
            gn = numpy.linalg.norm(F12)/numpy.sqrt(F12.shape[0])
            if gn > 1e-5:
                logging.warning("geometry of subsystem 2 is not a minimum!")
            logging.info("Ground state energy: {} Ha".format(E2))
            logging.info("Gradient norm: {} Ha".format(gn))

        # Some constants
        kBT = T*constants.kb / constants.hartree_to_ev
        beta = 1.0 / (kBT + 1e-14)
        n = len(self.sys.M)
        n1 = len(self.sys1.M)
        n2 = len(self.sys2.M)

        nout = len(quantities)
        fits = get_fits(quantities, beta, self.sys.I, fit_params)
        matvec = get_matvec(self.sys.mol, self.computer, diff=diff)
        matvec1 = get_matvec(self.sys1.mol, self.computer, diff=diff)
        matvec2 = get_matvec(self.sys2.mol, self.computer, diff=diff)

        coeffs = [X.coef for X in fits]

        def Xv(v): return eval_chebyshev_matvec_batch(
            v, matvec, coeffs, fits[0].domain)

        def Xv1(v): return eval_chebyshev_matvec_batch(
            v, matvec1, coeffs, fits[0].domain)

        def Xv2(v): return eval_chebyshev_matvec_batch(
            v, matvec2, coeffs, fits[0].domain)

        out = trace.exact_trace_bind(
            nout, n, n1, n2, Xv, Xv1, Xv2,
            P=self.P, P1=self.P1, P2=self.P2, names=quantities)
        return out


def harmonic(sys, computer, o=24, nsample=100,
             method="Rayleigh", T=298.15, wmin=50,
             wmax=8000, check=True, rotor=None, cutoff=None):
    logging.warning("This interface is deprecated!")
    calc = Harmonic(sys, computer)
    fit_params = {}
    fit_params["order"] = o
    fit_params["wmin"] = wmin
    fit_params["wmax"] = wmax
    if rotor is not None:
        fit_params["rotor"] = rotor
    if cutoff is not None:
        fit_params["cutoff"] = cutoff
    if method == "exact":
        return calc.run_exact(fit_params, T=T, check=check)
    else:
        return calc.run(
            fit_params, T=T, nsample=nsample, method=method, check=check)


def harmonic_ref(sys, ref, computer, o=24, nsample=100,
                 method="Rayleigh", T=298.15, wmin=50,
                 wmax=8000, check=True, rotor=None, cutoff=None):

    logging.warning("This interface is deprecated!")
    calc = HarmonicDiff(sys, ref, computer)
    fit_params = {}
    fit_params["order"] = o
    fit_params["wmin"] = wmin
    fit_params["wmax"] = wmax
    if rotor is not None:
        fit_params["rotor"] = rotor
    if cutoff is not None:
        fit_params["cutoff"] = cutoff
    if method == "exact":
        return calc.run_exact(fit_params, T=T, check=check)
    else:
        return calc.run(fit_params, T=T, nsample=nsample, method=method, check=check)
