import numpy
from numpy.polynomial import chebyshev as cby


def chebyshev_fit(func, wmin, wmax, order, nx=1000, full=False, points="linear"):
    if points == "linear":
        wmax2 = wmax*wmax
        wmin2 = wmin*wmin if wmin > 0 else -wmin*wmin
        xs = numpy.linspace(wmin2, wmax2, num=(nx), endpoint=True)
    elif points == "quadratic":
        xls = numpy.linspace(wmin, wmax, num=(nx), endpoint=True)
        xs = [numpy.sign(x)*x*x for x in xls]
    else:
        raise Exception("Unrecognized 'points' argument: {}".format(points))
    ys = numpy.asarray([func(x) for x in xs])
    out, stats = cby.Chebyshev.fit(xs, ys, order, full=True)
    if full:
        return out, stats
    else:
        return out


def zpe(x):
    sx = numpy.sqrt(x) if x > 0 else numpy.sqrt(-x)
    return sx*0.5


def Hfunc(x, beta):
    sx = numpy.sqrt(x) if x > 0 else numpy.sqrt(-x)
    ex = numpy.exp(-beta*sx)
    return sx*(ex/(1.0 - ex))


def Afunc(x, beta):
    sx = numpy.sqrt(x) if x > 0 else 0.0
    ex = numpy.exp(-beta*sx)
    return 1.0/beta*numpy.log(1.0 - ex)


def Sfunc(x, beta):
    return beta*(Hfunc(x, beta) - Afunc(x, beta))


def switch(w, sthr):
    return 1.0/(1.0+(sthr/w)**4)


def Sfunc_rot(x, beta, I, sthr):
    sx = numpy.sqrt(x) if x > 0 else 0.0
    sx = numpy.sqrt(x)
    avmom = numpy.trace(I)/3.0
    mu = 0.5/(sx + 1e-15)
    mu = mu*avmom/(mu + avmom)
    svib = beta*(Hfunc(x, beta) - Afunc(x, beta))
    srot = 0.5 + numpy.log(numpy.sqrt(2.0*numpy.pi*mu/beta))
    wt = switch(sx, sthr)
    return wt*svib + (1.0 - wt)*srot


def Sfunc_cut(x, beta, cut, alpha):
    sx = numpy.sqrt(x) if x > 0 else 0.0
    sx = (sx - cut)*(numpy.arctan(alpha*(sx - cut))/numpy.pi + 0.5) + cut
    svib = beta*(Hfunc(sx*sx, beta) - Afunc(sx*sx, beta))
    return svib
