import numpy


def eval_mono(mat, coef):
    """Evaluate a matrix polynomial by Naive evaluation in monomial basis:

    mat (array): matrix (2d numpy ndarray)
    coef (iterable): coefficients of monomials in order of increasing degree
    """
    out = numpy.zeros(mat.shape)
    for i, c in enumerate(coef):
        out += c*numpy.linalg.matrix_power(mat, i)
    return out


def eval_horner(mat, coef):
    """Evaluate a matrix polynomial by Horner's method.

    mat (array): matrix (2d numpy ndarray)
    coef (iterable): coefficients of monomials in order of increasing degree
    """
    crev = coef[::-1]
    I = numpy.eye(mat.shape[0])
    b = crev[0]*I
    for c in crev[1:]:
        b = c*I + numpy.matmul(b, mat)
    return b


def eval_horner_matvec(vec, matvec, coef):
    crev = coef[::-1]
    b = crev[0]*vec
    for c in crev[1:]:
        b = c*vec + matvec(b)
    return b


def eval_chebyshev(mat, coeff, domain=[-1, 1]):
    # shift and scale
    n = mat.shape[0]
    shift = (domain[1] + domain[0])/2.0
    scale = 2.0/(domain[1] - domain[0])
    temp = scale*(mat - shift*numpy.eye(n))

    o = len(coeff)
    T0 = numpy.eye(n)
    T1 = temp
    Tn1 = T1
    Tn2 = T0
    out = coeff[0]*T0
    if o > 1:
        out += coeff[1]*T1

    for i in range(2, o):
        Tn = 2.0*numpy.matmul(temp, Tn1) - Tn2
        out += coeff[i]*Tn
        Tn2 = Tn1
        Tn1 = Tn
    return out


def eval_chebyshev_batch(mat, coeffs, domain=[-1, 1]):
    # shift and scale
    n = mat.shape[0]
    shift = (domain[1] + domain[0])/2.0
    scale = 2.0/(domain[1] - domain[0])
    temp = scale*(mat - shift*numpy.eye(n))
    nbatch = len(coeffs)

    o = len(coeffs[0])
    T0 = numpy.eye(n)
    T1 = temp
    Tn1 = T1
    Tn2 = T0
    out = [coeff[0]*T0 for coeff in coeffs]
    if o > 1:
        for j in range(nbatch):
            out[j] += coeffs[j][1]*T1

    for i in range(2, o):
        Tn = 2.0*numpy.matmul(temp, Tn1) - Tn2
        for j in range(nbatch):
            out[j] += coeffs[j][i]*Tn
        Tn2 = Tn1
        Tn1 = Tn
    return out


def eval_chebyshev_matvec(vec, matvec, coeff, domain=[-1, 1], proj=None):
    # shift and scale
    shift = (domain[1] + domain[0])/2.0
    scale = 2.0/(domain[1] - domain[0])
    if proj is not None:
        vec = numpy.matmul(proj, vec)

    o = len(coeff)
    T0 = vec
    Tn2 = T0
    out = coeff[0]*vec
    if o > 1:
        T1 = scale*(matvec(vec) - shift*vec)
        Tn1 = T1
        out += coeff[1]*T1

    for i in range(2, o):
        Tn = 2.0*scale*(matvec(Tn1) - shift*Tn1) - Tn2
        out += coeff[i]*Tn
        Tn2 = Tn1
        Tn1 = Tn
    return out


def eval_chebyshev_matvec_batch(vec, matvec, coeffs, domain=[-1, 1], proj=None):
    # shift and scale
    shift = (domain[1] + domain[0])/2.0
    scale = 2.0/(domain[1] - domain[0])
    nbatch = len(coeffs)
    if proj is not None:
        vec = numpy.matmul(proj, vec)

    o = len(coeffs[0])
    for c in coeffs:
        assert(len(c) == o)
    T0 = vec
    Tn2 = T0
    out = [coef[0]*vec for coef in coeffs]
    assert(len(out) == nbatch)
    if o > 1:
        T1 = scale*(matvec(vec) - shift*vec)
        Tn1 = T1
        for k in range(nbatch):
            out[k] += coeffs[k][1]*T1

    for i in range(2, o):
        Tn = 2.0*scale*(matvec(Tn1) - shift*Tn1) - Tn2
        for k in range(nbatch):
            out[k] += coeffs[k][i]*Tn
        Tn2 = Tn1
        Tn1 = Tn

    return out
